/*	$Id: object.c,v 1.14 2001/08/10 16:27:04 sandro Exp $	*/

/*
 * Copyright (c) 1997-2001 Sandro Sigala.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <assert.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "slisp.h"
#include "extern.h"

/*--------------------------------------------------------------------------
 * Object allocator functions.
 *--------------------------------------------------------------------------*/

objectp nil, t;

static objectp free_objs_list = NULL;
static objectp used_objs_list = NULL;

int free_objs = 0;
int used_objs = 0;

objectp new_object(int type)
{
	objectp p;

	if (free_objs_list == NULL) {
		p = (objectp)xmalloc(sizeof(struct object));
#ifdef DEBUG
		warnx(":: allocating cons %p", p);
#endif
	} else {
		p = free_objs_list;
		free_objs_list = free_objs_list->next;
		--free_objs;
#ifdef DEBUG
		warnx(":: recycling cons %p", p);
#endif
	}

	p->next = used_objs_list;
	used_objs_list = p;

	p->type = type;
	if (type == OBJ_CONS) {
		p->value.c.car = nil;
		p->value.c.cdr = nil;
	}
	p->gc = 0;

	++used_objs;

	return p;
}

objectp search_object_identifier(char *s)
{
	objectp p;

	for (p = used_objs_list; p != NULL; p = p->next)
		if (p->type == OBJ_IDENTIFIER && !strcmp(p->value.id, s))
			return p;

	return NULL;
}

objectp search_object_string(char *s)
{
	objectp p;

	for (p = used_objs_list; p != NULL; p = p->next)
		if (p->type == OBJ_STRING && !strcmp(p->value.s, s))
			return p;

	return NULL;
}

objectp search_object_integer(int in)
{
	objectp p;

	for (p = used_objs_list; p != NULL; p = p->next)
		if (p->type == OBJ_INTEGER && p->value.i == in)
			return p;

	return NULL;
}

void init_objects(void)
{
	nil = new_object(OBJ_NIL);
	t = new_object(OBJ_T);
}

/*--------------------------------------------------------------------------
 * Object set functions.
 *--------------------------------------------------------------------------*/

typedef struct object_pair *object_pairp;
struct object_pair {
	objectp		name;
	objectp		value;
	object_pairp	next;
};

static object_pairp setobjs_list = NULL;

void set_object(objectp name, objectp value)
{
	object_pairp p;

	if (name->value.id == NULL)
		return;

	for (p = setobjs_list; p != NULL; p = p->next)
		if (p->name->value.id != NULL &&
		    !strcmp(name->value.id, p->name->value.id)) {
			p->value = value;
			return;
		}

	p = (object_pairp)xmalloc(sizeof(struct object_pair));
	p->next = setobjs_list;
	setobjs_list = p;
	p->name = name;
	p->value = value;
}

objectp get_object(objectp name)
{
	object_pairp p;

	for (p = setobjs_list; p != NULL; p = p->next)
		if (p->name->value.id != NULL &&
		    !strcmp(name->value.id, p->name->value.id))
			return p->value;

	return nil;
}

void dump_objects(char *fname)
{
	object_pairp p;
	FILE *fout;

	if ((fout = fopen(fname, "w")) == NULL)
		err(1, "%s", fname);

	for (p = setobjs_list; p != NULL; p = p->next) {
		fprintf(fout, "(setq %s '", p->name->value.id);
 		princ_object(fout, p->value);
		fprintf(fout, ")\n");
	}

	fclose(fout);
}

#ifdef DEBUG
static char *obj_type_str(objectp p)
{
	switch (p->type) {
	case OBJ_NIL: return "nil";
	case OBJ_T: return "t";
	case OBJ_INTEGER: return "integer";
	case OBJ_IDENTIFIER: return "identifier";
	case OBJ_STRING: return "string";
	case OBJ_CONS: return "cons";
	default: assert(0);
	}
}

void print_obj_lists(void)
{
	objectp p;
	warnx(":: used objects");
	for (p = used_objs_list; p != NULL; p = p->next)
		warnx("::   %p (%s)", p, obj_type_str(p));
	warnx(":: free objects");
	for (p = free_objs_list; p != NULL; p = p->next)
		warnx("::   %p (%s)", p, obj_type_str(p));
}
#endif

/*--------------------------------------------------------------------------
 * Poor-man garbage collection functions.
 *--------------------------------------------------------------------------*/

/* The integer used for tagging the Lisp objects. */
static int gc_id = 0;

static void tag_tree(objectp p)
{
	if (p->gc == gc_id)
		return;

	p->gc = gc_id;

	if (p->type == OBJ_CONS) {
		tag_tree(p->value.c.car);
		tag_tree(p->value.c.cdr);
	}
}

static void tag_whole_tree(void)
{
	object_pairp p;

	for (p = setobjs_list; p != NULL; p = p->next) {
		tag_tree(p->name);
		tag_tree(p->value);
	}
}
       
static void do_garbage_collect(void)
{
	objectp p, new_used_objs_list = t, next;

	tag_whole_tree();

	/*
	 * Search in the object vector.
	 */
	for (p = used_objs_list; p != NULL && p != t; p = next) {
		next = p->next;
		if (p->gc != gc_id) {
			/* Remove unreferenced object. */
#ifdef DEBUG
			warnx(":: collecting cons %p", p);
#endif
			switch (p->type) {
			case OBJ_STRING:
				free(p->value.s);
				break;
			case OBJ_IDENTIFIER:
				free(p->value.id);
				break;
			}

			p->next = free_objs_list;
			free_objs_list = p;

			++free_objs;
			--used_objs;
		} else {
			/* The object is referenced somewhere. */
			p->next = new_used_objs_list;
			new_used_objs_list = p;
		}
	}

	used_objs_list = new_used_objs_list;
}

void garbage_collect(void)
{
	if (++gc_id == INT_MAX)
		gc_id = 1;
	do_garbage_collect();
}
