/*	$Id: funcs.c,v 1.18 2001/08/10 16:27:04 sandro Exp $	*/

/*
 * Copyright (c) 1997-2001 Sandro Sigala.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "slisp.h"
#include "extern.h"

static objectp F_eval(objectp);
static objectp F_progn(objectp);

static int count_list(objectp p)
{
	int i = 0;

	while (p != nil && p->type == OBJ_CONS)
		p = p->value.c.cdr, ++i;

	return i;
}

static inline objectp car(objectp p)
{
	if (p->type == OBJ_CONS)
		return p->value.c.car;

	if (opt_lint && p != nil && p != t)
		warnx("warning: getting the CAR of a non cons object");

	return nil;
}

/*
 * Syntax: (car expr)
 */
static objectp F_car(objectp args)
{
	return car(eval(car(args)));
}

static inline objectp cdr(objectp p)
{
	if (p->type == OBJ_CONS)
		return p->value.c.cdr;

	if (opt_lint && p != nil && p != t)
		warnx("warning: getting the CDR of a non cons object");

	return nil;
}

/*
 * Syntax: (cdr expr)
 */
static objectp F_cdr(objectp args)
{
	return cdr(eval(car(args)));
}

/*
 * Syntax: (+ expr...)
 */
static objectp F_add(objectp args)
{
	objectp p = args, p1;
	int v = 0;

	do {
		p1 = eval(car(p));
		if (p1->type == OBJ_INTEGER)
			v += p1->value.i;
		else if (opt_lint)
			warnx("warning: sum with a non integer operand");
		p = cdr(p);
	} while (p != nil);

	p1 = new_object(OBJ_INTEGER);
	p1->value.i = v;

	return p1;
}

/*
 * Syntax: (- expr...)
 */
static objectp F_sub(objectp args)
{
	objectp p = args, p1;
	int v = 0;

	do {
		p1 = eval(car(p));
		if (p1->type == OBJ_INTEGER) {
			if (p == args && cdr(p) != nil)
				v = p1->value.i;
			else
				v -= p1->value.i;
		} else if (opt_lint)
			warnx("warning: difference with a non integer operand");
		p = cdr(p);
	} while (p != nil);

	p1 = new_object(OBJ_INTEGER);
	p1->value.i = v;

	return p1;
}

/*
 * Syntax: (* expr...)
 */
static objectp F_mul(objectp args)
{
	objectp p = args, p1;
	int v = 1;

	do {
		p1 = eval(car(p));
		if (p1->type == OBJ_INTEGER)
			v *= p1->value.i;
		else if (opt_lint)
			warnx("warning: product with a non integer operand");
		p = cdr(p);
	} while (p != nil);

	p1 = new_object(OBJ_INTEGER);
	p1->value.i = v;

	return p1;
}

/*
 * Syntax: (/ expr...)
 */
static objectp F_div(objectp args)
{
	objectp p = args, p1;
	int v = 0;

	do {
		p1 = eval(car(p));
		if (p1->type == OBJ_INTEGER) {
			if (p == args && cdr(p) != nil)
				v = p1->value.i;
			else {
				if (p1->value.i == 0) {
					if (opt_lint)
						warnx("warning: division by zero");
					v = 0;
					break;
				} else
					v /= p1->value.i;
			}
		} else if (opt_lint)
			warnx("warning: quotient with a non integer operand");
		p = cdr(p);
	} while (p != nil);

	p1 = new_object(OBJ_INTEGER);
	p1->value.i = v;

	return p1;
}

/*
 * Syntax: (% expr1 expr2)
 */
static objectp F_mod(objectp args)
{
	objectp p1, p2, p3;

	p1 = eval(car(args));
	p2 = eval(car(cdr(args)));

	if (p1->type != OBJ_INTEGER || p2->type != OBJ_INTEGER) {
		if (opt_lint)
			warnx("warning: module with a non integer operand");
		return nil;
	}

	p3 = new_object(OBJ_INTEGER);
	if (p2->value.i == 0) {
		if (opt_lint)
			warnx("warning: module by zero");
		p3->value.i = 0;
	} else
		p3->value.i = p1->value.i % p2->value.i;

	return p3;
}

/*
 * Syntax: (< expr1 expr2)
 */
static objectp F_lt(objectp args)
{
	objectp p1, p2;

	p1 = eval(car(args));
	p2 = eval(car(cdr(args)));

	if (p1->type != OBJ_INTEGER || p2->type != OBJ_INTEGER) {
		if (opt_lint)
			warnx("warning: comparison with a non integer operand");
		return nil;
	}

	if (p1->value.i < p2->value.i)
		return t;

	return nil;
}

/*
 * Syntax: (> expr1 expr2)
 */
static objectp F_gt(objectp args)
{
	objectp p1, p2;

	p1 = eval(car(args));
	p2 = eval(car(cdr(args)));

	if (p1->type != OBJ_INTEGER || p2->type != OBJ_INTEGER) {
		if (opt_lint)
			warnx("warning: comparison with a non integer operand");
		return nil;
	}

	if (p1->value.i > p2->value.i)
		return t;

	return nil;
}

/*
 * Syntax: (<= expr1 expr2)
 */
static objectp F_le(objectp args)
{
	objectp p1, p2;

	p1 = eval(car(args));
	p2 = eval(car(cdr(args)));

	if (p1->type != OBJ_INTEGER || p2->type != OBJ_INTEGER) {
		if (opt_lint)
			warnx("warning: comparison with a non integer operand");
		return nil;
	}

	if (p1->value.i <= p2->value.i)
		return t;

	return nil;
}

/*
 * Syntax: (>= expr1 expr2)
 */
static objectp F_ge(objectp args)
{
	objectp p1, p2;

	p1 = eval(car(args));
	p2 = eval(car(cdr(args)));

	if (p1->type != OBJ_INTEGER || p2->type != OBJ_INTEGER) {
		if (opt_lint)
			warnx("warning: comparison with a non integer operand");
		return nil;
	}

	if (p1->value.i >= p2->value.i)
		return t;

	return nil;
}

/*
 * Syntax: (= expr1 expr2)
 */
static objectp F_numeq(objectp args)
{
	objectp p1, p2;

	p1 = eval(car(args));
	p2 = eval(car(cdr(args)));

	if (p1->type != OBJ_INTEGER || p2->type != OBJ_INTEGER) {
		if (opt_lint)
			warnx("warning: comparison with a non integer operand");
		return nil;
	}

	if (p1->value.i == p2->value.i)
		return t;

	return nil;
}

static void princ_string(FILE *fout, char *s)
{
	char *p;

	fputc('"', fout);
	for (p = s; *p != '\0'; ++p)
		switch (*p) {
		case '\a': fputc('\\', fout); fputc('a', fout); break;
		case '\b': fputc('\\', fout); fputc('b', fout); break;
		case '\f': fputc('\\', fout); fputc('f', fout); break;
		case '\n': fputc('\\', fout); fputc('n', fout); break;
		case '\r': fputc('\\', fout); fputc('r', fout); break;
		case '\t': fputc('\\', fout); fputc('t', fout); break;
		case '\v': fputc('\\', fout); fputc('v', fout); break;
		default: fputc(*p, fout);
		}
	fputc('"', fout);
}

void princ_object(FILE *fout, objectp p)
{
	objectp p1;

	switch (p->type) {
	case OBJ_NIL:
		fprintf(fout, "nil");
		break;
	case OBJ_T:
		fputc('t', fout);
		break;
	case OBJ_IDENTIFIER:
		fprintf(fout, "%s", p->value.id);
		break;
	case OBJ_STRING:
		princ_string(fout, p->value.s);
		break;
	case OBJ_INTEGER:
		fprintf(fout, "%d", p->value.i);
		break;
	case OBJ_CONS:
		fputc('(', fout);
		p1 = p;
		do {
			princ_object(fout, p1->value.c.car);
			p1 = p1->value.c.cdr;
			if (p1 != nil) {
				fputc(' ', fout);
				if (p1->type != OBJ_CONS) {
					fprintf(fout, ". ");
					princ_object(fout, p1);
				}
			}
		} while (p1 != nil && p1->type == OBJ_CONS);
		fputc(')', fout);
	}
}

/*
 * Syntax: (princ expr...)
 */
static objectp F_princ(objectp args)
{
	objectp p = args, p1;

	do {
		p1 = eval(car(p));
		if (p1->type == OBJ_STRING)
			printf("%s", p1->value.s);
		else
			princ_object(stdout, p1);
		p = cdr(p);
	} while (p != nil);

	return p1;
}

/*
 * Syntax: (atom expr)
 */
static objectp F_atom(objectp args)
{
	objectp p;

	p = eval(car(args));

	switch (p->type) {
	case OBJ_T:
	case OBJ_NIL:
	case OBJ_INTEGER:
	case OBJ_STRING:
	case OBJ_IDENTIFIER:
		return t;
	}

	return nil;
}

/*
 * Syntax: (cons expr1 expr2)
 */
static objectp F_cons(objectp args)
{
	objectp p;

	p = new_object(OBJ_CONS);
	p->value.c.car = eval(car(args));
	p->value.c.cdr = eval(car(cdr(args)));

	return p;
}

/*
 * Syntax: (list expr1...)
 */
static objectp F_list(objectp args)
{
	objectp p = args, first = NULL, prev = NULL, p1;

	if (p == nil)
		return nil;

	do {
		p1 = new_object(OBJ_CONS);
		p1->value.c.car = eval(car(p));
		if (first == NULL)
			first = p1;
		if (prev != NULL)
			prev->value.c.cdr = p1;
		prev = p1;
		p = cdr(p);
	} while (p != nil);

	return first;
}

/*
 * Syntax: (eq expr1 expr2)
 */
static objectp F_eq(objectp args)
{
	objectp p1, p2;

	p1 = eval(car(args));
	p2 = eval(car(cdr(args)));

	if (p1 == p2)
		return t;

	if (p1->type == OBJ_CONS || p2->type == OBJ_CONS)
		return nil;

	if (p1->type == p2->type)
		switch (p1->type) {
		case OBJ_IDENTIFIER:
			if (!strcmp(p1->value.id, p2->value.id))
				return t;
			return nil;
		case OBJ_STRING:
			if (!strcmp(p1->value.s, p2->value.s))
				return t;
			return nil;
		case OBJ_INTEGER:
			if (p1->value.i == p2->value.i)
				return t;
			return nil;
		}

	return nil;
}

/*
 * Syntax: (quote expr)
 */
static objectp F_quote(objectp args)
{
	return car(args);
}

/*
 * Syntax: (and expr...)
 */
static objectp F_and(objectp args)
{
	objectp p = args, p1;

	do {
		p1 = eval(car(p));
		if (p1 == nil)
			return nil;
		p = cdr(p);
	} while (p != nil);

	return p1;
}

/*
 * Syntax: (or expr...)
 */
static objectp F_or(objectp args)
{
	objectp p = args, p1;

	do {
		p1 = eval(car(p));
		if (p1 != nil)
			return p1;
		p = cdr(p);
	} while (p != nil);

	return nil;
}

/*
 * Syntax: (not expr)
 * Syntax: (null expr)
 */
static objectp F_not(objectp args)
{
	objectp p = eval(car(args));

	if (p != nil)
		return nil;

	return t;
}

/*
 * Syntax: (cond (expr1 [expr2])...)
 */
static objectp F_cond(objectp args)
{
	objectp p = args, p1, p2, p3;

	do {
		p1 = car(p);
		if ((p2 = eval(car(p1))) != nil) {
			if ((p3 = cdr(p1)) != nil)
				return F_progn(p3);
			return p2;
		}
		p = cdr(p);
	} while (p != nil);

	return nil;
}

/*
 * Syntax: (if expr then-expr else-expr...)
 */
static objectp F_if(objectp args)
{
	objectp p1, p2, p3;

	p1 = car(args);
	p2 = car(cdr(args));
	p3 = cdr(cdr(args));

	if (eval(p1) != nil)
		return eval(p2);

	return F_progn(p3);
}

/*
 * Syntax: (when expr then-expr...)
 */
static objectp F_when(objectp args)
{
	objectp p1, p2;

	p1 = car(args);
	p2 = cdr(args);
	if (eval(p1) != nil)
		return F_progn(p2);

	return nil;
}

/*
 * Syntax: (unless expr else-expr...)
 */
static objectp F_unless(objectp args)
{
	objectp p1, p2;

	p1 = car(args);
	p2 = cdr(args);
	if (eval(p1) == nil)
		return F_progn(p2);

	return nil;
}

/*
 * Syntax: (while expr exprs...)
 */
static objectp F_while(objectp args)
{
	objectp p1, p2;

	p1 = car(args);
	p2 = cdr(args);

	while (eval(p1) != nil)
		F_progn(p2);

	return nil;
}

/*
 * Syntax: (progn expr...)
 */
static objectp F_progn(objectp args)
{
	objectp p = args, p1;

	do {
		p1 = eval(car(p));
		p = cdr(p);
	} while (p != nil);

	return p1;
}

/*
 * Syntax: (prog1 expr...)
 */
static objectp F_prog1(objectp args)
{
	objectp p = args, first = NULL, p1;

	do {
		p1 = eval(car(p));
		if (first == NULL)
			first = p1;
		p = cdr(p);
	} while (p != nil);

	if (first == NULL)
		first = nil;

	return first;
}

/*
 * Syntax: (prog2 expr...)
 */
static objectp F_prog2(objectp args)
{
	objectp p = args, second = NULL, p1;
	int i = 0;

	do {
		++i;
		p1 = eval(car(p));
		if (i == 2)
			second = p1;
		p = cdr(p);
	} while (p != nil);

	if (second == NULL)
		second = nil;

	return second;
}

/*
 * Syntax: (set name value)
 */
static objectp F_set(objectp args)
{
	objectp p1 = eval(car(args)), p2 = eval(car(cdr(args)));

	if (p1 == nil) {
		if (opt_lint)
			warnx("warning: setting the value of a nil object");
	} else
		set_object(p1, p2);

	return p2;
}

/*
 * Syntax: (setq name value...)
 * Syntax: (setf name value...)
 * `name' is not evalled
 */
static objectp F_setq(objectp args)
{
	objectp p = args, p1, p2;

	do {
		p1 = car(p);
		p2 = eval(car(cdr(p)));
		set_object(p1, p2);
		p = cdr(cdr(p));
	} while (p != nil);

	return p2;
}

/*
 * Syntax: (defun name arglist expr...)
 * `name' is not evalled
 * `arglist' is not evalled
 */
static objectp F_defun(objectp args)
{
	objectp p1 = car(args), p2 = car(cdr(args)), p3 = cdr(cdr(args));
	objectp lexpr;

	lexpr = new_object(OBJ_CONS);
	lexpr->value.c.car = new_object(OBJ_IDENTIFIER);
	lexpr->value.c.car->value.id = xstrdup("lambda");
	lexpr->value.c.cdr = new_object(OBJ_CONS);
	lexpr->value.c.cdr->value.c.car = p2;
	lexpr->value.c.cdr->value.c.cdr = p3;

	set_object(p1, lexpr);

	return lexpr;
}

static objectp eval_func(objectp p, objectp args)
{
	objectp p1, p2, p3, p4, p5;
	objectp eval_objs[64], save_objs[64];
	int i;

	p1 = car(p);
	if (p1->type == OBJ_IDENTIFIER && !strcmp(p1->value.id, "lambda")) {
		p2 = car(cdr(p));
		p3 = args;

		if (count_list(p2) != count_list(p3)) {
			warnx("wrong number of parameters");
			return nil;
		}

		/*
		 * Save the new variable values.
		 */
		i = 0;
		do {
			p5 = eval(car(p3));
			eval_objs[i++] = p5;
			p3 = cdr(p3);
		} while (p3 != nil);

		/*
		 * Save the old variable values and set the new ones.
		 */
		i = 0;
		do {
			p4 = car(p2);
			save_objs[i] = get_object(p4);
			set_object(p4, eval_objs[i]);
			p2 = cdr(p2);
			++i;
		} while (p2 != nil);

		p5 = F_progn(cdr(cdr(p)));

		/*
		 * Restore the old variable values.
		 */
		p2 = car(cdr(p));
		i = 0;
		do {
			p4 = car(p2);
			set_object(p4, save_objs[i++]);
			p2 = cdr(p2);
		} while (p2 != nil);

		return p5;
	}

	return nil;
}

objectp F_gc(objectp args)
{
	garbage_collect();

	return t;
}

objectp F_dump_memory(objectp args)
{
	objectp p = car(args);

	if (p != nil && cdr(args) == nil && p->type == OBJ_STRING) {
		if (strlen(p->value.s) > 0) {
			dump_objects(p->value.s);
			return t;
		} else
			warnx("expected filename");
	} else
		warnx("wrong number of parameters (expected string)");

	return nil;
}

struct intrinsic {
	char *name;
	objectp (*func)(objectp args);
};

static struct intrinsic intrinsics[] = {
	{ "%", F_mod },
	{ "&dump-memory", F_dump_memory },
	{ "*", F_mul },
	{ "+", F_add },
	{ "-", F_sub },
	{ "/", F_div },
	{ "<", F_lt },
	{ "<=", F_le },
	{ "=", F_numeq },
	{ ">", F_gt },
	{ ">=", F_ge },
	{ "and", F_and },
	{ "atom", F_atom },
	{ "car", F_car },
	{ "cdr", F_cdr },
	{ "cond", F_cond },
	{ "cons", F_cons },
	{ "defun", F_defun },
	{ "eq", F_eq },
	{ "eval", F_eval },
	{ "garbage-collect", F_gc },
	{ "gc", F_gc },
	{ "if", F_if },
	{ "list", F_list },
	{ "not", F_not },
	{ "null", F_not },
	{ "or", F_or },
	{ "princ", F_princ },
	{ "prog1", F_prog1 },
	{ "prog2", F_prog2 },
	{ "progn", F_progn },
	{ "quote", F_quote },
	{ "set", F_set },
	{ "setf", F_setq },
	{ "setq", F_setq },
	{ "unless", F_unless },
	{ "when", F_when },
	{ "while", F_while },
};

static int compar(const void *p1, const void *p2)
{
	return strcmp(((struct intrinsic *)p1)->name,
		      ((struct intrinsic *)p2)->name);
}

static objectp eval_cons(objectp p)
{
	objectp p1 = car(p), p2 = cdr(p), p3;

	if (p1 != nil && p1->type == OBJ_IDENTIFIER) {
		struct intrinsic key, *item;

		if (!strcmp(p1->value.id, "lambda"))
			return p;
		key.name = p1->value.id;
		if ((item = bsearch(&key, intrinsics,
				    sizeof intrinsics / sizeof intrinsics[0],
				    sizeof intrinsics[0], compar)) != NULL)
			return item->func(p2);

		if ((p3 = get_object(p1)) != nil)
			return eval_func(p3, p2);
		else
			warnx("function `%s' is undefined", p1->value.id);
	}

	return nil;
}

objectp eval(objectp p)
{
	switch (p->type) {
	case OBJ_IDENTIFIER:
		return get_object(p);
	case OBJ_INTEGER:
	case OBJ_STRING:
		return p;
	case OBJ_CONS:
		return eval_cons(p);
	}

	return p;
}

static objectp F_eval(objectp args)
{
	return eval(eval(car(args)));
}
