;; *************************************************************************************** ;;
;; lisp debug v0.8  : source level debugger for lisp                                       ;;
;; Copyright (C) 1998 Marc Mertens                                                         ;;
;;                                                                                         ;;
;;     This program is free software; you can redistribute it and/or modify                ;;
;;    it under the terms of the GNU General Public License as published by                 ;;
;;    the Free Software Foundation; either version 2 of the License, or                    ;;
;;    (at your option) any later version.                                                  ;;
;;                                                                                         ;;
;;    This program is distributed in the hope that it will be useful,                      ;;
;;    but WITHOUT ANY WARRANTY; without even the implied warranty of                       ;;
;;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                        ;;
;;    GNU General Public License for more details.                                         ;;
;;                                                                                         ;;
;;    You should have received a copy of the GNU General Public License                    ;;
;;    along with this program; if not, write to the Free Software                          ;;
;;    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA            ;;
;;                                                                                         ;;
;; Contact me on mmertens@akam.be                                                          ;;
;; ********************************************************************************************
;;
;; The following functions must be defined to get a link to the interface of the debugger
;;
;; (process-incoming)  ;; Should read the commandos from the interface and process them
;; (send-command)      ;; Should send the commandos from the lisp system to the server
;; (start-interface)   ;; should start the interface
;; (stop-interface)    ;; Should stop the interface
;; (get-unix-env "var" "default") ;; Get unix sytemvariable
;;
;; ********************************************************************************************

;; ********************************************************************************************
;; PAckage stuff
;; ********************************************************************************************

(in-package "DEBUGGER")

;; *******************************************************************************************
;; Global variables used in this link
;; *******************************************************************************************

(defparameter **socket-stream** 0)
(defparameter **save-socket-stream** 0)
(defparameter **save-standard-input** 0)
(defparameter **generic-stream** 0)





;; *****************************************************************************************
;; Using gray streams to modify the working of input
;; *****************************************************************************************

;; Define a special class for the modified input stream

(defclass debugger-input-stream (fundamental-input-stream) ())

;; Define the different methods needed for this class

(defmethod stream-read-char ((strm debugger-input-stream))
  (let ((chr nil))
    (loop
      (when (setf chr (read-char-no-hang))
	(return chr))
      (process-incoming))))

(defmethod stream-unread-char ((strm debugger-input-stream) chr)
  (unread-char chr))

(defmethod stream-read-char-no-hang ((strm debugger-input-stream))
  (process-incoming)
  (if (listen)
      (read-char)
    nil))

(defmethod stream-listen ((strm debugger-input-stream))
  (process-incoming)
  (listen))

(defmethod stream-clear-input ((strm debugger-input-stream))
  (clear-input))



;; *****************************************************************************************
;; Create a gray stream to be used instead of the *standard-input* , this allows
;; us to listen to the socket to our interface using spare moments in the read-eval-print
;; toplevel loop
;; *****************************************************************************************

(setf **generic-stream** (make-instance 'debugger-input-stream))


(defun debug-toploop ()
  (terpri)
  (catch 'exit-debug
    (unwind-protect
	(progn
	  ;; Can't close socket in stop-interface so it has to be done here
	  (when (streamp **save-socket-stream**) (close **save-socket-stream**))
	  (sleep 1)
	  ;; Start the interface program which creates a socket on 9007
	  (shell "interface -9007&")
	  (sleep 1)
	  ;; Connect to socket 9007
	  (setf **save-socket-stream** (setf **socket-stream** (socket-connect 9007)))
	  ;; Loop in debug mode untill stop-interface is called
	  (loop
	    (princ "debugger>> ")
	    ;; Avoid exitting the toploop because of error
	    (handler-case
	     (when (eq (princ (eval (read **generic-stream**))) :stop)
	       (stop-interface))
	     (error (condition)
		    (format t "ERROR!!! ~A" condition)
		    (terpri)
		    nil))
	    (terpri)))
      (setf **socket-stream** 0))))


;; *****************************************************************************************
;; Process the incoming data
;; *****************************************************************************************

(defun process-incoming ()
  (if (streamp **socket-stream**)
      (when (listen **socket-stream**)
	(let ((str "")
	      (chr))
	  (setf str (with-output-to-string (h)
					   (loop
					     (setf chr (read-char **socket-stream**))
					      (when (char= chr #\newline)
						(return))
					      (princ chr h))))
	   (eval (read-from-string str NIL NIL))))
    (stop-interface)))

;; ******************************************************************************************
;; Main interface to the lisp system
;; ******************************************************************************************

(defun send-command (command &rest arg-lst)
  (when (streamp **socket-stream**)
    (princ command **socket-stream**)
    (mapc #'(lambda (arg)
	      (princ " " **socket-stream**)
	      (cond ((stringp arg)
		     (princ (length arg) **socket-stream**)
		     (princ " " **socket-stream**)
		     (princ arg **socket-stream**))
		    (T
		     (princ arg **socket-stream**))))
	  arg-lst)
    (terpri **socket-stream**)
    ))
		     
;; ******************************************************************************************
;; Start the interface
;; ******************************************************************************************

(defun start-interface ()
  ;; For CLISP we just start the debug toploop
  (debug-toploop)
  )

;; *******************************************************************************************
;; Stop the interface
;; *******************************************************************************************

(defun stop-interface ()
  (setf **socket-stream** 0)
  (end-debug-eventloop)
  (throw 'exit-debug nil))


;; ******************************************************************************************
;; Some missing functions in CLISP
;; ******************************************************************************************

(setf (symbol-function 'special-operator-p) #'special-form-p)


;; ******************************************************************************************
;; Get unix system environment variable
;; ******************************************************************************************

(defun get-unix-env (var default)
  (cond ((system::getenv var))
	(t default)))


