/*
 * lisp debug version 0.8 a source level debugger for lisp
 * Copyright (C) 1998 Marc Mertens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can reach me at : mmertens@akam.be
 */

/* 
 * There are 3 ways to start the interface 
 *
 * 1. 'interface' , the interface doesn't need a PID nr form the parent
 * 2. 'interface pid' , pid > 0 , pid is the PID of the parent , interface must send a signal to pid to
 *                      indicate processing
 * 3. 'interface -port' , port nummer to use , communication happens with port nr
 * 
 * 
 */

/* 
 * tkAppInit.c --
 *
 *	Provides a default version of the Tcl_AppInit procedure for
 *	use in wish and similar Tk-based applications.
 *
 * Copyright (c) 1993 The Regents of the University of California.
 * Copyright (c) 1994 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkAppInit.c 1.22 96/05/29 09:47:08
 */

#include "tk.h"
#include "hash.h"
#include "errno.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/un.h>
#include <unistd.h>
#include <search.h>  /* Routines to work with hash tables */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <math.h>

/*
 * The following variable is a special hack that is needed in order for
 * Sun shared libraries to be used for Tcl.
 */

extern int matherr();
int *tclDummyMathPtr = (int *) matherr;



#include <tcl.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <limits.h>
#include <varargs.h>


/* 
 * Structures used by the debugger
 */


/*
 * Structure to hold info on debugpoints , used in binary tree
 */

typedef struct debugpoint
{
  char type; /* Type of breakpoint 8 bits , B1=breakpoint , B2=tempory breakpoint , B3=Conditional Breakpoint */
  char *exp; /* Conditional exp in conditional breakpoint */
  long executed; /* Contains the number of time the breakpoint is passed */
  char *source; /* Source the breakpoint is residing in */
  long begin;   /* Begin of the debugpoint */
  long end; /* End of the debugpoint */
} DEBUGPOINT;

HASH *H_DEBUGPOINT; /* Root of the debugpoint tree to hold informations about debugpoints */

/*
 * Constants defined
 */ 

#define MAX_POS 50                                        /* Max length of position string in TCL/TK texts */
#define MAX_LINE_LENGTH 100000                   /* Max length of line of a source */
#define MAX_LINES 100000                               /* Max nr lines in a source */
#define MAX_STRING 1025                                /* Max string lengte if needed */
#define SIGUSR1 10                                           /* Signal to send to GCL */


/*
 * Linked list used 
 */

LIST *L_SOURCES=NULL;

/* 
 * Global variables used by the main program
 */

static Tcl_Interp *interp;   /* Interpreter for this aplication */
static int PPID;             /* PPID */
static int server_sockfd,client_sockfd; /* Socket handlers client , server */
Tk_Window mainWindow;        /* Handle of the main Window */

/* 
 * Global variables used to hold the information need to converse between line.char and absolute position
 */

long LengthText;
long LineAbsolute[MAX_LINES];

/*
 * Global variables used for status and configuration information of the debugger
 */

int COMPILE_CODE; /* If true , debugged code must be compiled before loaded */
int SAVE_ON_EXIT; /* If true , settings of the debugger must be saved on exit */
int DISPLAY_PREVIOUS; /* If true the result of a previous call is visable in the debug window */
int CHECK_ERROR=1; /* If true , debugged code will enter debugger during condinuable lisp errors , if false the lisp system handles the error */
int DEBUG_MACRO=0; /* If true debugging code is added to macros , if not it is disabled */

char CURRENT_SOURCE[FILENAME_MAX+1]; /* Current source , loaded in the debugger window */
char SOURCE[FILENAME_MAX+1];         /* Temp location of current source name */
char DEBUGPOINT_COLOR[MAX_STRING+1]="#ff0000";
char DEBUGPOINTIF_COLOR[MAX_STRING+1]="#808000";
char CURRENT_COLOR[MAX_STRING+1]="#00ffff";
char AFTER_COLOR[MAX_STRING+1]="#008080";
char PROFILE_COLOR[MAX_STRING+1]="#ffff00";
char FONT[MAX_STRING+1]="fixed";
char STEP_OVER_SOURCE[FILENAME_MAX+1]; /* Source used when executed step-over */

int STEP_MODE; /* If true we are stepping thru the source */
int STEP_OVER_MODE; /* IF true , STEP_MODE = 0 and we are stepping over */
long STEP_OVER_BEGIN=0; /* Begin position of the list we are stepping over in */
long STEP_OVER_END=0;  /* End position of the list we are stepping over in */
int IN_BREAKPOINT=0;   /* Indicate if we are in a breakpoint , this means that the lisp system is in a loop 
			  able to receive commands from the interface */
long CURRENT_BEGIN; /* Copy of the current begin point in a breakpoint */
long CURRENT_END;   /* Copy of the current end point in a breakpoint */
char CURRENT_IN_SOURCE[FILENAME_MAX+1]; /* Indicates the source of the last breakpoint */
int PROFILING=0;    /* Indicate profiling , 0 = not profiling , 1 = profiling */
int MAX_PROFILING_COUNT=0;  /* Max PROFILE_COUNT REACHED */
long PROFILE_BEGIN=0; /* Used in profiling to show begin of region reaching profile count */
long PROFILE_END=0; /* Used in profiling to show end of region reached profile count */
long EXECUTED=0; /* Used in profiling to define threshold value to highlight code */

char FIND_POS[MAX_POS+1];   /* Current position in find operation */
int SEARCH_TYPE; /* Search type in find operation */

/*
 * Function prototypes (damn these C compilers)
 */

int Tcl_Invoke();

/*
 * Correct the " character in a string by a \" and the \ character by a \\
 * This to have a correct communication of arguments to the lisp system
 * Usage :  char *new;
 *          correct_string(old,new);
 *          free(new);
 */


char *correct_string(char *in)
{
  int length,nr_corrections,i,j;
  char *out;

  /* First define nr of corrections , gives idea on length new string */
  length=0;
  nr_corrections=0;
  
  for (i=0;in[i]!='\0';i++)
    {
      length=length+1;
      if (in[i]=='\\')
	nr_corrections=nr_corrections+1;
      if (in[i]=='"')
	nr_corrections=nr_corrections+1;
    };
  
  /* Now allocate the new string and correct it */
  
  out=malloc(sizeof(char)*(length+nr_corrections+1));
  j=0;
  for (i=0;in[i]!='\0';i++)
    {
      if (in[i]=='\\')
	{
	  out[j]='\\';
	  j=j+1;
	  out[j]='\\';
	}
      else if (in[i]=='"')
	{
	  out[j]='\\';
	  j=j+1;
	  out[j]='"';
	}
      else
	{
	  out[j]=in[i];
	};
      j=j+1;
    };
  out[j]='\0';
  return out;
}




/*
 * Check input from client  
 */

int input_waiting()
{
  struct timeval timeout;
  int result;
  fd_set inputs;
  ClientData data;

  timeout.tv_sec=0;
  timeout.tv_usec=0;
  FD_ZERO(&inputs);
  FD_SET(client_sockfd,&inputs);
  result=select(FD_SETSIZE,&inputs,(fd_set *)0,(fd_set *)0,&timeout);
  return result;
};

/*
 * Read decimal number from the client input stream , we asume that after each argument there is a blanc or a return
 * this blanc or return is always consumed
 */

long read_number()
{
  long result;
  char character;

  /* First skip all leading blancs , normally this may not happen , just to be sure */
  read(client_sockfd,&character,1);
  while (character==' ')
    character==read(client_sockfd,&character,1);
  /* Start reading number and calculate the number */
  if (character < 47 || character > 58)
    {
      return -1; /* There is no number waiting */
    };
  result=0;
  while (character > 47 && character < 58)
    {
      result=result*10+character-48;
      read(client_sockfd,&character,1);
    };
  /* Returns the result */
  return result;
}

/* 
 * Read a string from the lisp system , first read length and then read rest
 */

char *Read_String_Arg()
{
  char *string;
  long len,i;
  char dummy;

  len=read_number();

  string=malloc(sizeof(char)*(len+1));
  for (i=0;i<len;i++)
    {
      read(client_sockfd,&dummy,1);
      string[i]=dummy;
    };
  string[len]='\0';
  /* Read remaining space or return */
  read(client_sockfd,&dummy,1);

  return string;
}

/*
 * Comparisation operator of debugpoints
 */

int c_debugpoint(const void *first,const void *second)
{
  DEBUGPOINT *one,*two;
  one=(DEBUGPOINT *) first;
  two=(DEBUGPOINT *) second;

  if (strcmp(one->source,two->source)==0)
    {
      /* Same source */
      if ((one->begin==two->begin) && (one->end==two->end))
	{
	  return 0;
	}
      else if ((one->begin==two->begin) && (one->end>two->end))
	{
	  return 1;
	}
      else if ((one->begin==two->begin) && (one->end<two->end))
	{
	  return -1;
	}
      else if (one->begin>two->begin)
	{
	  return 1;
	}
      else if (one->begin<two->begin)
	{
	  return -1;
	};
    }
  else
    {
      return strcmp(one->source,two->source);
    };
}




  
/*
 * Conversion of absolute position to x.y format
 */

void AbsoluteToIndex(long position,char *result)
{
  long i;
  long l,c;
  
  if (position<1)
    position=1;

  for (i=0;LineAbsolute[i]<position && i<LengthText;i++);
  if (i==LengthText)
    {
      result[0]='e';
      result[1]='n';
      result[2]='d';
      result[3]='\0'; /* We have reached the end of the text */
      return;
    };
  
  l=i;  /* Line in text , starts from 1 */;
  c=position-LineAbsolute[i-1]-1; /* Character on line starts from 0 */
  snprintf(result,MAX_POS,"%ld.%ld",l,c);
}

long IndexToAbsolute(char *index)
{
  long l,c;
  
  if (sscanf(index,"%ld.%ld",&l,&c)!=2)
    /* Incorrect input */
    return -1;
 
  if (l>=LengthText)
    return LineAbsolute[LengthText-1]+c;
  return LineAbsolute[l-1]+c+1;
}



/* 
 * SHow the set breakpoints in the source 
 */

void Show_Debugpoint_Walk(char *key,void *data)
{
  DEBUGPOINT *p_debugpoint;
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];

  p_debugpoint=(DEBUGPOINT *) data;
  /* Check if this is a debugpoint for the current source */
  if (strcmp(CURRENT_SOURCE,p_debugpoint->source)==0)
    {
      /* Check type of breakpoint */
      if ((p_debugpoint->type & 32) == 32)  /* Conditional breakpoint */
	{
	  AbsoluteToIndex(p_debugpoint->begin,s_begin);
	  AbsoluteToIndex(p_debugpoint->end,s_end);
	  Tcl_Invoke(interp,".text.text","tag","add","debugif",s_begin,s_end,NULL);
	};
      if ((p_debugpoint->type & 128) == 128) /* Normal breakpoint */
	{
	  AbsoluteToIndex(p_debugpoint->begin,s_begin);
	  AbsoluteToIndex(p_debugpoint->end,s_end);
	  Tcl_Invoke(interp,".text.text","tag","add","debug",s_begin,s_end,NULL);
	};
    };
}
      
  

/*
 * Read the source in the source windows and prepare the tables to convert between
 * line.char and absolute position
 */

void Read_Source (Tcl_Interp *interp,char FileName[])
{
  int i;
  FILE *h;
  unsigned char line[MAX_LINE_LENGTH+1];
  long length;

  length=0;
  LengthText=1;
  LineAbsolute[0]=0;
  if ((h=fopen(FileName,"r"))==NULL)
    {
      /* Problem open file for read , so skip reading the file" */
      Tcl_Eval(interp,"tk_dialog .m Message \"Could not open the file \" {} -1 Ok");
      return;
    };
  Tcl_Eval(interp,".text.text configure -state normal");
  Tcl_Eval(interp,".text.text delete 1.0 end");

  while ((fgets(line,MAX_LINE_LENGTH,h)!=NULL) && (LengthText < MAX_LINES))
    {
      LineAbsolute[LengthText]=strlen(line)+length; 
      length=LineAbsolute[LengthText];
      LengthText=LengthText+1;
      /*  Tcl_Eval(interp,line);  */
      Tcl_Invoke(interp,".text.text","insert","end",line,NULL); 
    };

  fclose(h);
  Tcl_Eval(interp,".text.text configure -state disabled");
  /* Show back the setted breakpoints for the source */
  walk_hash(H_DEBUGPOINT,Show_Debugpoint_Walk);
  /* Show color breakpoints */
  Tcl_Invoke(interp,".text.text","tag","configure","debug","-foreground",DEBUGPOINT_COLOR,NULL);
  Tcl_Invoke(interp,".text.text","tag","configure","debugif","-foreground",DEBUGPOINTIF_COLOR,NULL);
}

/* 
 * Code to modify the list of sources in the menu , should be called 
 * when L_SOURCES changes
 */

void Change_Source_Menu_Walk(char *key,char *source)
{
  char command[2*FILENAME_MAX+101];
  snprintf(command,2*FILENAME_MAX+100,".menu.source.source add command -label %s -command \"load-source %s\"",source,source);
  Tcl_Eval(interp,command);
}

void Change_Source_Menu()
{
  Tcl_Eval(interp,".menu.source.source delete 0 end");
  walk_list(L_SOURCES,Change_Source_Menu_Walk);
}

/*
 * Code to be executed when we leave a breakpoint 
 */
  
void Leave_Breakpoint()
{
  /* Stop highlighting and leave breakpoint */
  if (IN_BREAKPOINT!=0)
    {
      IN_BREAKPOINT=0;
      Tcl_Invoke(interp,".text.text","tag","delete","breakpoint",NULL);
      write(client_sockfd,"(DEBUGGER::end-debug-eventloop)\n",32);
      /* If it is a breakpoint after a call we must disable the return button */
      Tcl_Invoke(interp,".button2.return","configure","-state","disabled",NULL);
    };
}

/*
 * Code to test if a conditional breakpoint has been set , only exectuted in breakpoint
 */

void check_if_breakpoint(char *exp)
{
  char *new_exp;
  
  write(client_sockfd,"(DEBUGGER::if-breakpoint \"",26);
  write(client_sockfd,exp,strlen(exp));
  write(client_sockfd,"\")\n",3);
}

/*
 * Check if we have to reload the source in the source window
 */

void Change_Source()
{
  if (strcmp(CURRENT_SOURCE,CURRENT_IN_SOURCE)!=0)
    {
      /* We have to reload anouther source */
      strcpy(CURRENT_SOURCE,CURRENT_IN_SOURCE);
      Read_Source(interp,CURRENT_SOURCE);
    }
}


/*
 * From the current position of the insert point in de text select the innermost list
 */

void DebugSelectList(char s_start[])
{
  long nrHook;
  char s_pos[MAX_POS+1],s_pos_copy[MAX_POS+1],s_temp[MAX_POS+1];
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];
  nrHook=0;
  
  strncpy(s_pos,s_start,MAX_POS);
  s_pos[MAX_POS]='\0';

  Tcl_Invoke(interp,".text.text","index",s_pos,NULL);
  if (interp->result[0]!='0' && interp->result[0]!='1' && interp->result[0]!='2' &&
      interp->result[0]!='3' && interp->result[0]!='4' && interp->result[0]!='5' &&
      interp->result[0]!='4' && interp->result[0]!='5' && interp->result[0]!='6' &&
      interp->result[0]!='7' && interp->result[0]!='8' && interp->result[0]!='9')
    /* Incorrect position given */
    return;

  /* Search for the beginning of a the innermost list */

  while (1==1)
    {
      Tcl_Invoke(interp,".text.text","search","-backwards","-regexp","[()]",s_pos,"1.0",NULL);
      if (interp->result[0]=='\0')
	{
	  return; /* No opening hook found */
	};
      strncpy(s_pos,interp->result,MAX_POS);
      s_pos[MAX_POS]='\0';

      snprintf(s_temp,MAX_POS,"%s linestart",s_pos);

      Tcl_Invoke(interp,".text.text","search","-backwards",";",s_pos,s_temp,NULL);
      if (interp->result[0]=='\0')
	{
	  /* We are not in a lisp comment */
	  Tcl_Invoke(interp,".text.text","get",s_pos,NULL);
	  if (interp->result[0]==')')
	    {
	      /* Encountered a closing hook */
	      nrHook=nrHook+1;
	      continue;
	    }
	  /* Encountered a opening hook */
	  else if (nrHook==0)
	    {
	      /* Found the innermost opening hook */
	      strcpy(s_begin,s_pos);
	      break;
	    }
	  else
	    /* Found just a opening hook */
	    {
	      nrHook=nrHook-1;
	      continue;
	    };
	}
      else
	{
	  /* We are in a lisp comment */
	  continue;
	};
    };
  /* Search for the end of the innermost list */
  nrHook=0;
  snprintf(s_pos_copy,MAX_POS,"%s + 1 chars",s_pos);
  strcpy(s_pos,s_pos_copy); /* Workaround for a bug in snprintf */
  while (1==1)
    {
      Tcl_Invoke(interp,".text.text","search","-forwards","-regexp","[();]",s_pos,"end",NULL);
      if (interp->result[0]=='\0')
	{
	  /* Didn't found a closing hook */
	  return;
	};
      strncpy(s_pos,interp->result,MAX_POS);
      s_pos[MAX_POS]='\0';
      Tcl_Invoke(interp,".text.text","get",s_pos,NULL);
      if (interp->result[0]==';')
	/* Found a start of a comment */
	{
	  snprintf(s_pos_copy,MAX_POS,"%s lineend + 1 chars",s_pos);
	  strcpy(s_pos,s_pos_copy); /* Workaround for a bug in snprintf */
	  continue;
	}
      else if (interp->result[0]=='(')
	{
	  /* Found a opening hook */
	  nrHook=nrHook+1;
	  snprintf(s_pos_copy,MAX_POS,"%s + 1 chars",s_pos);
	  strcpy(s_pos,s_pos_copy); /* Workaround for a bug in snprintf */
	  continue;
	}
      else if (nrHook==0)
	{
	  /* Found the closing hook */
	  snprintf(s_end,MAX_POS,"%s + 1 chars",s_pos);
	  break;
	}
      else
	{
	  /* Found just a closing hook */
	  nrHook=nrHook-1;
	  snprintf(s_pos_copy,MAX_POS,"%s + 1 chars",s_pos);
	  strcpy(s_pos,s_pos_copy); /* Workaround for a bug in snprintf */
	  continue;
	};
    };
  /* Now select the text begin the begin and end field */
  
  Tcl_Invoke(interp,".text.text","tag","remove","sel","1.0","end",NULL);
  Tcl_Invoke(interp,".text.text","tag","add","sel",s_begin,s_end,NULL);
}

/*
 * Create a separate window to display the results of a watch expression
 */

void DisplayWatchpointsInWindow(char tag[],char exp[],char result[])
{
  char s_command[MAX_STRING+1];
  long i,j,width,height,text_width,text_height;
  

  /* Convert tag to lower characters */
  for (i=0;tag[i]!='\0';i++)
    tag[i]=tolower(tag[i]);
  /* Check if window exist */
  snprintf(s_command,MAX_STRING,"winfo exists .%s",tag);
  Tcl_Eval(interp,s_command);
  switch (interp->result[0])
    {
    case '0':
      /* Windows does not exist so it must be created */
      /* First get the font parameters */
      snprintf(s_command,MAX_STRING,"font metrics %s -ascent",FONT);
      Tcl_Eval(interp,s_command);
      height=atol(interp->result);
      snprintf(s_command,MAX_STRING,"font metrics %s -descent",FONT);
      Tcl_Eval(interp,s_command);
      height=atol(interp->result)+height;
      snprintf(s_command,MAX_STRING,"font measure %s M",FONT);
      Tcl_Eval(interp,s_command);
      width=atol(interp->result);
      /* Now determine the number of lines and the maximum width of the result */
      text_width=10; /* Minimum width = 10 characters */
      text_height=1;
      j=0;
      for (i=0;result[i]!='\0';i++)
	{
	  if (result[i]=='\n')
	    {
	      if (j>text_width)
		text_width=j;
	      j=0;
	      text_height=text_height+1;
	    }
	  else
	    {
	      j=j+1;
	    };
	};
      if (j>text_width)
	text_width=j;
      /* MAximum size of window is 80 char and 50 lines */
      if (text_width>80)
	text_width=80;
      if (text_height>50)
	text_height=50;
      /* Minimum size of window is 20 char and 5 lines */
      if (text_width<20)
	text_width=20;
      if (text_height<5)
	text_height=5;
      /* Now create a window of calculated size */
      snprintf(s_command,MAX_STRING,"toplevel .%s -width %ld -height %ld",tag,(text_width+2)*width,(text_height+2)*height);
      Tcl_Eval(interp,s_command);
      /* Register a procedure to be called when this window gets destroyed */
      snprintf(s_command,MAX_STRING,"bind .%s <Destroy> {if {\"%%W\" == \".%s\"} {attach-watchpoint %s}}",tag,tag,tag);
      Tcl_Eval(interp,s_command);
       /* Filter out " characters because they cause problems  */
      for (i=0;exp[i]!='\0';i++)
	if (exp[i]==34)
	  exp[i]=39;
      snprintf(s_command,MAX_STRING,"wm title .%s \"%s\"",tag,exp);
      Tcl_Eval(interp,s_command);
      snprintf(s_command,MAX_STRING,"scrollbar .%s.xtext -command \".%s.text xview\" -orient horizontal",tag,tag);
      Tcl_Eval(interp,s_command);
      snprintf(s_command,MAX_STRING,"scrollbar .%s.ytext -command \".%s.text yview\"",tag,tag);
      Tcl_Eval(interp,s_command);
      snprintf(s_command,MAX_STRING,"text .%s.text -font %s -xscrollcommand \".%s.xtext set\" -yscrollcommand \".%s.ytext set\"",tag,FONT,tag,tag);
      Tcl_Eval(interp,s_command);
      snprintf(s_command,MAX_STRING,"pack .%s.xtext -side bottom -fill x",tag);
      Tcl_Eval(interp,s_command);
      snprintf(s_command,MAX_STRING,"pack .%s.ytext -side right -fill y",tag);
      Tcl_Eval(interp,s_command);
      snprintf(s_command,MAX_STRING,"pack .%s.text -side left -expand yes -fill both",tag);
      Tcl_Eval(interp,s_command);
      snprintf(s_command,MAX_STRING,".%s.text",tag);
      Tcl_Invoke(interp,s_command,"insert","insert",result,NULL);
      Tcl_Invoke(interp,s_command,"configure","-state","disabled",NULL);
      /* Resize window */
      snprintf(s_command,MAX_STRING,"wm geometry .%s %ldx%ld",tag,(text_width+2)*width,(text_height+2)*height);
      Tcl_Eval(interp,s_command);
      break;
    case '1':
      /* Window exist already , so only the result must be changed */
      snprintf(s_command,MAX_STRING,".%s.text",tag);
      Tcl_Invoke(interp,s_command,"configure","-state","normal",NULL);
      Tcl_Invoke(interp,s_command,"configure","-font",FONT,NULL);
      Tcl_Invoke(interp,s_command,"delete","1.0","end",NULL);
      Tcl_Invoke(interp,s_command,"insert","insert",result,NULL);
      Tcl_Invoke(interp,s_command,"configure","-state","disabled",NULL);
      break;
    };
}
     
 
/*
 * Functions called from lisp
 */


/*
 * Highlight part of the source
 */

void highlight_source ()
{
  long begin,end;
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];
  char *type;
  char *color;

  /* Read arguments */

  begin=read_number();
  end=read_number();

  type=Read_String_Arg();
  color=Read_String_Arg();

  /* Execute code */

  AbsoluteToIndex(begin,s_begin);
  AbsoluteToIndex(end,s_end);
  Tcl_Invoke(interp,".text.text","tag","add",type,s_begin,s_end,NULL);
  Tcl_Invoke(interp,".text.text","tag","configure",type,"-background",color,NULL);

  /* Clean up , code */
  free(type);
  free(color);
}


/*
 * Sets a breakpoint of type type-breakpoint 
 */

int Set_Debug_Breakpoint(int type_breakpoint)
{
  long begin,end;
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];
  DEBUGPOINT *p_debugpoint;
  char key[MAX_STRING+1];
  char *exp;
  
  
  Tcl_Invoke(interp,".text.text","index","sel.first",NULL);
  begin=IndexToAbsolute(interp->result);
  Tcl_Invoke(interp,".text.text","index","sel.last",NULL);
  end=IndexToAbsolute(interp->result);
  
  if ((begin < 0) || (end < 0))
    {
      /* No selection made , return */
      return TCL_OK;
    };
  
  /* Sets up the key of the entry in the binary tree */
  snprintf(key,MAX_STRING,"%ld%ld%s",begin,end,CURRENT_SOURCE);
  
  if ((p_debugpoint=(DEBUGPOINT *) search_hash(H_DEBUGPOINT,key))==NULL)
    {
      Tcl_Eval(interp,"tk_dialog .m Message \"No breakpoint possible at selected part\" {} -1 Ok");
    }
  else
    {
      AbsoluteToIndex(begin,s_begin);
      AbsoluteToIndex(end,s_end);

      /* Take action dependend on type of breakpoint */
      switch (type_breakpoint)
	{
	case 1:
	  if ((p_debugpoint->type & 128) == 128)
	    {
	      /* Breakpoint is set , so unset breakpoint */
	      Tcl_Invoke(interp,".text.text","tag","remove","debug",s_begin,s_end,NULL);

	      p_debugpoint->type=p_debugpoint->type & 127; /* Disable the breakpoint */
	    }
	  else
	    {
	      /* Breakpoint is not set , so set breakpoint */
	      Tcl_Invoke(interp,".text.text","tag","add","debug",s_begin,s_end,NULL);
	      Tcl_Invoke(interp,".text.text","tag","configure","debug","-foreground",DEBUGPOINT_COLOR,NULL);
	      p_debugpoint->type=p_debugpoint->type | 128; /* Enable the breakpoint */
	    };
	  break;
	case 2:
	  /* Set tempory breakpoint , independent of the existance of a tempory breakpoint */
	  p_debugpoint->type=p_debugpoint->type | 64; /* Enable the breakpoint */
	  break;
	case 3:
	  if ((p_debugpoint->type & 32) == 32)
	    {
	      /* Breakpoint is set , so unset breakpoint */
	      Tcl_Invoke(interp,".text.text","tag","remove","debugif",s_begin,s_end,NULL);
	      p_debugpoint->type=p_debugpoint->type & 31; /* Disable the breakpoint */
	      free(p_debugpoint->exp);
	    }
	  else
	    {
	      /* Breakpoint is not set , so set breakpoint */
	      Tcl_Invoke(interp,".command.text","get","1.0","end - 1 chars",NULL);
	      if (interp->result[0]=='\0')
		return TCL_OK; /* No expression , so no conditional breakpoint */
	      /* We met conditions of a conditional breakpoint */
	      Tcl_Invoke(interp,".text.text","tag","add","debugif",s_begin,s_end,NULL);
	      Tcl_Invoke(interp,".text.text","tag","configure","debugif","-foreground",DEBUGPOINTIF_COLOR,NULL);
	      p_debugpoint->type=p_debugpoint->type | 32; /* Enable the breakpoint */
	      Tcl_Invoke(interp,".command.text","get","1.0","end - 1 chars",NULL);
	      /* exp=malloc(sizeof(char)*(strlen(interp->result) +1));
		 strcpy(exp,interp->result); */
	      exp=correct_string(interp->result);
	      p_debugpoint->exp=exp;
	    };
	  break;
	};
    };
  return TCL_OK;
};




/*
 * Called by the debugcode of lisp functions 
 */

void give_control_to_interface (int after)
{
  long begin,end;
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];
  char *source;
  char key[MAX_STRING+1];
  DEBUGPOINT *p_debugpoint;
  int stop_and_display; /* 0 = exit loop and delete highlight  , 1 = highlight and no exit loop  , 2 =  no exit loop */

  /* If it is a breakpoint after a call we must enable the return button */

  if (after)
    Tcl_Invoke(interp,".button2.return","configure","-state","normal",NULL);

  /* Indicate we are in a breakpoint */
  
  IN_BREAKPOINT=1;

  /* Read arguments */
  
  source=Read_String_Arg();
  begin=read_number();
  end=read_number();
  
  /* Communicate begin,end and source via global var to other functions , called in the interface */

  CURRENT_BEGIN=begin;
  CURRENT_END=end;
  strncpy(CURRENT_IN_SOURCE,source,FILENAME_MAX);
  CURRENT_IN_SOURCE[FILENAME_MAX]='\0';

  stop_and_display=0;

  /* Figure out if we must stop and display the current line */
  if (STEP_MODE)
    {
      /* In step mode we must stop , highlight the debugwindow and wait for further instructions */
      stop_and_display=1;
    };
  if (STEP_OVER_MODE)
    {
      /* In step over mode we stop and highlight only if we are in the boundaries */
      if ((STEP_OVER_BEGIN <= begin) && (end <= STEP_OVER_END) &&
	  (strcmp(CURRENT_IN_SOURCE,STEP_OVER_SOURCE)==0))
	{
	  stop_and_display=1;
	};
    };
  
  /* Check if breakpoint is set */
  snprintf(key,MAX_STRING,"%ld%ld%s",begin,end,source);
  if ((p_debugpoint=(DEBUGPOINT *) search_hash(H_DEBUGPOINT,key))!=NULL)
    {
      /* Update profile info if needed */
      if (PROFILING==1)
	{
	  p_debugpoint->executed=p_debugpoint->executed+1;
	  if (p_debugpoint->executed > MAX_PROFILING_COUNT)
	      MAX_PROFILING_COUNT=p_debugpoint->executed;
	};
      /* Check if the debugpoint is set */
      if ((p_debugpoint->type & 128) == 128)
	{
	  /* Breakpoint is set */
	  stop_and_display=1;
	}
      else if ((p_debugpoint->type & 64) == 64)
	{
	  /* Tempory breakpoint is set */
	  stop_and_display=1;
	  p_debugpoint->type=p_debugpoint->type & 63; /* Disable tempory breakpoint */
	}
      else if ((p_debugpoint->type & 32) == 32)
	{
	  /* Conditional breakpoint */
	  check_if_breakpoint(p_debugpoint->exp);
	  stop_and_display=2;
	};
    };

  /* We must stop and display a breakpoint , stopping is done automatically by loop in debugcode in lisp */
  if (stop_and_display==1)
    {
      /* Display the resultscreen */
      write(client_sockfd,"(DEBUGGER::display-watchpoints)\n",32);
      if (PPID!=0)
	{
	  kill(PPID,SIGUSR1);
	};
      /* Bring debugger to foreground */
      /* Tcl_Eval(interp,"wm deiconify ."); */
      /* Tcl_Eval(interp,"raise .");  Seems to slow down when called when window is already raised */

      /* Code to change source displayed if needed */
      Change_Source();
      /* Highlight code reached */
      AbsoluteToIndex(begin,s_begin);
      AbsoluteToIndex(end,s_end);
      Tcl_Invoke(interp,".text.text","tag","delete","breakpoint",NULL);
      Tcl_Invoke(interp,".text.text","tag","add","breakpoint",s_begin,s_end,NULL);
      Tcl_Invoke(interp,".text.text","yview","-pickplace",s_begin,NULL);
      /* The color highlighted , depends on the fact if we are before or after a call */
      if (after)
	{
	  Tcl_Invoke(interp,".text.text","tag","configure","breakpoint","-background",AFTER_COLOR,NULL);
	}
      else
	{
	  Tcl_Invoke(interp,".text.text","tag","configure","breakpoint","-background",CURRENT_COLOR,NULL);
	};
    }
  else if (stop_and_display==2)
    {
    }
  else
    {
      Leave_Breakpoint();
    };

 /* Clean up allocated strings */
  free(source);
}
  
/* 
 * Called by the lisp system to display the result of watch expressions and watch variables 
 * REMEMBER :: This function replaces display_result_in_interface
 */

void display_result()
{
  long i,type,nr_args;
  char *watch_exp;
  char *watch_value;
  char *watch_tag;

  /* Clear the field of all watchpoints */
  Tcl_Invoke(interp,".result.text","configure","-state","normal",NULL);
  Tcl_Invoke(interp,".result.text","delete","1.0","end",NULL);
  /* Reads the number of arguments */
  nr_args=read_number();
  /* Reads the argments and process them */
  for (i=1;i<=nr_args;i++)
    {
      /* reads the arguments */
      type=read_number();
      watch_tag=Read_String_Arg();
      watch_exp=Read_String_Arg();
      watch_value=Read_String_Arg();
      /* process the arguments */
      switch (type)
	{
	case 0:  /* variables */
	  Tcl_Invoke(interp,".result.text","insert","insert",watch_exp,watch_tag,NULL);
	  Tcl_Invoke(interp,".result.text","insert","insert","==>",watch_tag,NULL);
	  Tcl_Invoke(interp,".result.text","insert","insert",watch_value,watch_tag,NULL);
	  Tcl_Invoke(interp,".result.text","insert","insert","\n",watch_tag,NULL);
	  break;
	case 1: /* expressions */
	  Tcl_Invoke(interp,".result.text","insert","insert",watch_exp,watch_tag,NULL);
	  Tcl_Invoke(interp,".result.text","insert","insert","-->",watch_tag,NULL);
	  Tcl_Invoke(interp,".result.text","insert","insert",watch_value,watch_tag,NULL);
	  Tcl_Invoke(interp,".result.text","insert","insert","\n",watch_tag,NULL);
	  break;
	case 10: /* variable in separate window */
	  DisplayWatchpointsInWindow(watch_tag,watch_exp,watch_value);
	  break;
	case 11: /* Expression in separate window */
	  DisplayWatchpointsInWindow(watch_tag,watch_exp,watch_value);
	  break;
	};
      /* Cleanup the allocated strings */
      free(watch_tag);
      free(watch_value);
      free(watch_exp);
    };
  Tcl_Invoke(interp,".result.text","configure","-state","disabled",NULL);
}

/*
 * Called by LISP to display the result of evaluating a expression in a result window
 */

void display_exp_in_interface()
{
  char *exp,*result;

  /* Read the arguments */
  
  exp=Read_String_Arg();
  result=Read_String_Arg();
  
  /* Display the result */
  Tcl_Invoke(interp,".result.text","configure","-state","normal",NULL);

  Tcl_Invoke(interp,".result.text","insert","insert",exp,NULL);
  Tcl_Invoke(interp,".result.text","insert","insert"," --> ",NULL);
  Tcl_Invoke(interp,".result.text","insert","insert",result,NULL);
  Tcl_Invoke(interp,".result.text","insert","insert","\n",NULL);
  
  Tcl_Invoke(interp,".result.text","configure","-state","disabled",NULL);

  /* Cleanup the allocated strings */
  free(result);
  free(exp);
}

/*
 * Called by LISP to invoke a real breakpoint
 */

void if_breakpoint()
{
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];

  
  /* Display results in result window */
  write(client_sockfd,"(DEBUGGER::display-watchpoints)\n",32);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  
  /* Code to change source displayed if needed */
  Change_Source();
  /* Highlight code reached */
  AbsoluteToIndex(CURRENT_BEGIN,s_begin);
  AbsoluteToIndex(CURRENT_END,s_end);
  
  Tcl_Invoke(interp,".text.text","tag","delete","breakpoint",NULL);
  Tcl_Invoke(interp,".text.text","tag","add","breakpoint",s_begin,s_end,NULL);
  Tcl_Invoke(interp,".text.text","yview","-pickplace",s_begin,NULL);
  Tcl_Invoke(interp,".text.text","tag","configure","breakpoint","-background",CURRENT_COLOR,NULL);

}

/*
 * Called by lisp to display the current environment in time 
 */

void display_time_env ()
{
  long begin,end;
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];
  char *source;
  
  begin=read_number();
  end=read_number();
  source=Read_String_Arg();
  strcpy(CURRENT_IN_SOURCE,source);
  free(source); /* Cleanup the allocated string */

  /* Display the envrionment */

  CURRENT_BEGIN=begin;
  CURRENT_END=end;

  /* Display watchpoints */
  write(client_sockfd,"(DEBUGGER::display-watchpoints)\n",32);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  /* Change source if needed */
  Change_Source();
  AbsoluteToIndex(begin,s_begin);
  AbsoluteToIndex(end,s_end);
  Tcl_Invoke(interp,".text.text","tag","delete","breakpoint",NULL);
  Tcl_Invoke(interp,".text.text","tag","add","breakpoint",s_begin,s_end,NULL);
  Tcl_Invoke(interp,".text.text","yview","-pickplace",s_begin,NULL);
  Tcl_Invoke(interp,".text.text","tag","configure","breakpoint","-background",CURRENT_COLOR,NULL);
}  

/*
 * Establish the settings of the debugger
 */

void setting()
{
  char *var;
  int nr;
  
  var=Read_String_Arg();
  if (strcmp(var,"COMPILE_CODE")==0)
    {
      COMPILE_CODE=read_number();
      Tcl_UpdateLinkedVar(interp,"**compile-code**");
    }
  else if (strcmp(var,"DISPLAY_PREVIOUS")==0)
    {
      DISPLAY_PREVIOUS=read_number();
      Tcl_UpdateLinkedVar(interp,"**display-result-call**");
      /* set the setting for the lisp system */
      if (DISPLAY_PREVIOUS)
	{
	  write(client_sockfd,"(setf DEBUGGER::**display-result-call** t)\n",43);
	}
      else
	{
	  write(client_sockfd,"(setf DEBUGGER::**display-result-call** nil)\n",45);
	};
      if (PPID != 0 && IN_BREAKPOINT==0)
	{
	  kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
	};
    }
  else if (strcmp(var,"SAVE_ON_EXIT")==0)
    {
      SAVE_ON_EXIT=read_number();
      Tcl_UpdateLinkedVar(interp,"**save-on-exit**");    
    }
  else if (strcmp(var,"CHECK_ERROR")==0)
    {
      DISPLAY_PREVIOUS=read_number();
      Tcl_UpdateLinkedVar(interp,"**check-error**");
      /* Change the result in LISP */
      if (CHECK_ERROR)
	{
	  write(client_sockfd,"(setf DEBUGGER::**check-error** t)\n",35);
	}
      else
	{
	  write(client_sockfd,"(setf DEBUGGER::**check-error** nil)\n",37);
	};
      if (PPID != 0 && IN_BREAKPOINT==0)
	{
	  kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
	};
    }
  else if (strcmp(var,"DEBUG_MACRO")==0)
    {
      DISPLAY_PREVIOUS=read_number();
      Tcl_UpdateLinkedVar(interp,"**debug-macro**");
      /* Change the result in LISP */
      if (DEBUG_MACRO)
	{
	  write(client_sockfd,"(setf DEBUGGER::**debug-macro** t)\n",35);
	}
      else
	{
	  write(client_sockfd,"(setf DEBUGGER::**debug-macro** nil)\n",37);
	};
      if (PPID != 0 && IN_BREAKPOINT==0)
	{
	  kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
	};
    }
  else if (strcmp(var,"DEBUGPOINT_COLOR")==0)
    {
      free(var); /* Cleanup the previous allocated string */
      var=Read_String_Arg();
      strncpy(DEBUGPOINT_COLOR,var,MAX_STRING);
      DEBUGPOINT_COLOR[MAX_STRING];
    }
  else if (strcmp(var,"DEBUGPOINTIF_COLOR")==0)
    {
      free(var); /* Cleanup the previous allocated string */
      var=Read_String_Arg();
      strncpy(DEBUGPOINTIF_COLOR,var,MAX_STRING);
      DEBUGPOINTIF_COLOR[MAX_STRING]='\0';
    }
  else if (strcmp(var,"CURRENT_COLOR")==0)
    {
      free(var); /* Cleanup the previous allocated string */
      var=Read_String_Arg();
      strncpy(CURRENT_COLOR,var,MAX_STRING);
      CURRENT_COLOR[MAX_STRING]='\0';
    }
  else if (strcmp(var,"AFTER_COLOR")==0)
    {
      free(var); /* Cleanup the previous allocated string */
      var=Read_String_Arg();
      strncpy(AFTER_COLOR,var,MAX_STRING);
      AFTER_COLOR[MAX_STRING]='\0';
    }  
  else if (strcmp(var,"PROFILE_COLOR")==0)
    {
      free(var); /* Cleanup the previous allocated string */
      var=Read_String_Arg();
      strncpy(PROFILE_COLOR,var,MAX_STRING);
      PROFILE_COLOR[MAX_STRING]='\0';
    }
  else if (strcmp(var,"FONT")==0)
    {
      free(var); /* Cleanup the previous allocated string */
      var=Read_String_Arg();
      strncpy(FONT,var,MAX_STRING);
      FONT[MAX_STRING]='\0';
      
    }
  else
    {
      Tcl_Eval(interp,"tk_dialog .m Message \"Unknown setting encountered ,check .lispdebug\" {} -1 Ok");
    };

  /* Cleanup the allocated strings */
  free(var);
}


/*
 * Store a possible breakpoint in the tree of breakpoints
 */

void set_possible_breakpoint()
{
  long begin,end;
  char *source;
  char *key,skey[MAX_STRING+1];
  DEBUGPOINT *p_debugpoint,*p;

  /* Reads arguments */
  
  source=Read_String_Arg();
  begin=read_number();
  end=read_number();

  /* Execute code */
  
  /* Handle item for tree  */
  p_debugpoint=malloc(sizeof(DEBUGPOINT));
  p_debugpoint->source=malloc((strlen(source)+1)*sizeof(char));
  strcpy(p_debugpoint->source,source);
  p_debugpoint->begin=begin;
  p_debugpoint->end=end;
  p_debugpoint->type=0;
  p_debugpoint->executed=0;

  /* Make key */
  snprintf(skey,MAX_STRING,"%ld%ld%s",begin,end,source);
  key=malloc(sizeof(char)*(strlen(skey)+1));
  strcpy(key,skey);

  /* Insert in hashtable if needed */
  
  insert_hash(H_DEBUGPOINT,key,(void *) p_debugpoint);

  /* Cleanup the allocated strings */
  free(source);
}


/*
 * Display a message bok with a Ok button for the lisp system
 */

void display_message ()
{
  /* char message[MAX_STRING]; */
  
  char *message,*new_message,*command;
  static long n = 1;
  static long i;

 n=n+1; /* Modify counter of errormessage */
 
 /* Reads arguments */

 message=Read_String_Arg();

 /* Filter out " characters because they cause the format of the message to be wrong */

  new_message=correct_string(message);
 
 
 /* Process arguments */

 command=malloc(sizeof(char)*(strlen(new_message)+50));
 sprintf(command,"tk_dialog .m%ld Message \"%s\" {} -1 Ok",n,new_message);
 Tcl_Eval(interp,command);

 /* Cleanup the allocated strings */
 
 free(command);
 free(new_message);
 free(message);
}


void highlight_error ()
{
  long begin;
  char s_begin[MAX_POS+1];

  /* Read arguments */

  begin=read_number();

  /* Execute code */

  AbsoluteToIndex(begin,s_begin);
  Tcl_Invoke(interp,".text.text","tag","add","error",s_begin,"lineend",NULL);
  Tcl_Invoke(interp,".text.text","tag","configure","error","-background","red",NULL);
  
}


/*
 * Called by lisp to go to the debugger , in case of an error during execution of debugged code
 */

void lisp_goes_to_debug ()
{
  /* This function must make sure that the debugger stops with a warning message about the error */
  STEP_MODE=1;
  STEP_OVER_MODE=0;
  STEP_OVER_BEGIN=0;
  STEP_OVER_END=0;

  display_message();
}

/* 
 * Functions called via TCL/TK
 */

/*
 * Put a result displayed in a separated window back in the result pane 
 *
 */

int Attach_Watchpoint(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  char *tag;

  tag=Tcl_GetStringFromObj(objv[1],NULL);
  write(client_sockfd,"(DEBUGGER::debug-separate-watchpoint \"",38);
  write(client_sockfd,tag,strlen(tag));
  write(client_sockfd,"\")\n",3);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1); /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}

/*
 * Select a list in the source
 */

int Select_List(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long x,y;
  char s_pos[MAX_POS+1];

  Tcl_GetLongFromObj(interp,objv[1],&x);
  Tcl_GetLongFromObj(interp,objv[2],&y);
  
  snprintf(s_pos,MAX_POS,"@%ld,%ld",x,y);
  DebugSelectList(s_pos);
  return TCL_OK;
}

/*
 * Select a list in the source
 */

int Select_Word(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long x,y;
  char s_begin[MAX_POS+1];
  char s_end[MAX_POS+1];

  Tcl_GetLongFromObj(interp,objv[1],&x);
  Tcl_GetLongFromObj(interp,objv[2],&y);
  
  snprintf(s_begin,MAX_POS,"@%ld,%ld wordstart",x,y);
  snprintf(s_end,MAX_POS,"@%ld,%ld wordend",x,y);

  Tcl_Invoke(interp,".text.text","tag","add","sel",s_begin,s_end,NULL);
  
  return TCL_OK;
}

/*
 * Select a whole function
 */

int Select_Function(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long x,y;
  char s_pos[MAX_POS+1];
  char s_temp[MAX_POS+1];

  Tcl_GetLongFromObj(interp,objv[1],&x);
  Tcl_GetLongFromObj(interp,objv[2],&y);
  
  snprintf(s_pos,MAX_POS,"@%ld,%ld",x,y);

  while (1==1)
    {
      Tcl_Invoke(interp,".text.text","search","-backwards","-regexp","[dD][eE][fF][uU][nN]",s_pos,"1.0",NULL);
      if (interp->result[0]=='\0')
	{
	  /* Didn't find a result */
	  return TCL_OK;
	};
      strncpy(s_pos,interp->result,MAX_POS);
      s_pos[MAX_STRING]='\0';
      snprintf(s_temp,MAX_POS,"%s linestart",s_pos);
      Tcl_Invoke(interp,".text.text","search","-backwards",";",s_pos,s_temp,NULL);
      if (interp->result[0]!='\0')
	{
	  /* Found a comment , so skip comment */
	  snprintf(s_temp,MAX_POS,"%s - 1 chars",s_pos);
	  strcpy(s_pos,s_temp); /* Workaround for a bug in snprintf */
	  continue;
	}
      else
	{
	  /* Found a defun not in a comment */
	  DebugSelectList(s_pos);
	  return TCL_OK;
	};
    };
  return TCL_OK;
}

/*
 * Select a result to delete or to display in other window
 */

int Select_Result(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long x,y;
  char s_pos1[MAX_POS+1];
  char s_pos2[MAX_POS+1];
  char s_label[MAX_POS+1]; /* We suppos that labels are much smaller then 30 characters */
  char s_command[MAX_STRING+1];
  /* Get position selection */

  Tcl_GetLongFromObj(interp,objv[1],&x);
  Tcl_GetLongFromObj(interp,objv[2],&y);
  snprintf(s_pos1,MAX_POS,"@%ld,%ld",x,y);

  /* Make sure that there is no selection made */
  Tcl_Invoke(interp,".result.text","tag","remove","sel","1.0","end",NULL);
  /* Get the label of the selected result line */
  Tcl_Invoke(interp,".result.text","tag","names",s_pos1,NULL);
  /* Check if we have clicked on a result */
  if (interp->result=='\0')
    return TCL_OK;
  /* get the label */
  strncpy(s_label,interp->result,MAX_POS);
  s_label[MAX_POS]='\0';
  /* Select the line representing the label */
  snprintf(s_pos1,MAX_POS,"%s.first",s_label);
  snprintf(s_pos2,MAX_POS,"%s.last",s_label);
  Tcl_Invoke(interp,".result.text","tag","add","sel",s_pos1,s_pos2,NULL);
  /* Now ask what we have to do with the result */
  Tcl_Eval(interp,"tk_dialog .r \"Result pane\" \"What do you want to do with this result\" {} 0 Return Delete Separate");
  switch (interp->result[0])
    {
    case '0': /* User has asked to do nothing */
      return TCL_OK;
      break;
    case '1': /* User asked to delete the result from the watchpoint list */
      snprintf(s_command,MAX_STRING,"(DEBUGGER::debug-delete-watchpoint \"%s\")\n",s_label);
      write(client_sockfd,s_command,strlen(s_command));
      break;
    case '2': /*User has asked to print the result in a separate window */
      snprintf(s_command,MAX_STRING,"(DEBUGGER::debug-separate-watchpoint \"%s\")\n",s_label);
      write(client_sockfd,s_command,strlen(s_command));
      break;
    };
  /* Dislay anew the result window */
  write(client_sockfd,"(DEBUGGER::display-watchpoints)\n",32);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}


/*
 * Prepare a resizing operation op=0 ,
 * or resize the relative portions of text and result windows op=1
 * or resize the relative portions of result and command windows op=2
 */
 
int Resize_Window(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long op,Y;
  static long height_text;
  static long height_result;
  static long height_command;
  static long start_Y;
  long delta,new_height_text,new_height_result,new_height_command;
  float fdelta;
  char s_height[MAX_POS+1];

  /* Get my real args */
  Tcl_GetLongFromObj(interp,objv[1],&op);
  Tcl_GetLongFromObj(interp,objv[2],&Y);
  
  /* Switch depending on task */
  switch (op)
    {
    case 0:
      start_Y=Y;
      Tcl_Invoke(interp,".text.text","cget","-height",NULL);
      sscanf(interp->result,"%ld",&height_text);
      Tcl_Invoke(interp,".result.text","cget","-height",NULL);
      sscanf(interp->result,"%ld",&height_result);
      Tcl_Invoke(interp,".command.text","cget","-height",NULL);
      sscanf(interp->result,"%ld",&height_command);
      break;
    case 1:
      fdelta=((float)(start_Y-Y)/10.0);
      delta=(long)fdelta;
      new_height_result=height_result+delta;
      new_height_text=height_text-delta;
      if (new_height_result > 0 &&
	  new_height_text > 0 &&
	  delta!=0)
	{
	  snprintf(s_height,MAX_POS,"%ld",new_height_text);
	  Tcl_Invoke(interp,".text.text","configure","-height",s_height,NULL);
	  snprintf(s_height,MAX_POS,"%ld",new_height_result);
	  Tcl_Invoke(interp,".result.text","configure","-height",s_height,NULL);
	  height_text=new_height_text;
	  height_result=new_height_result;
	  start_Y=Y;
	  Tcl_Eval(interp,"pack .text.text .result.text .command.text");
	};
      break;
    case 2:
      fdelta=((float)(start_Y-Y)/10.0);
      delta=(long)fdelta;
      new_height_command=height_command+delta;
      new_height_result=height_result-delta;
      if (new_height_command > 0 &&
	  new_height_result > 0 &&
	  delta!=0)
	{
	  snprintf(s_height,MAX_POS,"%ld",new_height_result);
	  Tcl_Invoke(interp,".result.text","configure","-height",s_height,NULL);
	  snprintf(s_height,MAX_POS,"%ld",new_height_command);
	  Tcl_Invoke(interp,".command.text","configure","-height",s_height,NULL);
	  height_result=new_height_result;
	  height_command=new_height_command;
	  start_Y=Y;
	  Tcl_Eval(interp,"pack .text.text .result.text .command.text");
	};
      break;
    };
  return  TCL_OK;
}



/* 
 * Select a color for highlighting the code current executing 
 */

int Debug_Color_Break(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Tcl_Invoke(interp,"tk_chooseColor",NULL);
  if (interp->result[0]=='\0')
    {
      return TCL_OK;
    }
  else
    {
      strncpy(CURRENT_COLOR,interp->result,MAX_STRING);
      CURRENT_COLOR[MAX_STRING]='\0';
      return TCL_OK;
    };
}

/* 
 * Select a color for highlighting the code current executing after a call is done
 */

int Debug_Color_After(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Tcl_Invoke(interp,"tk_chooseColor",NULL);
  if (interp->result[0]=='\0')
    {
      return TCL_OK;
    }
  else
    {
      strncpy(AFTER_COLOR,interp->result,MAX_STRING);
      AFTER_COLOR[MAX_STRING]='\0';
      return TCL_OK;
    };
}

/* 
 * Select a color for the breakpoints
 */

int Debug_Color_Breakpoint(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Tcl_Invoke(interp,"tk_chooseColor",NULL);
  if (interp->result[0]=='\0')
    {
      return TCL_OK;
    }
  else
    {
      strncpy(DEBUGPOINT_COLOR,interp->result,MAX_STRING);
      DEBUGPOINT_COLOR[MAX_STRING]='\0';
      Tcl_Invoke(interp,".text.text","tag","configure","debug","-foreground",DEBUGPOINT_COLOR,NULL);
      return TCL_OK;
    };
}

/* 
 * Select a color for the conditional breakpoints 
 */

int Debug_Color_Breakpointif(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Tcl_Invoke(interp,"tk_chooseColor",NULL);
  if (interp->result[0]=='\0')
    {
      return TCL_OK;
    }
  else
    {
      strncpy(DEBUGPOINTIF_COLOR,interp->result,MAX_STRING);
      DEBUGPOINTIF_COLOR[MAX_STRING]='\0';
      Tcl_Invoke(interp,".text.text","tag","configure","debugif","-foreground",DEBUGPOINTIF_COLOR,NULL);
      return TCL_OK;
    };
}


/* 
 * Select a color for the conditional breakpoints 
 */

int Debug_Color_Profile(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Tcl_Invoke(interp,"tk_chooseColor",NULL);
  if (interp->result[0]=='\0')
    {
      return TCL_OK;
    }
  else
    {
      strncpy(PROFILE_COLOR,interp->result,MAX_STRING);
      PROFILE_COLOR[MAX_STRING]='\0';
      Tcl_Invoke(interp,".text.text","tag","configure","profile","-background",PROFILE_COLOR,NULL);
      return TCL_OK;
    };
}

/*
 * Select a line in the font selection list 
 */

int Choose_Font_Select_Line(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long x,y;
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];
  char font[MAX_STRING+1];

  Tcl_GetLongFromObj(interp,objv[1],&x);
  Tcl_GetLongFromObj(interp,objv[2],&y);
  
  snprintf(s_begin,MAX_POS,"@%ld,%ld linestart",x,y);
  snprintf(s_end,MAX_POS,"@%ld,%ld lineend",x,y);
  Tcl_Invoke(interp,".font.list.list","get",s_begin,s_end,NULL);
  strncpy(font,interp->result,MAX_STRING);
  font[MAX_STRING]='\0';

  
  Tcl_Invoke(interp,".font.list.list","tag","remove","sel","1.0","end",NULL);
  Tcl_Invoke(interp,".font.list.list","tag","add","sel",s_begin,s_end,NULL);
  
  Tcl_Invoke(interp,".font.text.text","configure","-font",font,NULL);
  
  return TCL_OK;
}

/*
 * Process confirmation of choosend font
 */

int Choose_Font_Ok(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  /* Find selected font */
  Tcl_Invoke(interp,".font.list.list","get","sel.first","sel.last",NULL);
  strncpy(FONT,interp->result,MAX_STRING);
  FONT[MAX_STRING]='\0';
  /* Change font */
  Tcl_Invoke(interp,".text.text","configure","-font",FONT,NULL);
  Tcl_Invoke(interp,".result.text","configure","-font",FONT,NULL);
  Tcl_Invoke(interp,".command.text","configure","-font",FONT,NULL);
  /* Display results in result window in new font */
  write(client_sockfd,"(DEBUGGER::display-watchpoints)\n",32);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  /* Cancel font window */
  Tcl_Invoke(interp,"destroy",".font",NULL);
  /* End */
  return TCL_OK;
}

/*
 * Process the cancel button in the selectfont screen
 */

int Choose_Font_Cancel(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  /* Cancel font window */
  Tcl_Invoke(interp,"destroy",".font",NULL);
  /* End */
  return TCL_OK;
}

/* 
 * Select a font for the debugger
 */

int Debug_Font(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  FILE *fp;
  char line[MAX_STRING+1];

  /* First create a listbox */
  Tcl_Eval(interp,"toplevel .font");
  Tcl_Eval(interp,"grab .font");
  Tcl_Eval(interp,"frame .font.button");
  Tcl_Eval(interp,"frame .font.list");
  Tcl_Eval(interp,"frame .font.text");
  Tcl_Eval(interp,"pack .font.button .font.list .font.text -side top");
  Tcl_Eval(interp,"button .font.button.ok -text Ok -command choose-font-ok -state disabled");
  Tcl_Eval(interp,"button .font.button.cancel -text Cancel -command choose-font-cancel -state disabled");
  Tcl_Eval(interp,"pack .font.button.ok .font.button.cancel -side left");
  Tcl_Eval(interp,"scrollbar .font.list.scroll -command \".font.list.list yview\"");
  Tcl_Eval(interp,"text .font.list.list -yscrollcommand \".font.list.scroll set\"");
  Tcl_Eval(interp,"pack .font.list.list -in .font.list -side left");
  Tcl_Eval(interp,"pack .font.list.scroll -in .font.list -side left -expand yes -fill both");
  Tcl_Eval(interp,"text .font.text.text -height 4 -width 40");
  Tcl_Eval(interp,"pack .font.text.text");
  Tcl_Eval(interp,"bind .font.list.list <ButtonRelease-1> {choose-font-select-line %x %y}");
  /* Create the referenced commandos */
  Tcl_CreateObjCommand(interp,"choose-font-select-line",Choose_Font_Select_Line,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"choose-font-ok",Choose_Font_Ok,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"choose-font-cancel",Choose_Font_Cancel,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  /* Configure the example text */
  Tcl_Invoke(interp,".font.text.text","insert","end","abcdefghijklmnopqrstuvwxyz\nABCDEFGHIJKLMNOPQRSTUVWXYZ\n01234567890\n!@#$%^&*()_",NULL);
  Tcl_Invoke(interp,".font.text.text","tag","add","font","1.0","end",NULL);
  /* Get a list of available components */
  fp=popen("xlsfonts|sort -u","r");
  /* insert fonts in their font in the list */
  while (fgets(line,MAX_STRING-1,fp)!=NULL)
    {
      Tcl_Invoke(interp,".font.list.list","insert","insert",line,NULL);
    };
  fclose(fp);
  Tcl_Invoke(interp,".font.list.list","configure","-state","disabled",NULL);
  Tcl_Invoke(interp,".font.button.ok","configure","-state","normal",NULL);
  Tcl_Invoke(interp,".font.button.cancel","configure","-state","normal",NULL);
  return TCL_OK;
}

/* 
 * Process of changing of displaying the previous result
 */

int Debug_Display_Previous_Call(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  if (DISPLAY_PREVIOUS)
    {
      write(client_sockfd,"(setf DEBUGGER::**display-result-call** t)\n",43);
    }
  else
    {
      write(client_sockfd,"(setf DEBUGGER::**display-result-call** nil)\n",45);
    };
  if (PPID != 0 && IN_BREAKPOINT==0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}

/* 
 * Process , check error by debugger
 */

int Debug_Check_Error(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  if (CHECK_ERROR)
    {
      write(client_sockfd,"(setf DEBUGGER::**check-error** t)\n",35);
    }
  else
    {
      write(client_sockfd,"(setf DEBUGGER::**check-error** nil)\n",37);
    };
  if (PPID != 0 && IN_BREAKPOINT==0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}

/* 
 * Enable debugging macro
 */

int Debug_Debug_Macro(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  if (DEBUG_MACRO)
    {
      write(client_sockfd,"(setf DEBUGGER::**debug-macro** t)\n",35);
    }
  else
    {
      write(client_sockfd,"(setf DEBUGGER::**debug-macro** nil)\n",37);
    };
  if (PPID != 0 && IN_BREAKPOINT==0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}


/*
 * Paste of code in the debugging system
 */

int Debug_Paste(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  static long nr=0;
  char filename[FILENAME_MAX+1];
  int fd;
  char s_pos[MAX_POS];
  char *key,*dat;

  /* First clear the source window and the current source */
  Tcl_Invoke(interp,".text.text","configure","-state","normal",NULL);

  Tcl_Invoke(interp,".text.text","delete","1.0","end",NULL);
  CURRENT_SOURCE[0]='\0';
  /* Now paste the contents in the source window */

  /* Tcl_Invoke(interp,"tk_textPaste",".text.text",NULL); */
  Tcl_Eval(interp,"tk_textPaste .text.text");
  /* Use this information now to write to a diskfile */
  snprintf(filename,FILENAME_MAX,"/usr/tmp/lisp-debugger-paste%ld.lisp",nr);
  nr=nr+1;
  
  fd=open(filename,O_WRONLY | O_CREAT | O_TRUNC,S_IRWXU);
  if (fd == -1)
    {
      Tcl_Eval(interp,"tk_dialog .m Message \"Can't create tempory file\" {} -1 Ok");
      return TCL_OK;
    };
  Tcl_Invoke(interp,".text.text","get","1.0","end",NULL);
  write(fd,interp->result,sizeof(char)*strlen(interp->result));
  close(fd);
  Tcl_Invoke(interp,".text.text","configure","-state","disabled",NULL);

  /* Send a command to the lisp system to handle the file */
  /* This should be the only place that the user interface is commanding the lisp system directly */

  write(client_sockfd,"(DEBUGGER::debug-open-file \"",28);
  write(client_sockfd,filename,strlen(filename));
  write(client_sockfd,"\"",1);
  if (COMPILE_CODE)
    write(client_sockfd," T ",3);
  write(client_sockfd,")\n",2);

  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };

  /* In the maintime read the file in the text area */

  Read_Source(interp,filename);

  /* Set the current source name */
  
  strncpy(CURRENT_SOURCE,filename,FILENAME_MAX);
  CURRENT_SOURCE[FILENAME_MAX]='\0';
  
  /* Register the tempsource name in a list of sources beeing debugged */
  
  key=malloc((strlen(filename)+1)*sizeof(char));
  strcpy(key,filename);
  dat=malloc((strlen(filename)+1)*sizeof(char));
  strcpy(dat,filename);
  L_SOURCES=cons(key,dat,L_SOURCES);
  
  /* Modify the sources menu to display the new source */
  
  Change_Source_Menu();

  return TCL_OK;
}

/*
 * Start profiling code 
 */

/* 
 * For profiling code , punt nr executed on 0 , called by twalk
 */

void Debug_Start_Profile_Walk(char *key,void *data)
{
  DEBUGPOINT *p_debugpoint;

  p_debugpoint=(DEBUGPOINT *) data;
  p_debugpoint->executed=0;
}

/*
 * Start the profiling code 
 */
int Debug_Start_Profile(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Tcl_Eval(interp,"scale .text.scale -from 0 -to 10 -command debug-show-profile-scale");
  Tcl_Eval(interp,"pack .text.scale");
  /* Initialise the profiling */
  MAX_PROFILING_COUNT=0;
  PROFILING=1;
  walk_hash(H_DEBUGPOINT,Debug_Start_Profile_Walk);
  return TCL_OK;
}

/*
 * Stop profiling code
 */

int Debug_Stop_Profile(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  PROFILING=0;
  Tcl_Eval(interp,"destroy .text.scale");
  Tcl_Invoke(interp,".text.text","tag","delete","profile",NULL);
  return TCL_OK;
}

/*
 * Show profile information
 */


/*
 * Function to traverse the tree for Debug_Show_Profile_Sclae_Walk
 */


void Debug_Show_Profile_Scale_Remove(char *key,void *data)
{
  DEBUGPOINT *p_debugpoint;
  char s_begin[MAX_POS+1],s_end[MAX_POS+1];

  p_debugpoint=(DEBUGPOINT *) data;
  if (strcmp(p_debugpoint->source,CURRENT_SOURCE)==0 && p_debugpoint->executed < EXECUTED
                                                     && p_debugpoint->begin > PROFILE_BEGIN
                                                     && p_debugpoint->end < PROFILE_END)
    {
      AbsoluteToIndex(p_debugpoint->begin,s_begin);
      AbsoluteToIndex(p_debugpoint->end,s_end);
      Tcl_Invoke(interp,".text.text","tag","remove","profile",s_begin,s_end,NULL);
    };
}	  

void Debug_Show_Profile_Scale_Add(char *key,void *data)
{
  DEBUGPOINT *p_debugpoint;
  char s_begin[15],s_end[15];

  p_debugpoint=(DEBUGPOINT *) data;
  if (strcmp(p_debugpoint->source,CURRENT_SOURCE)==0 && p_debugpoint->executed >= EXECUTED)
    {
      AbsoluteToIndex(p_debugpoint->begin,s_begin);
      AbsoluteToIndex(p_debugpoint->end,s_end);
      Tcl_Invoke(interp,".text.text","tag","add","profile",s_begin,s_end,NULL);
      /* Remove smaller parts which have not reached counts */
      PROFILE_BEGIN=p_debugpoint->begin;
      PROFILE_END=p_debugpoint->end;
      walk_hash(H_DEBUGPOINT,Debug_Show_Profile_Scale_Remove);
    };
}

      
/*
 * Main function to show profile information
 */

int Debug_Show_Profile_Scale(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  char s_begin[MAX_POS+1];

  /* Set maximum scale */
  if (MAX_PROFILING_COUNT > 10)
    {
      snprintf(s_begin,MAX_POS,"%ld",MAX_PROFILING_COUNT);
      Tcl_Invoke(interp,".text.scale","configure","-to",s_begin,NULL);
    }
  else
    {
       Tcl_Invoke(interp,".text.scale","configure","-to","10",NULL);
    };
  /* Get the scale */ 
  Tcl_Invoke(interp,".text.scale","get",NULL);
  sscanf(interp->result,"%ld",&EXECUTED);

  Tcl_Invoke(interp,".text.text","tag","delete","profile",NULL);
  /* Mark the part of the source to be highlighted */
  walk_hash(H_DEBUGPOINT,Debug_Show_Profile_Scale_Add);
  /* Mark the part of the source to be not highlighted */
  /* Highlight marked code */
  Tcl_Invoke(interp,".text.text","tag","configure","profile","-background",PROFILE_COLOR,NULL);

  return TCL_OK;
}


/*
 *   Go back in time
 */

int Debug_Back(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  write(client_sockfd,"(DEBUGGER::step-back-in-time)\n",30);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}

/* 
 * Go forwards in time
 */

int Debug_Forward(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  write(client_sockfd,"(DEBUGGER::step-forward-in-time)\n",33);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}

/*
 * Step through the code of a function
 */


int Debug_Step(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  STEP_MODE=1;
  /* Initialise STEP_OVER part */
  STEP_OVER_MODE=0;
  STEP_OVER_BEGIN=0;
  STEP_OVER_END=0;
  Leave_Breakpoint();
  return TCL_OK;
}

/*
 * Steps through the source , showing only the top level layer
 */

int Debug_Step_Over(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long begin,end;
  char s_pos[MAX_POS+1];

  /* Disable step mode */
  STEP_MODE=0;
  STEP_OVER_MODE=1;
  /* Check if we are still stepping over in an existing list */
  if (STEP_OVER_END==0)
    {
      /* This is a new stepping over try*/
      /* Copy the current source name */
      strncpy(STEP_OVER_SOURCE,CURRENT_SOURCE,FILENAME_MAX);
      /* Get the innermost's list containing highlighted text */
      Tcl_Invoke(interp,".text.text","index","breakpoint.first - 1 chars",NULL);
      DebugSelectList(interp->result);
      Tcl_Invoke(interp,".text.text","index","sel.first",NULL);
      begin=IndexToAbsolute(interp->result);
      Tcl_Invoke(interp,".text.text","index","sel.last",NULL);
      end=IndexToAbsolute(interp->result);
      if ((begin < 0) || (end < 0))
	{
	  /* Try stepping in the highlighted text */
	  Tcl_Invoke(interp,".text.text","index","breakpoint.first",NULL);
	  begin=IndexToAbsolute(interp->result);
	  Tcl_Invoke(interp,".text.text","index","breakpoint.last",NULL);
	  end=IndexToAbsolute(interp->result);
	  if ((begin < 0) || (end < 0))
	    {
	      Tcl_Eval(interp,"tk_dialog .m Message \"No stepover possible\" {} -1 Ok");
	      return TCL_OK;
	    }
	  STEP_OVER_BEGIN=begin;
	  STEP_OVER_END=end;
	}
      else
	{
	  STEP_OVER_BEGIN=begin;
	  STEP_OVER_END=end;
	};
    }
  else
    {
      /* This is an existing stepping over step , check if we have reached the end of the list */
      Tcl_Invoke(interp,".text.text","search","-forwards",")","breakpoint.last + 1 chars",NULL);
      end=IndexToAbsolute(interp->result);
      if (end < 0)
	{
	  /* Stop stepping */
	  STEP_OVER_MODE=0;
	  STEP_MODE=0;
	  STEP_OVER_BEGIN=0;
	  STEP_OVER_END=0;
	}
      else if (end >= STEP_OVER_END)
	{
	  /* Stop stepping */
	  STEP_OVER_MODE=0;
	  STEP_MODE=0;
	  STEP_OVER_BEGIN=0;
	  STEP_OVER_END=0;
	};
    };
  Leave_Breakpoint();
  return TCL_OK;
}
/*
 * Set a tempory breakpoint and continue processing untill the next breakpoint
 */

int Debug_Next(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  /* Set a tempory breakpoint , for selected part */
  
  STEP_OVER_MODE=0;
  STEP_MODE=0;
  STEP_OVER_BEGIN=0;
  STEP_OVER_END=0;
  Set_Debug_Breakpoint(2);
  Leave_Breakpoint();
  return TCL_OK;
};

/*
 * Continue processing until the next breakpoint
 */

int Debug_Continue(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  STEP_OVER_MODE=0;
  STEP_MODE=0;
  STEP_OVER_BEGIN=0;
  STEP_OVER_END=0;
  Leave_Breakpoint();
  return TCL_OK;
};


/* 
 * Sets a watchpoint on a variable
 */

int Debug_Watch(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long begin,end;
  char *var_name;

  /* Check if there is a selection defining variable */
  Tcl_Invoke(interp,".text.text","tag","range","sel",NULL);
  if (interp->result[0]=='\0')
    return TCL_OK;
  /* Copy the variable to a string variable */
  Tcl_Invoke(interp,".text.text","get","sel.first","sel.last",NULL);
 
  write(client_sockfd,"(DEBUGGER::debug-set-watchpoint 0 0 0 \"",39);
  write(client_sockfd,CURRENT_SOURCE,strlen(CURRENT_SOURCE));
  write(client_sockfd,"\" \"",3);
  var_name=correct_string(interp->result);
  write(client_sockfd,var_name,strlen(var_name));
  free(var_name);
  write(client_sockfd,"\")\n",3);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };

  return TCL_OK;
}


/* 
 * Sets a watchpoint on a expression , may only be executed in the context of a range
 */

int Debug_Watch_Exp(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  long begin,end;
  char lispcommand[MAX_STRING+1];
  
  /* Check if there is a selection defining variable */
  Tcl_Invoke(interp,".text.text","tag","range","sel",NULL);
  if (interp->result[0]=='\0')
    return TCL_OK;
  /* Get begin and end of of selection */
  Tcl_Invoke(interp,".text.text","index","sel.first",NULL);
  begin=IndexToAbsolute(interp->result);
  Tcl_Invoke(interp,".text.text","index","sel.last",NULL);
  end=IndexToAbsolute(interp->result);
  
  /* Get the expression to be evaluated from the command window */
  Tcl_Invoke(interp,".command.text","get","1.0","end - 1 chars",NULL);

  snprintf(lispcommand,MAX_STRING,"(DEBUGGER::debug-set-watchpoint 1 %ld %ld \"%s\" \"%s\")\n",begin,end,CURRENT_SOURCE,interp->result);
  write(client_sockfd,lispcommand,strlen(lispcommand));
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };

  return TCL_OK;
}

/*
 * Evaluate a expression and display the result
 */

int Debug_Eval(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  /* Get the expressions to be evaluated */
  Tcl_Invoke(interp,".command.text","get","1.0","end -1 chars",NULL);
  /* Ask the client to display the expression */
  write(client_sockfd,"(DEBUGGER::display-result-exp \"",31);
  write(client_sockfd,interp->result,sizeof(char)*strlen(interp->result));
  write(client_sockfd,"\")\n",3);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}


/*
 * Evaluate a expression and return the result as the result of the code
 */

int Debug_Return_Result (ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  /* Get the expressions to be evaluated */
  Tcl_Invoke(interp,".command.text","get","1.0","end -1 chars",NULL);
  /* Ask the client to return this result as the result of evaluating the expression */
  write(client_sockfd,"(DEBUGGER::return-result-exp \"",30);
  write(client_sockfd,interp->result,sizeof(char)*strlen(interp->result));
  write(client_sockfd,"\")\n",3);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  return TCL_OK;
}
  
/*
 * Sets a new breakpoint in the debugger
 * 1 = permanent breakpoint
 * 2 = tempory breakpoint
 * 3 = conditional breakpoint
 */

int Debug_Breakpoint(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Set_Debug_Breakpoint(1);
  return TCL_OK;
}

/* 
 * Sets a conditional breakpoint 
 */

int Debug_Breakif(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Set_Debug_Breakpoint(3);
  return TCL_OK;
}


/* 
 * Stops the program executing , by sending a break to the program ,
 *  
 */

int Debug_Stop(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  write(client_sockfd,"(break \"Program interrupted by lispdebugger\")\n",46);
  if (PPID !=0)
    {
      kill(PPID,SIGUSR1);  /* Send signal to client , to indicate there is a command waiting */
    };

  return TCL_OK;
}

/* 
 * FUnction used as test to delete a item in the hash table if the source is equal to SOURCE 
 */

int Is_Source(char *key,void *data)
{
  DEBUGPOINT *p_debugpoint;
  p_debugpoint=(DEBUGPOINT *) data;
  if (strcmp(p_debugpoint->source,SOURCE)==0)
    {
      return 1;
    }
  else
    {
      return 0;
    };
}

/* 
 * Load a new source in the debugger 
 */

int Debug_Open(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  char SelectedFile[FILENAME_MAX+1];
  char *key,*dat; 

  /* The way CLOS is implemented in CLISP makes that the debug code is called during the load fase */
  /* To avoid stopping the load we must do a continue */

  STEP_OVER_MODE=0;
  STEP_MODE=0;
  STEP_OVER_BEGIN=0;
  STEP_OVER_END=0;

  Tcl_Eval(interp,"tk_getOpenFile -filetypes {{lisp {.lsp}} {lisp {.lisp}}} -title \"Select a file\" -parent .");
  if (interp->result[0]=='\0')
    return TCL_OK; /* Cancel action to select file */

  strncpy(SelectedFile,interp->result,FILENAME_MAX);
  SelectedFile[FILENAME_MAX]='\0';

  /* If the file existed already in the debug system , delete the file info */

  if (search(SelectedFile,L_SOURCES)!=NULL)
    {
      /* Delete breakpoint info */
      strncpy(SOURCE,SelectedFile,FILENAME_MAX);
      delete_hash_condition(H_DEBUGPOINT,Is_Source);
    };
      
  /* Send a command to the lisp system to handle the file */
  /* This should be the only place that the user interface is commanding the lisp system directly */

  write(client_sockfd,"(DEBUGGER::debug-open-file \"",28);
  write(client_sockfd,SelectedFile,strlen(SelectedFile));
  write(client_sockfd,"\"",1);
  if (COMPILE_CODE)
    write(client_sockfd," t ",3);
  write(client_sockfd,")\n",2);

  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };

  /* In the maintime read the file in the text area */

  Read_Source(interp,SelectedFile);

  /* Set the current source name */
  
  strncpy(CURRENT_SOURCE,SelectedFile,FILENAME_MAX);
  CURRENT_SOURCE[FILENAME_MAX]='\0';

  /* Register the source name , in a list of sources beeing debugged */
  if (search(SelectedFile,L_SOURCES)==NULL)
    {
      key=malloc((strlen(SelectedFile)+1)*sizeof(char));
      strcpy(key,SelectedFile);
      dat=malloc((strlen(SelectedFile)+1)*sizeof(char));
      strcpy(dat,SelectedFile);
      L_SOURCES=cons(key,dat,L_SOURCES);
    };

  /* Modify the sources menu , to display the new source */
  
  Change_Source_Menu();

  /* End of function */
  return TCL_OK;
}

/* 
 * FUnction used as test to delete a item in the hash table if the source is equal to CURRENT_SOURCE 
 */

int Is_Current(char *key,void *data)
{
  DEBUGPOINT *p_debugpoint;
  p_debugpoint=(DEBUGPOINT *) data;
  if (strcmp(p_debugpoint->source,CURRENT_SOURCE)==0)
    {
      return 1;
    }
  else
    {
      return 0;
    };
}

/*
 * Close the current source
 */

int Debug_Close(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  if (CURRENT_SOURCE[0]=='\0')
    return TCL_OK;
  /* Load original source in the lisp system */
  write(client_sockfd,"(load \"",7);
  write(client_sockfd,CURRENT_SOURCE,strlen(CURRENT_SOURCE));
  write(client_sockfd,"\")\n",3);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
  /* Remove breakpoint info of source */
  delete_hash_condition(H_DEBUGPOINT,Is_Current);
  /* Clear the source window */
  Tcl_Invoke(interp,".text.text","configure","-state","normal",NULL);
  Tcl_Invoke(interp,".text.text","delete","1.0","end",NULL);
  Tcl_Invoke(interp,".text.text","configure","-state","normal",NULL);
  /* Remove the CURRENT_SOURCE from source list */
  L_SOURCES=delete_all(CURRENT_SOURCE,L_SOURCES);
  /* If there is still a source in the L_SOURCES list , display the first one and make this current */
  if (L_SOURCES!=NULL)
    {
      strcpy(CURRENT_SOURCE,L_SOURCES->key);
      Read_Source(interp,CURRENT_SOURCE);
    }
  else
    {
      /* Set CURRENT SOURCE on NULL */
      CURRENT_SOURCE[0]='\0';
    };

  /* Modify the sources menu , to display the new source */
  
  Change_Source_Menu();
  return TCL_OK;

}

/*
 * Save the settings of the debugger
 */

void Save_Setting()
{
  int fd;
  char string[MAX_STRING+1],*file;

  snprintf(string,MAX_STRING,"%s/.lispdebug.lisp",getenv("HOME"));
  fd=open(string,O_WRONLY | O_CREAT | O_TRUNC,S_IRWXU);
  if (fd == -1)
    {
      Tcl_Eval(interp,"tk_dialog .m Message \"Can't create personal configuration file\" {} -1 Ok");
    };

  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"COMPILE_CODE\" %ld)\n",COMPILE_CODE);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"SAVE_ON_EXIT\" %ld)\n",SAVE_ON_EXIT);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"CHECK_ERROR\" %ld)\n",CHECK_ERROR);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"DEBUG_MACRO\" %ld)\n",DEBUG_MACRO);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"DISPLAY_PREVIOUS\" %ld)\n",DISPLAY_PREVIOUS);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"DEBUGPOINT_COLOR\" \"%s\")\n",DEBUGPOINT_COLOR);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"DEBUGPOINTIF_COLOR\" \"%s\")\n",DEBUGPOINTIF_COLOR);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"CURRENT_COLOR\" \"%s\")\n",CURRENT_COLOR);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"AFTER_COLOR\" \"%s\")\n",AFTER_COLOR);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"PROFILE_COLOR\" \"%s\")\n",PROFILE_COLOR);
  write(fd,string,strlen(string));
  snprintf(string,MAX_STRING,"(DEBUGGER::setting \"FONT\" \"%s\")\n",FONT);
  write(fd,string,strlen(string));
  close(fd);

}

int Debug_Save_Setting(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  Save_Setting();
  return TCL_OK;
}

/*
 * Find next and previous entries
 */

int Find_Next(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  char end[MAX_POS+1];
  char *search;

  Tcl_Invoke(interp,".find.entry.find","get",NULL);
  search=malloc(sizeof(char)*(strlen(interp->result)+1));
  strcpy(search,interp->result);

  switch (SEARCH_TYPE)
    {
    case 1:
      Tcl_Invoke(interp,".text.text","search","-exact",search,FIND_POS,NULL);
      break;
    case 2:
      Tcl_Invoke(interp,".text.text","search","-nocase",search,FIND_POS,NULL);
      break;
    case 3:
      Tcl_Invoke(interp,".text.text","search","-regexp",search,FIND_POS,NULL);
      break;
    };
  
  free(search);

  if (interp->result[0]!='\0')
    {
      /* Find occurence , mark beginning */
      strncpy(FIND_POS,interp->result,MAX_POS);
      FIND_POS[MAX_POS]='\0';
      snprintf(end,MAX_POS,"%s + 1 chars",FIND_POS);
      Tcl_Invoke(interp,".text.text","tag","delete","find",NULL);
      Tcl_Invoke(interp,".text.text","tag","add","find",FIND_POS,end,NULL);
      Tcl_Invoke(interp,".text.text","tag","configure","find","-background","red",NULL);
      Tcl_Invoke(interp,".text.text","see",FIND_POS,NULL);
      strncpy(FIND_POS,end,MAX_POS);
      FIND_POS[MAX_POS]='\0';
    }
  else
    {
      /* Found nothing , display error */
      Tcl_Eval(interp,"tk_dialog .m Message \"Found no occurence of the string\" {} -1 Ok");
    };

  return TCL_OK;
}

int Find_Previous(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  char end[MAX_POS+1];
  char * search;

  Tcl_Invoke(interp,".find.entry.find","get",NULL);
  search=malloc(sizeof(char)*(strlen(interp->result)+1));
  strcpy(search,interp->result);

  switch (SEARCH_TYPE)
    {
    case 1:
      Tcl_Invoke(interp,".text.text","search","-backward","-exact",search,FIND_POS,NULL);
      break;
    case 2:
      Tcl_Invoke(interp,".text.text","search","-backward","-nocase",search,FIND_POS,NULL);
      break;
    case 3:
      Tcl_Invoke(interp,".text.text","search","-backward","-regexp",search,FIND_POS,NULL);
      break;
    };

  free(search);

  if (interp->result[0]!='\0')
    {
      /* Find occurence , mark beginning */
      strncpy(FIND_POS,interp->result,MAX_POS);
      FIND_POS[MAX_POS]='\0';
      snprintf(end,MAX_POS,"%s + 1 chars",FIND_POS);
      Tcl_Invoke(interp,".text.text","tag","delete","find",NULL);
      Tcl_Invoke(interp,".text.text","tag","add","find",FIND_POS,end,NULL);
      Tcl_Invoke(interp,".text.text","tag","configure","find","-background","red",NULL);
      Tcl_Invoke(interp,".text.text","see",FIND_POS,NULL);
    }
  else
    {
      /* Found nothing , display error */
      Tcl_Eval(interp,"tk_dialog .m Message \"Found no occurence of the string\" {} -1 Ok");
    };

  return TCL_OK;
}

/*
 * Change the source displayed in a window because the user aksed it
 */

int Load_Source(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  strncpy(CURRENT_SOURCE,Tcl_GetStringFromObj(objv[1],NULL),FILENAME_MAX);
  CURRENT_SOURCE[FILENAME_MAX]='\0';
  Read_Source(interp,CURRENT_SOURCE);
}
  

/*
 * Find in the source window to a specific text
 */
int Debug_Find(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])
{
  /* Display the find window */

  SEARCH_TYPE=2;

  Tcl_Eval(interp,"toplevel .find");
  Tcl_Eval(interp,"frame .find.entry");
  Tcl_Eval(interp,"frame .find.radio");
  Tcl_Eval(interp,"frame .find.button");
  Tcl_Eval(interp,"pack .find.entry .find.radio .find.button -fill both -expand true");
  Tcl_Eval(interp,"label .find.entry.lb -text \"Find : \"");
  Tcl_Eval(interp,"entry .find.entry.find");
  Tcl_Eval(interp,"pack .find.entry.lb .find.entry.find -fill both -side left -expand true");
  Tcl_Eval(interp,"radiobutton .find.radio.case -text Case -value 1 -variable **search-type**");
  Tcl_Eval(interp,"radiobutton .find.radio.nocase -text \"No case\" -value 2 -variable **search-type**");
  Tcl_Eval(interp,"radiobutton .find.radio.regular -text \"Regular expression\" -value 3 -variable **search-type**");
  Tcl_Eval(interp,"pack .find.radio.case .find.radio.nocase .find.radio.regular -fill both -side left");
  Tcl_Eval(interp,"button .find.button.next -text Next -command find-next");
  Tcl_Eval(interp,"button .find.button.previous -text Previous -command find-previous");
  Tcl_Eval(interp,"pack .find.button.next .find.button.previous -side left");

  /* Initialise find position */
  strcpy(FIND_POS,"1.0");

  return TCL_OK;
    
}


/* Function that gets called during the destroy of the top level window , should stop program */
void QuitProgramWalk(char *key,char *source)
{ 
  /* Load the original sources back in the lisp system */
  write(client_sockfd,"(load \"",7);
  write(client_sockfd,source,strlen(source));
  write(client_sockfd,"\")\n",3);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };
}
  
int QuitProgram(ClientData data,Tcl_Interp *interp,int objc,Tcl_Obj *CONST objv[])

{ 
  /* Load the orignal sources back */
  walk_list(L_SOURCES,QuitProgramWalk);
  
  /* Delete all tempfiles */

  system("rm /usr/tmp/lisp-debugger*.lisp 2> /dev/null");

  /* IF asked save settings */
  if (SAVE_ON_EXIT)
    Save_Setting();

  /* Instruct the client to stop the interface */
  write(client_sockfd,"(DEBUGGER::stop-interface)\n",27);
  if (PPID != 0)
    {
      kill(PPID,SIGUSR1);  /* Send a signal to client , to indicate there is a command waiting */
    };

  /* Close the connection with client , block if no unreceived data */
  close(client_sockfd);
  close(server_sockfd);
  exit(0);
  return TCL_OK;
}



/*
 * lisp_command_handler , handles the command send by the lisp system
 */

void lisp_command_handler(ClientData data,int mask)
{
  int function_no;
  char dummy[1];
  /* 
   * Commands from the lisp system is in the form "function-nr [args]*" , 
   * the number and type of the args is determined by the function itself 
   */
  while (input_waiting()>0)
    {
      function_no=read_number();
      switch (function_no)
	{
	case 0:
	  highlight_source();
	  break;
	case 1:
	  display_message();
	  break;
	case 2:
	  highlight_error();
	  break;
	case 3:
	  set_possible_breakpoint();
	  break;
	case 4:
	  give_control_to_interface(0); /* give-control-to-interface , before call */
	  break;
	case 5:
	  display_result();
	  break;
	case 6:
	  display_exp_in_interface();
	  break;
	case 7:
	  if_breakpoint();
	  break;
	case 8:
	  display_time_env();
	  break;
	case 9:
	  setting();
	  break;
	case 10:
	  give_control_to_interface(1); /* give-control-to-interface , after call */
	  break;
	case 11:
	  lisp_goes_to_debug();
	  break;
	default:
	  break;
	};
    };
}
      

 /* 
 * ---------------------------------------------------------------------
 * Create the debugger form layout
 * ---------------------------------------------------------------------
 */

int CreateDebugWindow_Init (interp)
     Tcl_Interp *interp;
{

  /* Link of variables used here */
  
  Tcl_LinkVar(interp,"**compile-code**",(char *) &COMPILE_CODE,TCL_LINK_BOOLEAN);
  Tcl_LinkVar(interp,"**save-on-exit**",(char *) &SAVE_ON_EXIT,TCL_LINK_BOOLEAN);
  Tcl_LinkVar(interp,"**check-error**",(char *) &CHECK_ERROR,TCL_LINK_BOOLEAN);
  Tcl_LinkVar(interp,"**debug-macro**",(char *) &DEBUG_MACRO,TCL_LINK_BOOLEAN);
  Tcl_LinkVar(interp,"**display-result-call**",(char *) &DISPLAY_PREVIOUS,TCL_LINK_BOOLEAN);
  Tcl_LinkVar(interp,"**search-type**",(char *) &SEARCH_TYPE,TCL_LINK_INT);

  /* TCL-TK Code */
  
  /*  Create the frames needed */
  Tcl_Eval(interp,"frame .menu");
  Tcl_Eval(interp,"frame .button1");
  Tcl_Eval(interp,"frame .button2");
  Tcl_Eval(interp,"frame .text");
  Tcl_Eval(interp,"frame .command");
  Tcl_Eval(interp,"frame .result");
  Tcl_Eval(interp,"pack .menu -expand no -fill both");
  Tcl_Eval(interp,"pack .button1 -expand no -fill both");
  Tcl_Eval(interp,"pack .button2 -expand no -fill both");
  Tcl_Eval(interp,"pack .text -expand yes -fill both");
  Tcl_Eval(interp,"pack .result -expand yes -fill both");
  Tcl_Eval(interp,"pack .command -expand yes -fill both");
  /* Add some grip above and below the result window , to change the relative sides of the windows */
  Tcl_Eval(interp,"frame .grip1 -background grey -width 10 -height 10 -bd 1 -relief raised -cursor crosshair");
  Tcl_Eval(interp,"frame .grip2 -background grey -width 10 -height 10 -bd 1 -relief raised -cursor crosshair");
  Tcl_Eval(interp,"place .grip1 -in .result -anchor c -relx 0.1");
  Tcl_Eval(interp,"place .grip2 -in .command -anchor c -relx 0.1");
  /* Create the different widgets */
  Tcl_Eval(interp,"menubutton .menu.file -text File -underline 0 -menu .menu.file.file");
  Tcl_Eval(interp,"menubutton .menu.source -text Source -underline 0 -menu .menu.source.source");
  Tcl_Eval(interp,"menubutton .menu.edit -text Edit -underline 0 -menu .menu.edit.edit");
  Tcl_Eval(interp,"menubutton .menu.options -text Options -underline 0 -menu .menu.options.options");
  Tcl_Eval(interp,"menubutton .menu.tools -text Tools -underline 0 -menu .menu.tools.tools");
  Tcl_Eval(interp,"menu .menu.file.file -tearoff 0");
  Tcl_Eval(interp,"menu .menu.source.source -tearoff 1");
  Tcl_Eval(interp,"menu .menu.edit.edit -tearoff 0");
  Tcl_Eval(interp,"menu .menu.options.options -tearoff 0");
  Tcl_Eval(interp,"menu .menu.tools.tools -tearoff 0");
  Tcl_Eval(interp,".menu.file.file add command -label Open -command debug-open");
  Tcl_Eval(interp,".menu.file.file add command -label Close -command debug-close");
  Tcl_Eval(interp,".menu.file.file add separator");
  Tcl_Eval(interp,".menu.file.file add command -label Exit -command quit_program");
  Tcl_Eval(interp,".menu.edit.edit add command -label Paste -command debug-paste");
  Tcl_Eval(interp,".menu.edit.edit add command -label Find -command debug-find");
  Tcl_Eval(interp,".menu.options.options add command -label \"Color Break\" -command debug-color-break");
  Tcl_Eval(interp,".menu.options.options add command -label \"Color After Break\" -command debug-color-after");
  Tcl_Eval(interp,".menu.options.options add command -label \"Color BreakPoint\" -command debug-color-breakpoint");
  Tcl_Eval(interp,".menu.options.options add command -label \"Color Breakpoint If\" -command debug-color-breakpointif");
  Tcl_Eval(interp,".menu.options.options add command -label \"Color Profiling\" -command debug-color-profile");
  Tcl_Eval(interp,".menu.options.options add separator");
  Tcl_Eval(interp,".menu.options.options add command -label Font -command debug-font");
  Tcl_Eval(interp,".menu.options.options add separator");
  Tcl_Eval(interp,".menu.options.options add check -label \"Compile debugged code\" -variable **compile-code**");
  Tcl_Eval(interp,".menu.options.options add separator");
  Tcl_Eval(interp,".menu.options.options add check -label \"Display result last call\" -variable **display-result-call** -command debug-display-previous-call");
  Tcl_Eval(interp,".menu.options.options add separator");
  Tcl_Eval(interp,".menu.options.options add check -label \"Enter debugger in case of error\" -variable **check-error** -command debug-check-error");
  Tcl_Eval(interp,".menu.options.options add separator");
  Tcl_Eval(interp,".menu.options.options add check -label \"Debug macro's\" -variable **debug-macro** -command debug-debug-macro");
  Tcl_Eval(interp,".menu.options.options add separator");
  Tcl_Eval(interp,".menu.options.options add command -label \"Save options\" -command debug-save-setting");
  Tcl_Eval(interp,".menu.options.options add separator");
  Tcl_Eval(interp,".menu.options.options add check -label \"Save on exit\" -variable **save-on-exit**");
  Tcl_Eval(interp,".menu.tools.tools add command -label \"Start profiling\" -command debug-start-profile");
  Tcl_Eval(interp,".menu.tools.tools add command -label \"Stop profiling\" -command debug-stop-profile");
  Tcl_Eval(interp,".menu.tools.tools add command -label \"Interrupt program\" -command debug-stop");
  Tcl_Eval(interp,"pack .menu.file .menu.source .menu.edit .menu.options .menu.tools -side left");
  /* Definitions of buttons */
  Tcl_Eval(interp,"button .button1.step -text Step -width 10 -command debug-step");
  Tcl_Eval(interp,"button .button1.stepover -text \"Step over\" -width 10 -command debug-step-over");
  Tcl_Eval(interp,"button .button1.next -text Next -width 10 -command debug-next");
  Tcl_Eval(interp,"button .button1.continue -text Continue -width 10 -command debug-continue");
  Tcl_Eval(interp,"button .button1.breakpoint -text Breakpoint -width 10 -command debug-breakpoint");
  Tcl_Eval(interp,"button .button1.breakif -text \"Break If\" -width 10 -command debug-breakif");
  Tcl_Eval(interp,"button .button2.watch -text Watch -width 10 -command debug-watch");
  Tcl_Eval(interp,"button .button2.watchexp -text \"Watch exp\" -width 10 -command debug-watch-exp");
  Tcl_Eval(interp,"button .button2.return -text \"Change Result\" -width 10 -state disabled -command debug-return-result");
  Tcl_Eval(interp,"button .button2.eval -text Eval -width 10 -command debug-eval");
  Tcl_Eval(interp,"button .button2.back -text Back -width 10 -command debug-back");
  Tcl_Eval(interp,"button .button2.forward -text Forward -width 10 -command debug-forward");
  Tcl_Eval(interp,"pack .button1.step .button1.stepover .button1.next .button1.continue .button1.breakpoint .button1.breakif -side left");
  Tcl_Eval(interp,"pack .button2.watch .button2.watchexp .button2.return .button2.eval .button2.back .button2.forward -side left");
  /* Define scrollbars for the 3 main windows and the text windows themself */
  Tcl_Eval(interp,"scrollbar .text.xtext -command \".text.text xview\" -orient horizontal");
  Tcl_Eval(interp,"scrollbar .text.ytext -command \".text.text yview\"");
  Tcl_Eval(interp,"text .text.text -xscrollcommand \".text.xtext set\" -yscrollcommand \".text.ytext set\" -state disabled");
  Tcl_Eval(interp,"pack .text.xtext -side bottom -fill x");
  Tcl_Eval(interp,"pack .text.text -side left -expand yes -fill both");
  Tcl_Eval(interp,"pack .text.ytext -side left -fill y");

  Tcl_Eval(interp,"scrollbar .command.xtext -command \".command.text xview\" -orient horizontal");
  Tcl_Eval(interp,"scrollbar .command.ytext -command \".command.text yview\"");
  Tcl_Eval(interp,"text .command.text -xscrollcommand \".command.xtext set\" -yscrollcommand \".command.ytext set\" -state normal -height 3");
  Tcl_Eval(interp,"pack .command.xtext -side bottom -fill x");
  Tcl_Eval(interp,"pack .command.text -side left -expand yes -fill both");
  Tcl_Eval(interp,"pack .command.ytext -side left -fill y");

  Tcl_Eval(interp,"scrollbar .result.xtext -command \".result.text xview\" -orient horizontal");
  Tcl_Eval(interp,"scrollbar .result.ytext -command \".result.text yview\"");
  Tcl_Eval(interp,"text .result.text -xscrollcommand \".result.xtext set\" -yscrollcommand \".result.ytext set\" -state disabled -height 3");
  Tcl_Eval(interp,"pack .result.xtext -side bottom -fill x");
  Tcl_Eval(interp,"pack .result.text -side left -expand yes -fill both");
  Tcl_Eval(interp,"pack .result.ytext -side left -fill y");

  /* Change font */
  Tcl_Invoke(interp,".text.text","configure","-font",FONT,NULL);
  Tcl_Invoke(interp,".result.text","configure","-font",FONT,NULL);
  Tcl_Invoke(interp,".command.text","configure","-font",FONT,NULL);

  /* Set default color for selection */

  Tcl_Invoke(interp,".text.text","tag","configure","-background","grey",NULL);

  /* Create of bindings to commands  */

  Tcl_Eval(interp,"bind .text.text <Double-ButtonRelease-1> {select-list %x %y}");
  Tcl_Eval(interp,"bind .text.text <ButtonRelease-1> {select-word %x %y}");
  Tcl_Eval(interp,"bind .text.text <ButtonPress-2> {select-function %x %y}");
  Tcl_Eval(interp,"bind .result.text <ButtonPress-1> {select-result %x %y}");

  

  /* Bindings to the grip windows */

  Tcl_Eval(interp,"bind .grip1 <ButtonPress-1> {resize-window 0 %Y}"); /* Initiate resizing operation */
  Tcl_Eval(interp,"bind .grip1 <B1-Motion> {resize-window 1 %Y}");     /* Resize the text/result pane */
  Tcl_Eval(interp,"bind .grip2 <ButtonPress-1> {resize-window 0 %Y}"); /* Initiate resizing operation */
  Tcl_Eval(interp,"bind .grip2 <B1-Motion> {resize-window 2 %Y}");     /* Resize the result/command pane */

  /* Create of c functions as commands */
  Tcl_CreateObjCommand(interp,"debug-open",Debug_Open,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-breakpoint",Debug_Breakpoint,(ClientData) 0 ,(Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-breakif",Debug_Breakif,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-stop",Debug_Stop,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-step",Debug_Step,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-step-over",Debug_Step_Over,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-next",Debug_Next,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-continue",Debug_Continue,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-watch",Debug_Watch,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-watch-exp",Debug_Watch_Exp,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-eval",Debug_Eval,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-return-result",Debug_Return_Result,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"select-list",Select_List,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"select-word",Select_Word,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"select-function",Select_Function,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"select-result",Select_Result,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"resize-window",Resize_Window,(ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-color-break",Debug_Color_Break, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-color-after",Debug_Color_After, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-color-breakpoint",Debug_Color_Breakpoint, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-color-breakpointif",Debug_Color_Breakpointif, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-color-profile",Debug_Color_Profile, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-font",Debug_Font, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-display-previous-call",Debug_Display_Previous_Call, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-check-error",Debug_Check_Error, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-debug-macro",Debug_Debug_Macro, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-paste",Debug_Paste, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-start-profile",Debug_Start_Profile, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-stop-profile",Debug_Stop_Profile, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-show-profile-scale",Debug_Show_Profile_Scale, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-back",Debug_Back, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-forward",Debug_Forward, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-close",Debug_Close, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-save-setting",Debug_Save_Setting, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"debug-find",Debug_Find, (ClientData) 0, (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"find-next",Find_Next,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"find-previous",Find_Previous,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"load-source",Load_Source,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  Tcl_CreateObjCommand(interp,"attach-watchpoint",Attach_Watchpoint,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);
  return 0;
}
/*
 *----------------------------------------------------------------------
 *
 * main --
 *
 *	This is the main program for the application.
 *
 * Results:
 *	None: Tk_Main never returns here, so this procedure never
 *	returns either.
 *
 * Side effects:
 *	Whatever the application does.
 *
 *----------------------------------------------------------------------
 */

int main(argc, argv)
     int argc;			/* Number of command-line arguments. */
     char **argv;		/* Values of command-line arguments. */
{
  int server_len,client_len;
  struct sockaddr_un server_address;
  struct sockaddr_un client_address;
  struct sockaddr_in i_server_address;
  struct sockaddr_in i_client_address;
  int port,on;


  /* Process arguments */

  if (argc!=1 && argc!=2)
    {
      fprintf(stderr,"interface started with wrong number of arguments\n");
      exit -1;
    };

  if (argc==2)
    {
      sscanf(argv[1],"%d",&PPID);
    }
  else
    {
      PPID=0;
    };

  /* Check if PID is port nr or pid */
  if (PPID<0)
    {
      port=-PPID;
      PPID=0;
    }
  else
    {
      port=0;
    };
     
  /* Initialise tree of breakpoints */
  
  H_DEBUGPOINT=create_hash(5000);

  /* Set up sockets */

  /* Remove any old trace of sockets */
  unlink("/tmp/lispdebugger");
  /* Create socket and name it */
  if (port==0)
    {
      server_sockfd = socket(AF_UNIX,SOCK_STREAM,0);
      server_address.sun_family=AF_UNIX;
      strcpy(server_address.sun_path,"/tmp/lispdebugger");
      server_len=sizeof(server_address);
      bind(server_sockfd,(struct sockaddr *)&server_address,server_len);
    }
  else
    {
      /* Create a socket */
      server_sockfd = socket(AF_INET,SOCK_STREAM,0);
      /* Set a socket options to reuse socket */
      on=1;
      if (setsockopt(server_sockfd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof(int))==-1)
	{
	  fprintf(stderr,"Failed to set option because of error %ld\n",errno);
	  fflush(stderr);
	  exit(0);
	};
      /* Set address */
      i_server_address.sin_family = AF_INET;
      i_server_address.sin_addr.s_addr = htonl(INADDR_ANY);
      i_server_address.sin_port = htons((ushort) port);
      server_len=sizeof(server_address);
      if (bind(server_sockfd,(struct sockaddr *)&i_server_address,server_len)<0)
	{
	  fprintf(stderr,"Failed to bind because of error %ld\n",errno);
	  fflush(stderr);
	  exit(0);
	};
    };
  /* create a connection queue and wait for client */
  listen(server_sockfd,1);
  client_sockfd=accept(server_sockfd,(struct sockaddr *)&client_address,&client_len);
  
  /* starts TCL/TK stuff */

  Tk_Main(1,argv,Tcl_AppInit); /* Use a argc of 1 to let Tk_Main not interpret the args */


  return 0;			/* Needed only to prevent compiler warning. */
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_AppInit --
 *
 *	This procedure performs application-specific initialization.
 *	Most applications, especially those that incorporate additional
 *	packages, will have their own version of this procedure.
 *
 * Results:
 *	Returns a standard Tcl completion code, and leaves an error
 *	message in interp->result if an error occurs.
 *
 * Side effects:
 *	Depends on the startup script.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_AppInit(local_interp)
    Tcl_Interp *local_interp;		/* Interpreter for application. */
{
    interp=local_interp;
    if (Tcl_Init(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }
    if (Tk_Init(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }
    Tcl_StaticPackage(interp, "Tk", Tk_Init, Tk_SafeInit);


    /*
     * Call the init procedures for included packages.  Each call should
     * look like this:
     *
     * if (Mod_Init(interp) == TCL_ERROR) {
     *     return TCL_ERROR;
     * }
     *
     * where "Mod" is the name of the module.
     */

    /* Specify a function to be executed it the top level window is destroyed */
    
    Tcl_Eval(interp,"wm protocol . WM_DELETE_WINDOW quit_program");
    Tcl_CreateObjCommand(interp,"quit_program",QuitProgram,(ClientData) 0 , (Tcl_CmdDeleteProc *) 0);

    /* Create the debugger form , and display it */

    if (CreateDebugWindow_Init(interp) == TCL_ERROR) {
      return TCL_ERROR;
    };

    /* Register a function to read the commands from lisp on the incoming pipe */

    Tcl_CreateFileHandler(client_sockfd,TCL_READABLE,lisp_command_handler,(ClientData) 0);

    return TCL_OK;
			 
			  
    /*
     * Call Tcl_CreateCommand for application-specific commands, if
     * they weren't already created by the init procedures called above.
     */

    /*
     * Specify a user-specific startup file to invoke if the application
     * is run interactively.  Typically the startup file is "~/.apprc"
     * where "app" is the name of the application.  If this line is deleted
     * then no user-specific startup file will be run under any conditions.
     */

    /*  Tcl_SetVar(interp, "tcl_rcFileName", "~/.wishrc", TCL_GLOBAL_ONLY);
    return TCL_OK;
    */
}
