;; ***************************************************************************************************
;; Latest  version of debugger , optimised for speed
;; ***************************************************************************************************

;; **************************************************************************************************
;; Package stuff
;; **************************************************************************************************



(when (not (find-package "DEBUGGER")) (make-package "DEBUGGER"))
(in-package "DEBUGGER")

#+:CMU
(use-package "DEBUG")
#+:CMU
(use-package "DEBUG-INTERNALS")
#+:CMU
(declaim (optimize (speed 3) (safety 0)) (extensions:optimize-interface (speed 3)))
#+gcl
(declaim (optimize (speed 3) (safety 0)))

;; ***************************************************************************************************
;; Code of various tools
;; ***************************************************************************************************

;; ********************************************************************************************
;; Different tools to be used in the lisp environment
;; ********************************************************************************************

;; ********************************************************************************************
;; Packages used
;; ********************************************************************************************


(use-package "TK")

;; ********************************************************************************************
;; Global var used
;; ********************************************************************************************

(defparameter **directory** "/")
(defparameter **extension** "*")
(defparameter **selection** "")
(defparameter **selected-dir** "")
(defparameter **callback** NIL)
(defparameter **color** NIL)
(defparameter **font** NIL)

;; ******************************************************************************************
;; constants used
;; ******************************************************************************************

(defparameter **null-output** NIL) ;; Handle to null output


;; *******************************************************************************************
;; Let's you choose a file using a listbox
;; *******************************************************************************************

(defun choose-file (callback &key (directory) (extension))
  (let ((dummy NIL) (ok NIL))
    (setf **callback** callback)
    (multiple-value-setq (dummy ok) (TK::winfo :exist "."))
    (when (null ok) (TK::tkconnect))
    (choose-file-1 directory extension)))


(defun choose-file-1 (directory extension)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    ;; Create a new toplevel window , with the name of the directory to search , grab the focus
    (when (= (TK::winfo :exist ".file" :return 'number) 1)
      (return-from choose-file-1))
    (tk::toplevel '.file)
    (tk::grab :set '.file)
    ;; Create the different frames for the window
    (tk::frame '.file.button)
    (tk::frame '.file.entry)
    (tk::frame '.file.list)
    (tk::frame '.file.list.folder)
    (tk::frame '.file.list.content)
    (tk::pack '.file.button '.file.entry '.file.list :fill 'both)
    (tk::pack '.file.list :fill 'both :expand 'yes)
    (tk::pack '.file.list.folder '.file.list.content :fill 'both :expand 'yes :side 'left)
    
    ;; Create the different widgets
    (tk::button '.file.button.ok :text "Ok" :command '(choose-file-ok) :state "disabled")
    (tk::button '.file.button.cancel :text "Cancel" :command '(choose-file-cancel) :state "disabled")
    (tk::pack '.file.button.ok '.file.button.cancel :fill 'both :expand 'yes :side 'left)
    
    (tk::label '.file.entry.lbselect :text "Select : ")
    (tk::entry '.file.entry.select)
    (tk::label '.file.entry.lbextension :text "Extension : ")
    (tk::entry '.file.entry.extension :textVariable '**extension**)
    (tk::pack '.file.entry.lbselect '.file.entry.select '.file.entry.lbextension '.file.entry.extension :fill 'both :expand 'yes :side 'left)
    
    (tk::listbox '.file.list.folder.lb :yscrollcommand ".file.list.folder.sb set")
    (tk::listbox '.file.list.content.lb :yscrollcommand ".file.list.content.sb set")
    (tk::scrollbar '.file.list.folder.sb :command ".file.list.folder.lb yview")
    (tk::scrollbar '.file.list.content.sb :command ".file.list.content.lb yview")
    (tk::pack '.file.list.folder.lb :fill 'both :expand 'yes :side 'left)
    (tk::pack '.file.list.folder.sb :side 'left :fill 'y :expand 'yes)
    (tk::pack '.file.list.content.lb :fill 'both :expand 'yes :side 'left)
    (tk::pack '.file.list.content.sb :side 'left :fill 'y :expand 'yes)
    
    ;; Initialise variables
    (when directory (setf **directory** directory))
    (if extension (setf **extension** extension) (setf **extension** "*"))
    (TK::wm :title '.file **directory**)
    (setf **selection** "")
    
    ;; Fill the listbox
    (choose-file-fill-folder)

    ;; Set bindings
    
    (TK::bind '.file.list.folder.lb "<Double-ButtonPress-1>" '(choose-file-new-dir))
    (TK::bind '.file.list.folder.lb "<ButtonPress-1>" '(choose-file-select-dir))
    (TK::bind '.file.list.content.lb "<ButtonPress-1>" '(choose-file-select-file))
    (TK::bind '.file.list.content.lib "<Double-ButtonPress-1>" '(choose-file-select-file-exit))
    ;; Enable buttons
    (.file.button.ok :configure :state "normal")
    (.file.button.cancel :configure :state "normal")
    ))

;; Process the double click on a directory

(defun choose-file-new-dir ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **directory** (namestring (truename (format NIL "~A~A" (choose-file-dir **directory**) (tk::selection :get)))))
    (TK::wm :title '.file **directory**)
    (choose-file-fill-folder)))

(defun choose-file-select-dir ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (let ((select (.file.list.folder.lb :curselection :return 'list)))
      (setf **selected-dir** (first select))
      (choose-file-fill-content (.file.list.folder.lb :get **selected-dir** :return 'string)))))

(defun choose-file-select-file ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (let ((select (tk::selection :get)))
      (.file.list.folder.lb :select :clear)
      (.file.list.folder.lb :select :set **selected-dir**)
      (setf **selection** (namestring (truename (format NIL "~A~A/~A" (choose-file-dir **directory**)
							(.file.list.folder.lb :get **selected-dir** :return 'string)
							select))))
      (.file.entry.select :delete 0 'end)
      (.file.entry.select :insert 0 (file-namestring **selection**)))))
  

(defun choose-file-select-file-exit ()
  (choose-file-select-file)
  (choose-file-ok))


;; Process Ok button
(defun choose-file-ok ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (.file :configure :cursor "watch")
    (funcall **callback**  **selection**)
    (.file :configure :cursor "")
    (tk::destroy '.file)))

;; Process Cancel button
(defun choose-file-cancel ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **selection** "")
    (tk::destroy '.file)))

;; Add a / to a directory (if needed)

(defun choose-file-dir (dir)
  (cond ((string= dir "/") dir)
	(T (format NIL "~A/" dir))))

;; Fill the folder box
#+gcl
(defun choose-file-fill-folder ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (.file.list.folder.lb :delete 0 'end)
    (.file.list.folder.lb :insert 'end ".")
    (.file.list.folder.lb :insert 'end "..")
    (mapc #'(lambda (dir)
	      (when (directory-p dir) (.file.list.folder.lb :insert 'end (file-namestring dir))))
	  (sort (directory (choose-file-dir **directory**)) #'(lambda (x y) (string<= (namestring x) (namestring y)))))
    (.file.list.folder.lb :select :clear)
    (.file.list.folder.lb :select :set 0)
    (setf **selected-dir** 0)
    (choose-file-fill-content (.file.list.folder.lb :get 0 :return 'string))))

#+:CMU
(defun choose-file-fill-folder ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (.file.list.folder.lb :delete 0 'end)
    (.file.list.folder.lb :insert 'end ".")
    (.file.list.folder.lb :insert 'end "..")
    (mapc #'(lambda (dir)
	      (when (directory-p dir)
		(setf dir (namestring dir))
		(.file.list.folder.lb :insert 'end (file-namestring (subseq dir 0 (1- (length dir)))))))
	  (sort (directory (choose-file-dir **directory**)) #'(lambda (x y) (string<= (namestring x) (namestring y)))))
    (.file.list.folder.lb :select :clear)
    (.file.list.folder.lb :select :set 0)
    (setf **selected-dir** 0)
    (choose-file-fill-content (.file.list.folder.lb :get 0 :return 'string))))

;; Fill the content box

(defun choose-file-fill-content (dir)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (.file.list.content.lb :delete 0 'end)
    (mapc #'(lambda (dir)
	      (when (not (directory-p dir)) (.file.list.content.lb :insert 'end (file-namestring dir))))
	  (sort (directory (format NIL "~A~A/~A" (choose-file-dir **directory**) dir **extension**))
		#'(lambda (x y) (string<= (namestring x) (namestring y)))))))



;; Test if a file is a directory (systemspecific)

#+gcl
(defun directory-p (path)
  (setf path (concatenate 'string (namestring path) "/."))
  (probe-file path))

#+:CMU
(defun directory-p (path)
  (eq (unix:unix-file-kind (namestring path)) :directory))

;; *****************************************************************************************************
;; Select a color
;; *****************************************************************************************************

(defun choose-color (callback color)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **callback** callback)
    (setf **color** color)
    (TK::toplevel '.color)
    (TK::grab '.color)
    (TK::frame '.color.button)
    (TK::frame '.color.list)
    (TK::pack '.color.button '.color.list :side "top")
    (TK::button '.color.button.ok :text "Ok" :command '(choose-color-ok) :state "disabled")
    (TK::button '.color.button.cancel :text "Cancel" :command '(choose-color-cancel) :state "disabled")
    (TK::pack '.color.button.ok '.color.button.cancel :side "left")
    (TK::scrollbar '.color.list.scroll :command ".color.list.list yview")
    (TK::text '.color.list.list :yscrollcommand ".color.list.scroll set")
    (TK::pack '.color.list.list :in '.color.list :side "left")
    (TK::pack '.color.list.scroll :in '.color.list :side "left" :expand "yes" :fill "both")
    (TK::bind '.color.list.list "<ButtonPress-1>" '(choose-color-select-line TK::%X TK::%Y))
    (.color.list.list :configure :cursor "watch")
    (let ((handle (open "/usr/lib/X11/rgb.txt" :direction :input)) (color NIL) (str) (pos 0) (color-lst NIL) (max 0) (col 0))
      (declare (fixnum pos) (fixnum max) (fixnum col))
      (loop (when (null (setf str (read-line handle NIL NIL))) (return))
	    (cond ((char= (elt str 0) #\!))
		  (T (setf pos 0)
		     (multiple-value-setq (color pos) (read-from-string str NIL NIL :start pos))
		     (multiple-value-setq (color pos) (read-from-string str NIL NIL :start pos))
		     (multiple-value-setq (color pos) (read-from-string str NIL NIL :start pos))
		     (multiple-value-setq (color pos) (read-from-string str NIL NIL :start pos)) ;; Gets the third value
		     (setf max (max max (length (setf color (princ-to-string color)))))
		     (push color color-lst))))
    (tk::grab '.color.button)
    (.color.list.list :configure :width (+ 15 max))
    (setf color-lst (sort color-lst #'string<=))
    (mapc #'(lambda (color) (.color.list.list :insert "end" (format NIL "            ~A~A" color #\NEWLINE))) color-lst)
    (setf col 1)
    (setf max 0)
    (mapc #'(lambda (color)
	      (.color.list.list :tag :add color (format NIL "~A.0" col) (format NIL "~A.10" col))
	      (.color.list.list :tag :configure color :background color)
	      (when (> max 15) (setf max 0) (tk::tkwait :visibility '.color.list.lst)) ;; Dummy wait to avoid timeout errors
	      (setf max (1+ max))
	      (setf col (1+ col)))
	  color-lst)
    (.color.list.list :configure :state "disabled")
    (tk::grab '.color)
    (.color.list.list :configure :cursor "")
    (.color.button.ok :configure :state "normal")
    (.color.button.cancel :configure :state "normal"))))


(defun choose-color-select-line (X Y)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf X (- X (TK::winfo :rootx ".color.list.list" :return 'number)))
    (setf Y (- Y (TK::winfo :rooty ".color.list.list" :return 'number)))
    (setf **color** (.color.list.list :get (format NIL "@~A,~A linestart + 12 chars" X Y) (format NIL "@~A,~A lineend" X Y)
				      :return 'string))
    (.color.list.list :tag :add 'sel (format NIL "@~A,~A linestart" X Y) (format NIL "@~A,~A lineend" X Y))))


(defun choose-color-ok ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (funcall **callback** **color**)
    (tk::destroy '.color)))

(defun choose-color-cancel ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (tk::destroy '.color)))


;; *************************************************************************************************
;; Select a font to use
;; *************************************************************************************************


(defun choose-font (callback font)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **callback** callback)
    (setf **font** font)
    (TK::toplevel '.font)
    (TK::grab '.font)
    (TK::frame '.font.button)
    (TK::frame '.font.list)
    (TK::frame '.font.text)
    (TK::pack '.font.button '.font.list '.font.text :side "top")
    (TK::button '.font.button.ok :text "Ok" :command '(choose-font-ok) :state "disabled")
    (TK::button '.font.button.cancel :text "Cancel" :command '(choose-font-cancel) :state "disabled")
    (TK::pack '.font.button.ok '.font.button.cancel :side "left")
    (TK::scrollbar '.font.list.scroll :command ".font.list.list yview")
    (TK::text '.font.list.list :yscrollcommand ".font.list.scroll set")
    (TK::pack '.font.list.list :in '.font.list :side "left")
    (TK::pack '.font.list.scroll :in '.font.list :side "left" :expand "yes" :fill "both")
    (TK::text '.font.text.text :height 4)
    (TK::pack '.font.text.text)
    (TK::bind '.font.list.list "<ButtonPress-1>" '(choose-font-select-line TK::%X TK::%Y))
    (.font.list.list :configure :cursor "watch")
    ;; Configure the example text
    (.font.text.text :insert "end" "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ01234567890!@#$%^&*()_")
    (.font.text.text :tag :add "font" "1.0" "end")
    (.font.text.text :tag :configure "font" :font font)
    ;; Get a list of the fonts on the system
    (system "xlsfonts > /usr/tmp/debug-fonts-00")
    (let ((handle (open "/usr/tmp/debug-fonts-00" :direction :input)) (str "") (max 0))
      (tk::grab '.font.button)
      (loop
       (when (null (setf str (read-line handle NIL NIL))) (return))
       (.font.list.list :insert "end" (format NIL "~A~A" str #\NEWLINE))
       (incf max)
       (when (> max 10)
	 (setf max 0)
	 (setf max 0) (tk::tkwait :visibility '.font.list.lst))) ;; Dummy wait o avoid time out errors
      (.font.list.list :configure :state "disabled")
      (tk::grab '.font)
      (.font.list.list :configure :cursor "")
      (.font.button.ok :configure :state "normal")
      (.font.button.cancel :configure :state "normal"))))

  


(defun choose-font-select-line (X Y)
  (let ((font "")
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf X (- X (TK::winfo :rootx ".font.list.list" :return 'number)))
    (setf Y (- Y (TK::winfo :rooty ".font.list.list" :return 'number)))
    (setf font (.font.list.list :get (format NIL "@~A,~A linestart" X Y) (format NIL "@~A,~A lineend" X Y)
				      :return 'string))
    (when (not (string= font "")) (setf **font** font))
    (.font.text.text :tag :configure "font" :font **font**)
    (.font.list.list :tag :add 'sel (format NIL "@~A,~A linestart" X Y) (format NIL "@~A,~A lineend" X Y))))


(defun choose-font-ok ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (funcall **callback** **font**)
    (tk::destroy '.font)))

(defun choose-font-cancel ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (tk::destroy '.font)))



;; ******************************************************************************************************
;; Presents a list of all debugged functions and let you choose one , the corresponding source is then loaded
;; ******************************************************************************************************

(defun choose-function (callback)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **callback** callback)
    (TK::toplevel '.function)
    (TK::grab '.function)
    (TK::frame '.function.button)
    (TK::frame '.function.list)
    (TK::pack '.function.button '.function.list :side "top")
    (TK::button '.function.button.ok :text "Ok" :command '(choose-function-ok) :state "disabled")
    (TK::button '.function.button.cancel :text "Cancel" :command '(choose-function-cancel) :state "disabled")
    (TK::pack '.function.button.ok '.function.button.cancel :side "left")
    (TK::scrollbar '.function.list.scroll :command ".function.list.list yview")
    (TK::text '.function.list.list :yscrollcommand ".function.list.scroll set")
    (TK::pack '.function.list.list :in '.function.list :side "left")
    (TK::pack '.function.list.scroll :in '.function.list :side "left" :expand "yes" :fill "both")
    (TK::bind '.function.list.list "<ButtonPress-1>" '(choose-function-select-line TK::%X TK::%Y))
    (.function.list.list :configure :cursor "watch")
    ;; Get a list of the functions on the system
    (let ((max 0) (function-lst NIL))
      (maphash #'(lambda (function source) (push function function-lst)) **fun-source-lst**)
      (mapc #'(lambda (function)
		(.function.list.list :insert "end" (format NIL "~A~A" function #\NEWLINE))
		(incf max)
		(when (> max 10)
		  (setf max 0)
		  (tk::tkwait :visibility '.function.list.lst))) ;; Dummy wait to avoid timeout errors
	    (sort function-lst #'string<=)))
    (.function.list.list :configure :state "disabled")
    (tk::grab '.function)
    (.function.list.list :configure :cursor "")
    (.function.button.ok :configure :state "normal")
    (.function.button.cancel :configure :state "normal")))


(defun choose-function-select-line (X Y)
  (let ((function "")
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf X (- X (TK::winfo :rootx ".function.list.list" :return 'number)))
    (setf Y (- Y (TK::winfo :rooty ".function.list.list" :return 'number)))
    (setf function (.function.list.list :get (format NIL "@~A,~A linestart" X Y) (format NIL "@~A,~A lineend" X Y)
				      :return 'string))
    (when (not (string= function "")) (setf **function** (read-from-string function)))
    (.function.list.list :tag :add 'sel (format NIL "@~A,~A linestart" X Y) (format NIL "@~A,~A lineend" X Y))))


(defun choose-function-ok ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (gethash **function** **fun-source-lst**)
      (funcall **callback** **function** (gethash **function** **fun-source-lst**)))
    (tk::destroy '.function)))

(defun choose-function-cancel ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (tk::destroy '.function)))


;; ***************************************************************************************************
;; Code of the debugger itself
;; ***************************************************************************************************

;; ***************************************************************************************************
;; Global var , used for speed raisons , gcl does not compile defun in let good
;; ***************************************************************************************************

(defparameter *y* 0)
(defparameter *p* 0)
(defparameter *end* NIL)
(defparameter *buffer* "")
(defparameter *max* 0)
(defparameter *nl-list* NIL)


;;*******************************************************************************************************
;; Global var's
;; ******************************************************************************************************

(defparameter **source-lst** (make-hash-table :test #'equal)) ;; Contains for each debugged source a hash table of
                                                                                           ;; valid breakpoints
(defparameter **fun-source-lst** (make-hash-table :test #'equal)) ;; List of functions and their source definition
(defparameter **current-source** "")  ;; Source current loaded in the debug window
(defparameter **function** NIL) ;; Function choosen in choose function window
(defparameter **lock** NIL)    ;; Lock to avoid execute a function two times at the same time
(defparameter **env** NIL)    ;; Current environment
(defparameter **step** NIL)   ;; Indicate that we are stepping
(defparameter **step-over** NIL) ;; Indicate that we are stepping over
(defparameter **begin** "0.1") ;; Begin position of current breakpoint
(defparameter **end** "0.1")    ;; End position of current breakpoint
(defparameter **breakpoint-lst** (make-hash-table :test #'equal)) ;; List of breakpoints set
(defparameter **ifbreakpoint-lst** (make-hash-table :test #'equal)) ;; List of conditional breakpoints set
(defparameter **watchpoint-lst** (make-hash-table :test #'equal)) ;; List of watchpoints
(defparameter **profile-lst** (make-hash-table :test #'equal)) ;; Profile information
(defparameter **profile-max** 0) ;; Max profile count
(defparameter **break** NIL) ;; Indicate that we may leave a breakpoint
(defparameter **break-color** "cadetblue1")   ;; Color used to indicate the point in the source we are executing
(defparameter **breakpoint-color** "red") ;; Color used to set breakpoints in the source
(defparameter **breakpointif-color** "brown") ;; Color used to set conditional breakpoints in the source 
(defparameter **profile-base-color** "white") ;; Color for a piece of code whe can profile
(defparameter **profile-color** "cyan") ;; Color to show piece of code exceding profile count
(defparameter **geometry** "593x573+13+7") ;; Geometry of window
(defparameter **debug-font** "-*-helvetica-medium-r-normal-*-120-*-*-*-*-*-*") ;; Font used
(defparameter **save-on-exit** "1") ;; Save settings of the debugger on exit
(defparameter **profile** NIL) ;; Do profile counting
(defparameter **begin-end-function** (make-hash-table :test #'equal)) ;; Contains the begin and endpoints of functions , methods and generic functions , used to accellerate the behavior in cmucl
(defparameter **UNKNOWN** (gensym)) ;; variable which don't exist in the source code

;; *******************************************************************************************************
;; Detects if a character is a whitespace
;; *******************************************************************************************************

(declaim (inline whitespace))
(defun whitespace (character)
  (or (char= character #\newline) (char= character #\space) (char= character #\page)
       (char= character #\tab) (char= character #\return) (char= character #\linefeed)))

;; **********************************************************************************************************
;; Function to load a source , and get position information about the lists in the sourcecode
;; **********************************************************************************************************

(defun load-source (source)
  (let ((buffer "") (nl-list NIL) (h NIL) (line "") (l 0))
    (declare (fixnum l) (string buffer) (list nl-list))
    (setf nl-list '(0))
    ;; Open the source
    (setf h (open source :direction :input :if-does-not-exist NIL))
    (when (null h)
      (format T "Can't open <~A> " source)
      (return-from load-source (values NIL NIL)))
    ;; Read the source in the buffer and update the nl-list
    (setf buffer
	  (with-output-to-string (h-str)
				 (loop (cond ((null (setf line (read-line h NIL NIL)))
					             ;; End of source reached
			       	                     (return))
					            (T
						     (write-line line h-str)   ;; Write the line to the buffer
						     (push (setf l (+ l 1 (length line))) nl-list)))))) ;; Update the list of new-lines
    ;; Close the source file
    (close h)
    ;; Prepare the parsing of the buffer
    (source-init (reverse nl-list) buffer (1- l))
    ;; Parse the source
    (parse-source (make-hash-table :test #'eq))))

(declaim (inline source-init))
(defun source-init (nl buf m)
  (setf *y* 1)
  (setf *p* 0)
  (setf *end* NIL)
  (setf *nl-list* nl)
  (setf *buffer* buf)
  (setf *max* m))

;; Release resources

(declaim (inline source-close))
(defun source-close ()
  (setf *nl-list* NIL)
  (setf *buffer* ""))

;; Check for end of buffer

(declaim (inline source-end))
(defun source-end ()
  *end*)

;; Get the coordinate pair (x.y) , x is position in line , y = lineno , x starts from 0 , y from y

(declaim (inline source-coordinates))
(defun source-coordinates ()
  (loop (when (>= (second *nl-list*) *p*)
	  (return (cons *y* (- *p* (first *nl-list*)))))
	(pop *nl-list*)
	(incf *y*)))

;; Test if the next characters are the beginning of a lambda expression and position pointer

(declaim (inline source-next-is-lambda))
(defun source-next-is-lambda ()
  (cond ((> (+ *p* 2) *max*)
	 (setf *end* T)
	 NIL)
	((and (char= (aref *buffer* *p*) #\#) (char= (aref *buffer* (+ *p* 1)) #\') (char= (aref *buffer* (+ *p* 2)) #\())
	 ;; We have found the beginning of a lambda expression
	 (setf *p* (+ *p* 3))) ;; Increase pointer , return non nil value
	(T
	 NIL)))

;; Test if the next character is an opening hook and position pointer

(declaim (inline source-next-is-open))
(defun source-next-is-open ()
  (cond ((> *p* *max*)
	 (setf *end* T)
	 NIL)
	((char= (aref *buffer* *p*) #\()
	 (incf *p*))  ;; Increase pointer , return non nil value
	(T
	 NIL)))

;; Test if the next character is an closing hook and position pointer

(declaim (inline source-next-is-close))
(defun source-next-is-close ()
    (cond ((> *p* *max*)
	   (setf *end* T)
	   NIL)
	  ((char= (aref *buffer* *p*) #\))
	   (incf *p*))  ;; Increase pointer , return non nil value
	  (T
	   NIL)))


;; Skip white spaces if they exist

(declaim (inline source-skip-white))
(defun source-skip-white ()
  (declare (inline whitespace))
  (loop (cond ((> *p* *max*)
	       (setf *end* T)
	       (return NIL))
	      ((whitespace (aref *buffer* *p*))
	       (incf *p*))
	      (T
	       (return T)))))

;; Skip the comment lines

(declaim (inline source-skip-comment))
(defun source-skip-comment ()
  (cond ((> *p* *max*)
	 (setf *end* T)
	 NIL)
	((char= (aref *buffer* *p*) #\;)
	 (incf *p*)
	 (loop (cond ((> *p* *max*)
		      (setf *end* T)
		      (return NIL))
		     ((char= (aref *buffer* *p*) #\newline)
		      (return T))
		     (T
		      (incf *p*)))))
	(T
	 NIL)))


;; Read the next lisp item

(defun source-lisp-read ()
  (let ((exp NIL))
    (cond ((> *p* *max*)
	   (setf *end* T)
	   (values NIL NIL))
	  ((and (<= (+ *p* 2) *max*)
		(char= (aref *buffer* *p*) #\#)
		(or (char= (aref *buffer* (+ *p* 1)) #\+) (char= (aref *buffer* (+ *p* 1)) #\-)))
	   ;; The next charaters are a features-description*
	   (let ((p *p*))
	     (multiple-value-setq (exp *p*) (read-from-string *buffer* NIL NIL :start (+ *p* 2)))
	     (cond ((read-from-string (format NIL "#~A~A T NIL" (aref *buffer* (+ p 1)) exp))
		    ;; Next entry should be parsed
		    (values NIL NIL))
		   (T
		    ;; Next entry should be skipped
		    (when (<= *p* *max*)
		      (source-lisp-read-next-form))
		    (values NIL NIL)))))
	  (T
	   (multiple-value-setq (exp *p*) (read-from-string *buffer* NIL NIL :start *p*))
	   (values exp T)))))


;; ***************************************************************************************************
;; Read the next form without causing errors on packages ...
;; ***************************************************************************************************

(defun source-lisp-read-next-form ()
  (loop
    (source-skip-white)
    (when (or (source-end) (not (source-skip-comment))) (return)))  ;; Skip white spaces and comments
  (cond ((char= (aref *buffer* *p*) #\()
	 (let ((hooks 0) (char NIL))
	   (loop
	     (when (source-end) (return))
	     (incf *p*)
	     (setf char (aref *buffer* *p*))
	     (cond ((char= char #\))
		    (cond ((zerop hooks)
			   (incf *p*)
			   (return))
			  (T
			   (decf hooks))))
		   ((char= char #\()
		    (incf hooks))
		   ((source-skip-comment))))))
	(T
	 (loop
	   (when (source-end) (return))
	   (incf *p*)
	   (when (source-skip-white) (return))))))
	
		    
    
	 
    
    

;; ****************************************************************************************************
;; Parse the source
;; ****************************************************************************************************

(defun parse-source (position)
  (declare (inline source-skip-comment source-next-is-open
		   source-next-is-close source-coordinates source-lisp-read))
  (let ((source NIL) (exp NIL) (begin NIL) (end NIL) (ok NIL))
    (loop (when (source-end)
	    (source-close)
	    (return (values source position)))
	  (source-skip-white)
	  (cond ((source-skip-comment))
		((source-next-is-open)
		 (setf begin (source-coordinates))
		 (push (setf exp (parse-source-list position)) source)
		 (setf end (source-coordinates))
		 (setf (gethash exp position) (cons begin end)))
		((source-next-is-close)
		 (format T "Unexpected closing parentheses found at ~A" (source-coordinates))
		 (return (values NIL NIL)))
		((source-next-is-lambda)
		 (setf begin (source-coordinates))
		 (push (setf exp (list 'function (parse-source-list position))) source)
		 (setf end (source-coordinates))
		 (setf (gethash exp position) (cons begin end)))
		(T
		 (multiple-value-setq (exp ok) (source-lisp-read))
		 (when ok (push exp source)))))))


(defun parse-source-list (position)
  (declare (inline source-skip-comment source-next-is-open
		   source-next-is-close source-coordinates source-lisp-read))
  (let ((source NIL) (exp NIL) (begin NIL) (end NIL) (ok NIL))
    (setf source NIL)
    (loop (when (source-end)
	    (format T "Unmatched closing parentheses found")
	    (return NIL))
	  (source-skip-white)
	  (cond ((source-skip-comment))
		((source-next-is-open)
		 (setf begin (source-coordinates))
		 (push (setf exp (parse-source-list position)) source)
		 (setf end (source-coordinates))
		 (setf (gethash exp position) (cons begin end)))
		((source-next-is-close)
		 (return (nreverse source)))
		((source-next-is-lambda)
		 (setf begin (source-coordinates))
		 (push (setf exp (list 'function (parse-source-list position))) source)
		 (setf end (source-coordinates))
		 (setf (gethash exp position) (cons begin end)))
		(T
		 (multiple-value-setq (exp ok) (source-lisp-read))
		 (when ok (push exp source)))))))


		       
;; *************************************************************************************************
;; Add debugging code to the sources
;; *************************************************************************************************

(defun add-code (name source position)
  ;; Initialize the breakpoint list of the source
  (setf (gethash name **source-lst**) (make-hash-table :test #'equal))
  ;; Add the debugging code to the source (reading it backwards)
  (let ((debugged-code NIL))
    (declare (list debugged-code))
    (mapc #'(lambda (exp)
	      (cond ((listp exp)
		     (cond ((eq (first exp) 'defun)
			    (push (add-code-function name exp position) debugged-code))
			   ((eq (first exp) 'user::defmethod)
			    (push (add-code-method name exp position) debugged-code))
			   (T
			    (push (add-code-exp name exp position) debugged-code))))
		    (T (push exp debugged-code))))
	  source)
    debugged-code))



;; ****************************************************************************************************
;; Add a breakpoint code
;; ****************************************************************************************************
(declaim (inline add-debug-point))
#+gcl
(defun add-debug-point (name pos)
  (let ((x-s (gensym)) (env-s (gensym)) (begin (format NIL "~A.~A" (first (first pos)) (rest (first pos))))
	 (end (format NIL "~A.~A" (first (rest pos)) (rest (rest pos)))))
    ;; Make a note in the **source-lst** table so we can have a list of breakpoints (x-begin y-begin x-end y-end)
    (setf (gethash (list begin end) (gethash name **source-lst**)) T)
    ;; Generate the debuggin code
    (list 'debug-point (list 'quote name) begin end
	                        (list 'let (list (list '*evalhook* (list 'function (list 'lambda (list x-s env-s) env-s))))
				            (list 'eval NIL)))))
#+:CMU
(defun add-debug-point (name pos)
  (let ((begin (format NIL "~A.~A" (first (first pos)) (rest (first pos))))
	 (end (format NIL "~A.~A" (first (rest pos)) (rest (rest pos)))))
    ;; Make a note in the **source-lst** table so we can have a list of breakpoints (x-begin y-begin x-end y-end)
    (setf (gethash (list begin end) (gethash name **source-lst**)) T)
    ;; Generate the debuggin code
    (list 'debug-point (list 'quote name) begin end (list 'top-frame))))
	  



;; *********************************************************************************************
;; Add breakpoint to code
;; *********************************************************************************************
(declaim (inline add-code-breakpoint))
(defun add-code-breakpoint (name exp position)
  (cond ((null exp) exp)
	((and (listp exp) (gethash exp position))
	 (list 'progn (add-debug-point name (gethash exp position)) (add-code-exp name exp position)))
	(T
	 exp)))


;; ****************************************************************************************************
;; Add debugging code to a function
;; ****************************************************************************************************


(defun add-code-function (name exp position)
  (let ((pos (gethash exp position)))
    (setf (gethash (second exp) **fun-source-lst**) name) ;; List function name
    (setf (gethash (cons name (format NIL "~A.~A" (first (first pos)) (1- (rest (first pos))))) ;;Update position info functions
		   **begin-end-function**)
	  (format NIL "~A.~A" (first (rest pos)) (rest (rest pos))))
    (cons  'defun (cons (second exp)
			(cons (third exp)
			      (add-code-body-function name (rest (rest (rest  exp))) exp position))))))


;; ****************************************************************************************************
;; Add debugging code to a method
;; ****************************************************************************************************
;;
;; (defmethod function-specifier {method-qualifier}*
;;            specialized-lambda-list
;;            {declaration | documentation}*
;;            {form}*
;; ****************************************************************************************************

(defun add-code-method (name exp position)
  (let ((pos (gethash exp position)))
    (setf (gethash (second exp) **fun-source-lst**) name)    ;; List function name
    (setf (gethash (cons name (format NIL "~A.~A" (first (first pos)) (1- (rest (first pos))))) ;;Update position info functions
		   **begin-end-function**)
	  (format NIL "~A.~A" (first (rest pos)) (rest (rest pos))))
    (let ((method NIL)
	  (qualifier-lst NIL)
	  (lambda-lst NIL)
	  (tmp (rest exp)))
      (setf method (pop tmp))
      (loop (cond ((atom (first tmp))
		   (push (pop tmp) qualifier-lst))
		  (T
		   (setf lambda-lst (pop tmp))
		   (return (cons 'user::defmethod
				 (cons method
				       (revappend qualifier-lst
						  (cons lambda-lst
							(add-code-body-function name tmp exp position))))))))))))


;; *********************************************************************************************
;; Add debugging code to method
;; *********************************************************************************************
;;
;; (defgeneric function-specifier lambda-list
;;                  [option | method-description]*)
;;
;;    option ::= .... (must not be modified) is a list
;;    method-description ::= (:method {method-qualifier}*
;;                            specialized-lambda-list
;;                            {declaration | documentation}*
;;                            {form}*)
;; *********************************************************************************************

(defun add-code-generic (name exp position)
  (let ((pos (gethash exp position)))
    (setf (gethash (second exp) **fun-source-lst**) name)
    (setf (gethash (cons name (format NIL "~A.~A" (first (first pos)) (1- (rest (first pos))))) ;;Update position info functions
		   **begin-end-function**)
	  (format NIL "~A.~A" (first (rest pos)) (rest (rest pos))))
    (let ((function-name NIL)
	  (lambda-lst NIL))
      (pop exp)
      (setf function-name (pop exp))
      (setf lambda-lst (pop exp))
      (cons 'defgeneric
	    (cons function-name
		  (cons lambda-lst
			(add-code-method-description name exp position)))))))
  
;; *********************************************************************************************
;; Add the code for method-descriptions
;; ********************************************************************************************
;; exp ::= [option | method-description]*
;; option ::= list
;; method-description ::= (:method {method-qualifier}*
;;                         specialized-lambda-list
;;                         {declaration | documentation}*
;;                         {form}*)
;; ********************************************************************************************

(defun add-code-method-description (name exp position)
  (mapcar
   #'(lambda (method)
       (cond ((and (listp method) (eql (first method) :method))
	      ;; method is indeed a method
	      (let ((tmp (rest method))
		    (qualifier-lst NIL)
		    (lambda-lst NIL))
		(loop (cond ((atom (first tmp))
			     (push (pop tmp) qualifier-lst))
			    (T
			     (setf lambda-lst (pop tmp))
			     (return
			      (cons :method
				    (revappend qualifier-lst
					       (cons lambda-lst
						     (add-code-body-function name tmp method position))))))))))
	     (T
	      method)))
   exp))


;; *********************************************************************************************
;; Add the debugging code to the body of a function
;; **********************************************************************************************

(defun add-code-body-function (name body exp position)
  (let ((code NIL))
    (setf code NIL)
    ;; Process comments and declarations
    (loop
     (cond ((stringp (first body)) (push (pop body) code)) ;; Process commentstring if it exist
	   ((and (listp (first body)) (eq (first (first body)) 'declare)) ;; Process declarations
	    (push (pop body) code))
	   (T  ;; End of declarations or comments
	    (return))))
    ;; Add code to indicate a call of the function
    (push (add-debug-point name (gethash exp position)) code)
    ;; Add code for the expressions in the body
    (mapc #'(lambda (e) (push (add-code-breakpoint name e position) code)) body)
    ;; Return the new body
    (nreverse code)))


 ;; **********************************************************************************************
;; Add the debugging code to the body of a function
;; **********************************************************************************************

(defun add-code-body (name body exp position)
  (let ((code NIL))
    (setf code NIL)
    ;; Process comments and declarations
    (loop
     (cond ((and (listp (first body)) (eq (first (first body)) 'declare)) ;; Process declarations
		(push (pop body) code))
	       (T  ;; End of declarations or comments
		(return))))
    ;; Add code to indicate a start of the body
    (add-debug-point name (gethash exp position))
    ;; Add code for the expressions in the body
    (mapc #'(lambda (e) (push (add-code-breakpoint name e position) code)) body)
    ;; Return the new body
    (nreverse code)))


;; ***************************************************************************************************
;; Add debugging code to an expression
;; ***************************************************************************************************

(defun add-code-exp (name exp position)
  (cond ((and (listp exp) (atom (first exp)))
	 (let ((operator (first exp)))
	   (cond ((endp exp) NIL)
		 ((or (special-operator-p operator) (macro-function operator))
		  (case operator
		    ('defun
		      (add-code-function name exp position))
		    ('user::defmethod
		     (add-code-method name exp position))
		    ('defgeneric
		      (add-code-generic name exp position))
		    ('with-accessors
		     (add-code-with-accessors-slots name exp position))
		    ('with-slots
		     (add-code-with-accessors-slots name exp position))
		    ('with-input-from-string
		     (add-code-with-form name exp position))
		    ('with-open-file
		     (add-code-with-form name exp position))
		    ('with-output-to-string
		     (add-code-with-form name exp position))
		    ('with-open-stream
		     (add-code-with-form name exp position))
		    ('symbol-macrolet
		     (add-code-symbol-macrolet name exp position))
		    ('block
			;; Syntax :  (block name {form}*)
			(cons 'block (cons (second exp) (add-code-body name (rest (rest exp)) exp position))))
		    ('catch
			;; Syntax :   (catch tag {form}*)
			(cons 'catch (cons (second exp) (add-code-body name (rest (rest exp)) exp position))))
		    ('compiler-let
		     ;; Syntax :  (compiler-let ({var | (var [value])}*) {form}*)
		     (cons 'compiler-let
			   (cons (mapcar #'(lambda (arg)
					     (cond ((listp arg)
						    (cond ((rest arg)
							   (list (first arg) (add-code-breakpoint  name (second arg) position)))
							  (T arg)))
						   (T arg)))
					 (second exp))
			      (add-code-body name (rest (rest exp)) exp position))))
		    ('eval-when
			;; Syntax : (eval-when ({situation}*) {form}*)
			;; Remark the forms are evaluated only if situation 'eval' is specified
			(cond ((member 'eval (second exp))
			       (cons operator (cons (second exp) (add-code-body name (rest (rest exp)) exp position))))
			      (T
			       exp)))
		    ('flet
			;; Syntax :  (flet ({(name lambda-list {decl | doc}* {form}*)}*) . body)
			(add-code-exp-special-flet name exp position))
		    ('function
		     ;; Syntax : (function x)
		     (add-code-exp-special-function name exp position))
		    ('if
			;; Syntax ::    (if test then [else])
			(if (fourth exp)
			    (list 'if (add-code-breakpoint name (second exp) position)
				  (add-code-breakpoint name (third exp) position)
				  (add-code-breakpoint name (fourth exp) position))
			  (list 'if (add-code-breakpoint name (second exp) position)
				(add-code-breakpoint name (third exp) position))))
		    ('labels
			;; Syntax :  (labels ({(name lambda-list {decl | doc}* {form}*)}*) . body) (same as flet)
			(add-code-exp-special-flet name exp position))
		    ('let
			;; Syntax :  (let ({var | (var [value])}*) {decl}* {form}*)
			(add-code-exp-special-let name exp position))
		    ('let*
			;; Syntax :  (let* ({var | (var [value])}*) {decl}* {form}*)
			(add-code-exp-special-let name exp position))
		    ('macrolet
			;; Syntax :  (macrolet ({(name defmacro-lambda-list {decl | doc}* . body)}*) {form}*)
			;; We only compile the part in the form
			(cons 'macrolet
			      (cons (second exp)
				    (add-code-body name (rest (rest exp)) exp position))))
		    ('multiple-value-call
		     ;; Syntax:   (multiple-value-call function {form}*)
		     (cond ((and (listp (second exp)) (eq (first (second exp)) 'function))
			    ;; function is a lambda expression
			    (cons 'multiple-value-call
				  (cons (add-code-exp-special-function name (second exp) position)
					(mapcar #'(lambda (arg) (add-code-breakpoint name arg position))
						(rest (rest exp))))))
			   (T
			    (cons 'multiple-value-call
				  (cons (second exp)
					(mapcar #'(lambda (arg) (add-code-breakpoint name arg position))
						(rest (rest exp))))))))
		    ('multiple-value-prog1
		     ;; Syntax :  (multiple-value-prog1 form {form}*)
		     (cons 'multiple-value-prog1
			   (add-code-body name (rest exp) exp position)))
		    ('progn
		      ;; Syntax :  (progn {form}*)
		      (cons 'progn
			    (add-code-body name (rest exp) exp position)))
		    ('progv
			;; Syntax :  (progv symbols values {form}*)
			(cons 'progv
			      (cons (add-code-breakpoint name (second exp) position)
				    (cons (add-code-breakpoint name (third exp) position)
				    (add-code-body name (rest (rest (rest exp))) exp position)))))
		    ('setq
		     ;; Syntax :  (setq {var form}*)
		     (add-code-special-exp-setq name exp position))
		    ('return-from
			;; Syntax :  (return-from name [result])
			(cond ((rest (rest exp))
			       (list 'return-from
				     (second exp)
				     (add-code-breakpoint name (third exp) position)))
			      (T exp)))
		    ('tagbody
		     ;; Syntax :   (tagbody {tag | statement}*)
		     (cons 'tagbody
			   (mapcar #'(lambda (arg)
				       (if (listp arg) (add-code-breakpoint name arg position) arg))
				   (rest exp))))
		    ('the
			;; Syntax :  (the value-type form)
			(list 'the
			      (second exp)
			      (add-code-breakpoint name (third exp) position)))
		    ('throw
		     ;; Syntax : (throw tag result)
		     (list 'throw
			   (second exp)
			   (add-code-breakpoint name (third exp) position)))
		    ('unwind-protect
			;; Syntax :    (unwind-protect protected-form {cleanup-form}*)
			(cons 'unwind-protect
			      (cons (add-code-breakpoint name (second exp) position)
				    (add-code-body name (rest (rest exp)) exp position))))
		    ('and
		     ;; Syntax :  (and {form}*)
		     (cons 'and (mapcar #'(lambda (arg) (add-code-breakpoint name arg position))
					(rest exp))))
		    ('or
		     ;; Syntax :  (or {form}*)
		     (cons 'or (mapcar #'(lambda (arg) (add-code-breakpoint name arg position))
				       (rest exp))))
		    ('case
			;;  (case keyform {({key | ({key}*)} {form}*)}*)
			(add-code-special-exp-case name exp position))
		    ('cond
		     ;;  (cond {(test {form}*)}*)
		     (cons 'cond (mapcar #'(lambda (arg)
					     (cons (add-code-breakpoint name (first arg) position)
						   (add-code-body name (rest arg) exp position)))
					 (rest exp))))
		    ('decf
		     ;;  (decf place [delta])
		     (cond ((rest (rest exp))
			    (list 'decf
				  (second exp)
				  (add-code-breakpoint name (third exp) position)))
			   (T exp)))
		    ('do
			(add-code-exp-special-do name exp position))
		    ('do*
			(add-code-exp-special-do name exp position))
		    ('dolist
			(add-code-exp-special-do-list name exp position))
		    ('dotimes
			(add-code-exp-special-do-list name exp position))
		    ('incf
		     ;;  (incf place [delta])
		     (cond ((rest (rest exp))
			    (list 'incf
				  (second exp)
				  (add-code-breakpoint name (third exp) position)))
			   (T exp)))
		    ('locally
		     ;; (locally {decl}* {form}*)
		     (cons 'locally (add-code-body name (rest exp) exp position)))
		    ('loop
		     ;; Syntax :  (loop {form}*)
		     (cons 'loop (add-code-body name (rest exp) exp position)))
		    ('multiple-value-bind
			;; Syntax :  (multiple-value-bind ({var}*) values-form {decl}* {form}*)
			(cons 'multiple-value-bind
			      (cons (second exp)
				    (cons (add-code-breakpoint name (third exp) position)
					  (add-code-body name (rest (rest (rest exp))) exp position)))))
		    ('multiple-value-list
		     ;; Syntax :  (multiple-value-list form)
		     (list 'multiple-value-list (add-code-breakpoint name (second exp) position)))
		    ('multiple-value-setq
			;; Syntax :  (multiple-value-setq variables form)
			(list 'multiple-value-setq (second exp) (add-code-breakpoint name (third exp) position)))
		    ('unless
			;; Syntax :  (unless test {form}*)
			(cons 'unless
			      (cons (add-code-breakpoint name (second exp) position)
				    (add-code-body name (rest (rest exp)) exp position))))
		    ('when
			;; Syntax :  (unless test {form}*)
			(cons 'when
			      (cons (add-code-breakpoint name (second exp) position)
				    (add-code-body name (rest (rest exp)) exp position))))
		    ('prog
		     ;; Syntax :  (prog ({var | (var [init])}*) {decl}* {tag | statement}*)
		     (add-code-exp-special-prog name exp position))
		    ('prog*
		     ;; Syntax :    (prog* ({var | (var [init])}*) {decl}* {tag | statement}*)
		     (add-code-exp-special-prog name exp position))
		    ('prog1
			;; Syntax :  (prog1 first {form}*)
			(cons 'prog1 (add-code-body name (rest exp) exp position)))
		    ('prog2
			;; Syntax : (prog2 first second {forms}*)
			(cons 'prog2 (add-code-body name (rest exp) exp position)))
		    ('psetq
		     ;; Syntax :   (psetq {var form}*)
		     (add-code-special-exp-setq name exp position))
		    ('push
		     ;; Syntax ::     (push item place)
		     (list 'push
			   (add-code-breakpoint name (second exp) position)
			   (third exp)))
		    ('return
		     ;; Syntax ::  (return [result])
		     (if (rest exp) (list 'return (add-code-breakpoint name (second exp) position))
		       exp))
		    ('setf
		     ;; Syntax :   (setf {place newvalue}*)
		     (add-code-special-exp-setq name exp position))
		    ('destructuring-bind
		     ;; Syntax :   (destructuring-bind lambda-list expression {declaration}* {form}*)
		     (add-code-destructuring-bind name exp position))	
		    (T
		     (cond ((macro-function operator)
			    (add-code-exp-macro name exp position))
			   (T
			    exp)))))
		 (T
		  (add-code-exp-function name exp position)))))
	(T
	 exp)))


;; ******************************************************************************************************
;; Add the debugging code to a with-input-from-string macro
;; ******************************************************************************************************
;; (with-input-from-string (var string {keyword value}*) {decl}*
;;          {form}*)
;; (with-open-file (stream filename {options}*) {decl}* {form}*)
;; (with-open-stream (var stream) {decl}* {form}*)
;; (with-output-to-string (var [string]) {decl}* {form}*)
;; ******************************************************************************************************

(defun add-code-with-form (name exp position)
  (cons (first exp)
	(cons (cons (first (second exp))
		    (mapcar #'(lambda (e)
				(if (keywordp e) e (add-code-breakpoint name e position)))
			    (rest (second exp))))
	      (add-code-body name (rest (rest exp)) exp position))))


;; ******************************************************************************************************
;; Add the debugging code to a with-accessors or with-slots macro
;; (with-accessors(slots) ({slot-entry}*) instance-form &body body)
;; ******************************************************************************************************

(defun add-code-with-accessors-slots (name exp position)
  (cons (first exp)
	(cons (second exp)
	      (cons (third exp)
		    (add-code-body name (rest (rest (rest exp))) exp position)))))


;; ******************************************************************************************************
;; Add the debugging code to symbol-macrolet
;; ******************************************************************************************************

(defun add-code-symbol-macrolet (name exp position)
  (cons (first exp)
	(cons (second exp)
	      (add-code-body name (rest (rest exp)) exp position))))
	      
;; ******************************************************************************************************
;; Add the debugging code to a function
;; ******************************************************************************************************

(defun add-code-exp-function (name exp position)
  (cons (first exp) (mapcar #'(lambda (arg)
				(add-code-breakpoint name arg position))
			    (rest exp))))


;; ********************************************************************************************************
;; Add the extra code for a prog or prog* expression 
;; Syntax :    (prog* ({var | (var [init])}*) {decl}* {tag | statement}*)
;; *******************************************************************************************************

(defun add-code-exp-special-prog (name exp position)
  (cons (first exp)
	(cons (mapcar #'(lambda (var)
			  (cond ((listp var)
				 (cond ((rest var)
					(list (first var)
					      (add-code-breakpoint name (second var) position)))
				       (T var)))
				(T
				 var)))
		      (second exp))
	      (add-code-exp-special-do-body name (rest (rest exp)) exp position))))

;; ********************************************************************************************************
;; Add the extra code for a dolist,dotime  expresssion
;; Sybtax : (dolist (var listform [result]) {decl}* {tag | statement}*)
;; ********************************************************************************************************

(defun add-code-exp-special-do-list  (name exp position)
  (cons (first exp)
	 (cons (cond ((rest (rest (second exp)))
		               (list (first (second exp)) 
				      (add-code-breakpoint name (second (second exp)) position)
				      (add-code-breakpoint name (third (second exp)) position)))
		              (T
			       (list (first (second exp))
				      (add-code-breakpoint name (second (second exp)) position))))
		     (add-code-exp-special-do-body name (rest (rest exp)) exp position))))

;; *********************************************************************************************************
;; Add the extra code for a do or do* expression
;; Syntax : (do ({(var [init [step]])}*) (endtest {result}*) {decl}* {tag | statement}*)
;; *********************************************************************************************************

(defun add-code-exp-special-do (name exp position)
    (cons (first exp)
	 (cons (mapcar #'(lambda (var)
					(cond ((rest (rest var))
						    (list (first var)
							   (add-code-breakpoint name (second var) position)
							   (add-code-breakpoint name (third var) position)))
					          ((rest var)
						   (list (first var)
							  (add-code-breakpoint name (second var) position)))
						  (T var)))
                                          (second exp))
	      (cons (cons (add-code-breakpoint name (first (third exp)) position)
			           (if (rest (third exp)) (add-code-body name (rest (third exp)) exp position) NIL))
		        (add-code-exp-special-do-body name (rest (rest (rest exp))) exp position)))))

(defun add-code-exp-special-do-body (name body exp position)
  (let ((code NIL))
    (setf code NIL)
    (loop
     ;; Process declarations if they exist
     (cond ((and (listp (first body)) (eq (first (first body)) 'declare))
		(push (pop body) code))
	       (T
		(return))))
    (push (add-debug-point name (gethash exp position)) code)
    ;; Next add the code for the expressions in the body
    (mapc #'(lambda (e) (if (listp e) (push (add-code-breakpoint name e position) code)
			                             (push e code)))
	        body)
    (nreverse code)))

		        
	      
;; **********************************************************************************************************
;; Add the extra code for a case expression
;; syntax::  (case keyform {({key | ({key}*)} {form}*)}*)
;; **********************************************************************************************************

(defun add-code-special-exp-case (name exp position)
  (cons (pop exp) ;; case
	(cons (add-code-breakpoint name (first exp) position)
	      (mapcar #'(lambda (arg)
			            (cons (first arg) (add-code-body name (rest arg) exp position)))
			     (rest exp)))))

;; **********************************************************************************************************
;; Add the extra code for an setq expression
;; **********************************************************************************************************

(defun add-code-special-exp-setq (name exp position)
  (let ((code NIL) (op (pop exp)))
    (loop (when (endp exp) (return))
	     (push (pop exp) code)
	     (push (add-code-breakpoint name (first exp) position)
                       code)
	     (pop exp))
    (cons op (nreverse code))))

		  
;;***********************************************************************************************************
;; Add the extra code for an destructuring bind
;; Syntax : (destructuring-bind lambda-list expression {declaration}* {form}*)
;; **********************************************************************************************************

(defun add-code-destructuring-bind (name exp position)
  (let ((tmp-exp exp))
    (cons (pop tmp-exp)
	  (cons (pop tmp-exp)
		(cons (add-code-breakpoint name (pop tmp-exp) position)
		      (add-code-body name tmp-exp exp position))))))
  
		         
	     
;; ***********************************************************************************************************
;; Add code to an expression of the form (function func)
;; ***********************************************************************************************************

(defun add-code-exp-special-function (name exp position)	     
  (cond ((and (listp (second exp)) (eq (first (second exp)) 'lambda)) ;; X is of the form (lambda lambda-list . body)
	     (cons 'function
		   (list (cons (first (second exp))
			       (cons (second (second exp))
				     (add-code-body name (rest (rest (second exp))) exp position))))))
	     (T exp)))

;; *****************************************************************************************************
;; Add the extra code to an let or let* expressions
;; *****************************************************************************************************
;; Syntax :  (let* ({var | (var [value])}*) {decl}* {form}*)

(defun add-code-exp-special-let (name exp position)
  (let ((code NIL))
    (setf code NIL)
    (mapc #'(lambda (var-value)
	                (cond ((listp var-value)
			           (cond ((second var-value)
					      (push (list (first var-value) (add-code-breakpoint name (second var-value) position))
						        code))
					    (T
					     (push (list (first var-value)) code))))
			          (T
				   (push var-value code))))
	      (second exp))
    (cons (first exp)
	       (cons (nreverse code)
		         (add-code-body name (rest (rest exp)) exp position)))))
			                    
    

;; ******************************************************************************************************
;; Add the extra code for an flet or labels  expression , also about the save of the functions defined here
;; ******************************************************************************************************
;; Syntax :  (flet ({(name lambda-list {decl | doc}* {form}*)}*) . body)

(defun add-code-exp-special-flet (name exp position)
  (let ((fun-code NIL))
    (setf fun-code NIL)
    (mapc #'(lambda (funct)
		       (push  (cons (first funct) (cons (second funct) (add-code-body name (rest (rest funct)) exp position)))
			          fun-code))
	      (second exp))
    (cons (first exp) (cons (nreverse fun-code)
			                 (add-code-body name (rest (rest exp)) exp position)))))




;; *****************************************************************************************************
;; Add the debugging code for macros
;; ****************************************************************************************************

(defun add-code-exp-macro (name exp position)
  (add-code-exp name (macroexpand-1 exp) position))


;; **********************************************************************************************************
;; Graphical part of the debugger , using the TCL/TK language
;; **********************************************************************************************************

(defun debug ()
  (setf **NULL-OUTPUT** (open "/dev/null" :direction :output))
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    ;; Load the configuration setting
    (debug-load-setting)
    ;; Connect to the TK system
    (TK::tkconnect)
    (TK::wm :iconify ".")
    (TK::toplevel '.d)
    (TK::wm :geometry ".d" **geometry**)
    ;; Create the frames needed
    (TK::frame '.d.menu)
    (TK::frame '.d.button)
    (TK::frame '.d.text)
    (TK::frame '.d.command)
    (TK::frame '.d.result)
    (TK::pack '.d.menu :expand 'yes :fill 'both)
    (TK::pack '.d.button :expand 'yes :fill 'both)
    (TK::pack '.d.text :expand "yes" :fill "both")
    (TK::pack '.d.result :expand "yes" :fill "both")
    (TK::pack '.d.command :expand "yes" :fill "both")
    
  ;; Create the different widgets
    
    (TK::menubutton '.d.menu.file :text "File" :underline 0 :menu '.d.menu.file.file)
    (TK::menubutton '.d.menu.edit :text "Edit" :underline 0 :menu '.d.menu.edit.edit)
    (TK::menubutton '.d.menu.options :text "Options" :underline 0 :menu '.d.menu.options.options)
    (TK::menubutton '.d.menu.tools :text "Tools" :underline 0 :menu '.d.menu.tools.tools)
    (TK::menu '.d.menu.file.file :tearoff 0)
    (TK::menu '.d.menu.edit.edit :tearoff 0)
    (TK::menu '.d.menu.options.options :tearoff 0)
    (TK::menu '.d.menu.tools.tools :tearoff 0)
    (.d.menu.file.file :add 'command :label "Open" :command '(debug-open))
    (.d.menu.file.file :add 'command :label "Open function" :command '(debug-open-function))
    (.d.menu.file.file :add 'command :label "Close" :command '(debug-close))
    (.d.menu.file.file :add 'separator)
    (.d.menu.file.file :add 'command :label "Exit" :command '(debug-exit))
    (.d.menu.edit.edit :add 'command :label "Paste" :command '(debug-paste))
    (.d.menu.options.options :add 'command :label "Color Break" :command '(debug-color-break))
    (.d.menu.options.options :add 'command :label "Color BreakPoint" :command '(debug-color-breakpoint))
    (.d.menu.options.options :add 'command :label "Color Breakpoint If" :command '(debug-color-breakpointif))
    (.d.menu.options.options :add 'separator)
    (.d.menu.options.options :add 'command :label "Font" :command '(debug-font))
    (.d.menu.options.options :add 'separator)
    (.d.menu.options.options :add 'command :label "Save options" :command '(debug-save-setting))
    (.d.menu.options.options :add 'separator)
    (.d.menu.options.options :add 'check :label "Save on exit" :variable '**save-on-exit**)
    (.d.menu.tools.tools :add 'command :label "Start profiling" :command '(debug-start-profile))
    (.d.menu.tools.tools :add 'command :label "Stop profiling" :command '(debug-stop-profile))
    (tk::pack '.d.menu.file '.d.menu.edit '.d.menu.options '.d.menu.tools :side 'left)
    
    (TK::button '.d.button.step :Text "Step" :command '(debug-step))
    (TK::button '.d.button.stepover :text "Step over" :command '(debug-step-over))
    (TK::button '.d.button.continue :text "Continue" :command '(debug-continue))
    (TK::button '.d.button.breakpoint :text "Breakpoint" :command '(debug-breakpoint))
    (TK::button '.d.button.breakif :text "Break If" :command '(debug-breakif))
    (TK::button '.d.button.watch :text "Watch" :command '(debug-watch))
    (TK::button '.d.button.unwatch :text "UnWatch" :command '(debug-unwatch))
    (TK::button '.d.button.eval :text "Eval" :command '(debug-eval))
    (TK::pack '.d.button.step '.d.button.stepover '.d.button.continue '.d.button.breakpoint '.d.button.breakif '.d.button.watch
	      '.d.button.unwatch
	      '.d.button.eval
	      :side 'left)
    
    (TK::scrollbar '.d.text.xtext :command ".d.text.text xview" :orient "horizontal")
    (TK::scrollbar '.d.text.ytext :command ".d.text.text yview")
    (TK::text '.d.text.text :xscrollcommand ".d.text.xtext set" :yscrollcommand ".d.text.ytext set" :state 'disabled
	      :font **debug-font**)
    (TK::pack '.d.text.xtext :side 'bottom :fill 'x)
    (TK::pack '.d.text.text :side 'left :expand 'yes :fill 'both)
    (TK::pack '.d.text.ytext :side 'left :fill 'y)
    
    (TK::scrollbar '.d.command.xtext :command ".d.command.text xview" :orient "horizontal")
    (TK::scrollbar '.d.command.ytext :command ".d.command.text yview")
    (TK::text '.d.command.text :xscrollcommand ".d.command.xtext set" :yscrollcommand ".d.command.ytext set" :state 'normal
	      :font **debug-font** :height 3)
    (TK::pack '.d.command.xtext :side 'bottom :fill 'x)
    (TK::pack '.d.command.text :side 'left :expand 'yes :fill 'both)
    (TK::pack '.d.command.ytext :side 'left :fill 'y)
    
    
    (TK::scrollbar '.d.result.xtext :command ".d.result.text xview" :orient "horizontal")
    (TK::scrollbar '.d.result.ytext :command ".d.result.text yview")
    (TK::text '.d.result.text :xscrollcommand ".d.result.xtext set" :yscrollcommand ".d.result.ytext set" :state 'disabled
	      :font **debug-font** :height 3)
    (TK::pack '.d.result.xtext :side 'bottom :fill 'x)
    (TK::pack '.d.result.text :side 'left :expand 'yes :fill 'both)
    (TK::pack '.d.result.ytext :side 'left :fill 'y)

    (tk::wm :title ".d" "<debug>")
    
    ;; Bindings of events to functions
    
    (TK::bind '.d.text.text "<ButtonPress-1>" '(debug-select-word TK::%X TK::%Y))
    (TK::bind '.d.text.text "<ButtonPress-2>" '(debug-select-function TK::%X TK::%Y))
    (TK::bind '.d.text.text "<Double-ButtonPress-1>" '(debug-select-list TK::%X TK::%Y))
    (TK::bind '.d.result.text "<ButtonPress-1>" '(debug-select-line TK::%X TK::%Y))

    ;; Sets default things
    (TK::setk **save-on-exit** "1")
    )
  T)



;; ***************************************************************************************************
;; Do a paste of the selection in the debug system , use tmp directory here
;; ***************************************************************************************************

(defun debug-paste ()
  (let ((tmpsource (format NIL "/usr/tmp/~A" (gensym))) (handle NIL) (ok NIL) (selection "")
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (multiple-value-setq (selection ok) (tk::selection :get :return 'string))
    (when ok
      (setf handle (open tmpsource :direction :output))
      (write-string selection handle)
      (close handle)
      (debug-open-file tmpsource))))

;; ***************************************************************************************************
;; Select the word clicked on
;; ***************************************************************************************************

(defun debug-select-word (X Y)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (null **lock**)
      (unwind-protect
	  (progn
	    (setf **lock** T)
	    (let ((begin NIL) (end NIL))
	      ;; Adjust for the position of the window
	      (setf X (- X (TK::winfo :rootx '.d.text.text :return 'number)))
	      (setf Y (- Y (TK::winfo :rooty '.d.text.text :return 'number)))
	      ;; Make the selection
	      (setf begin (.d.text.text :search :-backwards :regexp "[ ():\"]" (format NIL "@~A,~A" X Y) "0.0" :return 'string))
	      (if  (string= begin "") (setf begin "1.0") (setf begin (format NIL "~A + 1 chars" begin)))
	      (setf end (.d.text.text :search :-forwards :regexp "[ ():\"]" (format NIL "@~A,~A" X Y) "end" :return 'string))
	      (when (string= end "") (setf end "end"))
	      (.d.text.text :tag :add 'sel begin end)))
	(setf **lock** NIL)))))

;; **************************************************************************************************
;; Select the line clicked on , adn delete all watchpoints there
;; **************************************************************************************************

(defun debug-select-line (X Y)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf X (- X (TK::winfo :rootx ".d.result.text" :return 'number)))
    (setf Y (- Y (TK::winfo :rooty ".d.result.text" :return 'number)))
    (.d.result.text :tag :add 'sel (format NIL "@~A,~A linestart" X Y) (format NIL "@~A,~A lineend" X Y))))
  

(defun debug-unwatch ()
  (let ((tag-lst NIL) (line 1) (ret (coerce (list #\newline) 'string))
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf tag-lst (.d.result.text :tag :names "sel.first" :return 'list))
    (mapc #'(lambda (tag) (remhash tag **watchpoint-lst**)) tag-lst)
    (.d.result.text :configure :state "normal")
    (.d.result.text :delete "1.0" "end")
    (maphash #'(lambda (tag s-c-b-e)
		 (.d.result.text :insert (format NIL "~A.0" line) (format NIL ">~A " (second s-c-b-e)))
		 (.d.result.text :tag :add tag (format NIL "~A.0 linestart" line) (format NIL "~A.0 lineend" line))
		 (.d.result.text :insert "end" ret)
		 (incf line))
	     **watchpoint-lst**)))
  
  
;; **************************************************************************************************
;; Select the surrounding list in the text
;; **************************************************************************************************


(defun debug-select-list (X Y)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (null **lock**)
      (unwind-protect
	  (let ((pos "")
		(begin "")
		(end "")
		(char "")
		(hooks 0))
	    (declare (string pos begin end char) (fixnum hooks))
	    ;; Adjust for the position of the window
	    (setf X (- X (TK::winfo :rootx ".d.text.text" :return 'number)))
	    (setf Y (- Y (TK::winfo :rooty ".d.text.text" :return 'number)))
	    ;; Find the first opening bracket
	    (setf pos (TK::tk-conc "@" X "," Y))
	    (setf pos (.d.text.text :search :-backwards :regexp "[()]" pos "0.0" :return 'string))
	    (loop
	      (cond ((string= pos "") (return-from debug-select-list))
		    ((string= (.d.text.text :get pos :return 'string) "(")
		     (cond ((zerop hooks)
			    (setf begin pos)
			    (return))
			   (T
			    (setf hooks (1- hooks))
			    (setf pos (TK::tk-conc pos " - 1 chars"))
			    (setf pos (.d.text.text :search :-backwards :regexp "[()]" pos "0.0" :return 'string)))))
		    (T
		     (setf hooks (1+ hooks))
		     (setf pos (TK::tk-conc pos " - 1 chars"))
		     (setf pos (.d.text.text :search :-backwards :regexp "[()]" pos "0.0" :return 'string)))))
	    ;; Find the closing parentheses
	    (setf pos (TK::tk-conc pos "+ 1 chars"))
	    (setf pos (.d.text.text :search :-forwards :regexp "[(\";)]" pos "end" :return 'string))
	    (setf char (.d.text.text :get pos :return 'string))
	    (loop
	      (cond ((string= pos "")
		     (return-from debug-select-list))
		    ((string= char "\"")
	             (setf pos (TK::tk-conc pos " + 1 chars"))
		     (setf pos (.d.text.text :search :-forwards "\"" pos "end" :return 'string))
		     (when (string= pos "") (return-from debug-select-list))
		     (setf pos (TK::tk-conc pos "+ 1 chars")))
		    ((string= char ";")
		     (setf pos (TK::tk-conc pos " lineend + 1 chars")))
		    ((string= char ")")
		     (cond ((zerop hooks)
			    (setf end (TK::tk-conc pos " + 1 chars"))
			    (return))
			   (T
			    (setf hooks (1- hooks))
			    (setf pos (TK::tk-conc pos " + 1 chars")))))
		    (T
		     (setf hooks (1+ hooks))
		     (setf pos (TK::tk-conc pos " + 1 chars"))))
	      (setf pos (.d.text.text :search :-forwards :regexp "[(\";)]" pos "end" :return 'string))
	      (setf char (.d.text.text :get pos :return 'string)))
	    ;; Now select the list
	    (.d.text.text :tag :add 'sel begin end))
	(setf **lock** NIL)))))
	    
		     

;; ***************************************************************************************************
;; Select a function
;; ***************************************************************************************************

(defun debug-select-function (X Y)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (null **lock**)
      (unwind-protect
	  (let ((pos NIL) (begin NIL) (end NIL) (nr 0))
	    (setf **lock** T)
	    ;; Adjust for the position of the window
	    (setf X (- X (TK::winfo :rootx ".d.text.text" :return 'number)))
	    (setf Y (- Y (TK::winfo :rooty ".d.text.text" :return 'number)))
	    ;; Make the selection
	    (setf begin (.d.text.text :search :-backwards :nocase :regexp
				      "\\(defun|\\(defmethod|\\(defgeneric"
				      (format NIL "@~A,~A" X Y) "0.0" :return 'string))
	    (when (string= begin "") (return-from debug-select-function))
	    ;; Find the closing parentheses
	    (setf end (gethash (cons **current-source** begin) **begin-end-function**))
;;	    (setf pos (.d.text.text :search :-forwards :regexp "[(\";)]" (format NIL "~A + 2 chars" begin) "end" :return 'string))
;;	    (loop (cond ((string= pos "") (return-from debug-select-function))
;;			((string= (.d.text.text :get pos :return 'string) "\"")
;;			 (setf pos (.d.text.text :search :-forwards :regexp "[\"]"
;;						 (format NIL "~A + 1 chars" pos) "end" :return 'string))
;;			 (setf pos (format NIL "~A + 1 chars" pos)))
;;			((string= (.d.text.text :get pos :return 'string) ";")
;;			 (setf pos (.d.text.text :search :-forwards :regexp "[(\";)]"
;;						 (format NIL "~A  lineend" pos) "end" :return 'string)))
;;			((string= (.d.text.text :get pos :return 'string) ")")
;;			 (cond ((zerop nr)
;;				(setf end (format NIL "~A + 1 chars" pos))
;;				(return))
;;			       (T
;;				(setf pos (.d.text.text :search :-forwards :regexp "[(\";)]"
;;							(format NIL "~A + 1 chars" pos) "end" :return 'string))
;;				(setf nr (1- nr)))))
;;			(T
;;			 (when (string= (.d.text.text :get pos :return 'string) "(") (setf nr (1+ nr)))
;;			 (setf pos (.d.text.text :search :-forwards :regexp "[(\";)]"
;;						 (format NIL "~A + 1 chars" pos) "end" :return 'string)))))
	    ;; Do the actual selection
	    (.d.text.text :tag :remove 'sel "0.1" "end")
	    (.d.text.text :tag :add 'sel begin end :return))
	(setf **lock** NIL)))))




;; ***************************************************************************************************
;; Select the font ot be used in the debugger
;; ***************************************************************************************************

(defun debug-font ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (choose-font #'(lambda (font)
		     (setf **debug-font** font)
		     (.d.text.text :configure :font **debug-font**)
		     (.d.result.text :configure :font **debug-font**)
		     (.d.command.text :configure :font **debug-font**)) **debug-font**)))
  
;; ***************************************************************************************************
;; Select the colors to be used in the debugger
;; ***************************************************************************************************

(defun debug-color-break ()
  (choose-color #'(lambda (color) (setf **break-color** color))  **break-color**))

(defun debug-color-breakpoint ()
  (choose-color #'(lambda (color) (setf **breakpoint-color** color)) **breakpoint-color**))

(defun debug-color-breakpointif ()
  (choose-color #'(lambda (color) (setf **breakpointif-color** color)) **breakpointif-color**))
  
(defun debug-open-function ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (choose-function #'(lambda (function source)
			 (setf **current-source** source)
			 (TK::wm :title ".d" **current-source**)
			 (debug-open-file-load-source source)
			 (debug-show-breakpoints source)
			 (let ((begin (.d.text.text :search :-forwards :nocase :regexp
						    (format NIL "\\(defun( )*~A|\\(defmethod( )*~A" function function)
						    "0.0" "end" :return 'string)))
			   (when (not (string= begin ""))
			     (.d.text.text :yview :-pickplace begin)))))))

		               

;; ***************************************************************************************************
;; Open a source file , this is , parse it and load it in the debug window
;; ***************************************************************************************************

(defun debug-open ()
  (choose-file #'debug-open-file :directory (namestring (truename "."))))

;; Callback function of the choose-file widget

(defun debug-open-file (file)
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **current-source** "")
    (debug-close-source file)
    (setf **current-source** file)
    (tk::wm :title ".d" **current-source**)
    (when (directory-p **current-source**)
      (TK::tk-dialog '.dialog "User error" "Selected item is not a file" "" 0 "Ok")
      (return-from debug-open-file NIL))
    (debug-open-file-load **current-source**)))


;; Load the file , add the debugging code and display the file in the text window

(defun debug-open-file-load (file)
  (let ((source NIL) (position NIL))
    (multiple-value-setq (source position) (load-source file))
    (mapc #'(lambda (x) (eval x)) (add-code file source position))
    (debug-open-file-load-source file)))

(defun debug-open-file-load-source (file)
  (let ((line "") (ret (coerce (list #\newline) 'string))
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (.d.text.text :configure :state "normal")
    (.d.text.text :delete "1.0" "end")
    (with-open-file (h file :direction :input :if-does-not-exist NIL)
		    (loop (when (null (setf line (read-line h NIL NIL))) (return))
			  (.d.text.text :insert "end" line :return)
			  (.d.text.text :insert "end" ret :return)))
    (.d.text.text :configure :state "disabled")))

      
;; ************************************************************************************************
;; Step through the code
;; ************************************************************************************************

(defun debug-step ()
  (setf **step** T)
  (setf **step-over** NIL)
  (setf **break** NIL))

(defun debug-step-over ()
  (setf **step-over** T)
  (setf **step** NIL)
  (setf **break** NIL))

(defun debug-continue ()
  (setf **step** NIL)
  (setf **step-over** NIL)
  (setf **break** NIL))


;; ************************************************************************************************
;; Sets a breakpoint
;; ************************************************************************************************

(defun debug-breakpoint ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (let ((begin (.d.text.text :index "sel.first + 1 chars" :return 'string))
	  (end (.d.text.text :index "sel.last" :return 'string))
	  (tag NIL))
      ;; If we can't set a breakpoint , just return
      (when (null (gethash **current-source** **source-lst**)) (return-from debug-breakpoint))
      (when (null (gethash (list begin end) (gethash **current-source** **source-lst**)))
	(return-from debug-breakpoint))
      ;; Set or remove breakpoint
      (setf tag (gethash (list **current-source** begin end) **breakpoint-lst**))
      (cond (tag
	     (.d.text.text :tag :delete tag)
	     (remhash (list **current-source** begin end) **breakpoint-lst**))
	    (T
	     (setf tag (gensym))
	     (.d.text.text :tag :add tag (format NIL "~A - 1 chars" begin) end)
	     (.d.text.text :tag :configure tag :foreground **breakpoint-color** 1)
	     (setf (gethash (list **current-source** begin end) **breakpoint-lst**) tag))))))

  
    
;; ************************************************************************************************
;; Sets a conditional breakpoint
;; ************************************************************************************************

(defun debug-breakif ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (string= "1.0"  (.d.text.text :index "end - 1 chars" :return 'string)) (return-from debug-breakif))
    (let ((begin (.d.text.text :index "sel.first + 1 chars" :return 'string)) (end (.d.text.text :index "sel.last" :return 'string)) (tag NIL)
	  (*BREAK-ENABLE* NIL) (*error-output* (make-string-output-stream)) (ok NIL) (code NIL))
      ;; If we can't set a breakpoint , just return
      (when (null (gethash **current-source** **source-lst**)) (return-from debug-breakif))
      (when (null (gethash (list begin end) (gethash **current-source** **source-lst**)))
	(return-from debug-breakif))
      (unwind-protect
	  (progn
	    (setf code (read-from-string (.d.command.text :get "1.0" "end" :return 'string)))
	    ;; Set or remove breakpoint
	    (setf tag (first (gethash (list **current-source** begin end) **ifbreakpoint-lst**)))
	    (cond (tag
		   (.d.text.text :tag :delete tag)
		   (remhash (list **current-source** begin end) **ifbreakpoint-lst**))
		  (T
		   (setf tag (gensym))
		   (.d.text.text :tag :add tag (format NIL "~A - 1 chars" begin) end)
		   (.d.text.text :tag :configure tag :foreground **breakpointif-color** 1)
		   (setf (gethash (list **current-source** begin end) **ifbreakpoint-lst**) (cons tag code))))
	    (setf ok T))
	(when (null ok) (TK::tk-dialog ".error" "Error in condition" (get-output-stream-string *error-output*) "" 0 "Ok"))))))
  


;; ************************************************************************************************
;; Sets watchpoints
;; ************************************************************************************************


(defun debug-watch ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (string= "1.0"  (.d.text.text :index "end - 1 chars" :return 'string)) (return-from debug-watch))
    (let ((ok NIL)
#+gcl	  
	  (*BREAK-ENABLE* NIL)                     ;; Disable debugger gcl in read-from-string
#+:CMU
          (*debug-hook* #'(lambda (x y) T))        ;; Disable debugger cmucl in read-from-string
	  
	 )
      (unwind-protect
	  (let ((begin NIL)
		(end NIL)
		(variable (read-from-string (.d.text.text :get "sel.first" "sel.last" :return 'string)))
		(tag (read-from-string (princ-to-string (gensym)))) (line 1)
		(ret (coerce (list #\newline) 'string)))
	    (cond ((symbolp variable)
		   (setf ok T)
		   ;; Define the watching of an expression
		   (setf (gethash tag **watchpoint-lst**) (cons **current-source** variable))
		   ;; Redisplay all the watchpoints
		   (.d.result.text :configure :state "normal")
		   (.d.result.text :delete "1.0" "end")
		   (.d.command.text :delete "1.0" "end")
		   (maphash #'(lambda (tag s-v)
				(when (string= **current-source** (first s-v))
				  (.d.result.text :insert (format NIL "~A.0" line) (format NIL ">~A " (rest s-v)))
				  (.d.result.text :tag :add tag (format NIL "~A.0 linestart" line) (format NIL "~A.0 lineend" line))
				  (.d.result.text :insert "end" ret)
				  (incf line)))
			    **watchpoint-lst**)
		   (.d.result.text :configure :state "disabled")
		   ;; Display allready the value of the watchpoint
		   (when **break** (debug-watch-point **current-source** **env**)))
		  (T
		   (setf ok NIL)))
	    (when (null ok)
	      (TK::tk-dialog ".error" "Error in watchpoint" "You must select a variable for a watchpoint" "" 0 "Ok")))))))


;; ************************************************************************************************
;; Eval expression
;; ************************************************************************************************

#+gcl
(defun safe-evalhook (exp env)
  (let ((*debug-io* (open "/dev/null" :direction :IO))
	(*BREAK-ENABLE* NIL)
	(er NIL)
	(val NIL))
    (multiple-value-setq (er val) (si::error-set `(evalhook ',exp NIL NIL ',env)))
    (if er (values val er) (values val NIL))))

#+gcl
(defun debug-eval ()
  (let ((*ERROR-OUTPUT* (make-string-output-stream))
	(eval-exp NIL)
	(result-exp NIL)
	(ok NIL))
    (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
      (multiple-value-setq (eval-exp ok)
	(safe-evalhook '(read-from-string (.d.command.text :get "1.0" "end" :return 'string)) NIL))
;;      (multiple-value-setq (eval-exp ok)
;;	(ignore-errors (first (multiple-value-list
;;			       (read-from-string (.d.command.text :get "1.0" "end" :return 'string))))))
      (when ok (TK::tk-dialog ".error" "Error reading exp" (get-output-stream-string *ERROR-OUTPUT*) "" 0 "Ok")
	    (return-from debug-eval)))
;;    (multiple-value-setq (result-exp ok)
;;      (ignore-errors (multiple-value-list (evalhook eval-exp NIL NIL **env**))))
    (multiple-value-setq (result-exp ok)
      (safe-evalhook eval-exp **env**))
    (when ok (TK::tk-dialog ".error" "Error evaluating exp" (get-output-stream-string *ERROR-OUTPUT*) "" 0 "Ok")
	  (return-from debug-eval))
    (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**)
	  (ret (coerce (list #\newline) 'string)))
      (.d.result.text :configure :state "normal")
      (.d.result.text :insert "end" (princ-to-string result-exp))
      (.d.result.text :insert "end" ret)
      (.d.result.text :yview :-pickplace "end")
      (.d.result.text :configure :state "disabled"))))

#+:CMU
(defun debug-eval ()
  (let ((*ERROR-OUTPUT* (make-string-output-stream))
	(eval-exp NIL)
	(result-exp NIL)
	(ok NIL))
    (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
      (multiple-value-setq (eval-exp ok)
	(ignore-errors (first (multiple-value-list
			       (read-from-string (.d.command.text :get "1.0" "end" :return 'string))))))
      (when ok (TK::tk-dialog ".error" "Error reading exp" (get-output-stream-string *ERROR-OUTPUT*) "" 0 "Ok")
	    (return-from debug-eval)))
    (multiple-value-setq (result-exp ok)
      (ignore-errors (multiple-value-list (eval-in-frame **env** eval-exp))))
    (when ok (TK::tk-dialog ".error" "Error evaluating exp" (get-output-stream-string *ERROR-OUTPUT*) "" 0 "Ok")
	  (return-from debug-eval))
    (when (null (rest result-exp)) (setf result-exp (first result-exp)))
    (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**)
	  (ret (coerce (list #\newline) 'string)))
      (.d.result.text :configure :state "normal")
      (.d.result.text :insert "end" (princ-to-string result-exp))
      (.d.result.text :insert "end" ret)
      (.d.result.text :yview :-pickplace "end")
      (.d.result.text :configure :state "disabled"))))
	   
;; ************************************************************************************************
;; Process watch points
;; ************************************************************************************************


(defun debug-watch-point (source env)
  (let ((line 1)
	(ret (coerce (list #\newline) 'string))
	(*STANDARD-OUTPUT* **NULL-OUTPUT**)
	(*STANDARD-ERROR* **NULL-OUTPUT**))
    (.d.command.text :tag :remove 'sel "1.0" "end")
    (.d.result.text :configure :state "normal")
    (.d.result.text :delete "1.0" "end")
    (maphash #'(lambda (tag s-v)
		 (let ((w-source (first s-v))
		       (w-variable (rest s-v))
		       (ok NIL)
		       (result NIL))
		   (when (string= w-source source)
		     ;; Display the results
		     (if (not (eq (setf result (debug-var-content w-variable env)) **UNKNOWN**))
			 (.d.result.text :insert (format NIL "~A.1" line) (format NIL "~A::~A~A" w-variable result ret))
		       (.d.result.text :insert (format NIL "~A.1" line) (format NIL "~A IS UNDEFINED~A" w-variable ret)))
		     (incf line))))
	     **watchpoint-lst**)))

#+gcl
(defun debug-var-content (var env)
  (if (or (boundp var)               ;; variable is special 
	  (assoc var (first env)))   ;; variable is defined in environment
      (evalhook var NIL NIL env)
    **UNKNOWN**))
  
#+:CMU
(defun debug-var-content (exp env)
  (let ((result NIL)
	(error NIL)
	(value NIL)
	(location (frame-code-location env))
	(vars (debug-function-symbol-variables (frame-debug-function env) exp))
	(*ERROR-OUTPUT* **NULL-OUTPUT**))
    (mapc #'(lambda (x)
	      (when (and (eq (debug-variable-symbol x) exp)
			 (eq (debug-variable-validity x location) :valid))
		(push (debug-variable-value x env) result)))
	  vars)
    (cond ((null result) **UNKNOWN**)
	  ((null (rest result)) (first result))
	  (T (format NIL "AMBIGUOUS VALUES : ~A" result)))))




;; ************************************************************************************************
;; Debugging code executed in case of a breakpoint
;; ************************************************************************************************

(defun debug-point (source begin end env)
  (let ((*BREAK-ENABLE* NIL)  ;; Avoid that we generate a lisp breakpoint
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **env** env)                    ;; Set the current environment
    ;; Check if we have to do profiling code
    (when **profile** (debug-point-profile source begin end))
    ;; Check if we have really a breakpoint
    (when (or **step**  ;; We are stepping
	      (gethash (list source begin end) **breakpoint-lst**) ;; We have a set breakpoint
	      (and (gethash (list source begin end) **ifbreakpoint-lst**) ;; We have a conditional breakpoint
		   (debug-point-if (rest (gethash (list source begin end) **ifbreakpoint-lst**)) **env**))
	      (and **step-over**
		   (or (= (.d.text.text :compare **begin** ">" begin :return 'number) 1)
		       (= (.d.text.text :compare end ">" **end** :return 'number) 1))))
      ;; We have a breakpoint , make sure that the correct source is loaded in the text box
      (when (not (string= source **current-source**))
	(setf **current-source** source)
	(debug-open-file-load-source **current-source**)
	(debug-show-breakpoints **current-source**))
      ;; Check for possible watch points
      (debug-watch-point source env)
      ;; Set the line context so that the debug line is visible and mark the area so that we see were we are
      (.d.text.text :yview :-pickplace begin)
      (.d.text.text :tag :add "break" (format NIL "~A - 1 chars" begin) end)
      (.d.text.text :tag :configure "break" :background **break-color**)
      ;; Wait for the command to go to the next code
      (unwind-protect
	  (progn
	    (setf **break** T)
	    (loop (when (null **break**) (return)) (sleep 0.3)))
	(.d.text.text :tag :delete "break")
	(setf **begin** begin)
	(setf **end** end))
      (setf **break** NIL))))


;; **************************************************************************************
;; Test if we have fulfilled the conditional of a conditional breakpoint
;; **************************************************************************************


#+gcl
(defun debug-point-if (code env)
  (let ((ok NIL)
	(result NIL)
	(*error-output* (make-string-output-stream))
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (multiple-value-setq (result ok) (safe-evalhook code env))
    (cond (ok
	   (TK::tk-dialog ".error" "Eval condition error" (get-output-stream-string *error-output*) "" 0 "Ok")
	   NIL)
	  (T
	   result))))


#+:CMU
(defun debug-point-if (code env)
  (let ((ok NIL)
	(result NIL)
	(*ERROR-OUTPUT* (make-string-output-stream))
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (multiple-value-setq (result ok)
      (ignore-errors (first (multiple-value-list (eval-in-frame env code)))))
    (cond (ok
	   (TK::tk-dialog ".error" "Eval condition error" (get-output-stream-string *error-output*) "" 0 "Ok")
	   NIL)
	  (T
	   result))))

  

;; *****************************************************************************************
;; Remark the breakpoints and conditional breakpoints on the new source
;; *****************************************************************************************

(defun debug-show-breakpoints (source)
  ;; Remark the breakpoints
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (maphash #'(lambda (s-b-e tag)
		 (when (string= (first s-b-e) source)
		   (.d.text.text :tag :add tag (format NIL "~A - 1 chars" (second s-b-e)) (third s-b-e))
		   (.d.text.text :tag :configure tag :foreground **breakpoint-color** 1)))
	     **breakpoint-lst**)
    ;; Remark the conditional breakpoints
    (maphash #'(lambda (s-b-e tag-code)
		 (when (string= (first s-b-e) source)
		   (.d.text.text :tag :add (first tag-code) (format NIL "~A - 1 chars" (second s-b-e)) (third s-b-e))
		   (.d.text.text :tag :configure (first tag-code) :foreground **breakpointif-color** 1)))
	     **ifbreakpoint-lst**)))

;; *****************************************************************************************
;; Do some profile counting
;; *****************************************************************************************

(defun debug-point-profile (source begin end)
  (let ((count (gethash (list source begin end) **profile-lst**)))
    (when count
      (setf **profile-max** (max **profile-max** (1+ count)))
      (setf (gethash (list source begin end) **profile-lst**) (1+ count)))))


;; ****************************************************************************************
;; Start the profiling code
;; ****************************************************************************************

(defun debug-start-profile ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (not **profile**)
      (TK::scale '.d.text.scale :from 1 :to 10 :command '(debug-show-profile-scale))
      (tk::pack '.d.text.scale))
    (setf **profile-max** 0)
    (setf **profile** T)
    (setf **profile-lst** (make-hash-table :test #'equal))
    ;; For the current source , color all pieces of code which could be profiled
    (when (gethash **current-source** **source-lst**)
      (maphash #'(lambda (begin-end dummy)
		   (setf (gethash (list **current-source** (first begin-end) (second begin-end)) **profile-lst**) 0))
	       (gethash **current-source** **source-lst**)))))

(defun debug-stop-profile ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (setf **profile-max** 0)
    (setf **profile** NIL)
    (setf **profile-lst** (make-hash-table :test #'equal))
    (.d.text.text :tag :delete "profile")
    (.d.text.text :tag :delete "profileshow")
    (TK::destroy '.d.text.scale)))

;; *****************************************************************************************
;; Show all the code executed more or equal the shale count
;; *****************************************************************************************

(defun debug-show-profile-scale ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (unwind-protect
	(progn
	  (.d.text.scale :configure :state "disabled")
	  (let ((count (.d.text.scale :get :return 'number)) (next "") (ok NIL))
	    (.d.text.text :tag :delete "profileshow")
	    (.d.text.text :tag :delete "profile")
	    (.d.text.scale :configure :to (max 10 **profile-max**))
	    (maphash #'(lambda (source-begin-end cnt)
			 (when (and (string= **current-source** (first source-begin-end))
				    (>= cnt count))
			   (multiple-value-setq (next ok) (.d.text.text :search :-forwards :regexp "[ )]"
									(second source-begin-end) "end" :return 'string))
			   (when (not ok) (setf next "end"))
			   (.d.text.text :tag :add "profileshow" (second source-begin-end) next :return T)))
		     **profile-lst**)
	    (.d.text.text :tag :configure "profileshow" :background **profile-color**)))
      (.d.text.scale :configure :state "normal"))))

  
;; *****************************************************************************************
;; Exit the debugging sequence
;; *****************************************************************************************

(defun debug-exit ()
  (let ((*STANDARD-OUTPUT* **NULL-OUTPUT**))
    ;; If needed save the configuration settings
    (when (string= **save-on-exit** "1") (debug-save-setting))
    ;; Clean up all the sources , and reload the nondebugged versions
    (maphash #'(lambda (source dummy) (debug-close-source source) (load source)) **source-lst**)
    (close **NULL-OUTPUT**)
    ;; Reinitialize all the environment variables
    (setf **source-lst** (make-hash-table :test #'equal))
    (setf **fun-source-lst** (make-hash-table :test #'eq))
    (setf **breakpoint-lst** (make-hash-table :test #'equal))
    (setf **ifbreakpoint-lst** (make-hash-table :test #'equal))
    (setf **watchpoint-lst** (make-hash-table :test #'equal))
    ;; Disable the current breakpoints
    (debug-continue)
    ;; Close the debug screen
    (TK::destroy ".d" :return))
    T)




;; ******************************************************************************************************
;; Cleanup the source information in the debugging system
;; ******************************************************************************************************

(defun debug-close-source (source)
  ;; Correct current source
  (when (string= source **current-source**) (setf **current-source** ""))
  ;; Stop current breakpoint
  (debug-continue)
  ;; Cleanup all the hashtables
  (remhash source **source-lst**)
  (maphash #'(lambda (key src) (when (string= src source) (remhash key **fun-source-lst**))) **fun-source-lst**)
  (maphash #'(lambda (key tag) (when (string= (first key) source) (remhash key **breakpoint-lst**))) **breakpoint-lst**)
  (maphash #'(lambda (key tag) (when (string= (first key) source) (remhash key **ifbreakpoint-lst**))) **ifbreakpoint-lst**)
  (maphash #'(lambda (key tag) (when (string= (first tag) source) (remhash key **watchpoint-lst**))) **watchpoint-lst**)
  (maphash #'(lambda (key tag) (when (string= (first key) source) (remhash key **begin-end-function**))) **begin-end-function**))

;; *******************************************************************************************
;; Close a source
;; *******************************************************************************************

(defun debug-close ()
  (let ((source **current-source**)
	(*STANDARD-OUTPUT* **NULL-OUTPUT**))
    (when (string= source "") (return-from debug-close))
    (debug-close-source source)
    (.d.text.text :configure :state "normal")
    (.d.text.text :delete "1.0" "end")
    (.d.text.text :configure :state "disabled")
    (load source)))


;; *******************************************************************************************
;; Save all settings of the debug system in $HOME/.lispdebug
;; *******************************************************************************************

(defun debug-save-setting ()
  (with-open-file (h (format NIL "~A.lispdebug" (namestring (user-homedir-pathname))) :direction :output
		             :if-exists :overwrite :if-does-not-exist :create)
		  (format h "(setf DEBUGGER::**break-color** ~C~A~C)" #\" **break-color** #\") (terpri h)
		  (format h "(setf DEBUGGER::**breakpoint-color** ~C~A~C)" #\" **breakpoint-color** #\") (terpri h)
		  (format h "(setf DEBUGGER::**breakpointif-color** ~C~A~C)" #\" **breakpointif-color** #\") (terpri h)		  
		  (format h "(setf DEBUGGER::**geometry** ~C~A~C)" #\" (tk::wm :geometry ".d") #\") (terpri h)
		  (format h "(setf DEBUGGER::**font** ~C~A~C)" #\" **font** #\") (terpri h)))


(defun debug-load-setting ()
  (let ((code NIL) (h NIL))
    ;; Init debug system
    (setf **source-lst** (make-hash-table :test #'equal)) 
    (setf **fun-source-lst** (make-hash-table :test #'equal))
    (setf **begin-end-function** (make-hash-table :test #'equal))
    (setf **current-source** "")
    (setf **function** NIL)
    (setf **lock** NIL)
    (setf **env** NIL)
    (setf **step** NIL)
    (setf **step-over** NIL)
    (setf **begin** "0.1")
    (setf **end** "0.1")
    (setf **breakpoint-lst** (make-hash-table :test #'equal))
    (setf **ifbreakpoint-lst** (make-hash-table :test #'equal))
    (setf **watchpoint-lst** (make-hash-table :test #'equal))
    (setf **profile-lst** (make-hash-table :test #'equal))
    (setf **profile-max** 0)
    (setf **break** NIL)
    (setf **break-color** "cadetblue1")
    (setf **breakpoint-color** "red")
    (setf **breakpointif-color** "brown")
    (setf **profile-base-color** "white")
    (setf **profile-color** "cyan")
    (setf **geometry** "644x630+13+7")
    (setf **debug-font** "-*-helvetica-medium-r-normal-*-120-*-*-*-*-*-*")
    (setf **save-on-exit** "1")
    (setf **profile** NIL)
    ;; Load init file
    (setf h (open (format NIL "~A.lispdebug" (namestring (user-homedir-pathname))) :direction :input :if-does-not-exist NIL))
    (when (null h) (return-from debug-load-setting))
    (loop
     (when (null (setf code (read h NIL NIL))) (return))
     (eval code))
    (close h)))



;; *****************************************************************************
;; Package stuff
;; *****************************************************************************

(in-package "USER")
(SHADOWING-IMPORT 'DEBUGGER::debug "USER")
