;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime/Read/reader.scm              */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Tue Dec 27 11:16:00 1994                          */
;*    Last change :  Fri Nov 20 06:45:36 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    Bigloo's reader                                                  */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __reader

   (import  (__error                   "Llib/error.scm"))
   
   (use     (__type                    "Llib/type.scm")
	    (__bigloo                  "Llib/bigloo.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__structure               "Llib/struct.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__dsssl                   "Llib/dsssl.scm")
	    (__ucs2                    "Llib/ucs2.scm")
	    (__unicode                 "Llib/unicode.scm")
	    (__bexit                   "Llib/bexit.scm")
	    (__binary                  "Llib/binary.scm")

	    (__rgc                     "Rgc/rgc.scm")

	    (__r4_numbers_6_5_fixnum   "Ieee/fixnum.scm")
	    (__r4_numbers_6_5_flonum   "Ieee/flonum.scm")
	    (__r4_numbers_6_5          "Ieee/number.scm")
	    (__r4_characters_6_6       "Ieee/char.scm")
	    (__r4_equivalence_6_2      "Ieee/equiv.scm")
	    (__r4_booleans_6_1         "Ieee/boolean.scm")
	    (__r4_symbols_6_4          "Ieee/symbol.scm")
	    (__r4_strings_6_7          "Ieee/string.scm")
	    (__r4_pairs_and_lists_6_3  "Ieee/pair-list.scm")
	    (__r4_control_features_6_9 "Ieee/control.scm")
	    (__r4_vectors_6_8          "Ieee/vector.scm")
	    (__r4_ports_6_10_1         "Ieee/port.scm")
	    (__r4_input_6_10_2         "Ieee/input.scm")
	    (__r4_output_6_10_3        "Ieee/output.scm")
	    (__evenv                   "Eval/evenv.scm"))
   
   (foreign (macro obj unspec               "BUNSPEC")
	    (macro obj boptional            "BOPTIONAL")
	    (macro obj brest                "BREST")
	    (macro obj bkey                 "BKEY")
	    (macro obj make-cnst (long)     "BCNST")
	    (export *bigloo-case-sensitive* "bigloo_case_sensitive"))
   
   (export  *bigloo-interpreter*
	    *bigloo-case-sensitive*
	    (read . port)
	    (read-case-sensitive . port)
	    (read-case-insensitive . port)
	    (reader-reset!)))

;*---------------------------------------------------------------------*/
;*    *bigloo-case-senstive* ...                                       */
;*---------------------------------------------------------------------*/
(define *bigloo-case-sensitive* *bigloo-case-sensitive*)
   
;*---------------------------------------------------------------------*/
;*    *bigloo-interpreter* ...                                         */
;*---------------------------------------------------------------------*/
(define *bigloo-interpreter* #f)

;*---------------------------------------------------------------------*/
;*    Les variables de control du lecteur                              */
;*---------------------------------------------------------------------*/
(define *par-open*  0)
(define *bra-open*  0)

;*---------------------------------------------------------------------*/
;*    Parenthesis mismatch (or unclosing) errors.                      */
;*---------------------------------------------------------------------*/
(define *list-error-level* 20)
(define *list-errors*      (make-vector *list-error-level* #unspecified))
(define *vector-errors*    (make-vector *list-error-level* #unspecified))

;*---------------------------------------------------------------------*/
;*    Control variables.                                               */
;*---------------------------------------------------------------------*/
(define *position?*   #f)
(define *line-number* 1)
(define *dot-symbol*  (string->symbol ";"))

;*---------------------------------------------------------------------*/
;*    reader-reset! ...                                                */
;*---------------------------------------------------------------------*/
(define (reader-reset!)
   (set! *line-number* 1)
   (set! *par-open* 0)
   (set! *bra-open* 0))

;*---------------------------------------------------------------------*/
;*    *bigloo-grammar* ...                                             */
;*---------------------------------------------------------------------*/
(define *bigloo-grammar*
   (regular-grammar ((float    (or (: (* digit) "." (+ digit))
			 	   (: (+ digit) "." (* digit))))
		     (letter   (in ("azAZ") (#a128 #a255)))
		     (special  (in "!@~$%^&*></-_+\\|=?.:"))
		     (kspecial (in "!@~$%^&*></-_+\\|=?."))
		     (quote    (in "\",'`"))
		     (paren    (in "()[]{}"))
		     (id       (: (* digit)
				  (or letter special)
				  (* (or letter special digit (in ",'`")))))
		     (kid      (: (* digit)
				  (or letter kspecial)
				  (* (or letter kspecial digit (in ",'`")))))
		     (blank    (in #\Space #\Tab #a012 #a013)))

      ;; newlines
      ((+ #\Newline)
       (set! *line-number* (+fx *line-number* (the-length)))
       (ignore))

      ;; blank lines
      ((+ blank)
       (ignore))

      ;; comments
      ((: ";" (* all))
       (ignore))

      ;; the interpreter header or the dsssl named constants
      ((: "#!" (+ (in digit letter special ",'`")))
       (let* ((str (the-string)))
	  (cond
	     ((string=? str "#!optional")
	      boptional)
	     ((string=? str "#!rest")
	      brest)
	     ((string=? str "#!key")
	      bkey)
	     (else
	      (set! *bigloo-interpreter* #t)
	      (ignore)))))

      ;; characters
      ((: "#a" (= 3 xdigit))
       (let ((string (the-string)))
	  (if (not (=fx (the-length) 5))
	      (error/location "read"
			      "Illegal ascii character"
			      string
			      (input-port-name     (the-port))
			      (input-port-position (the-port)))
	      (integer->char (string->integer (the-substring 2 5))))))
      ((: "#\\" (or letter digit special (in "#; " quote paren)))
       (string-ref (the-string) 2))
      ((: "#\\" (>= 2 letter))
       (let ((char-name (string->symbol
			 (string-upcase!
			  (the-substring 2 (the-length))))))
	  (case char-name
	     ((NEWLINE)
	      #\Newline)
	     ((TAB)
	      #\tab)
	     ((SPACE)
	      #\space)
	     ((RETURN)
	      (integer->char 13))
	     (else
	      (error/location "read"
			      "Illegal character"
			      (the-string)
			      (input-port-name     (the-port))
			      (input-port-position (the-port)))))))

      ;; pointers
      ((: "#p" (+ xdigit))
       (let ((address (the-substring 2 (the-length))))
	  (pragma "(obj_t)(strtol(BSTRING_TO_STRING($1), 0, 16))" address)))

      ;; ucs-2 characters
      ((: "#u" (= 4 xdigit))
       (integer->ucs2 (string->integer (the-substring 2 6) 16)))

      ;; strings
      ((: "\"" (* (or (out #a000 #\\ #\") (: #\\ all))) "\"")
       (escape-scheme-string (the-substring 1 (-fx (the-length) 1))))
      ;; foreign strings of char
      ((: "#\"" (* (or (out #a000 #\\ #\") (: #\\ all))) "\"")
       (escape-C-string (the-substring 1 (-fx (the-length) 1))))
      ;; ucs2 strings
      ((: "#u\"" (* (or (out #a000 #\\ #\") (: #\\ all))) "\"")
       (utf8-string->ucs2-string (the-substring 2 (the-length))))

      ;; fixnums
      ((: (? (in "-+")) (+ digit))
       (the-fixnum))
      ((: "#o" (? (in "-+")) (+ (in ("07"))))
       (string->integer (the-substring 2 (the-length)) 8))
      ((: "#d" (? (in "-+")) (+ (in ("09"))))
       (string->integer (the-substring 2 (the-length)) 10))
      ((: "#x" (? (in "-+")) (+ (in (uncase (in ("09af"))))))
       (string->integer (the-substring 2 (the-length)) 16))
      ((: "#e" (? (in "-+")) (+ digit))
       (string->elong (the-substring 2 (the-length)) 10))
      ((: "#l" (? (in "-+")) (+ digit))
       (string->llong (the-substring 2 (the-length)) 10))

      ;; flonum
      ((: (? (in "-+"))
	  (or float
	      (: (or float (+ digit)) (in "eE") (? (in "+-")) (+ digit))))
       (the-flonum))

      ;; end of pairs
      ((when (rgc-context? 'pair) (: "." (* blank) ")"))
       (error/location "read"
		       "Illegal pair"
		       (the-string)
		       (input-port-name     (the-port))
		       (input-port-position (the-port))))

      ;; doted pairs
      ((when (rgc-context? 'pair) ".")
       *dot-symbol*)
      ("."
       (error/location "read"
		       "Illegal token"
		       #\.
		       (input-port-name     (the-port))
		       (input-port-position (the-port))))

      ;; unspecified and eof-object
      ((: "#" (in "ue") (+ (in "nspecified-objt")))
       (let ((symbol (string->symbol
		      (string-upcase!
		       (the-substring 1 (the-length))))))
	  (case symbol
	     ((UNSPECIFIED)
	      unspec)
	     ((EOF-OBJECT)
	      beof)
	     (else
	      (error/location "read"
			      "Illegal identifier"
			      symbol
			      (input-port-name     (the-port))
			      (input-port-position (the-port)))))))

      ;; booleans
      ((: "#" (uncase #\t))
       #t)
      ((: "#" (uncase #\f))
       #f)

      ;; constants
      ((: "#<" (+ (or digit (uncase (in "afAF")))) ">")
       (if (not (=fx (the-length) 7))
	   (error/location "read"
			   "Illegal constant"
			   (the-string)
			   (input-port-name     (the-port))
			   (input-port-position (the-port)))
	   (make-cnst (string->integer (the-substring 2 6) 16))))

      ;; keywords
      ((or (: ":" kid) (: kid ":"))
       ;; since the keyword expression is also matched by the id
       ;; rule, keyword rule has to be placed before the id rule.
       (the-keyword))

      ;; identifiers
      (id
       ;; this rule has to be placed after the rule matching the `.' char
       (the-symbol))

      ;; quotations 
      ("'"
       (if *position?*
	   (econs 'quote
		  (cons (ignore) '())
		  (list 'at
			(input-port-name (the-port))
			(input-port-position (the-port))
			*line-number*))
	   (cons 'quote (cons (ignore) '()))))
      ("`"
       (if *position?*
	   (econs 'quasiquote
		  (cons (ignore) '())
		  (list 'at
			(input-port-name (the-port))
			(input-port-position (the-port))
			*line-number*))
	   (cons 'quasiquote (cons (ignore) '()))))
      (","
       (if *position?*
	   (econs 'unquote
		  (cons (ignore) '())
		  (list 'at
			(input-port-name (the-port))
			(input-port-position (the-port))
			*line-number*))
	   (cons 'unquote (cons (ignore) '()))))
      (",@"
       (if *position?*
	   (econs 'unquote-splicing
		  (cons (ignore) '())
		  (list 'at
			(input-port-name (the-port))
			(input-port-position (the-port))
			*line-number*))
	   (cons 'unquote-splicing (cons (ignore) '()))))

      ;; lists
      ((in "([")
       (let ((open-key *par-open*)
	     (pos      (input-port-position (the-port)))
	     (line     *line-number*))
	  ;; if possible, we store the opening parenthesis.
	  (if (and (vector? *list-errors*)
		   (<fx open-key (vector-length *list-errors*)))
	      (c-vector-set! *list-errors* open-key pos))
	  ;; and then, we compute the result list...
	  (set! *par-open* (+fx 1 *par-open*))
	  (rgc-context 'pair)
	  (let loop-pair ((walk (ignore))
			  (pos  pos)
			  (line line))
	     (cond
		((eq? walk *dot-symbol*)
		 ;; une pair pointee
		 (rgc-context)
		 (let ((cdr (ignore)))
		    (ignore)
		    (if (=fx open-key *par-open*)
			(begin
			   (rgc-context 'pair)
			   cdr)
			(error/location "read"
					"Illegal pair"
					cdr
					(input-port-name     (the-port))
					(input-port-position (the-port))))))
		((=fx open-key *par-open*)
		 (if (=fx open-key 0)
		     (rgc-context))
		 '())
		(else
		 (let ((new-pos  (input-port-position (the-port)))
		       (new-line *line-number*))
		    (if *position?*
			;; we put position only on pairs.
			(econs walk
			       (loop-pair (ignore) new-pos new-line)
			       (list 'at
				     (input-port-name (the-port))
				     pos
				     line))
			(cons walk
			      (loop-pair (ignore) new-pos new-line)))))))))
      ((in ")]")
       (set! *par-open* (-fx *par-open* 1))
       (if (<fx *par-open* 0)
	   (begin
	      (set! *par-open* 0)
	      (ignore))
	   #f))

      ;; vectors
      ("#("
       (let ((open-key *par-open*))
	  ;; if possible, we store the opening parenthesis.
	  (if (and (vector? *vector-errors*)
		   (<fx open-key (vector-length *vector-errors*)))
	      (let ((pos (input-port-position (the-port))))
		 (c-vector-set! *vector-errors* open-key pos)))
	  ;; and then, we compute the result list...
	  (set! *par-open* (+fx 1 *par-open*))
	  (let loop-vector ((walk  (ignore))
			    (res  '())
			    (len   0))
	     (cond
		((=fx open-key *par-open*)
		 (let ((vect (c-create-vector len)))
		    (let loop-vector-inner ((i (-fx len 1))
					    (l res))
		       (if (=fx i -1)
			   vect
			   (begin
			      (vector-set! vect i (car l))
			      (loop-vector-inner (-fx i 1)
						 (cdr l)))))))
		(else
		 (loop-vector (ignore)
			      (cons walk res)
			      (+fx 1 len)))))))
      ((: "#" id "(")
       (let ((id (let ((str (the-substring 1 (-fx (the-length) 1))))
		    (string->symbol (if (not (eq? *bigloo-case-sensitive*
						  #unspecified))
					str
					(string-upcase! str)))))
	     (open-key *par-open*))
	  ;; if possible, we store the opening parenthesis.
	  (if (and (vector? *vector-errors*)
		   (<fx open-key (vector-length *vector-errors*)))
	      (let ((pos (input-port-position (the-port))))
		 (c-vector-set! *vector-errors* open-key pos)))
	  ;; then we compute the vector
	  (set! *par-open* (+fx 1 *par-open*))
	  (let ((l (let loop-tvector ((walk (ignore)))
		      (cond
			 ((=fx open-key *par-open*)
			  '())
			 (else
			  (cons walk (loop-tvector (ignore))))))))
	     (list->tvector id l))))

      ;; structures
      ("#{"
       (let ((open-key *bra-open*))
	  ;; if possible, we store the opening parenthesis.
	  (if (and (vector? *vector-errors*)
		   (<fx open-key (vector-length *vector-errors*)))
	      (let ((pos (input-port-position (the-port))))
		 (c-vector-set! *vector-errors* open-key pos)))
	  ;; then, we compute the structure
	  (set! *bra-open* (+fx 1 *bra-open*))
	  (cons '_structure_
		(let loop-struct ((walk (ignore)))
		   (cond
		      ((=fx open-key *bra-open*)
		       '())
		      (else
		       (cons walk (loop-struct (ignore)))))))))
      ("}"
       (set! *bra-open* (-fx *bra-open* 1))
       (if (<fx *bra-open* 0)
	   (begin
	      (set! *bra-open* 0)
	      (ignore))
	   #f))

      ;; error
      (else
       (let ((char (the-failure)))
	  (if (eof-object? char)
	      (cond
		 ((>fx *par-open* 0)
		  (let ((open-key (-fx *par-open* 1)))
		     (reader-reset!)
		     (if (and (<fx open-key (vector-length *list-errors*))
			      (fixnum? (c-vector-ref *list-errors* open-key)))
			 (error/location "read"
					 "Unclosed list"
					 char
					 (input-port-name (the-port))
					 (c-vector-ref *list-errors* open-key))
			 (error "read"
				"Unexpected end-of-file"
				"Unclosed list"))))
		 ((>fx *bra-open* 0)
		  (let ((open-key (-fx *bra-open* 1)))
		     (reader-reset!)
		     (if (and (<fx open-key (vector-length *vector-errors*))
			      (fixnum? (c-vector-ref *vector-errors*
						     open-key)))
			 (error/location "read"
					 "Unclosed vector or structure"
					 char
					 (input-port-name (the-port))
					 (c-vector-ref *vector-errors*
						       open-key))
			 (error "read"
				"Unexpected end-of-file"
				"Unclosed vector or structure"))))
		 (else
		  (reset-eof (the-port))
		  char))
	      (error/location "read"
			      "Illegal char"
			      (illegal-char-rep char)
			      (input-port-name     (the-port))
			      (input-port-position (the-port))))))))

;*---------------------------------------------------------------------*/
;*    read ...                                                         */
;*---------------------------------------------------------------------*/
(define (read . input-port)
   (set! *position?* #f)
   ;; read except an undocumented argument used by the compiler to
   ;; get line number associated with expressions.
   (cond
      ((null? input-port)
       (read/rp *bigloo-grammar* (current-input-port)))
      ((not (input-port? (car input-port)))
       (error "read" "type `input-port' expected" (car input-port)))
      ((null? (cdr input-port))
       (read/rp *bigloo-grammar* (car input-port)))
      (else
       (set! *position?* #t)
       (if (fixnum? (cadr input-port))
	   (set! *line-number* (cadr input-port)))
       (read/rp *bigloo-grammar* (car input-port)))))

;*---------------------------------------------------------------------*/
;*    read/case ...                                                    */
;*---------------------------------------------------------------------*/
(define (read/case case . input-port)
   (let ((old *bigloo-case-sensitive*))
      (set! *bigloo-case-sensitive* case)
      (unwind-protect
	 (apply read input-port)
	 (set! *bigloo-case-sensitive* old))))
   
;*---------------------------------------------------------------------*/
;*    read-case-sensitiie ...                                          */
;*    -------------------------------------------------------------    */
;*    Case sensitive read.                                             */
;*---------------------------------------------------------------------*/
(define (read-case-sensitive . input-port)
   (apply read/case (cons #t input-port)))

;*---------------------------------------------------------------------*/
;*    read-case-insensitive ...                                        */
;*    -------------------------------------------------------------    */
;*    Case unsensitive read.                                           */
;*---------------------------------------------------------------------*/
(define (read-case-insensitive . input-port)
   (apply read/case (cons #f input-port)))

