;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime/Ieee/string.scm              */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Mon Mar 20 19:17:18 1995                          */
;*    Last change :  Sun Aug 16 09:39:00 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    6.7. Strings (page 25, r4)                                       */
;*    -------------------------------------------------------------    */
;*    Source documentation:                                            */
;*       @path ../../manuals/body.texi@                                */
;*       @node Strings@                                                */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __r4_strings_6_7

   (import  (__error                   "Llib/error.scm"))
   
   (use     (__type                    "Llib/type.scm")
	    (__bigloo                  "Llib/bigloo.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__r4_numbers_6_5_fixnum   "Ieee/fixnum.scm")
	    (__r4_equivalence_6_2      "Ieee/equiv.scm")
	    (__r4_vectors_6_8          "Ieee/vector.scm")
	    (__r4_booleans_6_1         "Ieee/boolean.scm")
	    (__r4_characters_6_6       "Ieee/char.scm")
	    (__r4_symbols_6_4          "Ieee/symbol.scm")
	    (__r4_pairs_and_lists_6_3  "Ieee/pair-list.scm")

	    (__evenv                   "Eval/evenv.scm"))
    
   (foreign (macro bool c-string?       (obj)              "STRINGP")
	    (bstring    c-make-string   (long uchar)       "make_string")
	    
	    (macro long c-string-length (bstring)          "STRING_LENGTH")
	    
	    (macro uchar c-string-ref  (bstring long)      "STRING_REF")
	    (macro obj  c-string-set! (bstring long uchar) "STRING_SET")
	    
	    (bool    c-string=?    (bstring bstring)      "bigloo_strcmp")
	    (bool    c-substring=? (bstring bstring long) "bigloo_strncmp")
	    (bool    c-substring-ci=? (bstring bstring long) "bigloo_strncmp_ci")
	    (bool    strcicmp      (bstring bstring)      "strcicmp")
	    (bool    string_lt     (bstring bstring)      "string_lt")
	    (bool    string_le     (bstring bstring)      "string_le")
	    (bool    string_gt     (bstring bstring)      "string_gt")
	    (bool    string_ge     (bstring bstring)      "string_ge")
	    (bool    string_cilt   (bstring bstring)      "string_cilt")
	    (bool    string_cile   (bstring bstring)      "string_cile")
	    (bool    string_cigt   (bstring bstring)      "string_cigt")
	    (bool    string_cige   (bstring bstring)      "string_cige")
	    
	    (bstring c-string-copy (string)            "string_to_bstring")
	    (bstring c-substring (bstring long long)   "c_substring")
	    (bstring c-string-append (bstring bstring) "string_append")
	    
	    (bstring escape-C-string (string)          "escape_C_string")
	    (bstring escape-scheme-string (string)     "escape_scheme_string")
	    (bstring c-constant-string-to-string (string)
		     "c_constant_string_to_string")
	    
	    (macro bool string-bound-check? (long long) "BOUND_CHECK")
	    (bstring c-string-for-read (bstring)        "string_for_read")
	    (obj   c-blit-string (bstring bint bstring bint bint)
		   "blit_string"))
   
   (export  (inline string?::bool               ::obj)
	    (inline make-string::bstring        ::int . char)
	    (inline string::bstring             . chars)
	    (inline string-length::long         ::bstring)
	    (inline string-ref::uchar           ::bstring ::long)
	    (inline string-set!::obj            ::bstring ::long ::uchar)
	    (inline string-ref-ur::uchar        ::bstring ::long)
	    (inline string-set-ur!::obj         ::bstring ::long ::uchar)
	    (inline string=?::bool              ::bstring ::bstring)
	    (inline substring=?::bool           ::bstring ::bstring ::long)
	    (inline string-ci=?::bool           ::bstring ::bstring)
	    (inline substring-ci=?::bool        ::bstring ::bstring ::long)
	    (inline string<?::bool              ::bstring ::bstring)
	    (inline string>?::bool              ::bstring ::bstring)
	    (inline string<=?::bool             ::bstring ::bstring)
	    (inline string>=?::bool             ::bstring ::bstring)
	    (inline string-ci<?::bool           ::bstring ::bstring)
	    (inline string-ci>?::bool           ::bstring ::bstring)
	    (inline string-ci<=?::bool          ::bstring ::bstring)
	    (inline string-ci>=?::bool          ::bstring ::bstring)
	    (inline substring::bstring          ::bstring ::long ::long)
	    (inline substring-ur::bstring       ::bstring ::long ::long)
	    (string-append::bstring             . strings)
	    (string->list                       ::bstring)
	    (list->string::bstring              ::obj)
	    (inline string-copy::bstring        ::bstring)
	    (string-fill!::bstring              ::bstring ::uchar)
	    (string-upcase::bstring             ::bstring)
	    (string-downcase::bstring           ::bstring)
	    (string-upcase!::bstring            ::bstring)
	    (string-downcase!::bstring          ::bstring)
	    (string-capitalize!::bstring        ::bstring)
	    (string-capitalize::bstring         ::bstring)
	    (inline string-for-read::bstring    ::bstring)
	    (inline blit-string-ur!             ::bstring ::bint
		    ::bstring ::bint
		    ::bint)
	    (blit-string!                       ::bstring ::bint
						  ::bstring ::bint
						  ::bint))
    
   (pragma  (c-string? (predicate-of bstring) no-cfa-top)
	    (string? side-effect-free no-cfa-top)
	    (c-string-ref side-effect-free no-cfa-top)
	    (string-ref-ur side-effect-free no-cfa-top)
	    (string-ref side-effect-free no-cfa-top)
	    (c-string-length side-effect-free no-cfa-top)
	    (string-length side-effect-free no-cfa-top)
	    (string-bound-check? side-effect-free no-cfa-top)
	    (string=? side-effect-free)
	    (substring=? side-effect-free)
	    (substring-ci=? side-effect-free)
	    (string-ci=? side-effect-free)
	    (string<? side-effect-free)
	    (string>? side-effect-free)
	    (string<=? side-effect-free)
	    (string>=? side-effect-free)
	    (string-ci<? side-effect-free)
	    (string-ci>? side-effect-free)
	    (string-ci<=? side-effect-free)
	    (string-ci>=? side-effect-free)))
 
;*---------------------------------------------------------------------*/
;*    @deffn string?@ ...                                              */
;*---------------------------------------------------------------------*/
(define-inline (string? obj)
   (c-string? obj))

;*---------------------------------------------------------------------*/
;*    @deffn make-string@ ...                                          */
;*---------------------------------------------------------------------*/
(define-inline  (make-string k . char)
   (if (null? char)
       (c-make-string k #\space)
       (c-make-string k (car char))))
 
;*---------------------------------------------------------------------*/
;*    @deffn string@ ...                                               */
;*---------------------------------------------------------------------*/
(define-inline (string . chars)
   (list->string chars))

;*---------------------------------------------------------------------*/
;*    @deffn string-length@ ...                                        */
;*---------------------------------------------------------------------*/
(define-inline (string-length string)
   (c-string-length string))

;*---------------------------------------------------------------------*/
;*    @deffn string-ref@ ...                                           */
;*---------------------------------------------------------------------*/
(define-inline (string-ref string k)
   (if (string-bound-check? k (string-length string))
       (c-string-ref string k)
       (error "string-ref" "index out of range" k)))
 
;*---------------------------------------------------------------------*/
;*    @deffn string-set!@ ...                                          */
;*---------------------------------------------------------------------*/
(define-inline (string-set! string k char)
   (if (string-bound-check? k (string-length string))
       (c-string-set! string k char)
       (error "string-set!" "index out of range" k)))

;*---------------------------------------------------------------------*/
;*    @deffn string-ref-ur@ ...                                        */
;*---------------------------------------------------------------------*/
(define-inline (string-ref-ur string k)
   (c-string-ref string k))
 
;*---------------------------------------------------------------------*/
;*    @deffn string-set-ur!@ ...                                       */
;*---------------------------------------------------------------------*/
(define-inline (string-set-ur! string k char)
   (c-string-set! string k char))

;*---------------------------------------------------------------------*/
;*    @deffn string=?@ ...                                             */
;*---------------------------------------------------------------------*/
(define-inline (string=? string1 string2)
   (c-string=? string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn substring=?@ ...                                          */
;*---------------------------------------------------------------------*/
(define-inline (substring=? string1 string2 len)
   (c-substring=? string1 string2 len))

;*---------------------------------------------------------------------*/
;*    @deffn substring-ci=?@ ...                                       */
;*---------------------------------------------------------------------*/
(define-inline (substring-ci=? string1 string2 len)
   (c-substring-ci=? string1 string2 len))

;*---------------------------------------------------------------------*/
;*    @deffn string-ci=?@ ...                                          */
;*---------------------------------------------------------------------*/
(define-inline (string-ci=? string1 string2)
   (strcicmp string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string<?@ ...                                             */
;*---------------------------------------------------------------------*/
(define-inline (string<? string1 string2)
   (string_lt string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string>?@ ...                                             */
;*---------------------------------------------------------------------*/
(define-inline (string>? string1 string2)
   (string_gt string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string<=?@ ...                                            */
;*---------------------------------------------------------------------*/
(define-inline (string<=? string1 string2)
   (string_le string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string>=?@ ...                                            */
;*---------------------------------------------------------------------*/
(define-inline (string>=? string1 string2)
   (string_ge string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string-ci<?@ ...                                          */
;*---------------------------------------------------------------------*/
(define-inline (string-ci<? string1 string2)
   (string_cilt string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string-ci>?@ ...                                          */
;*---------------------------------------------------------------------*/
(define-inline (string-ci>? string1 string2)
   (string_cigt string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string-ci<=?@ ...                                         */
;*---------------------------------------------------------------------*/
(define-inline (string-ci<=? string1 string2)
   (string_cile string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn string-ci>=?@ ...                                         */
;*---------------------------------------------------------------------*/
(define-inline (string-ci>=? string1 string2)
   (string_cige string1 string2))

;*---------------------------------------------------------------------*/
;*    @deffn substring@ ...                                            */
;*---------------------------------------------------------------------*/
(define-inline (substring string start end)
   ;; no macro on inline so we don't use `and'
   (if (if (>=fx end start)
	   (if (string-bound-check? start (+fx (string-length string) 1))
	       (string-bound-check? end (+fx (string-length string) 1))
	       #f)
	   #f)
       (c-substring string start end)
       (error "substring" "Illegal index" (cons start end))))

;*---------------------------------------------------------------------*/
;*    @deffn substring-ur@ ...                                         */
;*---------------------------------------------------------------------*/
(define-inline (substring-ur string start end)
   (c-substring string start end))

;*---------------------------------------------------------------------*/
;*    @deffn string-append@ ...                                        */
;*---------------------------------------------------------------------*/
(define (string-append . list)
   (if (null? list)
       ""
       (let loop ((list list))
	  (if (null? (cdr list))
	      (car list)
	      (c-string-append (car list) (loop (cdr list)))))))

;*---------------------------------------------------------------------*/
;*    @deffn list->string@ ...                                         */
;*---------------------------------------------------------------------*/
(define (list->string list)
   (let* ((len    (length list))
	  (string (make-string len)))
      (let loop ((i 0)
		 (l list))
	 (if (=fx i len)
	     string
	     (begin
		(string-set! string i (car l))
		(loop (+fx i 1) (cdr l)))))))

;*---------------------------------------------------------------------*/
;*    @deffn string->list@ ...                                         */
;*---------------------------------------------------------------------*/
(define (string->list string)
   (let ((len (string-length string)))
      (let loop ((i   (-fx len 1))
		 (res '()))
	 (if (=fx i -1)
	     res
	     (loop (-fx i 1)
		   (cons (string-ref string i) res))))))

;*---------------------------------------------------------------------*/
;*    @deffn string-copy@ ...                                          */
;*---------------------------------------------------------------------*/
(define-inline (string-copy string)
   (c-string-copy string))

;*---------------------------------------------------------------------*/
;*    @deffn string-fill!@ ...                                         */
;*---------------------------------------------------------------------*/
(define (string-fill! string char)
   (let ((len (string-length string)))
      (let loop ((i 0))
	 (if (=fx i len)
	     string
	     (begin
		(string-set! string i char)
		(loop (+fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    @deffn string-upcase@ ...                                        */
;*---------------------------------------------------------------------*/
(define (string-upcase string)
   (let* ((len (string-length string))
	  (res (make-string len)))
      (let loop ((i 0))
	 (if (=fx i len)
	     res
	     (begin
		(string-set! res i (char-upcase (string-ref string i)))
		(loop (+fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    @deffn string-downcase@ ...                                      */
;*---------------------------------------------------------------------*/
(define (string-downcase string)
   (let* ((len (string-length string))
	  (res (make-string len)))
      (let loop ((i 0))
	 (if (=fx i len)
	     res
	     (begin
		(string-set! res i (char-downcase (string-ref string i)))
		(loop (+fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    @deffn string-upcase!@ ...                                       */
;*---------------------------------------------------------------------*/
(define (string-upcase! string)
   (let* ((len (string-length string))
	  (res string))
      (let loop ((i 0))
	 (if (=fx i len)
	     res
	     (begin
		(string-set! res i (char-upcase (string-ref string i)))
		(loop (+fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    @deffn string-downcase!@ ...                                     */
;*---------------------------------------------------------------------*/
(define (string-downcase! string)
   (let* ((len (string-length string))
	  (res string))
      (let loop ((i 0))
	 (if (=fx i len)
	     res
	     (begin
		(string-set! res i (char-downcase (string-ref string i)))
		(loop (+fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    @deffn string-capitalize!@ ...                                   */
;*    -------------------------------------------------------------    */
;*    "hELLO" -> "Hello"                                               */
;*    "*hello" -> "*Hello"                                             */
;*    "hello you" -> "Hello You"                                       */
;*---------------------------------------------------------------------*/
(define (string-capitalize! string)
   (let ((non-first-alpha #f)		 
	 (string-len (string-length string)))	 
      (do ((i 0 (+fx i 1)))			 
	    ((=fx i string-len) string)
	 (let ((c (string-ref string i)))
	    (if (char-alphabetic? c)
		(if non-first-alpha
		    (string-set! string i (char-downcase c))
		    (begin
		       (set! non-first-alpha #t)
		       (string-set! string i (char-upcase c))))
		(set! non-first-alpha #f))))))

;*---------------------------------------------------------------------*/
;*    @deffn string-capitalize@ ...                                    */
;*---------------------------------------------------------------------*/
(define (string-capitalize string)
   (string-capitalize! (string-copy string)))

;*---------------------------------------------------------------------*/
;*    @deffn string-for-read@ ...                                      */
;*---------------------------------------------------------------------*/
(define-inline (string-for-read string)
   (c-string-for-read string))

;*---------------------------------------------------------------------*/
;*    @deffn blit-string-ur!@ ...                                      */
;*---------------------------------------------------------------------*/
(define-inline (blit-string-ur! s1 o1 s2 o2 l)
   (c-blit-string s1 o1 s2 o2 l))

;*---------------------------------------------------------------------*/
;*    @deffn blit-string!@ ...                                         */
;*---------------------------------------------------------------------*/
(define (blit-string! s1 o1 s2 o2 l)
  (if (and (string-bound-check? (+fx l o1) (+fx (string-length s1) 1))
	   (string-bound-check? (+fx l o2) (+fx (string-length s2) 1)))
      (c-blit-string s1 o1 s2 o2 l)
      (error "blit-string!" "Index and length out of range" 
	     (list (string-length s1) o1 (string-length s2) o2 l))))

