;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime/Eval/eval.scm                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sat Oct 22 09:34:28 1994                          */
;*    Last change :  Fri Nov 13 15:37:08 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    L'evaluateur de Bigloo                                           */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __eval
   
   (export  (eval                       <expression> . <env>)
	    (scheme-report-environment  <version>)
	    (null-environment           <versino>)
	    (interaction-environment)
	    (load                       <string>)
	    (loadq                      <string>)
	    (loada                      <string>)
	    (repl)
	    (quit)
	    (module-declaration!        decls)
	    (expand-define-macro        <expression> <expander>)
	    (expand-define-hygien-macro <expression> <expander>)
	    (expand-define-expander     <expression> <expander>)
	    (expand-define-pattern      <expression>)
	    (set-prompter!              ::procedure)
	    (get-prompter::procedure)
	    *load-path*
	    *user-pass-name*
	    *user-pass*
	    (notify-assert-fail         vars body loc)
	    *nil*)
   
   (import  (__type                    "Llib/type.scm")
	    (__error                   "Llib/error.scm")
	    (__bigloo                  "Llib/bigloo.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__structure               "Llib/struct.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__bexit                   "Llib/bexit.scm")
	    (__os                      "Llib/os.scm")
	    
	    (__reader                  "Read/reader.scm")
	    
	    (__r4_numbers_6_5          "Ieee/number.scm")
	    (__r4_numbers_6_5_fixnum   "Ieee/fixnum.scm")
	    (__r4_numbers_6_5_flonum   "Ieee/flonum.scm")
	    (__r4_characters_6_6       "Ieee/char.scm")
	    (__r4_equivalence_6_2      "Ieee/equiv.scm")
	    (__r4_booleans_6_1         "Ieee/boolean.scm")
	    (__r4_symbols_6_4          "Ieee/symbol.scm")
	    (__r4_strings_6_7          "Ieee/string.scm")
	    (__r4_pairs_and_lists_6_3  "Ieee/pair-list.scm")
	    (__r4_input_6_10_2         "Ieee/input.scm")
	    (__r4_control_features_6_9 "Ieee/control.scm")
	    (__r4_vectors_6_8          "Ieee/vector.scm")
	    (__r4_ports_6_10_1         "Ieee/port.scm")
	    (__r4_output_6_10_3        "Ieee/output.scm")

	    (__macro                   "Eval/macro.scm")
	    (__install_expanders       "Eval/expanders.scm")
	    (__progn                   "Eval/progn.scm")
	    (__expand                  "Eval/expand.scm")
	    (__evcompile               "Eval/evcompile.scm")
	    (__evmeaning               "Eval/evmeaning.scm")
	    (__evprimop                "Eval/evprimop.scm")
	    (__evenv                   "Eval/evenv.scm")
	    (extend.r.macro-env __match_normalize "Match/normalize.scm")
   
	    (__r5_macro_4_3_init       "R5rs/init5.scm"))
   
   (extern  (include "signal.h")
	    (reset-console!::obj   (::obj) "reset_console")
	    (macro sigsetmask::int (::int) "sigsetmask")
	    (macro sigint::int             "SIGINT")))

;*---------------------------------------------------------------------*/
;*    Expanders setup.                                                 */
;*---------------------------------------------------------------------*/
(install-all-expanders!)

;*---------------------------------------------------------------------*/
;*    eval ...                                                         */
;*---------------------------------------------------------------------*/
(define (eval exp . environment)
   (let* ((env  (if (pair? environment)
		    (let ((env (car environment)))
		       (if (not (or (eq? env (scheme-report-environment 5))
				    (eq? env (null-environment 5))
				    (interaction-environment)))
			   (error "eval" "Illegal environment" environment)
			   env))
		    (interaction-environment)))
	  (loc  (find-loc exp #f))
	  (exp  (if (procedure? *user-pass*) (*user-pass* exp) exp))
	  (sexp (if *hygien?* (expand-syntax exp) exp)))
      (evmeaning-reset-error!)
      (evmeaning (evcompile (expand sexp) '() env 'nowhere #f loc) '())))

;*---------------------------------------------------------------------*/
;*    scheme-report-environment ...                                    */
;*---------------------------------------------------------------------*/
(define (scheme-report-environment version)
   (if (=fx version 5)
       'scheme-report-environment
       (error "scheme-report-environment"
	      "Version not supported"
	      version)))
   
;*---------------------------------------------------------------------*/
;*    null-environment ...                                             */
;*---------------------------------------------------------------------*/
(define (null-environment version)
   (if (=fx version 5)
       'null-environment
       (error "scheme-report-environment"
	      "Version not supported"
	      version)))
   
;*---------------------------------------------------------------------*/
;*    interaction-environment ...                                      */
;*---------------------------------------------------------------------*/
(define (interaction-environment)
   'interaction-environment)
   
;*---------------------------------------------------------------------*/
;*    prompt ...                                                       */
;*---------------------------------------------------------------------*/
(define *prompt* (lambda (num)
		    (display num)
		    (display ":=> ")
		    (flush-output-port (current-output-port))))

;*---------------------------------------------------------------------*/
;*    set-prompter! ...                                                */
;*---------------------------------------------------------------------*/
(define (set-prompter! proc)
   (if (not (correct-arity? proc 1))
       (error "set-prompter!"
	      "argument has to be a procedure of 1 argument"
	      proc)
       (set! *prompt* proc)))

;*---------------------------------------------------------------------*/
;*    get-prompter ...                                                 */
;*---------------------------------------------------------------------*/
(define (get-prompter::procedure)
   *prompt*)

;*---------------------------------------------------------------------*/
;*    Global repl parameters ...                                       */
;*---------------------------------------------------------------------*/
(define *repl-num* 0)
(define *repl-quit* bigloo-exit)

;*---------------------------------------------------------------------*/
;*    repl ...                                                         */
;*---------------------------------------------------------------------*/
(define (repl)
   (let ((repl-quit *repl-quit*)
	 (repl-num  *repl-num*))
      (bind-exit (quit)
	 (set! *repl-quit* quit)
	 (set! *repl-num* (+fx 1 *repl-num*))
	 (unwind-protect
	    (internal-repl)
	    (begin
	       (set! *repl-num* repl-num)
	       (set! *repl-quit* repl-quit))))
      (newline)
      (flush-output-port (current-output-port))))

;*---------------------------------------------------------------------*/
;*    internal-repl ...                                                */
;*---------------------------------------------------------------------*/
(define (internal-repl)
   (let ((old-intrhdl (get-signal-handler sigint)))
      (unwind-protect
	 (let loop ()
	    (bind-exit (re-enter-internal-repl)
	       ;; we setup ^C interupt
	       (letrec ((intrhdl (lambda (n)
				    (newline (current-error-port))
				    (fprint (current-error-port)
					    "*** INTERRUPT:bigloo:")
				    (flush-output-port (current-error-port))
				    ;; we flush current input port
				    (reader-reset!)
				    (reset-console! (current-input-port))
				    ;; we restore signal handling
				    (sigsetmask 0)
				    (signal n intrhdl)
				    (re-enter-internal-repl #unspecified))))
		  (signal sigint intrhdl))
	       ;; and we loop until eof
	       (newline)
	       (let loop ()
		  (*prompt* *repl-num*)
		  (let ((exp (try (read)
				  (lambda (escape proc obj msg)
				     (flush-output-port (current-error-port))
				     (if (eof-object? obj)
					 (reset-eof (current-input-port)))
				     (sigsetmask 0)
				     (escape #unspecified)))))
		     (if (eof-object? exp)
			 (quit)
			 (let ((v (try (eval exp)
				       (lambda (escape proc obj msg)
					  (evmeaning-notify-error proc obj msg)
					  (flush-output-port (current-error-port))
					  (sigsetmask 0)
					  (escape #unspecified)))))
			    (print v)
			    (set! exp #unspecified)
			    (loop))))))
	    (loop))
	 (if (procedure? old-intrhdl)
	     (signal sigint old-intrhdl)))))

;*---------------------------------------------------------------------*/
;*    quit ...                                                         */
;*---------------------------------------------------------------------*/
(define (quit)
   (*repl-quit* 0))

;*---------------------------------------------------------------------*/
;*    *load-path*                                                      */
;*---------------------------------------------------------------------*/
(define *load-path* '())

;*---------------------------------------------------------------------*/
;*    find-file ...                                                    */
;*---------------------------------------------------------------------*/
(define (find-file name)
   (if (file-exists? name)
       name
       (let loop ((path *load-path*))
	  (if (null? path)
	      name
	      (let ((try (string-append (car path) "/" name)))
		 (if (file-exists? try)
		     try
		     (loop (cdr path))))))))

;*---------------------------------------------------------------------*/
;*    load ...                                                         */
;*---------------------------------------------------------------------*/
(define (load file-name)
   (loadv file-name #t))

(define (loadq file-name)
   (loadv file-name #f))

(define (loadv file-name v?)
   (let ((port (open-input-file (find-file file-name))))
      (evmeaning-reset-error!)
      (if (input-port? port)
	  (try (let loop ((sexp         (read port #t))
			  (v            (unspecified))
			  (module-seen? #f)
			  (main         #f))
		  (cond
		     ((eof-object? sexp)
		      (close-input-port port)
		      (let ((pmain (if (symbol? main)
				       (eval main)
				       #f)))
			 (if (procedure? pmain)
			     (pmain (command-line))
			     v)))
		     ((and (pair? sexp) (eq? (car sexp) 'module))
		      (if module-seen?
			  (error "load" "module defined twice" sexp)
			  (let ((main (assq 'main (cddr sexp))))
			     (let ((v (eval sexp)))
				(if v?
				    (print v))
				(loop (read port #t)
				      v
				      #t
				      (if (pair? main)
					  (cadr main)
					  v))))))
		     (else
		      (let ((v (eval sexp)))
			 (evmeaning-reset-error!)
			 (if v?
			     (print v))
			 (loop (read port #t)
			       v
			       module-seen?
			       main)))))
	       (lambda (escape proc mes obj)
		  ;; on imprime le message d'erreur
		  (evmeaning-notify-error proc mes obj)
		  (error "load"
			 "error occured when loading"
			 file-name)))
	  (error "load" "Can't open file" file-name))))

;*---------------------------------------------------------------------*/
;*    loada ...                                                        */
;*---------------------------------------------------------------------*/
(define (loada file)
   (let ((port (open-input-file file)))
      (if (input-port? port)
	  (begin
	     (set! *afile-list* (append (read port #t) *afile-list*))
	     (close-input-port port))
	  (error "loada" "Can't open file" file))))
   
;*---------------------------------------------------------------------*/
;*    On met dans ce fichier les definitions de                        */
;*    `expand-define-expander' et `expand-define-macro' car elles      */
;*    contiennent des appels a `Eval'.                                 */
;*---------------------------------------------------------------------*/
 
;*---------------------------------------------------------------------*/
;*    expand-define-expander ...                                       */
;*---------------------------------------------------------------------*/
(define (expand-define-expander x e)
   (match-case x
      ((?- (and (? symbol?) ?name) . ?macro)
       (let* ((expd-lam     (normalize-progn macro))
	      (expd-lam/loc (replace! x expd-lam))
	      (expd-eval    (eval expd-lam/loc)))
	  (install-expander
	   name
	   (lambda (x e)
	      (if (not (procedure? expd-eval))
		  (error name "illegal expander" macro)
		  (if (not (correct-arity? expd-eval 2))
		      (error name "wrong number of argument for expand" macro)
		      (try (expd-eval x e)
			   (lambda (escape proc mes obj)
			      (evmeaning-notify-error proc mes obj)
			      (error "expand"
				     "error occured while expansing the call"
				     x))))))))
       (unspecified))
      (else
       (error "define-expander" "Illegal `define-expander' syntax" x))))

;*---------------------------------------------------------------------*/
;*    expand-define-macro ...                                          */
;*---------------------------------------------------------------------*/
(define (expand-define-macro x e)
   (match-case x
      ((or (?- (?name . ?args) . ?body)
	   (?- ?name (lambda ?args . ?body)))
       (install-expander
	name
	(let* ((expd-lam     `(lambda (x e)
				 (e (let ,(destructure args '(cdr x) '())
				       ,(normalize-progn body))
				    e)))
	       (expd-lam/loc (replace! x expd-lam))
	       (expd-eval    (eval expd-lam/loc)))
	   (lambda (x e)
	      (try (expd-eval x e) 
		   (lambda (escape proc mes obj)
		      (evmeaning-notify-error proc mes obj)
		      (error "expand"
			     "error occured while expansing the call"
			     x))))))
       (unspecified))
      (else
       (error "define-macro" "Illegal `define-macro' syntax" x))))

;*---------------------------------------------------------------------*/
;*    expand-define-hygien-macro ...                                   */
;*---------------------------------------------------------------------*/
(define (expand-define-hygien-macro x e)
   (match-case x
      ((?- (quote (?name . ?args)) . ?body)
       (let ((body (map cadr body)))
	  (install-expander
	   name
	   (let* ((expd-lam     `(lambda (x e)
				    (e (let ,(destructure args '(cdr x) '())
					  ,(normalize-progn body))
				       e)))
		  (expd-lam/loc (replace! x expd-lam))
		  (expd-eval    (eval expd-lam/loc)))
	      (lambda (x e)
		 (try (expd-eval x e) 
		      (lambda (escape proc mes obj)
			 (evmeaning-notify-error proc mes obj)
			 (error "expand"
				"error occured while expansing the call"
				x))))))
	  (unspecified)))
      (else
       (error "define-macro" "Illegal `define-macro' syntax" x))))

;*---------------------------------------------------------------------*/
;*    destructure ...                                                  */
;*---------------------------------------------------------------------*/
(define (destructure pat arg bindings)
   (cond
      ((null? pat)
       (cons `(,(gensym '.dummy.)
	       (if (not (null? ,arg))
		   (error "expand" "Too many arguments provided" ,arg)
		   '()))
	     bindings))
      ((symbol? pat)
       (cons `(,pat ,arg) bindings))
      ((pair? pat)
       (destructure (car pat)
		    `(car ,arg)
		    (destructure (cdr pat)
				 `(cdr ,arg)
				 bindings)))))

;*---------------------------------------------------------------------*/
;*    module-declaration! ...                                          */
;*---------------------------------------------------------------------*/
(define (module-declaration! decls)
   (let loop ((decls decls))
      (cond
	 ((null? decls)
	  'done)
	 ((not (pair? (car decls)))
	  (error "eval" "Illegal module declaration" decls))
	 ((eq? (car (car decls)) 'include)
	  (include! (cdr (car decls)))
	  (loop (cdr decls)))
	 ((eq? (car (car decls)) 'import)
	  (import! (cdr (car decls)))
	  (loop (cdr decls)))
	 ((eq? (car (car decls)) 'load)
	  (import! (cdr (car decls)))
	  (loop (cdr decls)))
	 (else
	  (loop (cdr decls))))))

;*---------------------------------------------------------------------*/
;*    *files* ...                                                      */
;*---------------------------------------------------------------------*/
(define *included-files* '())
(define *imported-files* '())
(define *afile-list*     '())

;*---------------------------------------------------------------------*/
;*    include! ...                                                     */
;*---------------------------------------------------------------------*/
(define (include! includes)
   (for-each (lambda (i)
		(if (not (member i *included-files*))
		    (begin
		       (set! *included-files* (cons i *included-files*))
		       (loadq i))))
	     includes))

;*---------------------------------------------------------------------*/
;*    import! ...                                                      */
;*---------------------------------------------------------------------*/
(define (import! iclauses)
   (let ((l (map (lambda (i)
		     (match-case i
			((?- ?second)
			 (if (string? second)
			     second
			     (let ((cell (assq second *afile-list*)))
				(if (pair? cell)
				    (cadr cell)
				    #f))))
			((?- ?- ?third)
			 third)
			(?module
			 (let ((cell (assq module *afile-list*)))
			    (if (pair? cell)
				(cadr cell)
				#f)))
		 	(else
			 #f)))
		  iclauses)))
      (for-each (lambda (i)
		   (if (and (string? i)
			    (not (member i *imported-files*)))
		       (begin
			  (set! *imported-files* (cons i *imported-files*))
			  (loadq i))))
		l)))

;*---------------------------------------------------------------------*/
;*    expand-define-pattern ...                                        */
;*---------------------------------------------------------------------*/
(define (expand-define-pattern x)
   (match-case x
      ((?- ?name ?var ?body)
       (extend.r.macro-env name (eval `(lambda ,var ,body)))
       ''dummy)
      (else
       (error "expand-define-pattern" "Illegal form" x))))

;*---------------------------------------------------------------------*/
;*    notify-assert-fail ...                                           */
;*---------------------------------------------------------------------*/
(define (notify-assert-fail vars fail-body loc)
   (let ((port (current-error-port))
	 (old  (get-write-length)))
      (set-write-length! 80)
      (try (if (pair? loc)
	       (error/location "assert"
			       "assertion failed"
			       fail-body
			       (car loc)
			       (cdr loc))
	       (error "assert"
		      "assertion failed"
		      fail-body))
	   (lambda (escape proc mes obj)
	      (notify-error proc mes obj)
	      (escape #unspecified)))
      (fprint port "-----------------------")
      (fprint port "Variables' value are : ")
      (for-each (lambda (f) (fprint port "   " f " : " (eval f))) vars)
      (fprint port "-----------------------")
      (set-write-length! old)
      (let ((old-prompter (get-prompter)))
	 (set-prompter! (lambda (num) (display "*:=> ")))
	 (repl)
	 (set-prompter! old-prompter))))

;*---------------------------------------------------------------------*/
;*    *nil* ...                                                        */
;*---------------------------------------------------------------------*/
(define *nil* #t)

;*---------------------------------------------------------------------*/
;*    *user-pass* ...                                                  */
;*---------------------------------------------------------------------*/
(define *user-pass*      (unspecified))  ;; l'eventuelle user passe 
(define *user-pass-name* "User")         ;; le nom de la user pass
