/*---------------------------------------------------------------------*/
/*   A pratical implementation for the Scheme programming language     */
/*                                                                     */
/*                                    ,--^,                            */
/*                              _ ___/ /|/                             */
/*                          ,;'( )__, ) '                              */
/*                         ;;  //   L__.                               */
/*                         '   \\   /  '                               */
/*                              ^   ^                                  */
/*                                                                     */
/*   Copyright (c) 1992-1999 Manuel Serrano                            */
/*                                                                     */
/*     Bug descriptions, use reports, comments or suggestions are      */
/*     welcome. Send them to                                           */
/*       bigloo-request@kaolin.unice.fr                                */
/*       http://kaolin.unice.fr/bigloo                                 */
/*                                                                     */
/*   This program is free software; you can redistribute it            */
/*   and/or modify it under the terms of the GNU General Public        */
/*   License as published by the Free Software Foundation; either      */
/*   version 2 of the License, or (at your option) any later version.  */
/*                                                                     */
/*   This program is distributed in the hope that it will be useful,   */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/*   GNU General Public License for more details.                      */
/*                                                                     */
/*   You should have received a copy of the GNU General Public         */
/*   License along with this program; if not, write to the Free        */
/*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
/*   MA 02111-1307, USA.                                               */
/*---------------------------------------------------------------------*/
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/csymbol.c               */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Feb 12 14:51:41 1992                          */
/*    Last change :  Sun Nov 29 19:13:14 1998 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Symbol handling (creation and hash tabling).                     */
/*=====================================================================*/
#include <string.h>
#include <bigloo2.0a.h>

/*---------------------------------------------------------------------*/
/*    Quelques prototypes                                              */
/*---------------------------------------------------------------------*/
extern obj_t string_to_symbol();
static obj_t make_symbol();
extern obj_t make_vector();

/*---------------------------------------------------------------------*/
/*    Quelques variables extern                                        */
/*---------------------------------------------------------------------*/
obj_t c_symtab;

/*---------------------------------------------------------------------*/
/*    init_symbol_table ...                                            */
/*---------------------------------------------------------------------*/
init_symbol_table()
{
   c_symtab = make_vector( SYMBOL_HASH_TABLE_SIZE, BNIL );
}
          
/*---------------------------------------------------------------------*/
/*    make_symbol ...                                                  */
/*    -------------------------------------------------------------    */
/*    Symbols are not allocated in the first generation because        */
/*    they can't be collected.                                         */
/*---------------------------------------------------------------------*/
static obj_t
make_symbol( name )
char *name;
{
   obj_t symbol;

   symbol = GC_MALLOC( SYMBOL_SIZE );
   
   symbol->symbol_t.header = MAKE_HEADER( SYMBOL_TYPE, SYMBOL_SIZE );
   symbol->symbol_t.string = string_to_bstring( name );
   symbol->symbol_t.cval   = BNIL;
   
   return BREF( symbol );
}
   
/*---------------------------------------------------------------------*/
/*    string_to_symbol ...                                             */
/*    char * --> obj_t                                                 */
/*---------------------------------------------------------------------*/
obj_t
string_to_symbol( name )
char *name;
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) )
   {
      obj_t symbol = make_symbol( name );
      obj_t pair   = MAKE_PAIR( symbol, BNIL );
      
      VECTOR_SET( c_symtab, hash_number, pair );
      
      return symbol;
   }
   else
   {
      obj_t run = bucket, back = bucket;
      
      while( !NULLP( run ) &&
	     strcmp( BSTRING_TO_STRING( SYMBOL( CAR( run ) ).string ), name ) )
         back = run, run = CDR( run );
      
      if( !NULLP( run ) )
         return CAR( run );
      else
      {
         obj_t symbol = make_symbol( name );
	 obj_t pair   = MAKE_PAIR( symbol, BNIL );
	 
         SET_CDR( back, pair );

         return symbol;
      }
   }
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    symbol_exists_p ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction retourne vrai s'il existe un symbol de nom        */
/*    `string' et faux sinon                                           */
/*---------------------------------------------------------------------*/
int
symbol_exists_p( name )
char *name;
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) )
   {
      return 0;
   }
   else
   {
      obj_t run = bucket;
      
      while( !NULLP( run ) &&
	     strcmp( BSTRING_TO_STRING( SYMBOL( CAR( run ) ).string ), name ) )
         run = CDR( run );

      return !NULLP( run );
   }
}

