/*---------------------------------------------------------------------*/
/*   A pratical implementation for the Scheme programming language     */
/*                                                                     */
/*                                    ,--^,                            */
/*                              _ ___/ /|/                             */
/*                          ,;'( )__, ) '                              */
/*                         ;;  //   L__.                               */
/*                         '   \\   /  '                               */
/*                              ^   ^                                  */
/*                                                                     */
/*   Copyright (c) 1992-1999 Manuel Serrano                            */
/*                                                                     */
/*     Bug descriptions, use reports, comments or suggestions are      */
/*     welcome. Send them to                                           */
/*       bigloo-request@kaolin.unice.fr                                */
/*       http://kaolin.unice.fr/bigloo                                 */
/*                                                                     */
/*   This program is free software; you can redistribute it            */
/*   and/or modify it under the terms of the GNU General Public        */
/*   License as published by the Free Software Foundation; either      */
/*   version 2 of the License, or (at your option) any later version.  */
/*                                                                     */
/*   This program is distributed in the hope that it will be useful,   */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/*   GNU General Public License for more details.                      */
/*                                                                     */
/*   You should have received a copy of the GNU General Public         */
/*   License along with this program; if not, write to the Free        */
/*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
/*   MA 02111-1307, USA.                                               */
/*---------------------------------------------------------------------*/
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/csocket.c               */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Mon Jun 29 18:18:45 1998                          */
/*    Last change :  Thu Jan 14 19:29:08 1999 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Scheme sockets                                                   */
/*    -------------------------------------------------------------    */
/*    This file is based on a contribution of                          */
/*    David Tolpin (dvd@pizza.msk.su)                                  */
/*                                                                     */
/*    Bugs correction (conversion between host and network byte order) */
/*    by Marc Furrer (Marc.Furrer@di.epfl.ch)                          */
/*                                                                     */
/*    Reworked  by Erick Gallesio for 2.2 release.                     */
/*    Some additions and simplifications (I hope).                     */
/*                                                                     */
/*    Win32 support by Caleb Deupree <cdeupree@erinet.com>             */  
/*=====================================================================*/
#ifdef WIN32
#  include <fcntl.h>
#  include <tclWinPort.h>
#  define BAD_SOCKET(s) ((s) == INVALID_SOCKET)
#  ifndef _O_WRONLY
#    define _O_WRONLY O_WRONLY
#  endif
#  ifndef _O_RDONLY
#    define _O_RDONLY O_RDONLY
#  endif 
#else 
#  include <sys/types.h>
#  include <sys/socket.h>
#  include <netinet/in.h>
#  include <arpa/inet.h>
#  include <netdb.h>
#  include <memory.h>
#  define BAD_SOCKET(s) ((s) < 0)
#endif
#include <errno.h>
#include <bigloo2.0a.h>

/*---------------------------------------------------------------------*/
/*    Importations ...                                                 */
/*---------------------------------------------------------------------*/
extern long default_io_bufsiz;
extern obj_t string_to_bstring( char * );
extern obj_t close_input_port( obj_t );
extern obj_t close_output_port( obj_t );
extern obj_t file_to_input_port( FILE * );

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    socket_error ...                                                 */
/*---------------------------------------------------------------------*/
static void
socket_error( char *who, char *message, obj_t object )
{
   FAILURE( string_to_bstring( who ), string_to_bstring( message ), object );
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    system_error ...                                                 */
/*---------------------------------------------------------------------*/
static void
system_error( char *who )
{
   char buffer[ 512 ]; 
  
   sprintf( buffer, "%d", errno );
   socket_error( who, buffer, BUNSPEC );
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    set_socket_io_ports ...                                          */
/*---------------------------------------------------------------------*/
void
set_socket_io_ports( int s, obj_t sock, char *who )
{
   int t, len, port;
   obj_t hostname;
   char *fname;
   FILE *fs, *ft;
   char buffer[ 200 ];

#ifdef WIN32
   {
      int r;

      if( ((t = _open_osfhandle(s, _O_WRONLY)) == -1) || 
	  ((r = _open_osfhandle(s, _O_RDONLY)) == -1) )
      {
	 sprintf( buffer, "%s: cannot open osfhandle", who );
	 socket_error( "set_socket_io_ports", buffer, BUNSPEC );
      }
    
      if( !((fs = fdopen(r, "r")) && (ft = fdopen(t, "w"))) )
      {
	 sprintf( buffer, "%s: cannot create socket io ports", who );
	 socket_error( "set_socket_io_ports", buffer, BUNSPEC );
      }
   }
#else
   /* duplicate handles so that we are able to access one */
   /* socket channel via two scheme ports.                */
   t = dup( s ); 

   if( t == -1 )
   {
      sprintf( buffer, "%s: cannot duplicate io port", who );
      socket_error( "set_socket_io_ports", buffer, BUNSPEC );
   }

   if( !((fs = fdopen(s, "r")) && (ft = fdopen(t, "w"))) )
   {
      sprintf( buffer, "%s: cannot create socket io ports", who );
      socket_error( "set_socket_io_ports", buffer, BUNSPEC );
  }

#endif

  port     = SOCKET( sock ).portnum;
  hostname = SOCKET( sock ).hostname;
  len      = STRING_LENGTH( hostname ) + 20;
  fname    = (char *)GC_MALLOC( len );
  sprintf( fname, "%s:%d", BSTRING_TO_STRING( hostname ), port );

  /* Create input port */
  SOCKET( sock ).input = file_to_input_port( fs );
  SOCKET( sock ).input->input_port_t.kindof = KINDOF_CONSOLE;
  SOCKET( sock ).input->input_port_t.name   = fname;

  /* Create output port */
  SOCKET( sock ).output = make_output_port( fname, ft, KINDOF_FILE );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    make_client_socket ...                                           */
/*---------------------------------------------------------------------*/
obj_t
make_client_socket( obj_t hostname, int port )
{
   char str[] = "make-client-socket";
   struct hostent *hp;
   struct sockaddr_in server;
   struct in_addr local_ip;
   obj_t local_host;
   int s;
   obj_t a_socket;
 
   /* Locate the host IP address */
   if( (hp = gethostbyname( BSTRING_TO_STRING( hostname ) )) == NULL )
      socket_error( str, "unknown or misspelled host name", hostname );

   /* Get a socket */
   if( BAD_SOCKET( s = socket( AF_INET, SOCK_STREAM, 0 )) )
      socket_error( str, "cannot create socket", BUNSPEC );
  
   /* Setup a connect address */
   memset( &server, 0, sizeof( server ) );
   memcpy( (char*)&server.sin_addr, hp->h_addr, hp->h_length );
   server.sin_family = AF_INET;
   server.sin_port   = htons( port );

   /* Try to connect */
   if( connect( s, (struct sockaddr *)&server, sizeof( server ) ) < 0 )
   {
      close( s );
      system_error( str );
   }

   /* Create a new Scheme socket object */
   a_socket                       = GC_MALLOC( SOCKET_SIZE );
   a_socket->socket_t.header      = MAKE_HEADER( SOCKET_TYPE, 0 );
   a_socket->socket_t.portnum     = ntohs( server.sin_port );
   a_socket->socket_t.hostname    = string_to_bstring( hp->h_name );
   a_socket->socket_t.hostip      = string_to_bstring( inet_ntoa(server.sin_addr) );
   a_socket->socket_t.fd 	  = s;
   a_socket->socket_t.input 	  = BFALSE;
   a_socket->socket_t.output 	  = BFALSE;
   a_socket->socket_t.ready_event = BFALSE;

   set_socket_io_ports( s, a_socket, str );
   return a_socket;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    make_server_socket ...                                           */
/*---------------------------------------------------------------------*/
obj_t
make_server_socket( int port )
{
   char msg[] = "make-server-socket";
   struct sockaddr_in sin;
   int s, portnum, len;
   obj_t local_host;
   struct in_addr local_ip;
   obj_t a_socket;

   /* Determine port to use */
   portnum = port;
   if( portnum < 0 )
      socket_error( "make-server-socket", "bad port number", BINT( port ) );

   /* Create a socket */
   if( BAD_SOCKET( s = socket(AF_INET, SOCK_STREAM, 0) ) )
      socket_error( "make-server-socket", "Cannot create socket", BUNSPEC );
  
   /* Bind the socket to a name */
   sin.sin_family      = AF_INET;
   sin.sin_port        = htons( portnum );
   sin.sin_addr.s_addr = INADDR_ANY;

   if( bind( s, (struct sockaddr *)&sin, sizeof( sin ) ) < 0 )
   {
      close( s );
      system_error( msg );
   }

   /* Query the socket name (permits to get the true socket number if 0 was given */
   len = sizeof( sin );
   if( getsockname(s, (struct sockaddr *)&sin, (int *)&len ) < 0 )
   {
      close( s );
      system_error( msg );
   }

   /* Indicate that we are ready to listen */
   if( listen( s, 5 ) < 0 )
   {
      close( s );
      system_error( msg );
   }

   /* Now we can create the socket object */
   a_socket                        = GC_MALLOC( SOCKET_SIZE );
   a_socket->socket_t.header       = MAKE_HEADER( SOCKET_TYPE, 0 );
   a_socket->socket_t.portnum      = ntohs( sin.sin_port );
   a_socket->socket_t.hostname     = string_to_bstring( "localhost" );
   a_socket->socket_t.hostip	   = string_to_bstring( "localhost" );
   a_socket->socket_t.fd	   = s;
   a_socket->socket_t.input	   = BFALSE;
   a_socket->socket_t.output	   = BFALSE;
   a_socket->socket_t.ready_event  = BFALSE;
   
   return a_socket;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_accept_connection ...                                     */
/*---------------------------------------------------------------------*/
obj_t
socket_accept_connection( obj_t sock )
{
   char buff[ 50 ], *s;
   char str[] = "socket-accept-connection";
   struct sockaddr_in sin;
   struct hostent *host;
   int len = sizeof( sin );
   int new_s;

   if( BAD_SOCKET( new_s = accept( SOCKET( sock ).fd,
				   (struct sockaddr *)&sin,
				   &len ) ) )
      system_error( str );

   /* Set the client info (if possible its name, otherwise its IP number) */
   host = gethostbyaddr( (char *)&sin.sin_addr, sizeof( sin.sin_addr ), AF_INET );
   s    = (char *)inet_ntoa( sin.sin_addr );

   SOCKET( sock ).hostip   = string_to_bstring( s );
   SOCKET( sock ).hostname = string_to_bstring( host ? host->h_name : s );
   
   set_socket_io_ports( new_s, sock, str );
   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_shutdown ...                                              */
/*---------------------------------------------------------------------*/
obj_t
socket_shutdown( obj_t sock, int close_socket )
{
   obj_t tmp1, tmp2;

  if( close_socket && SOCKET( sock ).fd > 0)
  {
     int fd = SOCKET( sock ).fd;

     shutdown( fd, 2 );
     SOCKET( sock ).fd = -1;
  }
 
  /* Warning: input and output can have already be garbaged :if the socket is   */
  /* no more used, the input and output are not marked as used and can          */
  /* (eventually) be released before the call to shutdown (through free_socket) */
  /* be done. One way could be to just set SOCKET(sock).{in|out}put to #t       */
  /* and wait that next GC frees the ports if not already down. However,        */
  /* this will really disconnect the peer when the GC occurs rather than when   */
  /* the call to shutdown is done. This is not important if this function       */
  /* is called by the GC, but could be annoying when it is called by the user   */
  if( INPUT_PORTP( SOCKET(sock).input ) )
  {
     close_input_port( SOCKET( sock ).input );
     SOCKET( sock ).input = BFALSE;
  }
  if( OUTPUT_PORTP( SOCKET(sock).output ) )
  {
     close_output_port( SOCKET( sock ).output );
     SOCKET( sock ).output = BFALSE;
  }
  
  return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_dup ...                                                   */
/*---------------------------------------------------------------------*/
obj_t
socket_dup( obj_t socket )
{
  obj_t a_socket;
  int new_fd;
#ifdef WIN32
  HANDLE process;
#endif

#ifdef WIN32
  process = GetCurrentProcess();
  if( !DuplicateHandle( process, 
			(HANDLE)SOCKET(socket).fd, process,
			(HANDLE*)&new_fd,
			0, 
			TRUE, 
			DUPLICATE_SAME_ACCESS ) )
     socket_error( "socket-dup",
		   "cannot duplicate socket",
		   BINT( GetLastError() ) );
#else
  if( (new_fd=dup( SOCKET(socket).fd )) < 0 )
     socket_error("socket-dup", "cannot duplicate socket", socket );
#endif

  a_socket                  = GC_MALLOC( SOCKET_SIZE );
  a_socket->socket_t.header = MAKE_HEADER( SOCKET_TYPE, 0 );

  SOCKET( a_socket ) = SOCKET( socket );
  SOCKET( a_socket ).fd = new_fd;  

  return a_socket;
}


/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    socket_local_addr ...                                            */
/*---------------------------------------------------------------------*/
obj_t
socket_local_addr( obj_t sock )
{
   struct sockaddr_in sin;
   int len = sizeof( sin );

   if( getsockname( SOCKET( sock ).fd, (struct sockaddr *)&sin, &len ) )
      socket_error( "socket-local-address", "cannot get socket name", sock );

   return string_to_bstring( (char *) inet_ntoa( sin.sin_addr ) );
}
