/*---------------------------------------------------------------------*/
/*   A pratical implementation for the Scheme programming language     */
/*                                                                     */
/*                                    ,--^,                            */
/*                              _ ___/ /|/                             */
/*                          ,;'( )__, ) '                              */
/*                         ;;  //   L__.                               */
/*                         '   \\   /  '                               */
/*                              ^   ^                                  */
/*                                                                     */
/*   Copyright (c) 1992-1999 Manuel Serrano                            */
/*                                                                     */
/*     Bug descriptions, use reports, comments or suggestions are      */
/*     welcome. Send them to                                           */
/*       bigloo-request@kaolin.unice.fr                                */
/*       http://kaolin.unice.fr/bigloo                                 */
/*                                                                     */
/*   This program is free software; you can redistribute it            */
/*   and/or modify it under the terms of the GNU General Public        */
/*   License as published by the Free Software Foundation; either      */
/*   version 2 of the License, or (at your option) any later version.  */
/*                                                                     */
/*   This program is distributed in the hope that it will be useful,   */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/*   GNU General Public License for more details.                      */
/*                                                                     */
/*   You should have received a copy of the GNU General Public         */
/*   License along with this program; if not, write to the Free        */
/*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
/*   MA 02111-1307, USA.                                               */
/*---------------------------------------------------------------------*/
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/crgc.c                  */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Sun Sep 13 11:58:32 1998                          */
/*    Last change :  Thu Feb 11 21:28:15 1999 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Rgc runtime (mostly port handling).                              */
/*=====================================================================*/
#include <stdio.h>
#include <errno.h> 
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#if( !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   include <termio.h>
#endif
#if( !defined( sony_news ) && \
     !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   include <unistd.h>
#endif
#include <sys/file.h>
/*---------------------------------------------------------------------*/
/*    The file `gc_private.h' must not be included into this file. To  */
/*    prevent this inclusion, we define the GC_PRIVATE_H macro. We     */
/*    have to protect this file, because `gc_private.h' use an         */
/*    incorrect prototype for `sbrk' that is incompatible we some      */
/*    operating system (such as Linux). Linux uses a prototype in      */
/*    the file `unisys.h' which is not compatible with the one used    */
/*    in file `gc_private.h'.                                          */
/*---------------------------------------------------------------------*/
#if( defined( i386 ) )
#   define GC_PRIVATE_H
#endif
#include <bigloo2.0a.h>
#if( defined( sony_news ) || (defined( NeXT ) && defined( mc68000 )) )
#   include <ctype.h>
#endif

/*---------------------------------------------------------------------*/
/*    isascii                                                          */
/*---------------------------------------------------------------------*/
#if( !defined( isascii ) )
#   define isascii( c ) (!((c) & ~0177))
#endif

/*---------------------------------------------------------------------*/
/*    DEBUG                                                            */
/*---------------------------------------------------------------------*/
#define DEBUG

#if( !defined( DEBUG ) )
#   define assert( exp ) (0)
#endif

/*---------------------------------------------------------------------*/
/*    C importations                                                   */
/*---------------------------------------------------------------------*/
extern obj_t bigloo_case_sensitive;
extern obj_t string_to_bstring_len( char *, int );
extern obj_t string_to_symbol( char * );
extern obj_t string_to_keyword( char * );

/*---------------------------------------------------------------------*/
/*    Stub code (for Tk)                                               */
/*---------------------------------------------------------------------*/
int (*the_getc)(FILE *) = fgetc;

#if( defined( feof) )
int my_feof( FILE *stream ) {
   return feof( stream );
}
int (*the_feof)(FILE *) = my_feof;
#else
int (*the_feof)(FILE *) = feof;
#endif

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    rgc_enlarge_buffer ...                                           */
/*    -------------------------------------------------------------    */
/*    This function double the size of a port's buffer. An error is    */
/*    raised if there is not enough room for the allocation.           */
/*---------------------------------------------------------------------*/
static void
rgc_enlarge_buffer( obj_t port )
{
   long bufsize = INPUT_PORT( port ).bufsiz;
   char *buffer = (char *)GC_MALLOC_ATOMIC( 2 * bufsize );
   
#if defined( RGC_DEBUG )
   printf( "rgc_enlarge_buffer: bufsize: %d\n", bufsize );
#endif

   if( !buffer )
      C_FAILURE( "rgc_enlarge_buffer_port", "Can't enlarge buffer", port );
      
   memcpy( buffer, RGC_BUFFER( port ), bufsize );

   INPUT_PORT( port ).bufsiz = 2 * bufsize;
   RGC_BUFFER( port )      = buffer;
}
  
/*---------------------------------------------------------------------*/
/*    static long                                                      */
/*    fread_con ...                                                    */
/*    -------------------------------------------------------------    */
/*    This function implements a non blocking fread on the console.    */
/*---------------------------------------------------------------------*/
static long
fread_con( char *ptr, long size, long nmemb, FILE *stream )
{
   long  num = size * nmemb;
   char *buf = ptr;
   int   c;
   
   while( ((c = the_getc( stream )) != EOF) )
   {
      *buf++ = c;

      if( c == '\n' ) break;
      if( --num <= 0 ) break;
   }

   return (long)(buf - ptr);
}
   
/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_size_fill_con_buffer ...                                     */
/*---------------------------------------------------------------------*/
static bool_t
rgc_size_fill_con_buffer( obj_t port, int abufsize, int size )
{
   unsigned char *buffer = RGC_BUFFER( port );

#if defined( RGC_DEBUG )
   printf( "rgc_size_fill_console_buffer: abufsize: %d  size: %d\n", abufsize, size );
   assert( (abufsize + size) == INPUT_PORT( port ).bufsiz );
   assert( size > 0 );
#endif

   /* we start reading at ABUFSIZE - 1 because we have */
   /* to remove the '\0' sentinel that ends the buffer */
   abufsize += fread_con( &buffer[ abufsize - 1 ],
 			  1,
			  size,
			  INPUT_PORT( port ).file );
   buffer[ abufsize - 1 ] = '\0';
   
   INPUT_PORT( port ).abufsiz = abufsize;
   assert( INPUT_PORT( port ).abufsiz <= INPUT_PORT( port ).bufsiz );
   
   if( the_feof( INPUT_PORT( port ).file ) )
#if defined( RGC_DEBUG )
      printf( "rgc_size_fill_console_buffer: EOF\n"),
#endif
      INPUT_PORT( port ).eof = 1;

   if( ferror( INPUT_PORT( port ).file ) )
      C_FAILURE( "rgc_size_fill_con_buffer",
		 "Error while reading on console",
		 BINT( ferror( INPUT_PORT( port ).file ) ) );
   
#if defined( RGC_DEBUG )
   printf( "FIN de fill: size: %d  asize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   INPUT_PORT( port ).bufsiz, INPUT_PORT( port ).abufsiz,
	   INPUT_PORT( port ).forward, 
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
   printf( "buffer: [%s]\n", RGC_BUFFER( port ) );
#endif

   return (INPUT_PORT( port ).abufsiz > 0);
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_fill_console_buffer ...                                      */
/*---------------------------------------------------------------------*/
static bool_t
rgc_fill_console_buffer( obj_t port )
{
   long bufsize    = INPUT_PORT( port ).bufsiz;
   long abufsize   = INPUT_PORT( port ).abufsiz;
   long forward    = INPUT_PORT( port ).forward;
   long matchstart = INPUT_PORT( port ).matchstart;

#if defined( RGC_DEBUG )
   printf( "rgc_fill_console_buffer: bufsize: %d  abufsize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   bufsize, abufsize, INPUT_PORT( port ).forward,
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
#endif
   
   /* if the buffer is not full, we fill it */
   if( abufsize < bufsize )
      return rgc_size_fill_con_buffer( port, abufsize, bufsize - abufsize );
   else
   {
      if( matchstart > 0 )
      {
	 unsigned char *buffer = RGC_BUFFER( port );

	 assert( abufsize > 0 );
	 
	 /* we shift the buffer left and we fill the buffer */
	 strcpy( buffer, buffer + matchstart );
	 
	 abufsize                     -= matchstart;
	 INPUT_PORT( port ).matchstart = 0;
	 INPUT_PORT( port ).matchstop -= matchstart;
	 INPUT_PORT( port ).forward   -= matchstart;
	 INPUT_PORT( port ).lastchar   = RGC_BUFFER( port )[ matchstart - 1 ];
	 
	 return rgc_size_fill_con_buffer( port, abufsize, bufsize-abufsize );
      }
      else
      {
	 /* we current token is too large for the buffer */
	 /* we have to enlarge it.                       */
	 rgc_enlarge_buffer( port );
	 
	 return rgc_fill_console_buffer( port );
      }
   }
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_size_fill_file_buffer ...                                    */
/*---------------------------------------------------------------------*/
static bool_t
rgc_size_fill_file_buffer( obj_t port, int abufsize, int size )
{
   unsigned char *buffer = RGC_BUFFER( port );
   
#if defined( RGC_DEBUG )
   assert( abufsize >= 1 );
   assert( (abufsize + size) == INPUT_PORT( port ).bufsiz );

   printf( "rgc_size_fill_file_buffer: abufsize: %d  size: %d\n", abufsize, size );
   assert( size > 0 );
#endif

   /* we start reading at ABUFSIZE - 1 because we have */
   /* to remove the '\0' sentinel that ends the buffer */
   abufsize += fread( &buffer[ abufsize - 1 ],
		      1,
		      size,
		      INPUT_PORT(port).file );
   
   INPUT_PORT( port ).abufsiz = abufsize;
   assert( INPUT_PORT( port ).abufsiz <= INPUT_PORT( port ).bufsiz );
   
   if( feof( INPUT_PORT( port ).file ) )
      INPUT_PORT( port ).eof = 1;

   if( ferror( INPUT_PORT( port ).file ) )
      C_FAILURE( "rgc_size_fill_file_buffer",
		 "Error while reading on file",
		 BINT( ferror( INPUT_PORT( port ).file ) ) );
   

#if defined( RGC_DEBUG )
   printf( "FIN de fill: size: %d  asize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   INPUT_PORT( port ).bufsiz, INPUT_PORT( port ).abufsiz,
	   INPUT_PORT( port ).forward, 
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
   printf( "buffer: [%s]\n", buffer );
#endif
   
   if( abufsize > 0 )
   { 
      buffer[ abufsize - 1 ] = '\0';

      return 1;
   }
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    void *                                                           */
/*    movemem ...                                                      */
/*---------------------------------------------------------------------*/
void *movemem( char *dest, char *src, long n )
{
   while( n-- )
      *dest++ = *src++;
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    rgc_fill_file_buffer ...                                         */
/*---------------------------------------------------------------------*/
static bool_t
rgc_fill_file_buffer( obj_t port )
{
   long bufsize    = INPUT_PORT( port ).bufsiz;
   long abufsize   = INPUT_PORT( port ).abufsiz;
   long forward    = INPUT_PORT( port ).forward;
   long matchstart = INPUT_PORT( port ).matchstart;
   unsigned char *buffer = RGC_BUFFER( port );

#if defined( RGC_DEBUG )
   assert( (abufsize >= 0) && (abufsize <= bufsize) );
   
   printf( "rgc_fill_file_buffer: bufsize: %d  abufsize: %d  forward: %d  mstart: %d  mstop: %d\n",
	   bufsize, abufsize, INPUT_PORT( port ).forward,
	   INPUT_PORT( port ).matchstart, INPUT_PORT( port ).matchstop );
#endif

   if( matchstart > 0 )
   {
      long movesize = abufsize - matchstart;
	 
      /* we shift the buffer left and we fill the buffer */
      movemem( &buffer[ 0 ], &buffer[ matchstart ], movesize );
	 
      abufsize                     -= matchstart;
      INPUT_PORT( port ).matchstart = 0;
      INPUT_PORT( port ).matchstop -= matchstart;
      INPUT_PORT( port ).forward   -= matchstart;
      INPUT_PORT( port ).lastchar   = buffer[ matchstart - 1 ];
	 
      return rgc_size_fill_file_buffer( port, abufsize, bufsize - abufsize );
   }
   else
   {
      if( abufsize < bufsize )
	 return rgc_size_fill_file_buffer( port, abufsize, bufsize-abufsize );
      else
      {
	 /* we current token is too large for the buffer */
	 /* we have to enlarge it.                       */
	 rgc_enlarge_buffer( port );
	 
	 return rgc_fill_file_buffer( port );
      }
   }
}
      
/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_fill_buffer ...                                              */
/*---------------------------------------------------------------------*/
bool_t
rgc_fill_buffer( obj_t port )
{
#if defined( RGC_DEBUG ) 
   puts( "~~~~~ rgc_fill_buffer ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~" );
   printf( "eof: %d  mstart: %d  mstop: %d\n",
	   INPUT_PORT( port ).eof,
	   INPUT_PORT( port ).matchstart,
	   INPUT_PORT( port ).matchstop );
#endif
   /* In every case, forward has to be unwinded */
   /* because forward has reached the sentinel  */
   INPUT_PORT( port ).forward--;
   
   /* an input port that has seen its eof       */
   /* cannot be filled anymore                  */
   if( INPUT_PORT( port ).eof )
      return 0;
   else
   {
      if( (INPUT_PORT( port ).kindof == KINDOF_FILE) ||
	  (INPUT_PORT( port ).kindof == KINDOF_PIPE) )
	 return rgc_fill_file_buffer( port );
      else
	 return rgc_fill_console_buffer( port );
   }
}
   
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_substring ...                                         */
/*    -------------------------------------------------------------    */
/*    This function makes no bound checks because these tests have     */
/*    already been performed in the grammar.                           */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_substring( obj_t ip, long offset, long end )
{
   long start = INPUT_PORT( ip ).matchstart;
   long len   = end - offset;

#if defined( RGC_DEBUG )
   printf( "buffer-substring: start: %d  stop: %d  forward: %d  abufsiz: %d\n",
	   start, INPUT_PORT( ip ).matchstop,
	   INPUT_PORT( ip ).forward, INPUT_PORT( ip ).abufsiz );
#endif

   return string_to_bstring_len( &RGC_BUFFER( ip )[ start + offset ], len );
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    rgc_buffer_fixnum ...                                            */
/*---------------------------------------------------------------------*/
long
rgc_buffer_fixnum( obj_t ip )
{
   return atol( &RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart ] );
}

/*---------------------------------------------------------------------*/
/*    double                                                           */
/*    rgc_buffer_flonum ...                                            */
/*---------------------------------------------------------------------*/
double
rgc_buffer_flonum( obj_t ip )
{
   return strtod( &RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart ], 0 );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_symbol ...                                            */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_symbol( obj_t ip )
{
   long size  = RGC_BUFFER_LENGTH( ip );
   long start = INPUT_PORT( ip ).matchstart;
   long stop  = INPUT_PORT( ip ).matchstop;
   char bck;
   char *aux;
   obj_t sym;
   
   bck = RGC_BUFFER( ip )[ stop ];
   RGC_BUFFER( ip )[ stop ] = '\0';
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   if( (BUNSPEC == bigloo_case_sensitive) || !CBOOL( bigloo_case_sensitive ) )
   {
      unsigned char *walk;

      for( walk = aux; *walk; walk++ )
	 if( isascii( *walk ) )
	    *walk = toupper( *walk );
   }

   sym = string_to_symbol( aux );
   RGC_BUFFER( ip )[ stop ] = bck;

   return sym;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    rgc_buffer_keyword ...                                           */
/*---------------------------------------------------------------------*/
obj_t
rgc_buffer_keyword( obj_t ip )
{
   long size  = RGC_BUFFER_LENGTH( ip );
   long start = INPUT_PORT( ip ).matchstart;
   long stop  = INPUT_PORT( ip ).matchstop;
   char bck;
   char *aux;
   obj_t key;
   
   bck = RGC_BUFFER( ip )[ stop ];
   RGC_BUFFER( ip )[ stop ] = '\0';
   
   aux = &RGC_BUFFER( ip )[ start ];
   
   if( (BUNSPEC == bigloo_case_sensitive) || !CBOOL( bigloo_case_sensitive ) )
   {
      unsigned char *walk;

      for( walk = aux; *walk; walk++ )
	 if( isascii( *walk ) )
	    *walk = toupper( *walk );
   }

   key = string_to_keyword( aux );
   RGC_BUFFER( ip )[ stop ] = bck;

   return key;
}
 
/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    rgc_buffer_unget_char ...                                        */
/*---------------------------------------------------------------------*/
int
rgc_buffer_unget_char( obj_t ip, int c )
{
   if( INPUT_PORT( ip ).matchstop > 0 )
      INPUT_PORT( ip ).matchstop--;
   else
   {
      RGC_BUFFER( ip )[ 0 ] = c;
      if( INPUT_PORT( ip ).abufsiz == 0 )
      {
	 INPUT_PORT( ip ).abufsiz == 1;
	 RGC_BUFFER( ip )[ 1 ] = '\0';
      }
   }

   return c;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_bol_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the matchstart position located at the beginning of a line?   */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_bol_p( obj_t ip )
{
#if( defined( RGC_DEBUG ) )   
   printf( "RGC_BUFFER_BOL_P: mstart: %d  [mstart]: %d  lastchar: %d  --> %d\n",
	   INPUT_PORT( ip ).matchstart, 
	   RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart - 1 ],
	   INPUT_PORT( ip ).lastchar,
	   INPUT_PORT( ip ).lastchar == '\n' );
#endif
   
   if( INPUT_PORT( ip ).matchstart > 0 )
      return RGC_BUFFER( ip )[ INPUT_PORT( ip ).matchstart - 1 ] == '\n';
   else
      return INPUT_PORT( ip ).lastchar == '\n';
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_eol_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Does the buffer contain, at its first non match position, a `\n' */
/*    character?                                                       */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_eol_p( obj_t ip )
{
   int c = RGC_BUFFER_GET_CHAR( ip );
   
#if( defined( RGC_DEBUG ) )   
   long f = INPUT_PORT( ip ).forward;
#endif
   
#if( defined( RGC_DEBUG ) )   
   printf( "RGC_BUFFER_EOL_P: forward: %d %d", f, c );
#endif
   
   if( !c )
   {
      if( !RGC_BUFFER_EMPTY( ip ) )
      {
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   not empty --> 0" );
#endif
	 return 0;
      }

      if( INPUT_PORT( ip ).kindof == KINDOF_CONSOLE )
      {
#if( defined( RGC_DEBUG ) )   
	 puts( "  kindof == CONSOLE --> 1" );
#endif
	 return 1;
      }
      if( rgc_fill_buffer( ip ) )
	 return rgc_buffer_eol_p( ip );
      else
      {
#if( defined( RGC_DEBUG ) )   
	 puts( "   not rgc_fill_buffer --> 0" );
#endif
	 return 0;
      }
   }
   else
   {
      INPUT_PORT( ip ).forward--;
#if( defined( RGC_DEBUG ) )   
      printf( "   --> %d\n", c == '\n' );
#endif
      return c == '\n';
   }
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_bof_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the match position at the beginning of the file?              */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_bof_p( obj_t ip )
{
   return INPUT_PORT( ip ).filepos == 1;
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    rgc_buffer_eof_p ...                                             */
/*    -------------------------------------------------------------    */
/*    Is the input port at its end-of-file position?                   */
/*---------------------------------------------------------------------*/
bool_t
rgc_buffer_eof_p( obj_t ip )
{
   int c = RGC_BUFFER_GET_CHAR( ip );

#if( defined( RGC_DEBUG ) )   
   long f = INPUT_PORT( ip ).forward;
#endif
   
#if( defined( RGC_DEBUG ) )   
   printf( "RGC_BUFFER_EOF_P: forward: %d %d", f, c );
#endif
   
   if( !c )
   {
      if( !RGC_BUFFER_EMPTY( ip ) )
      {
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   not empty --> 0" );
#endif
	 return 0;
      }
      else
	 INPUT_PORT( ip ).forward--;
	 
#if( defined( RGC_DEBUG ) )   
	 puts( "   --> 1" );
#endif
	 return 1;
   }
   else
   {
      INPUT_PORT( ip ).forward--;
#if( defined( RGC_DEBUG ) )   
      puts( "   not empty --> 0" );
#endif
      return 0;
   }
}
      
      

   
