/*---------------------------------------------------------------------*/
/*   A pratical implementation for the Scheme programming language     */
/*                                                                     */
/*                                    ,--^,                            */
/*                              _ ___/ /|/                             */
/*                          ,;'( )__, ) '                              */
/*                         ;;  //   L__.                               */
/*                         '   \\   /  '                               */
/*                              ^   ^                                  */
/*                                                                     */
/*   Copyright (c) 1992-1999 Manuel Serrano                            */
/*                                                                     */
/*     Bug descriptions, use reports, comments or suggestions are      */
/*     welcome. Send them to                                           */
/*       bigloo-request@kaolin.unice.fr                                */
/*       http://kaolin.unice.fr/bigloo                                 */
/*                                                                     */
/*   This program is free software; you can redistribute it            */
/*   and/or modify it under the terms of the GNU General Public        */
/*   License as published by the Free Software Foundation; either      */
/*   version 2 of the License, or (at your option) any later version.  */
/*                                                                     */
/*   This program is distributed in the hope that it will be useful,   */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/*   GNU General Public License for more details.                      */
/*                                                                     */
/*   You should have received a copy of the GNU General Public         */
/*   License along with this program; if not, write to the Free        */
/*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
/*   MA 02111-1307, USA.                                               */
/*---------------------------------------------------------------------*/
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/cprocess.c              */
/*    -------------------------------------------------------------    */
/*    Author      :  Erick Gallesio                                    */
/*    Creation    :  Mon Jan 19 17:35:12 1998                          */
/*    Last change :  Fri Sep 18 15:33:22 1998 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Process handling C part. This part is mostly compatible with     */
/*    STK. This code is extracted from STK by Erick Gallesio.          */
/*=====================================================================*/
#include <fcntl.h>
#include <errno.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <bigloo2.0a.h>

#define MSG_SIZE 1024

#if( defined( HAVE_SIGCHLD ) )
#   define PURGE_PROCESS_TABLE()	/* Nothing to do */
#else
#   define PURGE_PROCESS_TABLE() process_terminate_handler( 0 )
#endif

/*---------------------------------------------------------------------*/
/*    Importations                                                     */
/*---------------------------------------------------------------------*/
extern obj_t make_input_port( char *, FILE *, obj_t, long );
extern obj_t make_output_port( char *, FILE *, obj_t );
extern int   default_io_bufsiz;
extern obj_t close_input_port( obj_t );
   
/*---------------------------------------------------------------------*/
/*    process table                                                    */
/*---------------------------------------------------------------------*/
#define MAX_PROC_NUM 40                    /* (simultaneous processes) */
static obj_t proc_arr[ MAX_PROC_NUM ];     /* process table            */

static char *std_streams[ 3 ] = {
  "input",	
  "output",	
  "error",
};

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    c_process_alivep ...                                             */
/*---------------------------------------------------------------------*/
obj_t
c_process_alivep( obj_t process )
{
   if( PROCESS( process ).exited ) 
      return BFALSE;
   else
   {
      int info, res;

      /* Use waitpid to gain the info. */
      res = waitpid( PROCESS_PID( process ), &info, WNOHANG );
      
      if( res == 0 ) 
	 /* process is still running */
	 return BTRUE;
      else
      {
	 if( res == PROCESS_PID( process ) )
	 {
	    /* process has terminated and we must save this information */
	    PROCESS(process).exited      = 1;
	    PROCESS(process).exit_status = info;
	    return BFALSE;
	 }
	 else
	    return BFALSE;
      }
   }
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    c_unregister_process ...                                         */
/*---------------------------------------------------------------------*/
void
c_unregister_process( obj_t proc )
{
   int i;

   for( i = 0; i < 3; i++ )
   {
      obj_t p = PROCESS( proc ).stream[ i ];
      
      if( INPUT_PORTP( p ) )
	 close_input_port( p );

      if( OUTPUT_PORTP( p ) )
	 close_output_port( p );
   }
   
   proc_arr[ PROCESS( proc ).index ] = BUNSPEC;
}
   
/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    process_terminate_handler ...                                    */
/*---------------------------------------------------------------------*/
static void
process_terminate_handler( int sig )
{
  register int i;
  obj_t proc;

#if( defined( HAVE_SIGCHLD ) && !defined( HAVE_SIGACTION ) )
  static int in_handler = 0;

  /* Necessary on System V */
  signal( SIGCHLD, process_terminate_handler ); 
  if( in_handler++ ) /* Execution is re-entrant */ return;
  
  do {
#endif
     /* Find the process which is terminated                          */
     /* Note that this loop can find:                                 */
     /*      - nobody: if the process has been destroyed by GC        */
     /*      - 1 process: This is the normal case                     */
     /*	    - more than one process: This can arise when:             */
     /*		- we use signal rather than sigaction                 */
     /*		- we don't have SIGCHLD and this function is called   */
     /*		  by PURGE_PROCESS_TABLE                              */
     /* Sometimes I think that life is a little bit complicated (ndrl */
     /* sic Erick Gallesio :-)                                        */
     for( i = 0; i < MAX_PROC_NUM; i++ )
     {
	proc = proc_arr[ i ];
	if( PROCESSP( proc ) && !c_process_alivep( proc ) )
	   /* This process has exited. We can delete it from the table*/
	   c_unregister_process( proc );
     }

#if( defined( HAVE_SIGCHLD ) && !defined( HAVE_SIGACTION ) )
     /* Since we can be called recursively, we have perhaps forgot to */
     /* delete some dead process from the table. So, we have perhaps  */
     /* to scan the process array another time                        */
  } while ( --in_handler > 0 );
#endif
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    init_process ...                                                 */
/*---------------------------------------------------------------------*/
static void
init_process()
{
   static initializedp = 0;
   int i;
   
   if( initializedp )
      return;
   else
      initializedp = 1;

   /* we first initialize the process table */
   for( i = 0; i < MAX_PROC_NUM; i++ ) proc_arr[ i ] = BUNSPEC;

#if( defined( HAVE_SIGCHLD ) )
   /* On systems which support SIGCHLD, the processes table is cleaned */
   /* up as soon as a process terminate. On other systems this is done */
   /* from time to time to avoid filling the table too fast.           */

# if( defined( HAVE_SIGACTION ) )
  {
    /* Use the secure Posix.1 way */
    struct sigaction sigact;
    
    sigemptyset( &(sigact.sa_mask) );
    sigact.sa_handler = process_terminate_handler;
    /* Ignore SIGCHLD generated by SIGSTOP */
    sigact.sa_flags   = SA_NOCLDSTOP;     
#  if( defined( SA_RESTART ) )
    /* Thanks to Harvey J. Stein <hjstein@MATH.HUJI.AC.IL> for the fix */
    sigact.sa_flags  |= SA_RESTART;
#  endif
    sigaction( SIGCHLD, &sigact, NULL );
  }
# else
  /* Use "classical" way. (Only Solaris 2 seems to have problem with it */
  signal( SIGCHLD, process_terminate_handler );
# endif
#endif

  return;
}

/*---------------------------------------------------------------------*/
/*    static int                                                       */
/*    find_process ...                                                 */
/*---------------------------------------------------------------------*/
static int
find_process()
{
   int i;

   for( i = 0; i < MAX_PROC_NUM; i++ )
      if( BUNSPEC == proc_arr[ i ] ) return i;
   return (-1);
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    cannot_run ...                                                   */
/*---------------------------------------------------------------------*/
static void
cannot_run( int pipes[ 3 ][ 2 ], obj_t bcommand, char *msg )
{
   int i;

   for( i = 0; i < 3; i++ )
   {
      if( pipes[ i ][ 0 ] != -1 ) close( pipes[ i ][ 0 ] );
      if( pipes[ i ][ 1 ] != -1 ) close( pipes[ i ][ 1 ]);
   }
  
   FAILURE( string_to_bstring( "run-process" ),
	    string_to_bstring( msg ),
	    bcommand );
}
 
/*---------------------------------------------------------------------*/
/*    static obj_t                                                     */
/*    make_process ...                                                 */
/*---------------------------------------------------------------------*/
static obj_t
make_process()
{
   int   i;
   obj_t a_proc;

   init_process();
   PURGE_PROCESS_TABLE();

   /* find slot */
   i = find_process();
   if( i < 0 )
      C_FAILURE( "make-process", "too many processes", BUNSPEC );

   a_proc = GC_MALLOC( PROCESS_SIZE );
   a_proc->process_t.header      = MAKE_HEADER( PROCESS_TYPE, 0 );
   a_proc->process_t.index       = i;
   a_proc->process_t.stream[ 0 ] = BFALSE;
   a_proc->process_t.stream[ 1 ] = BFALSE;
   a_proc->process_t.stream[ 2 ] = BFALSE;
   a_proc->process_t.exit_status = 0;
   a_proc->process_t.exited      = 0;
   
   /* Enter this process in the process table */
   proc_arr[ i ] = BREF( a_proc );
   
   return BREF( a_proc );
}
  
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_run_process ...                                                */
/*---------------------------------------------------------------------*/
obj_t
c_run_process( obj_t bhost, obj_t bfork, obj_t bwaiting,
	       obj_t binput, obj_t boutput, obj_t berror,
	       obj_t bcommand, obj_t bargs )
{
   bool_t     waiting;
   int        pid, i, argc;
   obj_t      redirection[ 3 ];
   int        pipes[ 3 ][ 2 ];
   char       msg[ MSG_SIZE ], **argv, **argv_start;
   obj_t      runner;
   obj_t      proc;
   
   /* misc initializations */
   waiting = CBOOL( bwaiting );

   /* redirection initializations */
   redirection[ 0 ] = binput;
   redirection[ 1 ] = boutput;
   redirection[ 2 ] = berror;
   for (i = 0; i < 3; i++)
   {
      pipes[ i ][ 0 ] = pipes[ i ][ 1 ] = -1;
   }

   /* First try to look if this redirecttion has not already done       */
   /* This can arise by doing                                           */
   /*     output: "out" error: "out"       which is correct             */
   /*     output: "out" input: "out"       which is obviously incorrect */
   for( i = 0; i < 3; i++ )
   {
      if( STRINGP( redirection[ i ] ) )
      {
	 /* redirection to a file */
	 int j;
	 char *ri = BSTRING_TO_STRING( redirection[ i ] );
	 
	 for( j = 0; j < 3; j++ )
	 {
	    if( j != i && STRINGP( redirection[ j ] ) )
	    {
	       struct stat stat_i, stat_j;
	       char *rj = BSTRING_TO_STRING( redirection[ j ] );
	       /* Do a stat to see if we try to open the same file 2    */
	       /* times. If stat == -1 this is probably because file    */
	       /* doesn't exist yet.                                    */
	       if( stat( ri, &stat_i ) == -1 )
		  continue;
	       if( stat( rj, &stat_j) == -1 )
		  continue;
		
	       if( stat_i.st_dev==stat_j.st_dev &&
		   stat_i.st_ino==stat_j.st_ino)
	       {
		  /* Same file was cited 2 times */
		  if( i == 0 || j == 0 )
		  {
		     sprintf( msg, "read/write on the same file: %s", ri );
		     cannot_run( pipes, bcommand, msg );
		  }
		  
		  /* assert(i == 1 && j == 2 || i == 2 && j == 1); */
		  pipes[ i ][ 0 ] = dup( pipes[ j ][ 0 ] );
		  break;
	       }
	    }
	 }
	    
	 /* Two cases are possible here:                                     */
	 /* - we have stdout and stderr redirected on the same file (j != 3) */
	 /* - we have not found current file in list of redirections (j == 3)*/
	 if( j == 3 )
	 {
	    pipes[ i ][ 0 ] = open( ri,
				    i==0 ? O_RDONLY:(O_WRONLY|O_CREAT|O_TRUNC),
				    0666 );
	 }
	    
	 if( pipes[ i ][ 0 ] < 0 )
	 {
	    sprintf( msg,
		     "can't redirect standard %s to file %s",
		     std_streams[ i ],
		     ri );
	    cannot_run( pipes, bcommand, msg );
	 }
      }
      else
      {
	 if( KEYWORDP( redirection[ i ] ) )
	 {
	    /* redirection in a pipe */
	    if( pipe( pipes[ i ] ) < 0 )
	    {
	       sprintf( msg,
			"can't create stream for standard %s",
			std_streams[ i ] );

	       cannot_run( pipes, bcommand, msg );
	    }
	 }
      }
   }

   /* command + arguments initializations    */
   /* 4 = null + rsh + host + command + args */
   argc = 0;
   argv_start = (char **)GC_MALLOC_ATOMIC( (list_length(bargs) + 4) *
					   sizeof(char *) );
   argv = argv_start + 2;

   argv[ argc++ ] = BSTRING_TO_STRING( bcommand );
   for( runner = bargs; PAIRP( runner ); runner = CDR( runner ) ) 
      argv[ argc++ ] = BSTRING_TO_STRING( CAR( runner ) );
   argv[ argc ] = 0L;
   
   /* rsh initialization */
   if( STRINGP( bhost ) )
   {
      argc += 2;
      argv[ 0 ] = "rsh";
      argv[ 1 ] = BSTRING_TO_STRING( bhost );
   }

   /* proc object creation */
   proc = make_process();

   switch( CBOOL( bfork ) && (pid = fork()) )
   {
      case -1:
	 cannot_run( pipes, bcommand, "can't create child process" ); 
	 break;

      case 0:
	 /* The child process */
	 for( i = 0; i < 3; i++ )
	 {
	    if( STRINGP( redirection[ i ] ) )
	    {
	       /* redirection in a file */
	       close( i );
	       dup( pipes[ i ][ 0 ] );
	       close( pipes[ i ][ 0 ] );
	    }
	    else
	    {
	       if( KEYWORDP( redirection[ i ] ) )
	       {
		  /* redirection in a pipe */
		  close( i );
		  dup( pipes[ i ][ i == 0 ? 0 : 1 ] );
		  close( pipes[ i ][ 0 ] );
		  close( pipes[ i ][ 1 ] );
	       }
	    }
	 }

	 for( i = 3; i < NOFILE; i++ ) close( i );

	 /* and now we do the exec */
	 execvp( *argv, argv );

	 /* if we are here, it is because we cannot exec */
	 cannot_run( pipes, bcommand, "can't execute" ); 
	 exit( 1 );

      default:
	 /* The is the parent process */
	 PROCESS( proc ).pid = pid;
	 for( i = 0; i < 3; i++ )
	 {
	    if( STRINGP( redirection[ i ] ) )
	       /* redirection in a file */
	       close( pipes[ i ][ 0 ] );
	    else
	    {
	       if( KEYWORDP( redirection[ i ] ) )
	       {
		  FILE *f;
		  
		  /* redirection in a pipe */
		  close( pipes[ i ][ i == 0 ? 0 : 1 ] );

		  /* make a new file descriptor to access the pipe */
		  f = ((i == 0) ?
		       fdopen( pipes[ i ][ 1 ], "w" ) :
		       fdopen( pipes[ i ][ 0 ], "r"));
		     
		  if( f == NULL )
		     cannot_run( pipes, bcommand, "cannot fdopen" );

		  sprintf( msg, "pipe-%s-%d", std_streams[ i ], pid);
		  
		  if( i != 1 )
		     PROCESS( proc ).stream[ i ] =
			make_output_port( msg, f, KINDOF_PIPE );
		  else
		     PROCESS( proc ).stream[ i ] =
			make_input_port( msg,f,KINDOF_PIPE,default_io_bufsiz );
	       }
	    }
	 }
	 
	 if( waiting )
	 {
	    waitpid( pid, &(PROCESS( proc ).exit_status), 0 );
	    PROCESS( proc ).exited = 1;
	 }
   }

   return proc;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_process_list ...                                               */
/*---------------------------------------------------------------------*/
obj_t
c_process_list()
{
   int   i;
   obj_t lst = BNIL;

   PURGE_PROCESS_TABLE();

   for( i = 0; i < MAX_PROC_NUM; i++ )
      if( proc_arr[ i ] != BUNSPEC )
	 lst = MAKE_PAIR( proc_arr[ i ], lst );
   
   return lst;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_process_wait ...                                               */
/*---------------------------------------------------------------------*/
obj_t
c_process_wait( obj_t proc )
{
   PURGE_PROCESS_TABLE();

   if( PROCESS( proc ).exited )
      return BFALSE;
   else
   {
      int ret = waitpid( PROCESS_PID(proc), &(PROCESS(proc).exit_status), 0 );

      PROCESS( proc ).exited = 1;
      return (ret == 0) ? BFALSE : BTRUE;
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_process_xstatus ...                                            */
/*---------------------------------------------------------------------*/
obj_t
c_process_xstatus( obj_t proc )
{
   int info, n;

   PURGE_PROCESS_TABLE();

   if( PROCESS(proc).exited )
      n = PROCESS(proc).exit_status;
   else
   {
      if( waitpid( PROCESS_PID( proc ), &info, WNOHANG ) == 0 )
      {
	 /* process is still running */
	 return BFALSE;
      }
      else
      {
	 /* process is now terminated */
	 PROCESS( proc ).exited      = 1;
	 PROCESS( proc ).exit_status = info;
	 n = WEXITSTATUS(info);
      }
   }
   
   return BINT( n );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_process_send_signal ...                                        */
/*---------------------------------------------------------------------*/
obj_t
c_process_send_signal( obj_t proc, int signal )
{
   PURGE_PROCESS_TABLE();
   
   kill( PROCESS_PID( proc), signal );
   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_process_kill ...                                               */
/*---------------------------------------------------------------------*/
obj_t
c_process_kill( obj_t proc )
{
#if( defined( SIGTERM ) )
   return c_process_send_signal( proc, SIGTERM );
#else
   return BUNSPEC;
#endif
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_process_stop ...                                               */
/*---------------------------------------------------------------------*/
obj_t
c_process_stop( obj_t proc )
{
#if( defined( SIGSTOP ) )
   return c_process_send_signal( proc, SIGSTOP );
#else
   return BUNSPEC;
#endif
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    c_process_continue ...                                           */
/*---------------------------------------------------------------------*/
obj_t
c_process_continue( obj_t proc )
{
#if( defined( SIGCONT ) )
   return c_process_send_signal( proc, SIGCONT );
#else
   return BUNSPEC;
#endif
}


