;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Type/tenv.scm               */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun Dec 25 11:32:49 1994                          */
;*    Last change :  Fri Dec 25 14:56:39 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The Type environment manipulation                                */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module type_env

   (include "Tools/trace.sch"
	    "Type/coercer.sch")
   
   (import  tools_shape
	    tools_error
	    ast_ident
	    ast_var
	    engine_param
	    module_module
	    type_type
	    type_tools
	    type_cache
	    object_class)

   (static  (bind-type!::type         ::symbol ::bool)
	    (uninitialized-types))

   (export  (initialize-Tenv!)
	    (set-Tenv!                <Tenv>)
	    (add-Tenv!                <Tenv>)
	    (get-Tenv)
	    (find-type::type          ::symbol)
	    (use-type!::type          ::symbol)
	    (use-foreign-type!::type  ::symbol)
	    (type-exists?::bool       ::symbol)
	    (declare-type!::type      ::symbol ::bstring ::symbol)
	    (declare-subtype!::type   ::symbol ::bstring symbol* ::symbol)
	    (declare-aliastype!::type ::symbol ::bstring ::symbol ::type)
	    (for-each-type!           ::procedure)
	    (check-types)
	    (sub-type?::bool          ::type ::type)))

;*---------------------------------------------------------------------*/
;*    *Tenv* ...                                                       */
;*    -------------------------------------------------------------    */
;*    The Global environment (for global variable definitions).        */
;*---------------------------------------------------------------------*/
(define *Tenv* 'the-global-environment)

;*---------------------------------------------------------------------*/
;*    get-hash-number ...                                              */
;*---------------------------------------------------------------------*/
(define (get-hash-number o)
   (string->0..2^x-1 (symbol->string o) 10))

;*---------------------------------------------------------------------*/
;*    set-Tenv! ...                                                    */
;*---------------------------------------------------------------------*/
(define (set-Tenv! Tenv)
   (set! *Tenv* Tenv)
   (struct-set! *Tenv* 2 get-hash-number)
   (struct-set! *Tenv* 3 type-id)
   (struct-set! *Tenv* 5 eq?))

;*---------------------------------------------------------------------*/
;*    add-Tenv! ...                                                    */
;*---------------------------------------------------------------------*/
(define (add-Tenv! Tenv)
   (define (adjust-type-coercers! type)
      (for-each (lambda (coercer)
		   (let ((from (coercer-from coercer))
			 (to   (coercer-to coercer)))
		      (coercer-from-set! coercer (find-type (type-id from)))
		      (coercer-to-set! coercer (find-type (type-id to)))))
		(type-coerce-to type)))
   (define (find-coercer from to)
      (let loop ((coercer (type-coerce-to from)))
	 (cond
	    ((null? coercer)
	     #f)
	    ((eq? (coercer-to (car coercer)) to)
	     (car coercer))
	    (else
	     (loop (cdr coercer))))))
   (define (add-type-coercers! old new)
      (for-each (lambda (coercer)
		   (let* ((from        (coercer-from coercer))
			  (to          (coercer-to coercer))
			  (tid         (type-id to))
			  (tid-exists? (type-exists? tid)))
		      (if (or (not tid-exists?)
			      (not (find-coercer old (find-type tid))))
			  (type-coerce-to-set! old
					       (cons coercer
						     (type-coerce-to old))))))
		(type-coerce-to new)))
   ;; in a first stage, we bind all new types (not rebinding already binded
   ;; types)
   (let ((remember-list '()))
      ;; the remember list is used to store class type. In have to make
      ;; two traversals over classes. A first one for defining them,
      ;; a second one, when all classes are correctly set up, to
      ;; fix the its-super class fields. Without the fix, new installed
      ;; classes would have a its-super field that points to the old
      ;; hash table (the one restored, not the current compiler's one).
      (for-each-hash (lambda (new)
			(let* ((id  (type-id new))
			       (old (get-hash id *Tenv*)))
			   (cond
			      ((not (type? old))
			       (put-hash! new *Tenv*)
			       (if (class? new)
				   (begin
				      (heap-add-class! new)
				      (set! remember-list
					    (cons new remember-list)))))
			      ((not (type-init? old))
			       (error "add-Tenv"
				      "Illegal type heap redefinition"
				      id)
			       (bigloo-exit 255))
			      (else
			       ;; we have to store the new coercers 
			       ;; for the old type
			       (add-type-coercers! old new)))))
		     Tenv)
      ;; we have to walk thru the remember list in order to
      ;; setup the correct super class fields
      (for-each (lambda (new)
		   (if (class? new)
		       (let ((super (class-its-super new)))
			  (if (class? super)
			      (let* ((super-id (class-id super))
				     (old-s    (find-type super-id)))
				 (if (not (class? old-s))
				     (error "add-Tenv"
					    "Can't find super class of"
					    (class-name new))
				     (class-its-super-set! new old-s)))))))
		remember-list))
   ;; in a second stage, we have to reset the all coercer for _all_ types.
   ;; This is mandatory because some types have not been rebound and
   ;; then we need to adjust the coercer fields of freshly bound types.
   ;; we have to walk thru all types, not only the freshly defined ones
   ;; because old types may have coercion to new types (for instance, for
   ;; fresh classes).
   (for-each-hash (lambda (new)
		     (adjust-type-coercers! new))
		  *Tenv*))
		 
;*---------------------------------------------------------------------*/
;*    get-Tenv ...                                                     */
;*---------------------------------------------------------------------*/
(define (get-Tenv)
   (struct-set! *Tenv* 2 'get-hash-number)
   (struct-set! *Tenv* 3 'type-id)
   (struct-set! *Tenv* 5 'eq?)
   *Tenv*)

;*---------------------------------------------------------------------*/
;*    initialize-Tenv! ...                                             */
;*---------------------------------------------------------------------*/
(define (initialize-Tenv!)
   ;; the global environment
   (set! *Tenv* (make-hash-table 1024 get-hash-number type-id eq? 256)))

;*---------------------------------------------------------------------*/
;*    find-type ...                                                    */
;*---------------------------------------------------------------------*/
(define (find-type::type id::symbol)
   (let ((type (get-hash id *Tenv*)))
      (if (not (type? type))
	  (error "find-type" "Can't find type" id)
	  type)))

;*---------------------------------------------------------------------*/
;*    type-exists? ...                                                 */
;*    -------------------------------------------------------------    */
;*    Returns #t if the type exists _and_ is initialized.              */
;*---------------------------------------------------------------------*/
(define (type-exists?::bool id::symbol)
   (let ((type (get-hash id *Tenv*)))
      (if (not (type? type))
	  #f
	  (type-init? type))))

;*---------------------------------------------------------------------*/
;*    bind-type! ...                                                   */
;*---------------------------------------------------------------------*/
(define (bind-type!::type id::symbol init?::bool)
   (let ((type (get-hash id *Tenv*)))
      (if (type? type)
	  (if (and (not *lib-mode*) (type-init? type))
	      (user-error "bind-type!" "Type redefinition" (shape type))
	      (begin
		 ;; the type has already been allocated, we mark it
		 ;; has initialized.
		 (if init? (type-init?-set! type #t))
		 ;; and we return it.
		 type))
	  (let ((new (instantiate::type (id id) (init? init?))))
	     (put-hash! new *Tenv*)
	     new))))

;*---------------------------------------------------------------------*/
;*    use-type! ...                                                    */
;*---------------------------------------------------------------------*/
(define (use-type!::type id::symbol)
   (trace (ast 2) "~~~ use-type!: " id #\Newline)
   (let ((type (get-hash id *Tenv*)))
      (cond
	 ((type? type)
	  type)
	 (*types-already-checked?*
	  (error "use-type!" "Can't find type" id))
	 (else
	  (trace (ast 3) "    TYPE BOUND " id #\Newline)
	  (bind-type! id #f)))))

;*---------------------------------------------------------------------*/
;*    use-foreign-type! ...                                            */
;*    -------------------------------------------------------------    */
;*    I have changed the syntax for the foreign declaration. In order  */
;*    to be consisten now, extern clauses have to be written using     */
;*    the :: notation (e.g. (print::int (::string ::int) "printf")).   */
;*    Since I also want a backward compatibility Bigloo accepts the    */
;*    two syntaxes. This function implement the compatibility.         */
;*---------------------------------------------------------------------*/
(define (use-foreign-type!::type id::symbol)
   (trace (ast 2) "~~~ use-foreign-type!: " id #\Newline)
   (let ((tid (parse-id id)))
      ;; parse-id calls  use-type! so, here we have to call use-type!
      ;; if and only if parse-id did do it with a real type.
      ;; That is, if the cdr of the result of parse-id is not
      ;; the default type.
      (if (eq? (cdr tid) (get-default-type))
	  ;; This works only because the default type is not a legal type
	  ;; that one can use in a foreign clause.
	  (use-type! (car tid))
	  (cdr tid))))

;*---------------------------------------------------------------------*/
;*    declare-type! ...                                                */
;*---------------------------------------------------------------------*/
(define (declare-type!::type id::symbol name::bstring class::symbol)
   (trace (ast 2) "~~~ declare-type!: " id #\Newline)
   (if (not (memq class '(bigloo C _)))
       (user-error "declare-type!"
		   "Illegal type class"
		   class)
       (let ((type (bind-type! id #t)))
	  (type-name-set!   type name)
	  (type-$-set!      type ($-in-name? name))
	  (type-class-set!  type class)
	  type)))
 
;*---------------------------------------------------------------------*/
;*    declare-subtype! ...                                             */
;*    -------------------------------------------------------------    */
;*    Subtype inherit from coercion of their parents.                  */
;*---------------------------------------------------------------------*/
(define (declare-subtype!::type id::symbol name::bstring parents class::symbol)
   (trace (ast 2) "~~~ declare-subtype!: " id #\Newline)
   [assert (parents) (list? parents)]
   (let ((type    (bind-type! id #t))
	 (parents (map find-type parents)))
      (type-name-set!    type name)
      (type-$-set!       type ($-in-name? name))
      (type-class-set!   type class)
      (type-parents-set! type parents)
      type))

;*---------------------------------------------------------------------*/
;*    declare-aliastype! ...                                           */
;*---------------------------------------------------------------------*/
(define (declare-aliastype! id name class::symbol alias::type)
   (trace (ast 2) "~~~ declare-aliastype!: " id #\Newline)
   (let ((type (declare-type! id name class)))
      (type-alias-set! type alias)
      type)) 

;*---------------------------------------------------------------------*/
;*    for-each-type! ...                                               */
;*---------------------------------------------------------------------*/
(define (for-each-type! proc)
   (for-each-hash proc *Tenv*))

;*---------------------------------------------------------------------*/
;*    uninitialized-types ...                                          */
;*    -------------------------------------------------------------    */
;*    We build the list of the unitialized types.                      */
;*---------------------------------------------------------------------*/
(define (uninitialized-types)
   (let ((uninit '()))
      (for-each-type! (lambda (t)
			 (if (not (type-init? t))
			     (set! uninit (cons t uninit)))))
      uninit))

;*---------------------------------------------------------------------*/
;*    *types-already-checked?* ...                                     */
;*---------------------------------------------------------------------*/
(define *types-already-checked?* #f)

;*---------------------------------------------------------------------*/
;*    check-types ...                                                  */
;*    -------------------------------------------------------------    */
;*    We check that all types are initialized.                         */
;*    -------------------------------------------------------------    */
;*    After this function is called, `use-type' does not tolerate the  */
;*    usage of undefined types (this is implemented using the          */
;*    `types-already-checked?*' variable).                             */
;*---------------------------------------------------------------------*/
(define (check-types)
   (let ((ut (uninitialized-types)))
      (if (pair? ut)
	  (error *module*
		 "These types are used but not defined"
		 (shape ut))
	  (set! *types-already-checked?* #t))))

;*---------------------------------------------------------------------*/
;*    sub-type? ...                                                    */
;*    -------------------------------------------------------------    */
;*    Is a type a subtype of `obj'?                                    */
;*---------------------------------------------------------------------*/
(define (sub-type? minor major)
   (cond
      ((eq? minor major)
       #t)
      ((memq major (type-parents minor))
       #t)
      (else
       #f)))
