;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Module/module.scm           */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri May 31 10:29:03 1996                          */
;*    Last change :  Tue Dec 22 17:10:16 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The compilation of a Module clause                               */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module module_module

   (include "Ast/unit.sch"
	    "Engine/pass.sch")
   
   (import  engine_param
	    engine_pass
	    tools_error
	    heap_restore
	    module_main
	    module_statexp
	    module_impuse
	    module_include
	    module_with
	    module_type
	    module_foreign
	    module_eval
	    module_load
	    module_pragma
	    module_checksum
	    module_option
	    module_alibrary)
   
   (export  (class ccomp::object
	       ;; the compiler identifier
	       (id::symbol read-only)
	       ;; the compilation method when compiling the module
	       ;; which contains this clause
	       (producer::procedure read-only (default (lambda (c) '())))
	       ;; what to do when importing a module which contains
	       ;; this clause
	       (consumer::procedure read-only (default (lambda (m c) '()))) 
	       ;; what to do after the overall module compilation
	       (finalizer::procedure read-only (default (lambda () 'void)))
	       ;; we do so module checksumming in order to ensure
	       ;; modules coherence at initialization time
	       (checksummer::procedure read-only (default (lambda (m c) c))))
	   
	    (produce-module!                    <module-clause>)
	    (produce-module-clause!             <clause>)
	    (consume-module!                    name::symbol <module-clause>)
	    (consume-module-clause!             name::symbol <clause>)
	    (checksum-module                    <clause>)

	    (module-checksum-object)
	    
	    (module-initialization-id::symbol ::symbol)

	    *main*             
	    *module*
	    *module-clause*
	    *module-checksum*))

;*---------------------------------------------------------------------*/
;*    Informations about the module being compiled                     */
;*---------------------------------------------------------------------*/
(define *module*               #f)    ;; module identifer
(define *module-clause*        #f)    ;; module clause (for pretty-printing)
(define *main*                 #f)    ;; the main entry point identifier
(define *module-checksum*      #f)    ;; the checksum of the current module

;*---------------------------------------------------------------------*/
;*    produce-module! ...                                              */
;*    -------------------------------------------------------------    */
;*    The library load order is very tricky. To be short, library      */
;*    heaps have to be loaded before the module is parsed (because     */
;*    module parsing may declare some classes). Thus we make a kind    */
;*    of dummy pre-parsing to get the library clauses then we          */
;*    restore the additional heaps (that may also be requested         */
;*    by compiler options) and we start the real module processing.    */
;*---------------------------------------------------------------------*/
(define (produce-module! mclause)
   (pass-prelude "Module")
   (match-case mclause
      ((module (and (? symbol?) ?name) . ?clauses)
       (let ((clauses (if (symbol? *main*)
			  (cons `(main ,*main*) clauses)
			  clauses)))
	  (if (not (legal-module-name? name))
	      (user-error "Parse error"
			  "Illegal module name"
			  mclause)
	      (begin
		 (set! *module* name)
		 (set! *module-clause* mclause)
		 (install-clauses-compiler!)
		 (let ((clauses (produce-library-clauses clauses)))
		    ;; one library clauses have been parser
		    ;; we must restore additional heap
		    (restore-additional-heaps)
		    ;; now we resume the module parsing process
		    (for-each produce-module-clause! clauses)
		    (set! *module-checksum* (checksum-module mclause))
		    (pass-postlude (finalize-clause-compilations)))))))
      (else
       (user-error "Parse error"
		   "Illegal module declaration"
		   mclause))))

;*---------------------------------------------------------------------*/
;*    produce-library-clauses ...                                      */
;*---------------------------------------------------------------------*/
(define (produce-library-clauses clauses)
   (let ((producer (find-clause-producer 'library '(library _))))
      (let loop ((clauses clauses)
		 (res     '()))
	 (cond
	    ((null? clauses)
	     (reverse! res))
	    (else
	     (let ((clause (car clauses)))
		(match-case clause
		   ((library . ?-)
		    (producer clause)
		    (loop (cdr clauses) res))
		   (else
		    (loop (cdr clauses) (cons clause res))))))))))

;*---------------------------------------------------------------------*/
;*    finalize-clause-compilations ...                                 */
;*---------------------------------------------------------------------*/
(define (finalize-clause-compilations)
   (let loop ((cc    *clause-compilers*)
	      (units '()))
      (if (null? cc)
	  units
	  (let* ((finalizer (ccomp-finalizer (car cc)))
		 (finalres  (finalizer)))
	     (loop (cdr cc)
		   (if (pair? finalres)
		       (append finalres units)
		       units))))))

;*---------------------------------------------------------------------*/
;*    produce-module-clause! ...                                       */
;*---------------------------------------------------------------------*/
(define (produce-module-clause! clause)
   (match-case clause
      (((and (? symbol?) ?id) . ?-)
       ((find-clause-producer id clause) clause))
      (else
       (user-error "Parse error"
		   "Illegal module clause"
		   clause
		   '()))))
	     
;*---------------------------------------------------------------------*/
;*    *clause-compilers* ...                                           */
;*---------------------------------------------------------------------*/
(define *clause-compilers* '())

;*---------------------------------------------------------------------*/
;*    install-clauses-compiler ...                                     */
;*---------------------------------------------------------------------*/
(define (install-clauses-compiler!)
   ;; the order of the compilers is important. Don't change it
   (set! *clause-compilers*
	 (list (make-eval-compiler)
	       (make-main-compiler)
	       (make-load-compiler)
	       (make-use-compiler)
	       (make-import-compiler)
	       (make-from-compiler)
	       (make-static-compiler)
	       (make-export-compiler)
	       (make-include-compiler)
	       (make-with-compiler)
	       (make-foreign-compiler)
	       (make-extern-compiler)
	       (make-type-compiler)
	       (make-pragma-compiler)
	       (make-option-compiler)
	       (make-alibrary-compiler))))

;*---------------------------------------------------------------------*/
;*    find-clause-producer ...                                         */
;*---------------------------------------------------------------------*/
(define (find-clause-producer keyword clause)
   (define (unknown-clause-producer values)
      (user-error "Parse error"
		  "Unknown module clause"
		  clause
		  '()))
   (let loop ((cc *clause-compilers*))
      (cond
	 ((null? cc)
	  unknown-clause-producer)
	 ((eq? (ccomp-id (car cc)) keyword)
	  (ccomp-producer (car cc)))
	 (else
	  (loop (cdr cc))))))

;*---------------------------------------------------------------------*/
;*    module-initialization-id ...                                     */
;*---------------------------------------------------------------------*/
(define (module-initialization-id id)
   'module-initialization)

;*---------------------------------------------------------------------*/
;*    consume-module! ...                                              */
;*---------------------------------------------------------------------*/
(define (consume-module! pname mclause)
   (match-case mclause
      ((module (and (? symbol?) ?name) . ?clauses)
       (cond
	((not (legal-module-name? name))
	 (user-error "Parse error" "Illegal module name" mclause '()))
	((not (eq? pname name))
	 (user-error "Module declaration"
		     (string-append "conflict in module's name: "
				    (symbol->string name) " vs "
				    (symbol->string pname))
		     mclause
		     '()))
	(else
	 (apply append (map (lambda (c) (consume-module-clause! name c))
			    clauses)))))
      (else
       (user-error "Parse error"
		   "Illegal module declaration"
		   mclause
		   '()))))

;*---------------------------------------------------------------------*/
;*    consume-module-clause! ...                                       */
;*---------------------------------------------------------------------*/
(define (consume-module-clause! module clause)
   (match-case clause
      (((and (? symbol?) ?id) . ?values)
       ((find-clause-consumer id clause) module clause))
      (else
       (user-error "Parse error"
		   "Illegal module clause"
		   clause
		   '()))))
	     
;*---------------------------------------------------------------------*/
;*    find-clause-consumer ...                                         */
;*---------------------------------------------------------------------*/
(define (find-clause-consumer keyword clause)
   (define (unknown-clause-consumer module values)
      (user-error "Parse error"
		  "Unknown module clause"
		  clause
		  '()))
   (let loop ((cc *clause-compilers*))
      (cond
	 ((null? cc)
	  unknown-clause-consumer)
	 ((eq? (ccomp-id (car cc)) keyword)
	  (ccomp-consumer (car cc)))
	 (else
	  (loop (cdr cc))))))

;*---------------------------------------------------------------------*/
;*    legal-module-name? ...                                           */
;*---------------------------------------------------------------------*/
(define (legal-module-name? name)
   (not (memq name '(eval foreign))))

;*---------------------------------------------------------------------*/
;*    checksum-module ...                                              */
;*---------------------------------------------------------------------*/
(define (checksum-module mclause)
   (if *unsafe-version*
       0
       (match-case mclause
	  ((module (and (? symbol?) ?name) . ?clauses)
	   (cond
	      ((not (legal-module-name? name))
	       (user-error "Parse error" "Illegal module name" mclause '()))
	      (else
	       (let loop ((clauses  clauses)
			  (checksum (initial-checksum name)))
		  (if (null? clauses)
		      checksum
		      (loop (cdr clauses)
			    (checksum-module-clause (car clauses)
						    checksum)))))))
	  (else
	   (user-error "Parse error"
		       "Illegal module declaration"
		       mclause
		       '())))))

;*---------------------------------------------------------------------*/
;*    checksum-module-clause ...                                       */
;*---------------------------------------------------------------------*/
(define (checksum-module-clause clause checksum::long)
   (match-case clause
      (((and (? symbol?) ?id) . ?-)
       ((find-clause-checksummer id clause) clause checksum))
      (else
       (user-error "Parse error"
		   "Illegal module clause"
		   clause
		   '()))))
	     
;*---------------------------------------------------------------------*/
;*    find-clause-checksummer ...                                      */
;*---------------------------------------------------------------------*/
(define (find-clause-checksummer keyword clause)
   (define (unknown-clause-checksummer module values)
      (user-error "Parse error"
		  "Unknown module clause"
		  clause
		  '()))
   (let loop ((cc *clause-compilers*))
      (cond
	 ((null? cc)
	  unknown-clause-checksummer)
	 ((eq? (ccomp-id (car cc)) keyword)
	  (ccomp-checksummer (car cc)))
	 (else
	  (loop (cdr cc))))))

;*---------------------------------------------------------------------*/
;*    module-checksum-object ...                                       */
;*---------------------------------------------------------------------*/
(define (module-checksum-object)
   (pass-prelude "Module checksum object")
   (let ((checksum *module-checksum*)
	 (dest     (if (not (string? *dest*))
		       (if (and (pair? *src-files*)
				(string? (car *src-files*)))
			   (string-append (prefix (car *src-files*)) ".mco")
			   #f)
		       (if (eq? *pass* 'mco)
			   *dest*
			   (string-append (prefix *dest*) ".mco")))))
      (define (generate-mco)
	 (if (string? dest)
	     (let ((oport (open-output-file dest)))
		(if (output-port? oport)
		    (begin
		       (fprint oport checksum)
		       (close-output-port oport))
		    (error "module checksum"
			   "Can't open file for output"
			   dest)))
	     (print checksum)))
      ;; we check if the mco file already exists
      (if (and (string? dest) (file-exists? dest))
	  ;; if it exists we check if it alread
	  ;; contains the correct checksum
	  (let ((iport (open-input-file dest)))
	     (if (not (input-port? iport))
		 (user-error "module checksum"
			     "Can't open file for input"
			     dest)
		 (let ((cs (read iport)))
		    (close-input-port iport)
		    (if (not (=fx cs checksum))
			(begin
			   (delete-file dest)
			   (generate-mco))))))
	  (generate-mco)))
   ;; we are done
   (pass-postlude #t))
      

