;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Module/foreign.scm          */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Tue Jun  4 16:28:03 1996                          */
;*    Last change :  Sat Dec 19 18:52:14 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The foreign and extern clauses compilation. Foreign and extern   */
;*    clauses only differs by their syntax. They play the same role    */
;*    and have equivalent semantic. I kept foreign clauses for upward  */
;*    compatibility with Bigloo 1.8.                                   */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module module_foreign
   (include "Ast/unit.sch"
	    "Tools/trace.sch")
   (import  module_module
	    module_checksum
	    engine_param
	    ast_glo-decl
	    tools_error
	    tools_shape
	    type_type
	    type_env
	    type_tools
	    ast_var
	    ast_env
	    ast_ident
	    foreign_ctype
	    foreign_access)
   (export  (make-foreign-compiler)
	    (make-extern-compiler)))

;*---------------------------------------------------------------------*/
;*    make-foreign-compiler ...                                        */
;*---------------------------------------------------------------------*/
(define (make-foreign-compiler)
   (instantiate::ccomp (id 'foreign)
		       (producer foreign-producer)
		       (consumer (lambda (m c) (foreign-producer c)))
		       (finalizer foreign-finalizer)
		       (checksummer clause-checksummer)))

;*---------------------------------------------------------------------*/
;*    make-extern-compiler ...                                         */
;*---------------------------------------------------------------------*/
(define (make-extern-compiler)
   (instantiate::ccomp (id 'extern)
		       (producer extern-producer)
		       (consumer (lambda (m c) (extern-producer c)))
		       (checksummer clause-checksummer)))

;*---------------------------------------------------------------------*/
;*    foreign-producer ...                                             */
;*---------------------------------------------------------------------*/
(define (foreign-producer clause)
   (match-case clause
      ((?- . ?protos)
       (for-each foreign-parser protos)
       '())
      (else
       (user-error "Parse error"
		   (string-append "Illegal `foreign' clause")
		   clause
		   '()))))

;*---------------------------------------------------------------------*/
;*    extern-producer ...                                              */
;*---------------------------------------------------------------------*/
(define (extern-producer clause)
   (match-case clause
      ((?- . ?protos)
       (for-each extern-parser protos)
       '())
      (else
       (user-error "Parse error"
		   (string-append "Illegal `extern' clause")
		   clause
		   '()))))

;*---------------------------------------------------------------------*/
;*    check-c-args? ...                                                */
;*---------------------------------------------------------------------*/
(define (check-c-args? proto)
   (let loop ((proto proto))
      (cond
	 ((null? proto)
	  #t)
	 ((symbol? proto)
	  #t)
	 ((not (pair? proto))
	  #f)
	 ((not (symbol? (car proto)))
	  #f)
	 (else
	  (loop (cdr proto))))))

;*---------------------------------------------------------------------*/
;*    foreign-parser ...                                               */
;*---------------------------------------------------------------------*/
(define (foreign-parser foreign)
   (trace (ast 2) "foreign parser: " foreign #\Newline)
   (match-case foreign
      ((include ?string)
       (if (not (string? string))
	   (user-error "Parse error" "Illegal `include' clause" foreign '())
	   (if (not (member string *include-foreign*))
	       (set! *include-foreign* (cons string *include-foreign*)))))
      ((type . ?-)
       (parse-c-foreign-type foreign))
      ((export (and (? symbol?) ?bname) (and (? string?) ?cname))
       (set! *foreign-exported* (cons foreign *foreign-exported*)))
      (((or export type include) . ?-)
       ;; an error
       (user-error "Parse error" "Illegal foreign form" foreign '()))
      ((or (macro ?type ?l-name ?proto ?c-name)
	   (infix macro ?type ?l-name ?proto ?c-name))
       (if (not (and (string? c-name)
		     (symbol? type)
		     (symbol? l-name)
		     (check-c-args? proto)))
	   (user-error "Parse error" "Illegal `macro' form" foreign '())
	   (declare-global-cfun! l-name
				 c-name
				 type
				 proto
				 (eq? (car foreign) 'infix)
				 #t
				 foreign)))
      ((macro ?type ?l-name ?c-name)
       (if (not (and (string? c-name)
		     (symbol? type)
		     (symbol? l-name)))
	   (user-error "Parse error" "Illegal `macro' form" foreign '())
	   (declare-global-cvar! l-name c-name type #t foreign)))
      ((macro . ?-)
       (user-error "Parse error" "Illegal foreign form" foreign '()))
      ((?type ?l-name ?proto ?c-name)
       (if (not (and (string? c-name)
		     (symbol? type)
		     (symbol? l-name)
		     (check-c-args? proto)))
	   (user-error "Parse error" "Illegal `function' form" foreign '())
	   (declare-global-cfun! l-name
				 c-name
				 type
				 proto
				 #f
				 #f
				 foreign)))
      ((?type ?l-name ?c-name)
       (if (not (and (string? c-name)
		     (symbol? type)
		     (symbol? l-name)))
	   (user-error "Parse error" "Illegal `variable' form" foreign '())
	   (declare-global-cvar! l-name c-name type #f foreign)))
      (else
       (user-error "Parse error" "Illegal foreign form" foreign '()))))

;*---------------------------------------------------------------------*/
;*    extern-parser ...                                                */
;*---------------------------------------------------------------------*/
(define (extern-parser extern)
   (trace (ast 2) "extern parser: " extern #\Newline)
   (match-case extern
      ((type . ?-)
       ;; type clauses
       (parse-c-extern-type extern))
      (((or export include) . ?-)
       (foreign-parser extern))
      ((or (macro (and (? symbol?) ?id) ?proto (and (? string?) ?cn))
	   (infix macro (and (? symbol?) ?id) ?proto (and (? string?) ?cn)))
       ;; macro function definitions
       (let* ((pid  (parse-id id))
	      (ln   (car pid))
	      (type (type-id (cdr pid))))
	  (if (or (not (check-id pid extern))
		  (not (check-c-args? proto)))
	      (user-error "Parse error" "Illegal extern form" extern '())
	      (let ((infix? (eq? (car extern) 'infix)))
		 (declare-global-cfun! ln cn type proto infix? #t extern)))))
      ((macro (and (? symbol?) ?id) (and (? string?) ?c-name))
       ;; macro variable definitions
       (let* ((pid    (parse-id id))
	      (l-name (car pid))
	      (type   (type-id (cdr pid))))
	  (if (not (check-id pid extern))
	      (user-error "Parse error" "Illegal extern form" extern '())
	      (declare-global-cvar! l-name c-name type #t extern))))
      ((macro . ?-)
       (user-error "Parse error" "Illegal extern form" extern '()))
      (((and (? symbol?) ?id) ?proto (and (? string?) ?cn))
       ;; function definitions
       (let* ((pid  (parse-id id))
	      (ln   (car pid))
	      (type (type-id (cdr pid))))
	  (if (or (not (check-id pid extern))
		  (not (check-c-args? proto)))
	      (user-error "Parse error" "Illegal extern form" extern '())
	      (declare-global-cfun! ln cn type proto #f #f extern))))
      (((and (? symbol?) ?id) (and (? string?) ?c-name))
       ;; variable definitions
       (let* ((pid    (parse-id id))
	      (l-name (car pid))
	      (type   (type-id (cdr pid))))
	  (if (not (check-id pid extern))
	      (user-error "Parse error" "Illegal extern form" extern '())
	      (declare-global-cvar! l-name c-name type #f extern))))
      (else
       (user-error "Parse error" "Illegal extern form" extern '()))))

;*---------------------------------------------------------------------*/
;*    parse-c-foreign-type ...                                         */
;*---------------------------------------------------------------------*/
(define (parse-c-foreign-type type)
   (match-case type
      ((type (and (? symbol?) ?id) (and (? string?) ?name))
       (declare-type! id name 'C))
      ((type (and (? symbol?) ?id) ?t-exp (and (? string?) ?name))
       (if (check-c-foreign-type-exp? t-exp)
	   (let ((ctype (declare-c-type! type id t-exp name)))
	      ;; declare-c-type! can return otherthing that a type
	      ;; (for instance on type redefinition).
	      (if (type? ctype)
		  (let ((accesses (make-ctype-accesses! ctype ctype)))
		     (set! *foreign-accesses*
			   (append accesses *foreign-accesses*))
		     ;; if the declared type is an alias to a structure
		     ;; we automatically create the pending corresponding
		     ;; aliasing.
		     (if (and (calias? ctype) (cstruct? (type-alias ctype)))
			 (parse-c-foreign-type
			  `(type ,(symbol-append id '*)
				 ,(symbol-append t-exp '*)
				 ,(make-pointer-to-name ctype))))))
	      #unspecified)
	   (user-error "Parse error" "Illegal `C foreign type'" type '())))
      (else
       (user-error "Parse error" "Illegal `C foreign type'" type '()))))
 
;*---------------------------------------------------------------------*/
;*    check-c-foreign-type-exp? ...                                    */
;*---------------------------------------------------------------------*/
(define (check-c-foreign-type-exp? t-exp)   
   (match-case t-exp
      ((? symbol?)
       #t)
      (((or union struct) . ?slots)
       (let loop ((slots slots))
	  (if (null? slots)
	      #t
	      (match-case (car slots)
		 (((? symbol?) (? symbol?) (? string?))
		  (loop (cdr slots)))
		 (else
		  #f)))))
      ((pointer (? symbol?))
       #t)
      (((or struct* union*) (? symbol?))
       #t)
      ((array (? symbol?))
       #t)
      ((function (? symbol?) (and (or () (? pair?)) ?t-exp))
       (check-c-args? t-exp))
      ((enum . ?slots)
       (let loop ((slots slots))
	  (if (null? slots)
	      #t
	      (match-case (car slots)
		 (((? symbol?) (? string?))
		  (loop (cdr slots)))
		 (else
		  #f)))))
      (else
       #f)))

;*---------------------------------------------------------------------*/
;*    parse-c-extern-type ...                                          */
;*---------------------------------------------------------------------*/
(define (parse-c-extern-type type)
   (match-case type
      ((type (and (? symbol?) ?id) (and (? string?) ?name))
       (parse-c-foreign-type type))
      ((type (and (? symbol?) ?id) ?t-exp (? string?))
       (let ((foreign-type (c-extern-type->c-foreign-type t-exp)))
	  (if foreign-type
	      (parse-c-foreign-type type)
	      (user-error "Parse error" "Illegal `C extern type'" type '()))))
      (else
       (user-error "Parse error" "Illegal `C extern type'" type '()))))
 
;*---------------------------------------------------------------------*/
;*    c-extern-type->c-foreign-type ...                                */
;*---------------------------------------------------------------------*/
(define (c-extern-type->c-foreign-type t-exp)   
   (match-case t-exp
      ((? symbol?)
       t-exp)
      (((or union struct) . ?slots)
       (let loop ((slots slots))
	  (if (null? slots)
	      t-exp
	      (match-case (car slots)
		 (((and ?id (? symbol?)) (and ?name (? string?)))
		  (let* ((pid  (parse-id id))
			 (sid  (car pid))
			 (type (type-id (cdr pid))))
		     (if (not (check-id pid t-exp))
			 #f
			 (begin
			    (set-cdr! (car slots) (cons sid (cons name '())))
			    (set-car! (car slots) type)
			    (loop (cdr slots))))))
		 (else
		  #f)))))
      ((pointer (? symbol?))
       t-exp)
      (((or struct* union*) (? symbol?))
       t-exp)
      ((array (? symbol?))
       t-exp)
      ((function (? symbol?) (and (or () (? pair?)) ?t-args))
       (if (check-c-args? t-args)
	   t-exp
	   #f))
      ((enum . ?slots)
       (let loop ((slots slots))
	  (if (null? slots)
	      t-exp
	      (match-case (car slots)
		 (((? symbol?) (? string?))
		  (loop (cdr slots)))
		 (else
		  #f)))))
      (else
       #f)))

;*---------------------------------------------------------------------*/
;*    *foreign-accesses* ...                                           */
;*---------------------------------------------------------------------*/
(define *foreign-accesses* '())

;*---------------------------------------------------------------------*/
;*    *foreign-exported* ...                                           */
;*---------------------------------------------------------------------*/
(define *foreign-exported* '())

;*---------------------------------------------------------------------*/
;*    foreign-finalizer ...                                            */
;*---------------------------------------------------------------------*/
(define (foreign-finalizer)
   ;; we patch bigloo foreign exported variables
   (for-each (lambda (foreign)
                (let ((global (find-global (cadr foreign)))
		      (name   (caddr foreign)))
                   (cond
		      ((not (global? global))
                       (user-error "Foreign"
                                   "Unbound global variable"
                                   foreign
				   '()))
		      ((string? (global-name global))
		       (user-warning
			"Foreign"
			"Re-exportation of global variable (ignored)"
			foreign))
		      (else
                       (global-name-set! global name)))))
             *foreign-exported*)
   (set! *foreign-exported* '())
   ;; and we build a unit if Foreign types have generated codes
   (if (null? *foreign-accesses*)
       'void
       (let ((accesses (reverse! *foreign-accesses*)))
	  (set! *foreign-accesses* '())
	  (list (unit 'foreign 48 accesses #t)))))
	  
       
