;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Cgen/emit-cop.scm           */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Tue Jul  2 14:39:37 1996                          */
;*    Last change :  Mon Sep 21 13:52:26 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The emission of cop code.                                        */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module cgen_emit-cop
   (include "Tools/location.sch")
   (import  type_type
	    type_tools
	    type_cache
	    engine_param
	    ast_var
	    ast_node
	    ast_env
	    cgen_emit
	    cgen_cop)
   (export  (generic emit-cop::bool ::cop)
	    (reset-bdb-loc!)
	    (emit-bdb-loc ::obj)
	    (emit-newline . ::obj)
	    (untrigraph::bstring ::bstring)))

;*---------------------------------------------------------------------*/
;*    emit-cop ...                                                     */
;*    -------------------------------------------------------------    */
;*    If emit-cop emit an expression with a `;' it returns #f,         */
;*    otherwise it returns #t.                                         */
;*---------------------------------------------------------------------*/
(define-generic (emit-cop::bool cop::cop))
 
;*---------------------------------------------------------------------*/
;*    emit-cop ::clabel ...                                            */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::clabel)
   (with-access::clabel cop (used? name body loc)
      (if used?
	  (begin
	     (emit-bdb-loc loc)
	     (display name *c-port*)
	     (write-char #\: *c-port*)
	     (emit-newline)))
      (emit-cop body)))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cgoto ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cgoto)
   (with-access::cgoto cop (label loc)
      (emit-bdb-loc loc)
      (fprin *c-port* "goto " (clabel-name label) #\;)
      (emit-newline)
      #f))

;*---------------------------------------------------------------------*/
;*    emit-cop ::block ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::block)
   (with-access::block cop (body loc)
      (if (block? body)
	  (emit-cop body)
	  (begin
	     (emit-bdb-loc loc)
	     (write-char #\{ *c-port*)
	     (emit-newline)
	     (if (emit-cop body)
		 (begin
		    (write-char #\; *c-port*)
		    (emit-newline)))
	     (write-char #\} *c-port*)
	     (emit-newline)
	     #f))))

;*---------------------------------------------------------------------*/
;*    emit-cop ::creturn ...                                           */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::creturn)
   (with-access::creturn cop (value loc)
      (emit-bdb-loc loc)
      (display "return " *c-port*)
      (if (emit-cop value)
	  (begin
	     (write-char #\; *c-port*)
	     (emit-newline)))
      #f))

;*---------------------------------------------------------------------*/
;*    emit-cop ::catom ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::catom)
   (with-access::catom cop (value)
      (emit-atom-value value)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cvoid ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cvoid)
   (with-access::cvoid cop (value)
      (emit-cop value)))

;*---------------------------------------------------------------------*/
;*    emit-cop ::varc ...                                              */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::varc)
   (with-access::varc cop (variable)
      (display (variable-name variable) *c-port*)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cpragma ...                                           */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cpragma)
   (with-access::cpragma cop (args format loc)
      (emit-bdb-loc loc)
      (if (null? args)
	  (display format *c-port*)
	  (let* ((sport  (open-input-string format))
		 (args   (list->vector args))
		 (parser (regular-grammar ()
			    ((: #\$ (+ (in (#\0 #\9))))
			     (let* ((str   (the-string))
				    (len   (the-length))
				    (index (string->number
					    (substring str 1 len))))
				(emit-cop (vector-ref args (-fx index 1)))
				(ignore)))
			    ((+ (out #\$))
			     (display (the-string) *c-port*)
			     (ignore))
			    (else
			     (the-failure)))))
	     (read/rp parser sport)
	     #t))))

;*---------------------------------------------------------------------*/
;*    emit-cop ::ccast ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::ccast)
   (with-access::ccast cop (arg type loc)
      (emit-bdb-loc loc)
      (display "((" *c-port*)
      (display (type-name type) *c-port*)
      (write-char #\) *c-port*)
      (emit-cop arg)
      (write-char #\) *c-port*)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-cop ::csequence ...                                         */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::csequence)
   (with-access::csequence cop (c-exp? cops loc)
      (if c-exp?
	  (begin
	     (if (null? cops)
		 (emit-atom-value #unspecified)
		 (begin
		    (write-char #\( *c-port*)
		    (let liip ((exp cops))
		       (if (null? (cdr exp))
			   (begin
			      (emit-cop (car exp))
			      (write-char #\) *c-port*)
			      #t)
			   (begin
			      (emit-cop (car exp))
			      (if (cfail? (car exp))
				  (begin
				     (write-char #\) *c-port*)
				     #t)
				  (begin
				     (write-char #\, *c-port*)
				     (emit-newline)
				     (liip (cdr exp))))))))))
	  (let liip ((exp cops))
	     (if (null? exp)
		 #f
		 (let ((e (car exp)))
		    (if (emit-cop e)
			(begin
			   (write-char #\; *c-port*)
			   (emit-newline)))
		    (if (cfail? e)
			(liip '())
			(liip (cdr exp)))))))))

;*---------------------------------------------------------------------*/
;*    emit-cop ::nop ...                                               */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::nop)
   (write-char #\; *c-port*)
   (emit-newline)
   #f)

;*---------------------------------------------------------------------*/
;*    emit-cop ::stop ...                                              */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::stop)
   (with-access::stop cop (value loc)
      (if (emit-cop value)
	  (begin
	     (write-char #\; *c-port*)
	     (emit-newline)))
      #f))

;*---------------------------------------------------------------------*/
;*    emit-cop ::csetq ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::csetq)
   (with-access::csetq cop (var value loc)
      ;; we first emit a location for this node
      (emit-bdb-loc loc)
      (emit-cop var)
      ;; don't omit to put space sourrounding `=' otherwise
      ;; it could become an ambiguous assignement (e.g. x=-1).
      (display " = " *c-port*)
      (emit-cop value)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cif ...                                               */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cif)
   (with-access::cif cop (test true false loc)
      (emit-bdb-loc loc)
      (display "if(" *c-port*)
      (emit-cop test)
      (write-char #\) *c-port*)
      (emit-cop true)
      (display " else " *c-port*)
      (emit-cop false)))

;*---------------------------------------------------------------------*/
;*    emit-cop ::local ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::local-var)
   (with-access::local-var cop (vars loc)
      (emit-bdb-loc loc)
      (for-each (lambda (local)
		   (fprin *c-port*
			  (make-typed-declaration (local-type local)
						  (local-name local))
			  (if (and (>fx *bdb-debug* 0)
				   (eq? (type-class (local-type local))
					'bigloo))
			      (string-append " = (("
					     (make-typed-declaration
					      (local-type local)
					      "")
					     ")BUNSPEC)")
			      "")
			  #\;)
		   (emit-newline))
		vars)
      #f))

;*---------------------------------------------------------------------*/
;*    emit-cop ::bdb-block ...                                         */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::bdb-block)
   (with-access::bdb-block cop (loc body)
      (emit-bdb-loc loc)
      (fprin *c-port* "int bigloo_dummy_bdb; bigloo_dummy_bdb = 0; {")
      (emit-bdb-loc loc)
      (emit-cop body)
      (write-char #\} *c-port*)
      (emit-newline)))
   
;*---------------------------------------------------------------------*/
;*    emit-cop ::cfuncall ...                                          */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cfuncall)
   (labels ((emit-extra-light-cfuncall (cop)
               (let ((actuals (cfuncall-args cop)))
		  (emit-cop (cfuncall-fun cop))
		  (write-char #\( *c-port*)
		  (let loop ((actuals actuals))
		     ;; actuals are never empty because their are always
		     ;; the EOA.
		     (if (null? (cddr actuals))
			 (begin
			    (emit-cop (car actuals))
			    (write-char #\) *c-port*)
			    #t)
			 (begin
			    (emit-cop (car actuals))
			    (display ", " *c-port*)
			    (loop (cdr actuals)))))))
	    (emit-light-cfuncall (cop)
               (let ((actuals (cfuncall-args cop)))
		  (display "PROCEDURE_L_ENTRY(" *c-port*)
		  (emit-cop (cfuncall-fun cop))
		  (display ")(" *c-port*)
		  (let loop ((actuals actuals))
		     ;; actuals are never empty because their are always
		     ;; the function and EOA.
		     (if (null? (cddr actuals))
			 (begin
			    (emit-cop (car actuals))
			    (display ")" *c-port*)
			    #t)
			 (begin
			    (emit-cop (car actuals))
			    (display ", " *c-port*)
			    (loop (cdr actuals)))))))
	    (emit-regular-cfuncall/eoa (cop)
	       (let ((actuals (cfuncall-args cop)))
		  (display "PROCEDURE_ENTRY(" *c-port*)
		  (emit-cop (cfuncall-fun cop))
		  (display ")(" *c-port*)
		  (let loop ((actuals actuals))
		     ;; actuals are never empty because their are always
		     ;; the function and EOA.
		     (if (null? (cdr actuals))
			 (begin
			    (emit-cop (car actuals))
			    (display ")" *c-port*)
			    #t)
			 (begin
			    (emit-cop (car actuals))
			    (display ", " *c-port*)
			    (loop (cdr actuals)))))))
	    (emit-regular-cfuncall/oeoa (cop)
	       (let ((actuals (cfuncall-args cop)))
		  (display "PROCEDURE_ENTRY(" *c-port*)
		  (emit-cop (cfuncall-fun cop))
		  (display ")(" *c-port*)
		  (let loop ((actuals actuals))
		     ;; actuals are never empty because their are always
		     ;; the function and EOA.
		     (if (null? (cddr actuals))
			 (begin
			    (emit-cop (car actuals))
			    (display ")" *c-port*)
			    #t)
			 (begin
			    (emit-cop (car actuals))
			    (display ", " *c-port*)
			    (loop (cdr actuals)))))))
	    (emit-stdc-regular-cfuncall (cop)
	       (begin
		  (display "(VA_PROCUDUREP( " *c-port*)
		  (emit-cop (cfuncall-fun cop))
		  (display " ) ? " *c-port*)
		  (emit-regular-cfuncall/eoa cop)
		  (display " : " *c-port*)
		  (emit-regular-cfuncall/oeoa cop)
		  (display " )" *c-port*)
		  #t)))
      (emit-bdb-loc (cop-loc cop))
      (case (cfuncall-strength cop)
	 ((elight)
	  (emit-extra-light-cfuncall cop))
	 ((light)
	  (emit-light-cfuncall cop))
	 (else
	  (if *stdc*
	      (emit-stdc-regular-cfuncall cop)
	      (emit-regular-cfuncall/eoa cop))))))

;*---------------------------------------------------------------------*/
;*    emit-cop ::capply ...                                            */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::capply)
   (with-access::capply cop (fun arg loc)
      (emit-bdb-loc loc)
      (display "apply(" *c-port*)
      (emit-cop fun)
      (display ", " *c-port*)
      (emit-cop arg)
      (write-char #\) *c-port*)
      #t))
	     
;*---------------------------------------------------------------------*/
;*    emit-cop ::capp ...                                              */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::capp)
   (define (emit-infix-capp)
      (let ((actuals (capp-args cop)))
	 (write-char #\( *c-port*)
	 (emit-cop (car actuals))
	 (emit-cop (capp-fun cop))
	 (emit-cop (cadr actuals))
	 (write-char #\) *c-port*)
	 #t))
   (define (emit-prefix-capp)
      (let ((actuals (capp-args cop)))
	 (emit-cop (capp-fun cop))
	 (write-char #\( *c-port*)
	 (if (null? actuals)
	     (begin
		(write-char #\) *c-port*)
		#t)
	     (let loop ((actuals actuals))
		(if (null? (cdr actuals))
		    (begin
		       (emit-cop (car actuals))
		       (write-char #\) *c-port*)
		       #t)
		    (begin
		       (emit-cop (car actuals))
		       (display ", " *c-port*)
		       (loop (cdr actuals))))))))
   (let ((fun (varc-variable (capp-fun cop)))
	 (loc (capp-loc cop)))
      (emit-bdb-loc loc)
      (if (and (cfun? (global-value fun)) (cfun-infix? (global-value fun)))
	  (emit-infix-capp)
	  (emit-prefix-capp))))

;*---------------------------------------------------------------------*/
;*    *bfalse*                                                         */
;*    -------------------------------------------------------------    */
;*    A local cache for the C false macro.                             */
;*---------------------------------------------------------------------*/
(define *bfalse* #f)

;*---------------------------------------------------------------------*/
;*    emit-cop ::cfail ...                                             */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cfail)
   (with-access::cfail cop (proc msg obj loc)
      (emit-bdb-loc loc)
      (if (not *bfalse*)
	  (set! *bfalse* (find-global 'bfalse 'foreign)))
      (cond
	 ((and (varc? proc) (eq? (varc-variable proc) *bfalse*)
	       (varc? msg) (eq? (varc-variable msg) *bfalse*)
	       (varc? obj) (eq? (varc-variable obj) *bfalse*))
	  (display "exit( -1 );" *c-port*))
	 ((<=fx *bdb-debug* 0)
	  (display "FAILURE(" *c-port*)
	  (emit-cop proc)
	  (write-char #\, *c-port*)
	  (emit-cop msg)
	  (write-char #\, *c-port*)
	  (emit-cop obj)
	  (display ");" *c-port*))
	 (else
	  (display "the_failure(" *c-port*)
	  (emit-cop proc)
	  (write-char #\, *c-port*)
	  (emit-cop msg)
	  (write-char #\, *c-port*)
	  (emit-cop obj)
	  (display "), exit( -1 );" *c-port*)))
      #f))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cswitch ...                                           */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cswitch)
   (with-access::cswitch cop (test clauses loc)
      (emit-bdb-loc loc)
      (display "switch " *c-port*)
      (write-char #\( *c-port*)
      (emit-cop test)
      (write-char #\) *c-port*)
      (write-char #\{ *c-port*)
      (emit-newline)
      (let loop ((clauses clauses))
	 (let ((clause (car clauses)))
	    (if (eq? (car clause) 'else)
		(begin
		   (display "default: " *c-port*)
		   (emit-newline)
		   (if (emit-cop (cdr clause))
		       (begin
			  (write-char #\; *c-port*)
			  (emit-newline)))
		   (write-char #\} *c-port*)
		   (emit-newline)
		   #f)
		(begin
		   (for-each (lambda (t)
				(display "case " *c-port*)
				(emit-atom-value t)
				(display " : " *c-port*)
				(emit-newline))
			     (car clause))
		   (if (emit-cop (cdr clause))
		       (begin
			  (write-char #\; *c-port*)
			  (emit-newline)))
		   (display "break;" *c-port*)
		   (emit-newline)
		   (loop (cdr clauses))))))))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cmake-box ...                                         */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cmake-box)
   (with-access::cmake-box cop (value loc)
      (emit-bdb-loc loc)
      (display "MAKE_CELL(" *c-port*)
      (emit-cop value)
      (write-char #\) *c-port*)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cbox-ref ...                                          */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cbox-ref)
   (with-access::cbox-ref cop (var loc)
      (emit-bdb-loc loc)
      (display "CELL_REF(" *c-port*)
      (emit-cop var)
      (write-char #\) *c-port*)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cbox-set! ...                                         */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cbox-set!)
   (with-access::cbox-set! cop (var value loc)
      (emit-bdb-loc loc)
      (display "CELL_SET(" *c-port*)
      (emit-cop var)
      (display ", " *c-port*)
      (emit-cop value)
      (write-char #\) *c-port*)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cset-ex-it ...                                        */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cset-ex-it)
   (with-access::cset-ex-it cop (exit jump-value body loc)
      (emit-bdb-loc loc)
      (display "if( SET_EXIT(" *c-port*)
      (emit-cop exit)
      (display ") ) {" *c-port*)
      (display "RESTORE_TRACE(); " *c-port*)
      (emit-cop jump-value)
      (emit-bdb-loc loc)
      (display "} else {" *c-port*)
      (emit-cop body)
      (emit-bdb-loc loc)
      (write-char #\} *c-port*)
      (emit-newline)
      #f))

;*---------------------------------------------------------------------*/
;*    emit-cop ::cjump-ex-it ...                                       */
;*---------------------------------------------------------------------*/
(define-method (emit-cop cop::cjump-ex-it)
   (with-access::cjump-ex-it cop (exit value loc)
      (emit-bdb-loc loc)
      (display "JUMP_EXIT( " *c-port*)
      (emit-cop exit)
      (write-char #\, *c-port*)
      (emit-cop value)
      (write-char #\) *c-port*)
      #t))

;*---------------------------------------------------------------------*/
;*    emit-atom-value ...                                              */
;*---------------------------------------------------------------------*/
(define (emit-atom-value value)
   (cond
      ((boolean? value)
       (display "((" *c-port*)
       (display (string-sans-$ (type-name *bool*)) *c-port*)
       (display #\) *c-port*)
       (display (if value 1 0) *c-port*)
       (display #\) *c-port*))
      ((null? value)
       (display "BNIL" *c-port*))
      ((char? value)
       (display "((" *c-port*)
       (display (string-sans-$ (type-name *char*)) *c-port*)
       (display ")" *c-port*)
       (if (=fx (char->integer value) 0)
	   (display "'\\000'" *c-port*)
	   (begin
	      (write-char #\' *c-port*)
	      (if (=fx (char->integer value) 39)
		  (display "\\''" *c-port*)
		  (begin
		     (case value
			((#\return)
			 (write-char #\\ *c-port*)
			 (write-char #\r *c-port*))
			((#\tab)
			 (write-char #\\ *c-port*)
			 (write-char #\t *c-port*))
			((#\newline)
			 (write-char #\\ *c-port*)
			 (write-char #\n *c-port*))
			((#\\)
			 (write-char #\\ *c-port*)
			 (write-char #\\ *c-port*))
			(else
			 (write-char value *c-port*)))
		     (write-char #\' *c-port*)))))
       (write-char #\) *c-port*))
      ((eq? value #unspecified)
       (display "BUNSPEC" *c-port*))
      ((cnst? value)
       (display "BCNST(" *c-port*)
       (display (cnst->integer value) *c-port*)
       (display #\) *c-port*))
      ((string? value)
       (display #\" *c-port*)
       (display (untrigraph (string-for-read value)) *c-port*)
       (display #\" *c-port*))
      ((fixnum? value)
       (display "((" *c-port*)
       (display (string-sans-$ (type-name *long*)) *c-port*)
       (display ")" *c-port*)
       (display value *c-port*)
       (display ")" *c-port*))
      ((flonum? value)
       (display "((" *c-port*)
       (display (string-sans-$ (type-name *real*)) *c-port*)
       (display ")" *c-port*)
       (display value *c-port*)
       (display ")" *c-port*))
      (else
       (display value *c-port*))))

;*---------------------------------------------------------------------*/
;*    untrigraph ...                                                   */
;*    -------------------------------------------------------------    */
;*    We remove ?? and replace it by \077\077\--- (the octal ascii     */
;*    code of ?) in order to avoir C trigraph confusions.              */
;*---------------------------------------------------------------------*/
(define (untrigraph from)
   (let* ((len   (string-length from))
	  (len-3 (-fx len 3)))
      ;; first we count how many collision we have
      (let ((nb-col (let loop ((i      0)
			       (nb-col 0))
		       (cond
			  ((>fx i len-3)
			   nb-col)
			  ((not (char=? (string-ref from i) #\?))
			   (loop (+fx i 1) nb-col))
			  ((not (char=? (string-ref from (+fx i 1)) #\?))
			   (loop (+fx i 2) nb-col))
			  ;; yes, we have one
			  (else
			   (loop (+fx i 3) (+fx nb-col 1)))))))
	 (if (=fx nb-col 0)
	     ;; there is no trigraph clashes
	     from
	     ;; there is some, we allocate a new string. Each trigraph
	     ;; require 4 times its size.
	     (let ((res   (make-string (+fx len (*fx 3 (*fx nb-col 3)))))
		   (len-1 (-fx len 1)))
		(let loop ((r 0)
			   (w 0))
		   (cond
		      ((=fx r len)
		       res)
		      ((or (not (char=? (string-ref from r) #\?))
			   (>fx r len-3))
		       (string-set! res w (string-ref from r))
		       (loop (+fx r 1) (+fx w 1)))
		      ((not (char=? (string-ref from (+fx r 1)) #\?))
		       (string-set! res w #\?)
		       (string-set! res (+fx w 1) (string-ref from (+fx r 1)))
		       (loop (+fx r 2) (+fx w 2)))
		      (else
		       ;; this is a trigraph
		       (string-set! res w #\\)
		       (string-set! res (+fx w 1) #\0)
		       (string-set! res (+fx w 2) #\7)
		       (string-set! res (+fx w 3) #\7)
		       (string-set! res (+fx w 4) #\\)
		       (string-set! res (+fx w 5) #\0)
		       (string-set! res (+fx w 6) #\7)
		       (string-set! res (+fx w 7) #\7)
		       (string-set! res (+fx w 8) #\\)
		       (let ((code (integer->string
				    (char->integer (string-ref from (+fx r 2)))
				    8)))
			  (if (=fx (string-length code) 4)
			      (begin
				 (string-set! res (+fx w 9) #\0)
				 (string-set! res (+fx w 10)
					      (string-ref code 2))
				 (string-set! res (+fx w 11)
					      (string-ref code 3)))
			      (begin
				 (string-set! res (+fx w 9)
					      (string-ref code 2))
				 (string-set! res (+fx w 10)
					      (string-ref code 3))
				 (string-set! res (+fx w 11)
					      (string-ref code 4)))))
		       (loop (+fx r 3) (+fx w 12))))))))))

;*---------------------------------------------------------------------*/
;*    *bdb-loc* ...                                                    */
;*    -------------------------------------------------------------    */
;*    The current bdb source location information.                     */
;*---------------------------------------------------------------------*/
(define *bdb-loc* #unspecified)

;*---------------------------------------------------------------------*/
;*    reset-bdb-loc! ...                                               */
;*---------------------------------------------------------------------*/
(define (reset-bdb-loc!)
   (set! *bdb-loc* #unspecified))

;*---------------------------------------------------------------------*/
;*    emit-bdb-loc ...                                                 */
;*    -------------------------------------------------------------    */
;*    This function emits a bdb location information (that is a        */
;*    C # line information.) This function emits this information      */
;*    only if we are dumping the C code for a different Scheme source  */
;*    code line that the previous line dump.                           */
;*---------------------------------------------------------------------*/
(define (emit-bdb-loc cur-loc)
   (cond
      ((<=fx *bdb-debug* 0)
       #unspecified)
      ((not (location? cur-loc))
       (if (location? *bdb-loc*)
	   (emit-bdb-loc *bdb-loc*)))
      (else
       (let ((cur-fname  (location-fname cur-loc))
	     (cur-line   (location-lnum cur-loc))
	     (prev-loc   *bdb-loc*))
	  (if (or (not (location? prev-loc))
		  (let ((prev-fname (location-fname prev-loc)))
		     (or (eq? cur-fname prev-fname)
			 (string=? cur-fname prev-fname))))
	      ;; we have to emit a # line information only if the code
	      ;; we are emitting belongs to the current function (that is,
	      ;; we do not have to emit line information for inlined
	      ;; bodies).
	      (begin
		 (fprint *c-port* #"\n# " cur-line " \"" cur-fname #\")
		 (set! *bdb-loc* cur-loc)))))))

;*---------------------------------------------------------------------*/
;*    emit-newline ...                                                 */
;*---------------------------------------------------------------------*/
(define (emit-newline . mandatory)
   (cond
      ((<=fx *bdb-debug* 0)
       (newline *c-port*))
      ((pair? mandatory)
       (if (location? *bdb-loc*)
	   (emit-bdb-loc *bdb-loc*)
	   (newline *c-port*)))
      (else
       (write-char #\space *c-port*))))

;*---------------------------------------------------------------------*/
;*    fprin ...                                                        */
;*---------------------------------------------------------------------*/
(define (fprin port . values)
   (for-each (lambda (value) (display value port)) values))
