;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Cfa/set.scm                 */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Feb 23 17:02:23 1995                          */
;*    Last change :  Sun Nov 29 09:47:40 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The `set' package.                                               */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module cfa_set
   (import  type_type
	    ast_var
	    ast_node
	    tools_shape
	    tools_error
	    (node-key cfa_approx)
	    (node-key-set! cfa_approx))
   (export  (declare-set! ::vector)
	    (make-set!    <set>)
	    (set?::bool   <obj>)
	    (set-extend!  <set>    <obj>)
	    (set-union!   <set>  . <set>*)
	    (set-for-each ::procedure <set>)
	    (set-length   <set>)
	    (set->list    <set>)
	    (set->vector  <set>)))

;*---------------------------------------------------------------------*/
;*    The `set' and `meta-set' structures                              */
;*---------------------------------------------------------------------*/
(define-struct meta-set  table compacted-size)
(define-struct large-set the-set meta)
(define-struct small-set the-set meta)

;*---------------------------------------------------------------------*/
;*    max sizes                                                        */
;*---------------------------------------------------------------------*/
(define max-small-set-size (let ((ptr-align (pragma::long "PTR_ALIGNMENT")))
			      (-fx (bit-lsh 1 (+fx ptr-align 3)) ptr-align)))
(define max-large-size     10000)

;*---------------------------------------------------------------------*/
;*    declare-set! ...                                                 */
;*---------------------------------------------------------------------*/
(define (declare-set! table)
   (let* ((cardinal  (vector-length table))
	  (quotient  (quotient  cardinal 8))
	  (remainder (remainder cardinal 8))
	  (size      (cond
			((<fx cardinal max-small-set-size)
			 0)
			((=fx remainder 0)
			 (+fx quotient 1))
			(else
			 (+fx quotient 2)))))
      (cond
	 ((>=fx cardinal max-large-size)
	  (internal-error "define-set!"
			  "Too many element in set"
			  (shape table)))
	 ((=fx 0 size)
	  (let loop ((i   0)
		     (pow 1))
	     (cond
		((=fx i cardinal)
		 (meta-set table size))
		(else
		 (node-key-set! (vector-ref table i) pow)
		 (loop (+fx i 1)
		       (*fx 2 pow))))))
	 (else
	  (let loop ((i         0)
		     (quotient  0)
		     (mask      1))
	     (cond
		((=fx i cardinal)
		 (meta-set table size))
		((=fx mask 256)
		 (loop i (+fx quotient 1) 1))
		(else
		 (node-key-set! (vector-ref table i) (cons quotient mask))
		 (loop (+fx i 1) quotient (*fx mask 2)))))))))

;*---------------------------------------------------------------------*/
;*    make-set! ...                                                    */
;*---------------------------------------------------------------------*/
(define (make-set! meta-set)
   (cond
      ((not (meta-set? meta-set))
       (internal-error "make-set" "Not a meta-set" (shape meta-set)))
      ((=fx (meta-set-compacted-size meta-set) 0)
       (small-set 0 meta-set))
      (else
       (large-set (make-string (meta-set-compacted-size meta-set) #a000)
		  meta-set))))

;*---------------------------------------------------------------------*/
;*    set? ...                                                         */
;*---------------------------------------------------------------------*/
(define (set? obj)
   (or (small-set? obj) (large-set? obj)))

;*---------------------------------------------------------------------*/
;*    set-extend! ...                                                  */
;*---------------------------------------------------------------------*/
(define (set-extend! set obj)
   (define (small-set-extend!)
      (small-set-the-set-set! set
			      (bit-or (small-set-the-set set)
				      (node-key obj)))
      #unspecified)
   (define (large-set-extend!)
      (let* ((key       (node-key obj))
	     (the-set   (large-set-the-set set))
	     (quotient  (car key))
	     (mask      (cdr key)))
	 (string-set! the-set
		      quotient
		      (char-or (integer->char mask)
			       (string-ref the-set quotient)))
	 #unspecified))
   (cond
      ((small-set? set)
       (small-set-extend!))
      ((large-set? set)
       (large-set-extend!))
      (else
       (internal-error "set-extend!" "Not a set" (shape set)))))
       
;*---------------------------------------------------------------------*/
;*    set-member? ...                                                  */
;*---------------------------------------------------------------------*/
(define (set-member? set obj)
   (define (small-set-member?)
      (>fx (bit-and (small-set-the-set set) (node-key obj)) 0))
   (define (large-set-member?)
      (let* ((key       (node-key obj))
	     (the-set   (large-set-the-set set))
	     (quotient  (car key))
	     (mask      (cdr key)))
	 (>fx (bit-and mask (char->integer (string-ref the-set quotient)))
	      0)))
   (cond
      ((small-set? set)
       (small-set-member?))
      ((large-set? set)
       (large-set-member?))
      (else
       (internal-error "set-member?" "Not a set" (shape set)))))
       
;*---------------------------------------------------------------------*/
;*    set-union-2! ...                                                 */
;*    -------------------------------------------------------------    */
;*    This function returns #t if nothing as been added. Otherwise,    */
;*    it returns #f.                                                   */
;*---------------------------------------------------------------------*/
(define (set-union-2! dst src)
   (define (small-set-union!)
      (let ((old (small-set-the-set dst)))
	 (small-set-the-set-set! dst
				 (bit-or (small-set-the-set dst)
					 (small-set-the-set src)))
	 (not (eq? old (small-set-the-set dst)))))
   (define (large-set-union!)
      (let ((the-dst (large-set-the-set dst))
	    (the-src (large-set-the-set src)))
	 (let loop ((i   (-fx (meta-set-compacted-size (large-set-meta dst))
			      1))
		    (res #f))
	    (if (=fx i -1)
		res
		(let ((old (string-ref the-dst i))
		      (new (char-or (string-ref the-dst i)
				    (string-ref the-src i))))
		   (if (char=? new old)
		       (loop (-fx i 1) res)
		       (begin
			  (string-set! the-dst i new)
			  (loop (-fx i 1) #t))))))))
   (cond
      ((small-set? dst)
       (if (not (small-set? src))
	   (internal-error "set-union!" "Incompatible sets" (shape src))
	   (small-set-union!)))
      ((large-set? dst)
       (if (not (large-set? src))
	   (internal-error "set-union!" "Incompatible sets" (shape src))
	   (if (not (=fx (string-length (large-set-the-set dst))
			 (string-length (large-set-the-set src))))
	       (internal-error "set-union!" "Incompatible sets" (shape src))
	       (large-set-union!))))
      (else
       (internal-error "set-union!" "Not a set" (shape dst)))))

;*---------------------------------------------------------------------*/
;*    set-union! ...                                                   */
;*    -------------------------------------------------------------    */
;*    This function returns #t if nothing as been added. Otherwise,    */
;*    it returns #f.                                                   */
;*---------------------------------------------------------------------*/
(define (set-union! dst . src*)
   (cond
      ((null? src*)
       dst)
      ((null? (cdr src*))
       (set-union-2! dst (car src*)))
      (else
       (let loop ((src* src*)
		  (res  #f))
	  (if (null? src*)
	      res
	      (loop (cdr src*)
		    (or (set-union-2! dst (car src*)) res)))))))
		    
;*---------------------------------------------------------------------*/
;*    set-for-each ...                                                 */
;*    -------------------------------------------------------------    */
;*    This function should be improve to avoid the invocation          */
;*    of `set-member?'. The call to this function should be            */
;*    inlined.                                                         */
;*---------------------------------------------------------------------*/
(define (old-set-for-each proc set)
   [assert check (set) (set? set)]
   (let* ((meta (cond
		   ((small-set? set)
		    (small-set-meta set))
		   ((large-set? set)
		    (large-set-meta set))
		   (else
		    (internal-error "set-for-each"
				    "argument not a set"
				    (shape set)))))
	  (table (meta-set-table meta)))
      (let loop ((i (-fx (vector-length table) 1)))
	 (cond
	    ((=fx i -1)
	     #unspecified)
	    ((set-member? set (vector-ref table i))
	     (proc (vector-ref table i))
	     (loop (-fx i 1)))
	    (else
	     (loop (-fx i 1)))))))

;*---------------------------------------------------------------------*/
;*    set-for-each ...                                                 */
;*---------------------------------------------------------------------*/
(define (set-for-each proc set)
   (define (small-set-member? set obj)
      (>fx (bit-and (small-set-the-set set) (node-key obj)) 0))
   (define (small-set-for-each)
      (if (=fx (small-set-the-set set) 0)
	  #unspecified
	  (let* ((meta  (small-set-meta set))
		 (table (meta-set-table meta)))
	     [assert (table)
		     (<fx (small-set-the-set set) (bit-lsh 1 (vector-length table)))]
	     (let loop ((i (-fx (vector-length table) 1)))
		(cond
		   ((=fx i -1)
		    #unspecified)
		   ((>fx (bit-and (small-set-the-set set) (bit-lsh 1 i)) 0)
		    [assert (i) (set-member? set (vector-ref table i))]
		    [assert (i)
			    (let* ((obj (vector-ref table i))
				   (key (node-key obj)))
			       (=fx key (bit-lsh 1 i)))]
		    (proc (vector-ref table i))
		    (loop (-fx i 1)))
		   (else
		    [assert (i) (not (set-member? set (vector-ref table i)))]
		    [assert (i)
			    (let* ((obj (vector-ref table i))
				   (key (node-key obj)))
			       (=fx key (bit-lsh 1 i)))]
		    (loop (-fx i 1))))))))
   (define (large-set-for-each)
      (let* ((meta  (large-set-meta set))
	     (table (meta-set-table meta)))
	 (let loop ((i (-fx (vector-length table) 1)))
	    (cond
	       ((=fx i -1)
		#unspecified)
	       ((set-member? set (vector-ref table i))
		(proc (vector-ref table i))
		(loop (-fx i 1)))
	       (else
		(loop (-fx i 1)))))))
   (cond
      ((small-set? set)
       (small-set-for-each))
      ((large-set? set)
       (large-set-for-each))
      (else
       (internal-error "set-for-each" "Not a set" (shape set)))))

;*---------------------------------------------------------------------*/
;*    set-length ...                                                   */
;*---------------------------------------------------------------------*/
(define (set-length set)
   (define (small-set-length)
      (let loop ((the-set (small-set-the-set set))
		 (num     0))
	 (cond
	    ((=fx the-set 0)
	     num)
	    (else
	     (loop (bit-rsh the-set 1) (+fx num (bit-and the-set 1)))))))
   (define (large-set-length)
      (let* ((the-set (large-set-the-set set))
	     (the-len (string-length the-set)))
	 (let loop ((offset 0)
		    (num    0))
	    (if (=fx offset the-len)
		num
		(let liip ((char (char->integer (string-ref the-set offset)))
			   (num  num))
		   (cond
		      ((=fx char 0)
		       (loop (+fx 1 offset) num))
		      (else
		       (liip (bit-rsh char 1)
			     (+fx num (bit-and char 1))))))))))
   (cond
      ((small-set? set)
       (small-set-length))
      ((large-set? set)
       (large-set-length))
      (else
       (internal-error "set-length" "Not a set" (shape set)))))
    
;*---------------------------------------------------------------------*/
;*    set->list ...                                                    */
;*---------------------------------------------------------------------*/
(define (set->list set)
   (let ((meta (cond
		  ((small-set? set)
		   (small-set-meta set))
		  ((large-set? set)
		   (large-set-meta set))
		  (else
		   (internal-error "set-for-each"
				   "argument not a set"
				   (shape set))))))
      (let* ((table (meta-set-table meta))
	     (size  (vector-length table)))
	 (let loop ((i 0)
		    (l '()))
	    (cond
	       ((=fx i size)
		l)
	       ((set-member? set (vector-ref table i))
		(loop (+fx i 1) (cons (vector-ref table i) l)))
	       (else
		(loop (+fx i 1) l)))))))

;*---------------------------------------------------------------------*/
;*    set->vector ...                                                  */
;*---------------------------------------------------------------------*/
(define (set->vector set)
   (list->vector (set->list set)))
		
		
