;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Ast/venv.scm                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun Dec 25 11:32:49 1994                          */
;*    Last change :  Sat Dec 19 14:29:59 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The global environment manipulation                              */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module ast_env
   (import  tools_shape
	    engine_param
	    tools_error
	    type_type
	    type_cache
	    type_env
	    ast_var
	    ast_node
	    ast_hrtype
	    (*module* module_module))
   (export  (initialize-Genv!)
	    (set-Genv!              <Genv>)
	    (add-Genv!              <Genv>)
	    (get-Genv) 
	    (find-global            ::symbol . <symbol>)
	    (bind-global!::global   ::symbol ::symbol ::value ::symbol ::obj)
	    (unbind-global!         ::symbol ::symbol)
	    (for-each-global!       ::procedure)
	    (global-bucket-position ::symbol ::symbol)))

;*---------------------------------------------------------------------*/
;*    *Genv* ...                                                       */
;*    -------------------------------------------------------------    */
;*    The Global environment (for global variable definitions).        */
;*---------------------------------------------------------------------*/
(define *Genv* 'the-global-environment)

;*---------------------------------------------------------------------*/
;*    get-hash-number ...                                              */
;*---------------------------------------------------------------------*/
(define (get-hash-number o)
   (string->0..2^x-1 (symbol->string o) 12))

;*---------------------------------------------------------------------*/
;*    set-Genv! ...                                                    */
;*---------------------------------------------------------------------*/
(define (set-Genv! Genv)
   (set! *Genv* Genv)
   (struct-set! *Genv* 2 get-hash-number)
   (struct-set! *Genv* 3 car)
   (struct-set! *Genv* 5 eq?))
		 
;*---------------------------------------------------------------------*/
;*    add-Genv! ...                                                    */
;*    -------------------------------------------------------------    */
;*    When adding a new environment we have to mark that all global    */
;*    bindings are library ones.                                       */
;*---------------------------------------------------------------------*/
(define (add-Genv! Genv)
   (for-each-hash
    (lambda (bucket)
       (for-each (lambda (new)
		    (let* ((id      (global-id new))
			   (module  (global-module new))
			   (type    (global-type new))
			   (value   (global-value new))
			   (bucket  (get-hash id *Genv*))
			   (type-id (type-id type)))
		       (global-library?-set! new #t)
		       ;; we restore type result
		       (global-type-set! new (find-type type-id))
		       ;; the parameters type
		       (restore-value-types! value)
		       (cond
			  ((not (pair? bucket))
			   (put-hash! (list id new) *Genv*))
			  ((eq? module *module*)
			   (let ((new-bucket (cons new (cdr bucket))))
			      (set-cdr! bucket new-bucket)))
			  (else
			   (set-cdr! (cdr bucket) (cons new (cddr bucket)))))))
		 (cdr bucket)))
    Genv))

;*---------------------------------------------------------------------*/
;*    restore-value-types! ...                                         */
;*---------------------------------------------------------------------*/
(define-generic (restore-value-types! value::value)
   #unspecified)

;*---------------------------------------------------------------------*/
;*    restore-value-types! ::sfun ...                                  */
;*---------------------------------------------------------------------*/
(define-method (restore-value-types! value::sfun)
   (with-access::sfun value (args)
      (let loop ((args args))
	 (cond
	    ((pair? args)
	     (let ((arg (car args)))
		(cond
		   ((type? arg)
		    (set-car! args (find-type (type-id arg))))
		   ((local? arg)
		    (let ((new-type (find-type (type-id (local-type arg)))))
		       (local-type-set! arg new-type)))
		   (else
		    (error "restore-value-types(sfun)"
			   "Illegal argument"
			   (shape arg))))
		(loop (cdr args))))
	    ((null? args)
	     (let ((body (sfun-body value)))
		;; we still have to restore the body types
		(if (node? body)
		    (hrtype-node! body))))
	    (else
	     (error "restore-value-types"
		    "Illegal non pair argument"
		    (shape args)))))))

;*---------------------------------------------------------------------*/
;*    restore-value-types! ::cfun ...                                  */
;*---------------------------------------------------------------------*/
(define-method (restore-value-types! value::cfun)
   (with-access::cfun value (args-type)
      (let loop ((args args-type))
	 (if (pair? args)
	     (begin
		(set-car! args (find-type (type-id (car args))))
		(loop (cdr args)))))))
   
;*---------------------------------------------------------------------*/
;*    get-Genv ...                                                     */
;*---------------------------------------------------------------------*/
(define (get-Genv)
   (struct-set! *Genv* 2 'get-hash-number)
   (struct-set! *Genv* 3 'car)
   (struct-set! *Genv* 5 'eq?)
   *Genv*)

;*---------------------------------------------------------------------*/
;*    initialize-Genv! ...                                             */
;*---------------------------------------------------------------------*/
(define (initialize-Genv!)
   (set! *Genv* (make-hash-table 4096 get-hash-number car eq? 1024)))

;*---------------------------------------------------------------------*/
;*    find-global ...                                                  */
;*---------------------------------------------------------------------*/
(define (find-global id::symbol . module)
   [assert (module) (or (null? module) (symbol? (car module)))]
   (let ((bucket (get-hash id *Genv*))
	 (module (if (null? module) '() (car module))))
      (cond
	 ((not (pair? bucket))
	  #f)
	 ((null? (cdr bucket))
	  #f)
	 ((null? module)
	  (cadr bucket))
	 (else
	  (let loop ((globals (cdr bucket)))
	     (cond
		((null? globals)
		 #f)
		((eq? (global-module (car globals)) module)
		 (car globals))
		(else
		 (loop (cdr globals)))))))))

;*---------------------------------------------------------------------*/
;*    bind-global! ...                                                 */
;*    -------------------------------------------------------------    */
;*    When binding a global, if a previous global with the same id     */
;*    has already been bound, we follow the two rules:                 */
;*       1- if module is the name of the current module, the global    */
;*          is added at the head of the list.                          */
;*       2- if module is not the name of the current module, the       */
;*          global is not added at the head of the list (practically,  */
;*          it is added in second position).                           */
;*    Moreover, because we have add a lot of confusion because of this */
;*    we always check if we are redefining a foreign function with a   */
;*    Scheme function. In such a situation, we raise a warning.        */
;*---------------------------------------------------------------------*/
(define (bind-global!::global id::symbol
			      module::symbol
			      value::value
			      import::symbol
			      src::obj)
   (let ((global (find-global id module)))
      ;; If the current module if not foreign we make the foreign check
      ;; descibed above
      (if (not (eq? module 'foreign))
	  (let ((old-foreign (find-global id 'foreign)))
	     (if (global? old-foreign)
		 (if (and (number? *warning*) (>=fx *warning* 2))
		     (user-warning id
				   "Scheme declaration overrides foreign declaration"
				   src)))))
      ;; Now we keep going we the other check.
      (if (global? global)
	  (if (not *lib-mode*)
	      (user-error id "Illegal global redefinition" src)
	      global)
	  (let ((new    (instantiate::global (module module)
					     (id id)
					     (value value)
					     (src src)
					     (import import)))
		(bucket (get-hash id *Genv*)))
	     (cond
		((not (pair? bucket))
		 (put-hash! (list id new) *Genv*))
		((eq? module *module*)
		 (let ((new-bucket (cons new (cdr bucket))))
		    (set-cdr! bucket new-bucket)))
		(else
		 (set-cdr! (cdr bucket) (cons new (cddr bucket)))))
	     new))))
 
;*---------------------------------------------------------------------*/
;*    unbind-global! ...                                               */
;*---------------------------------------------------------------------*/
(define (unbind-global! id::symbol module::symbol)
   (let ((global (find-global id module)))
      (if (not (global? global))
	  (user-error "unbind-global!" "Can't find global" `(@ ,id ,module))
	  (let ((bucket (get-hash id *Genv*)))
	     (let loop ((cur  (cdr bucket))
			(prev bucket))
		(if (eq? (car cur) global)
		    (set-cdr! prev (cdr cur))
		    (loop (cdr cur) (cdr prev))))))))
   
;*---------------------------------------------------------------------*/
;*    for-each-global! ...                                             */
;*---------------------------------------------------------------------*/
(define (for-each-global! proc::procedure)
   (for-each-hash (lambda (bucket) (for-each proc (cdr bucket)))
		  *Genv*))
   
;*---------------------------------------------------------------------*/
;*    global-bucket-position                                           */
;*---------------------------------------------------------------------*/
(define (global-bucket-position id module)
   (let ((bucket (get-hash id *Genv*)))
      (if (not (pair? bucket))
	  -1
	  (let loop ((globals (cdr bucket))
		     (pos     0))
	     (cond
		((null? globals)
		 -1)
		((eq? (global-module (car globals)) module)
		 pos)
		(else
		 (loop (cdr globals)
		       (+fx pos 1))))))))
   
