;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bmacs/ude/ude-html.el                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Jan 21 06:00:40 1999                          */
;*    Last change :  Fri Feb 12 08:30:28 1999 (serrano)                */
;*    -------------------------------------------------------------    */
;*    This file implements emacs->html conversion. Mostly it is used   */
;*    to produce HTML version of fontified source codes.               */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(provide 'ude-html)
(require 'ude-autoload)

;*---------------------------------------------------------------------*/
;*    ude->html ...                                                    */
;*---------------------------------------------------------------------*/
(defun ude->html ()
  (interactive)
  (let ((name (concat (ude-string-prefix (buffer-name)) ".html")))
    (ude-html-source-buffer->html name)
    (switch-to-buffer-other-frame (get-buffer name))))

;*---------------------------------------------------------------------*/
;*    ude->html-file ...                                               */
;*---------------------------------------------------------------------*/
(defun ude->html-file ()
  (interactive)
  (let ((old-buf (current-buffer)))
    (let ((name (concat (ude-string-prefix (buffer-name)) ".html")))
      (ude-html-source-buffer->html name)
      (let ((buffer (get-buffer name)))
	(set-buffer buffer)
	(write-file name)))
    (set-buffer old-buf)))

;*---------------------------------------------------------------------*/
;*    ude-html-source-buffer->html ...                                 */
;*    -------------------------------------------------------------    */
;*    Copy the content of the current buffer into a new buffer         */
;*    named to-buffer. That buffer is a fontified HTML buffer.         */
;*---------------------------------------------------------------------*/
(defun ude-html-source-buffer->html (to-name)
  (let ((to-buffer (get-buffer-create to-name))
	(from-buffer (current-buffer)))
    ;; first make some cleaning into the destination buffer
    (set-buffer to-buffer)
    (erase-buffer)
    (insert "<pre>")
    ;; then start parsing the input file
    (set-buffer from-buffer)
    (save-excursion
      (let ((extents (extent-list))
	    (read    (point-min)))
	(while (consp extents)
	  (let ((extent (car extents)))
	    (let ((start (extent-start-position extent))
		  (end   (extent-end-position extent)))
	      (if (and (>= start read) (extent-face extent))
		  (progn
		    (ude-html-copy-region from-buffer to-buffer read start)
		    (let* ((tags   (ude-html-extent->html-tag extent "" ""))
			   (tstart (car tags))
			   (tstop  (cdr tags)))
		      (setq extents (cdr extents))
		      (while (and (consp extents)
				  (eq (extent-start-position (car extents))
				      start))
			(setq extents (cdr extents)))
		      (if (consp extents)
			  (let ((e (car extents)))
			    (if (< (extent-start-position (car extents)) end)
				(setq end
				      (extent-start-position (car extents))))))
		      (ude-html-insert to-buffer tstart)
		      (ude-html-copy-region from-buffer to-buffer start end)
		      (ude-html-insert to-buffer tstop)
		      (setq read end)))
		(setq extents (cdr extents))))))
	(ude-html-copy-region from-buffer to-buffer read (point-max)))
      (set-buffer to-buffer)
      (goto-char (point-max))
      (insert "</pre>")
      (untabify (point-min) (point-max)))))

;*---------------------------------------------------------------------*/
;*    ude-html-copy-region ...                                         */
;*---------------------------------------------------------------------*/
(defun ude-html-copy-region (from-buffer to-buffer start end)
  (if (> end start)
      (let ((string (buffer-substring start end from-buffer)))
	(ude-html-insert to-buffer string))))

;*---------------------------------------------------------------------*/
;*    ude-html-insert ...                                              */
;*---------------------------------------------------------------------*/
(defun ude-html-insert (buffer string)
  (let ((buf (current-buffer)))
    (set-buffer buffer)
    (goto-char (point-max))
    (insert string)
    (set-buffer buf)))

;*---------------------------------------------------------------------*/
;*    ude-html-extent->html-tag ...                                    */
;*---------------------------------------------------------------------*/
(defun ude-html-extent->html-tag (extent start stop)
  (let ((face (extent-face extent)))
    (let ((fg-name (face-foreground-name face))
	  (rgb     (color-rgb-components (face-foreground face)))
	  (und     (face-underline-p face)))
      (let ((color (if (stringp fg-name)
		       (let* ((r (/ (car rgb) 256))
			      (g (/ (car (cdr rgb)) 256))
			      (b (/ (car (cdr (cdr rgb))) 256))
			      (num (concat (ude-html-hex->string r)
					   (ude-html-hex->string g)
					   (ude-html-hex->string b))))
			 (format "<font color=\"#%s\"><b>" num))
		     nil)))
	(if (stringp color)
	    (progn
	      (setq start (concat start color))
	      (setq stop (concat "</b></font>" stop))))
	(if und
	    (progn
	      (setq start (concat start "<underline>"))
	      (setq stop (concat "</underline>" stop))))
	(cons start stop)))))

;*---------------------------------------------------------------------*/
;*    ude-html-hex->string ...                                         */
;*---------------------------------------------------------------------*/
(defun ude-html-hex->string (num)
  (cond
   ((> num 16)
    (format "%x" num))
   ((> num 0)
    (format "0%x" num))
   (t
    "00")))

;*---------------------------------------------------------------------*/
;*    ude-html-fontify-html-buffer ...                                 */
;*---------------------------------------------------------------------*/
(defun ude-html-fontify-html-buffer (mode &optional pred)
  "Fontify an html buffer according to MODE. The optional
argument PRED is a predicate that must satisfy a region in
order to get fontified. PRED is a function of two arguments,
the beginning and the end of the region"
  (interactive "aMode: ")
  (ude-html-fontify-html-region (point-min) (point-max) mode pred))

;*---------------------------------------------------------------------*/
;*    ude-html-fontify-html-region ...                                 */
;*    -------------------------------------------------------------    */
;*    This function fontify the current buffer region according to     */
;*    MODE.                                                            */
;*---------------------------------------------------------------------*/
(defun ude-html-fontify-html-region (beg end mode pred)
  "Fontify the current buffer region according to MODE and PRED.
See also UDE-HTML-FONTIFY-HTML-BUFFER."
  (save-excursion
    (goto-char beg)
    ;; we search for <pre> ... </pre> block that are supposed to
    ;; contain source code.
    (let* ((regexp (format "%s\\(\\(?:[^<]\\|[<][^/]\\|[<]/[^p]\\|[<]/p[^r]\\|[<]/pr[^e]\\|[<]/pre[^>]\\)+\\)%s"
			   (regexp-quote "<pre>")
			   (regexp-quote "</pre>")))
	   (curbuf (current-buffer))
	   (tmpnm  "*ude-fontify-html*")
	   (tmphnm "*ude-fontify-html.html*")
	   (tmpbuf (get-buffer-create tmpnm)))
      (set-buffer tmpbuf)
      (erase-buffer)
      (funcall mode)
      (set-buffer curbuf)
      (while (re-search-forward regexp end t)
	(let* ((start (match-beginning 0))
	       (stop  (match-end 0))
	       (mbeg  (match-beginning 1))
	       (mend  (match-end 1))
	       (delta (- mend mbeg)))
	  (if (or (not pred) (funcall pred mbeg mend))
	      (progn
		(kill-region mbeg mend)
		(set-buffer tmpbuf)
		(yank)
		(font-lock-fontify-buffer)
		(ude-html-source-buffer->html tmphnm)
		(set-buffer tmpbuf)
		(erase-buffer)
		(let ((buf (get-buffer tmphnm)))
		  (set-buffer buf)
		  (let ((new-delta (- (point-max) (point-min))))
		    ;; we have to push forward the end of the fontification
		    ;; because fontified region are larger than non
		    ;; fontified ones
		    (setq end (+ end (- new-delta delta))))
		  (kill-region (point-min) (point-max))
		  (set-buffer curbuf)
		  (goto-char start)
		  (delete-char 11)
		  (yank)
		  (set-buffer buf)
		  (erase-buffer)
		  (set-buffer curbuf)
		  (goto-char start)
		  (search-forward "</pre>" end t)))
	    ;; when the predicate has failed, we simply skip that region
	    (goto-char stop)))))))
	
