;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bmacs/dbg/dbg-mode.el                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun Apr 19 15:12:44 1998                          */
;*    Last change :  Fri Dec 11 18:32:50 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The bdb mode (not to be confused with the bdb function).         */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(provide 'dbg-mode)
(require 'comint)
(require 'dbg-config)
(require 'dbg-filter)
(require 'dbg-complete)
(require 'dbg-toolbar)
(require 'dbg-about-icon)
(require 'dbg-autoload)
(require 'ude-autoload)
(require 'ude-config)

;*---------------------------------------------------------------------*/
;*    minibuffer local map                                             */
;*---------------------------------------------------------------------*/
(defvar dbg-minibuffer-local-map nil
  "Keymap for minibuffer prompting of bdb startup command.")

(if (not dbg-minibuffer-local-map)
    (progn
      (setq dbg-minibuffer-local-map (copy-keymap minibuffer-local-map))
      (define-key
	dbg-minibuffer-local-map "\C-i" 'comint-dynamic-complete-filename)))

;*---------------------------------------------------------------------*/
;*    dbg-release ...                                                  */
;*---------------------------------------------------------------------*/
(defun dbg-release ()
  (interactive)
  (ude-about (format "Ude release: %s\nDbg release: %s\n\n%s\n\n%s"
		     ude-version
		     dbg-version
		     ude-author
		     ude-url)
	     dbg-about-icon))
;  
;*---------------------------------------------------------------------*/
;*    dbg-make-customize-menu ...                                      */
;*---------------------------------------------------------------------*/
(defun dbg-make-customize-menu ()
  '(["Release..." dbg-release t]
     "--:shadowEtchedOut"
     ["Ude..." ude-customize t]
     ["Dbg..." dbg-customize t]))

;*---------------------------------------------------------------------*/
;*    dbg-output-buffer ...                                            */
;*    -------------------------------------------------------------    */
;*    The buffer for process output.                                   */
;*---------------------------------------------------------------------*/
(defvar dbg-output-buffer nil)

;*---------------------------------------------------------------------*/
;*    dbg-mode ...                                                     */
;*---------------------------------------------------------------------*/
(defun dbg-mode ()
  "Major mode for running Bdb withing XEmacs.
The following command are available:

\\{comint-mode-map}."
  (interactive)
  (comint-mode)
  (setq major-mode 'dbg-mode)
  (setq mode-name dbg-binary)
  (setq mode-line-process '(":%s"))
  (use-local-map (copy-keymap comint-mode-map))
  ;; local bindings
  (local-set-key "\C-i" 'dbg-complete-command)
  (local-set-key "\C-m" 'dbg-send-input)
  (local-set-key "\C-l" 'dbg-refresh)
  (local-set-key dbg-mouse-binding 'ude-predicate-mouse-event)
  ;; the popup menu
  (ude-add-menu '(lambda (event) t)
		'dbg-popup-menu)
  ;; the menu bar
  (if (featurep 'menubar)
      (progn
	(set-buffer-menubar current-menubar)
	(add-submenu nil (cons "config" (dbg-make-customize-menu)))))
  ;; the bdb prompt
  (setq comint-prompt-regexp dbg-prompt-regexp)
  (setq paragraph-start comint-prompt-regexp)
  (make-local-variable 'comint-prompt-regexp)
  (make-local-variable 'paragraph-start)
  ;; the prompt marker is defined in dbg-filter file.
  (make-local-variable 'dbg-delete-prompt-marker)
  (setq dbg-delete-prompt-marker (make-marker))
  ;; same for dbg-last-frame
  (make-variable-buffer-local 'dbg-last-frame)
  (setq dbg-last-frame nil)
  ;; and so it is for dbg-marker-acc
  (make-variable-buffer-local 'dbg-marker-acc)
  ;; the mouse shape when flying over non text
  (set-glyph-image nontext-pointer-glyph "left_ptr")
  ;; we set up kill buffer hook
  (make-local-hook 'kill-buffer-hook)
  (add-hook 'kill-buffer-hook 'dbg-quit)
  ;; the toolbar
  (dbg-toolbar-init)
  ;; the modline
  (dbg-modeline-init)
  ;; we are done thus we run bdb hooks
  (run-hooks 'dbg-mode-hook))

;*---------------------------------------------------------------------*/
;*    dbg-quit ...                                                     */
;*---------------------------------------------------------------------*/
(defun dbg-quit ()
  (if (eq (current-buffer) dbg-comint-buffer)
      (progn
	;; we have to disconnect all connected buffer
	(dbg-disconnect-all-buffers)
	;; we now close all additional frames
	(if (dbg-installed-hook-p 'dbg-stack-hook)
	    (dbg-stack-quit))
	(if (dbg-installed-hook-p 'dbg-args-hook)
	    (dbg-args-quit))
	(if (dbg-installed-hook-p 'dbg-locals-hook)
	    (dbg-locals-quit))
	(if (dbg-installed-hook-p 'dbg-display-hook)
	    (dbg-display-quit))
	;; if bdb was the only one in its frame, we close the frame
	(let* ((frame (selected-frame))
	       (wins  (get-buffer-window-list dbg-comint-buffer)))
	  (while (consp wins)
	    (if (null (cdr wins))
		;; we have to delete the whole frame
		(progn
		  (setq wins '())
		  (delete-frame frame))
	      (progn
		(delete-window (car wins))
		(setq wins (cdr wins)))))))))

;*---------------------------------------------------------------------*/
;*    dbg-popup-menu ...                                               */
;*    ------------------------------------------------------------     */
;*    The main window bdb popup menu.                                  */
;*---------------------------------------------------------------------*/
(defun dbg-popup-menu (event)
  (interactive "e")
  (popup-menu
   (list "bdb"
	 ;; clearance
	 ["Clear line" dbg-clear-line t]
	 ["Clear window" dbg-clear-window t]
	 "--:shadowEtchedOut"
	 ;; buffer connection
	 ["Connect buffer" dbg-connect-buffer t]
	 ["Disconnect buffer" dbg-disconnect-buffer t]
	 ["Connect file" dbg-connect-file t]
	 "--:shadowEtchedOut"
	 ;; gdb hooking
	 ["Show stack"
	 dbg-stack-toggle
	 :style toggle
	 :selected (dbg-installed-hook-p 'dbg-stack-hook)]
	 ["Show args"
	 dbg-args-toggle
	 :style toggle
	 :selected (dbg-installed-hook-p 'dbg-args-hook)]
	 ["Show locals"
	 dbg-locals-toggle
	 :style toggle
	 :selected (dbg-installed-hook-p 'dbg-locals-hook)]
	 ["Show display"
	 dbg-display-toggle
	 :style toggle
	 :selected (dbg-installed-hook-p 'dbg-display-hook)]
	 ;; console logs
	 "--:shadowEtchedOut"
	 ["Console log" dbg-pop-console t]
	 ["Erase console log" dbg-erase-console t]
	 ["Recenter console log" dbg-recenter-console t])))

;*---------------------------------------------------------------------*/
;*    dbg-modeline-glyph ...                                           */
;*---------------------------------------------------------------------*/
(defvar dbg-modeline-glyph
  (let ((glyph (make-glyph "/* XPM */
static char *gnus-pointer[] = {
/* width height num_colors chars_per_pixel */
\"16 14 5 1\",
\"       c None s None\",
\".      c black\",
\"X      c gray50\",
\"o      c gray85\",
\"#      c white\",
\"      .   .     \",
\"       ...      \",
\"     X.....X    \",
\"    XX.....XX   \",
\"    ..#...X..   \",
\"   X.#.XXX...X  \",
\"  ...#..o...... \",
\"   .....o.....  \",
\"   .....o.....  \",
\"  .X....o....X. \",
\"    ....o....   \",
\"   .X...o...X.  \",
\"     X..o..X    \",
\"       XoX      \"};
")))
    (set-glyph-face glyph 'modeline-buffer-id)
    glyph))

(defvar dbg-modeline-busy-glyph
  (let ((glyph (make-glyph "/* XPM */
static char *gnus-pointer[] = {
/* width height num_colors chars_per_pixel */
\"16 14 6 1\",
\"       c None s None\",
\".      c black\",
\"X      c gray50\",
\"o      c gray85\",
\"#      c white\",
\"r      c #ff0000\",
\"      .   .     \",
\"       ...      \",
\"     X.....X    \",
\"    XX.....XX   \",
\"    ..#...X..   \",
\"   X.#.XXX...X  \",
\"  rrr#rrorr.rrr \",
\"   rrr.ror...r  \",
\"   rr...orr.rr  \",
\"  rXrr.rorrrrXr \",
\"    rrrror.rr   \",
\"   rXrrro...Xr  \",
\"     Xrror.X    \",
\"       XoX      \"
};")))
    (set-glyph-face glyph 'modeline-buffer-id)
    glyph))

;*---------------------------------------------------------------------*/
;*    dbg-modeline-glyph-cons ...                                      */
;*---------------------------------------------------------------------*/
(defvar dbg-modeline-glyph-cons nil)

;*---------------------------------------------------------------------*/
;*    dbg-modeline-init ...                                            */
;*    -------------------------------------------------------------    */
;*    This function change the left glyph for the modeline.            */
;*---------------------------------------------------------------------*/
(defun dbg-modeline-init ()
  (setq dbg-modeline-glyph-cons
	(cons (copy-extent modeline-buffer-id-left-extent)
	      dbg-modeline-glyph))
  (setq mode-line-buffer-identification
	(cons dbg-modeline-glyph-cons
	      (mapcar '(lambda (prop)
			 (let ((name (cdr prop)))
			   (if (string= "XEmacs%N:" (cdr prop))
			       (cons (car prop) "%N:")
			     prop)))
		      mode-line-buffer-identification))))

;*---------------------------------------------------------------------*/
;*    dbg-modeline-busy ...                                            */
;*    -------------------------------------------------------------    */
;*    This function change the left glyph for the modeline.            */
;*---------------------------------------------------------------------*/
(defun dbg-modeline-busy ()
  (if (consp dbg-modeline-glyph-cons)
      (rplacd dbg-modeline-glyph-cons dbg-modeline-busy-glyph))
  (set-buffer-modified-p (buffer-modified-p)))
  
;*---------------------------------------------------------------------*/
;*    dbg-modeline-free ...                                            */
;*    -------------------------------------------------------------    */
;*    This function change the left glyph for the modeline.            */
;*---------------------------------------------------------------------*/
(defun dbg-modeline-free ()
  (if (consp dbg-modeline-glyph-cons)
      (rplacd dbg-modeline-glyph-cons dbg-modeline-glyph))
  (set-buffer-modified-p (buffer-modified-p)))
  

