;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bmacs/bee/bee-usage.el               */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sat Nov 14 16:04:11 1998                          */
;*    Last change :  Wed Dec 16 07:12:29 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    Bee variable usage informations.                                 */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(provide 'bee-usage)
(require 'bee-autoload)
(require 'ude-config)
(require 'ude-custom)
(require 'ude-toolbar)

;*---------------------------------------------------------------------*/
;*    bee-find-local-definition ...                                    */
;*---------------------------------------------------------------------*/
(defun bee-find-local-definition (var)
  "Find a local variable definition."
  nil)

;*---------------------------------------------------------------------*/
;*    make-define-entry ...                                            */
;*---------------------------------------------------------------------*/
(defun make-define-entry (pos buffer)
  (let ((define (buffer-substring pos
				  (save-excursion
				    (end-of-line)
				    (point)))))
    (vector (concat "def   : " define)
	    `(let ((pop-up-frames t))
	       (pop-to-buffer ,buffer)
	       (goto-char ,pos))
	    t)))

;*---------------------------------------------------------------------*/
;*    make-module-entry ...                                            */
;*---------------------------------------------------------------------*/
(defun make-module-entry (module buffer)
  (if (consp module)
      (vector (concat "module: " (car module))
	      `(let ((pop-up-frames t))
	       (pop-to-buffer ,buffer)
	       (beginning-of-buffer))
	      t)
    (vector "????" '() t)))

;*---------------------------------------------------------------------*/
;*    make-decl-entry ...                                              */
;*---------------------------------------------------------------------*/
(defun make-decl-entry (module buffer)
  (if (consp (cdr module))
      (let* ((start (cdr (cdr module)))
	     (decl (id-select-sexp-start start)))
	(vector (concat "decl  : " (buffer-substring (car decl) (cdr decl)))
		`(let ((pop-up-frames t))
		   (pop-to-buffer ,buffer)
		   (goto-char ,(car decl)))
		t))
    (vector "decl  : implicit" '() t)))

;*---------------------------------------------------------------------*/
;*    make-export-entry ...                                            */
;*---------------------------------------------------------------------*/
(defun make-export-entry (module)
  (if (consp (cdr module))
      (concat "scope : " (car (cdr module)))
    "scope : static"))

;*---------------------------------------------------------------------*/
;*    make-use-entry ...                                               */
;*---------------------------------------------------------------------*/
(defun make-use-entry (ident module)
  (let ((static (or (not (consp (cdr module)))
		    (string= (car (cdr module)) "static"))))
    (vector "Calls and refs..." `(bee-find-usage ,ident ,static) t)))
  
;*---------------------------------------------------------------------*/
;*    make-assert-entry ...                                            */
;*---------------------------------------------------------------------*/
(defun make-assert-entry (asserts)
  (if (null asserts)
      asserts
    (list (cons "Assertions..."
		(mapcar '(lambda (assert)
			   (vector (format "%S" assert) '() t))
			asserts)))))

;*---------------------------------------------------------------------*/
;*    bee-usage-find ...                                               */
;*    -------------------------------------------------------------    */
;*    This function is not exact because it does not open the include  */
;*    file in order to read the directives clauses.                    */
;*---------------------------------------------------------------------*/
(defun bee-usage-find (ident)		     
  "Find documentation."
  (interactive (ude-interactive-ident (point) "Usage: "))
  (let ((buffer (save-excursion (bee-tags-find-variable-noselect ident))))
    (if (bufferp buffer)
	;; we have found the definition for this variable
	(progn
	  (set-buffer buffer)
	  ;; we fetch the s-expression that defines the variable
	  (let ((sexp (id-select-sexp-start (point))))
	    ;; if check for errors
	    (if (not (consp sexp))
		(error (format "Illegal definition -- %S" ident))
	      (let* ((assert (save-excursion (bee-assert-find sexp)))
		     (define-point (point))
		     (module (bee-module-declaration-find ident))
		     (menu   (append
			      (list (make-module-entry module buffer)
				    (make-export-entry module)
				    (make-decl-entry module buffer)
				    (make-define-entry define-point buffer)
				    (make-use-entry ident module))
			      (make-assert-entry assert))))
		(popup-menu (cons ident menu)))))))))

;*---------------------------------------------------------------------*/
;*    grep keymap                                                      */
;*---------------------------------------------------------------------*/
(defvar bee-grep-mouse-map (make-sparse-keymap))
(define-key bee-grep-mouse-map [(button2)]
  (function bee-grep-visit))
(defvar bee-local-root-directory "./"
  "This variable IS NOT buffer local.
It is used to bypass the buffer local variable mechanism!")

;*---------------------------------------------------------------------*/
;*    bee-find-usage ...                                               */
;*---------------------------------------------------------------------*/
(defun bee-find-usage (ident static)
  ;; we have to set the BEE-LOCAL-ROOT-DIRECTORY while we are
  ;; in the source buffer. It is not possible to use the
  ;; UDE-ROOT-DIRECTORY in the grep buffer because that variable
  ;; will be incorrect over there.
  (setq bee-local-root-directory ude-root-directory)
  (let ((files (if static
		   (file-relative-name (buffer-file-name (current-buffer))
				       ude-root-directory)
		 (ude-compile-makefile-entry ude-makefile-getsources-entry))))
    (if (or (string= files "")
	    (string-match ude-makefile-getsources-entry files))
	(error "Can't find source files")
      (let* ((cmd (format "cd %s; %s \"^[^;]+\\(?%s[ \t\n)]\" %s"
			  ude-root-directory
			  ude-egrep-n
			  ident
			  files))
	     (res (exec-to-string cmd)))
	(if (not (string= res ""))
	    (let ((src-file (file-relative-name
			     (buffer-file-name
			      (current-buffer))
			     ude-root-directory))
		  (buffer (get-buffer-create (format "*Calls-and-Refs-%s*"
						     ident))))
	      (set-buffer buffer)
	      (erase-buffer)
	      (insert res)
	      (let* ((height (save-excursion
			       (+ 2 (count-lines (point-min) (point-max)))))
		     (cur-height (frame-height (selected-frame)))
		     (new-height (if (> height cur-height)
				     cur-height
				   height))
		     (default-frame-alist (cons 'height
						(cons new-height
						      default-frame-alist)))
		     (pop-up-frames t)
		     (file-regexp "^\\([a-zA-Z]?:?[^:( \t\n]+\\)[:( \t]+\\([0-9]+\\)[:) \t].*$")
		     (nofile-regexp "^\\([0-9]+\\)[:) \t].*$"))
		;; the grep toolbar
		(bee-usage-init-toolbar (current-buffer))
		;; we seek `file:line: expression'
		(beginning-of-buffer)
		(while (re-search-forward file-regexp (point-max) t)
		  (let* ((file (buffer-substring (match-beginning 1)
						 (match-end 1)))
			 (line (buffer-substring (match-beginning 2)
						 (match-end 2)))
			 (end  (match-end 0))
			 (extent (make-extent (match-beginning 0) end)))
		    (set-extent-property extent
					 'grep (cons file line))
		    (set-extent-property extent
					 'mouse-face 'highlight)
		    (set-extent-property extent
					 'keymap bee-grep-mouse-map)
		    (goto-char end)))
		;; we seek `line: expression'
		(beginning-of-buffer)
		(while (re-search-forward nofile-regexp (point-max) t)
		  (let* ((line (buffer-substring (match-beginning 1)
						 (match-end 1)))
			 (end  (match-end 0))
			 (extent (make-extent (match-beginning 0) end)))
		    (set-extent-property extent
					 'grep (cons src-file line))
		    (set-extent-property extent
					 'mouse-face 'highlight)
		    (set-extent-property extent
					 'keymap bee-grep-mouse-map)
		    (goto-char end)))
		(pop-to-buffer buffer)))
	  (error (format "Nothing found about %S" ident)))))))

;*---------------------------------------------------------------------*/
;*    bee-grep-visit ...                                               */
;*---------------------------------------------------------------------*/
(defun bee-grep-visit (event)
  (interactive "e")
  (let* ((point  (event-closest-point event))
	 (buffer (event-buffer event))
	 (extent (extent-at point buffer 'grep))
	 (prop   (extent-property extent 'grep)))
    (if (consp prop)
	(let ((file (car prop))
	      (line (cdr prop)))
	  (if (stringp file)
	      (let* ((default-directory bee-local-root-directory)
		     (buffer (find-file-noselect file)))
		(if (bufferp buffer)
		    (progn
		      (set-buffer buffer)
		      (beginning-of-buffer)
		      (goto-line (string-to-int line))
		      (let ((pop-up-frames t))
			(pop-to-buffer buffer)))
		  (error "Can't find buffer for %S" file))))))))

;*---------------------------------------------------------------------*/
;*    bee-assert-find ...                                              */
;*    -------------------------------------------------------------    */
;*    This function search for all assertion in SEXP.                  */
;*---------------------------------------------------------------------*/
(defun bee-assert-find (sexp)
  (let ((start  (car sexp))
	(end    (cdr sexp))
	(assert '()))
    (save-excursion
      (goto-char start)
      (while (re-search-forward "[(\\[]assert[ \n\t]" end t)
	(let ((sexp (id-select-sexp-start (match-beginning 0))))
	  (if (consp sexp)
	      (progn
		(setq assert (cons (buffer-substring (car sexp) (cdr sexp))
				   assert))
		(goto-char (cdr sexp))))))
      (nreverse assert))))

;*---------------------------------------------------------------------*/
;*    Various profile toolbar button                                   */
;*---------------------------------------------------------------------*/
(defvar bee-usage-quit-button
  (toolbar-make-button-list ude-quit-icon))
(defvar bee-usage-open-button
  (toolbar-make-button-list ude-open-icon))

;*---------------------------------------------------------------------*/
;*    bee-usage-opened-toolbar ...                                     */
;*---------------------------------------------------------------------*/
(defvar bee-usage-opened-toolbar 
  '(;;close button
    [ude-close-toolbar-button bee-close-profile-toolbar t "Close toolbar"]
    [:style 2d :size 2]
    
    ;; the quit button
    [bee-usage-quit-button bee-usage-delete-frame t "Close Usage Frame"]
    [:style 2d :size 2]))

;*---------------------------------------------------------------------*/
;*    bee-usage-closed-toolbar ...                                     */
;*---------------------------------------------------------------------*/
(defvar bee-usage-closed-toolbar
  '([ude-open-toolbar-button bee-open-profile-toolbar t "Open toolbar"]))

;*---------------------------------------------------------------------*/
;*    Opening/closing toolbars ...                                     */
;*---------------------------------------------------------------------*/
(defun bee-close-profile-toolbar ()
  (ude-open-close-toolbar bee-usage-closed-toolbar))

(defun bee-open-profile-toolbar ()
  (ude-open-close-toolbar bee-usage-opened-toolbar))

;*---------------------------------------------------------------------*/
;*    bee-usage-buffer ...                                             */
;*---------------------------------------------------------------------*/
(defvar bee-usage-buffer nil)

;*---------------------------------------------------------------------*/
;*    bee-usage-delete-frame ...                                       */
;*---------------------------------------------------------------------*/
(defun bee-usage-delete-frame ()
  (interactive)
  (if (bufferp bee-usage-buffer)
      (let ((window (get-buffer-window bee-usage-buffer t)))
	(if (windowp window)
	    (let ((frame (window-frame window)))
	      (if (framep frame)
		  (delete-frame frame)))))))

;*---------------------------------------------------------------------*/
;*    bee-usage-init-toolbar ...                                       */
;*---------------------------------------------------------------------*/
(defun bee-usage-init-toolbar (buffer)
  (setq bee-usage-buffer buffer)
  (set-specifier default-toolbar-visible-p t)
  (set-specifier default-toolbar bee-usage-opened-toolbar buffer))

