;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bmacs/bee/bee-tags.el                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu May 28 12:00:29 1998                          */
;*    Last change :  Wed Dec 16 07:13:25 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    This module implements the search functions:                     */
;*      - a module definition (using the afile)                        */
;*      - a local variable definition                                  */
;*      - a global imported variable (using the tags).                 */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(provide 'bee-tags)
(require 'etags)
(require 'bee-autoload)
(require 'bee-config)
(require 'ude-autoload)
(require 'ude-config)

;*---------------------------------------------------------------------*/
;*    bee-find-file-create-p ...                                       */
;*    -------------------------------------------------------------    */
;*    This function searches the file FNAME in the root                */
;*    directory. If that file does not exists it is created. If after  */
;*    that creation FNAME still does not exists, nil is returned.      */
;*---------------------------------------------------------------------*/
(defun bee-find-file-create-p (fname cmd)
  (if (file-exists-p fname)
      fname
    (progn
      (message "calling `%s'..." cmd)
      (let ((res (exec-to-string cmd)))
	(message "")
	(if (string-match ude-make-error-output res)
	    nil
	  fname)))))

;*---------------------------------------------------------------------*/
;*    bee-find-afile-create-p ...                                      */
;*---------------------------------------------------------------------*/
(defun bee-find-afile-create-p ()
  (bee-find-file-create-p (concat ude-root-directory ude-afile-name)
			  bee-make-afile))

;*---------------------------------------------------------------------*/
;*    bee-find-tagsfile-create-p ...                                   */
;*    -------------------------------------------------------------    */
;*    This function searches the UDE-TAGSFILE-NAME file in the root    */
;*    directory. If that file does not exists it is created. If after  */
;*    that creation the tagsfile still does not exists, FIND-TAGSFILE  */
;*    returns nil.                                                     */
;*---------------------------------------------------------------------*/
(defun bee-find-tagsfile-create-p ()
  (bee-find-file-create-p (concat ude-root-directory ude-tagsfile-name)
			  bee-make-tags))
  
;*---------------------------------------------------------------------*/
;*    bee-find-module-nokeymap ...                                     */
;*    -------------------------------------------------------------    */
;*    Find a module implementation.                                    */
;*    This function is not bound to any key binding.                   */
;*---------------------------------------------------------------------*/
(defun bee-find-module-nokeymap (module)
  "Find a module implementation."
  (interactive (ude-interactive-ident (point) "Module: "))
  (if (not (bee-find-afile-create-p))
      (ude-error "Find module: no afile file loaded.")
    (let* ((cmd (concat ude-uncase-grep " \"" module " \" "
			ude-root-directory ude-afile-name))
	   (res (exec-to-string cmd)))
      (if (> (length res) 0)
	  (let ((cell (read res)))
	    (if (consp cell)
		(let ((fname (car (cdr cell))))
		  (find-file-other-frame
		   (concat ude-root-directory fname)))
	      (ude-error (format "Find module: corrupted afile -- %S [%S]"
				 (concat ude-root-directory ude-afile-name)
				 cmd))))
	(ude-error "Find module: can't find module -- %S" module)))))

;*---------------------------------------------------------------------*/
;*    bee-find-module ...                                              */
;*    -------------------------------------------------------------    */
;*    Find a module implementation.                                    */
;*---------------------------------------------------------------------*/
(defun bee-find-module (module)
  "Find a module implementation."
  (interactive (ude-interactive-ident (point) "Module: "))
  (bee-find-module-nokeymap module))

;*---------------------------------------------------------------------*/
;*    bee-find-module-list ...                                         */
;*    -------------------------------------------------------------    */
;*    If the afile exists, that function return the list of the        */
;*    defined modules.                                                 */
;*---------------------------------------------------------------------*/
(defun bee-find-module-list ()
  (if (file-exists-p (concat ude-root-directory ude-afile-name))
      (let ((old-buffer (find-buffer-visiting (concat ude-root-directory
						      ude-afile-name))))
	(if (bufferp old-buffer)
	    (sort (mapcar '(lambda (x) (symbol-name (car x)))
			  (read old-buffer))
		  'string<)
	  (let ((buffer (find-file-noselect (concat ude-root-directory
						    ude-afile-name))))
	    (if (bufferp buffer)
		(let ((res (sort (mapcar '(lambda (x) (symbol-name (car x)))
					 (read buffer))
				 'string<)))
		  (kill-buffer buffer)
		  res)
	      '()))))
    '()))

;*---------------------------------------------------------------------*/
;*    bee-find-definition ...                                          */
;*    -------------------------------------------------------------    */
;*    Search for a variable definition. This function first searches   */
;*    for a local variable and then searches for a global one.         */
;*---------------------------------------------------------------------*/
(defun bee-find-definition (var)
  "Find a variable or class definition."
  (interactive (ude-interactive-ident (point) "Identifier: "))
  (if (not (bee-find-local-definition var))
      ;; there is no local variable called VAR in the context
      (if (not (bee-find-tagsfile-create-p))
	  (ude-error "Find variable: no tags file loaded.")
	(let ((tags-file-name (concat ude-root-directory ude-tagsfile-name))
	      (tags-add-tables t))
	  (condition-case err
	      (find-tag-other-frame var)
	    (error (apply 'ude-error "Can't find definition for %S"
			  (cdr err))))))))

;*---------------------------------------------------------------------*/
;*    bee-tags-find-variable-noselect ...                              */
;*    -------------------------------------------------------------    */
;*    Find a buffer defining VAR in a buffer that is not selected.     */
;*---------------------------------------------------------------------*/
(defun bee-tags-find-variable-noselect (var)
  "Find a variable or class definition."
  (interactive "sVariable: ")
  (if (not (bee-find-tagsfile-create-p))
      (ude-error "Find variable: no tags file loaded.")
    (condition-case err
	(let ((tags-file-name (concat ude-root-directory ude-tagsfile-name))
	      (tags-add-tables t))
	  (message "Seeking definition `%S'" var)
	  (find-tag-noselect var))
      (error (apply 'ude-error "Can't find variable %S" (cdr err))))))

;*---------------------------------------------------------------------*/
;*    bee-tags-find-variable ...                                       */
;*    -------------------------------------------------------------    */
;*    Find a variable or class definition.                             */
;*---------------------------------------------------------------------*/
(defun bee-tags-find-variable (var)
  "Find a variable or class definition."
  (interactive "sVariable: ")
  (if (not (bee-find-tagsfile-create-p))
      (ude-error "Find variable: no tags file loaded.")
    (condition-case err
	(let ((tags-file-name (concat ude-root-directory ude-tagsfile-name))
	      (tags-add-tables t))
	  (find-tag-other-frame var))
      (error (apply 'ude-error "Can't find variable %S" (cdr err))))))

;*---------------------------------------------------------------------*/
;*    bee-tags-find-class ...                                          */
;*---------------------------------------------------------------------*/
(defun bee-tags-find-class (class)
  (if (not (bee-find-tagsfile-create-p))
      (ude-error "Find variable: no tags file loaded.")
    (condition-case err
	(let ((tags-file-name (concat ude-root-directory ude-tagsfile-name))
	      (tags-add-tables t))
	  (find-tag-other-frame (concat "class " class)))
      (error (apply 'ude-error "Can't find class %S" (cdr err))))))
  
;*---------------------------------------------------------------------*/
;*    bee-tags-find ...                                                */
;*    -------------------------------------------------------------    */
;*    This function is a dispatcher. It checks (with the current       */
;*    position) if it should look for a module or a variable           */
;*    definition. It looks for a module if (point) is inside an        */
;*    `import' or `use' module clause.                                 */
;*---------------------------------------------------------------------*/
(defun bee-tags-find ()
  "Find a variable/class/module definition."
  (interactive)
  (let* ((pos (point))
	 (ident (ude-fetch-identifier pos)))
    (if (stringp ident)
	(if (bee-module-declaration-p pos "(\\(import\\|use\\)[ \n]")
	    (bee-find-module ident)
	  (bee-tags-find-variable ident))
      (let ((class (let ((symbol (id-select-symbol (point))))
		     (if (consp symbol)
			 (if (save-excursion
			       (goto-char (car symbol))
			       (re-search-forward ".*::\\(.+\\)" (cdr symbol)))
			     (buffer-substring (match-beginning 1)
					       (match-end 1))
			   nil)
		       nil))))
	(if (stringp class)
	    (bee-tags-find-class class)
	    (let ((ident (read-string "Find definition: ")))
	      (if (not (string= ident ""))
		  (bee-tags-find-variable ident))))))))

;*---------------------------------------------------------------------*/
;*    bee-tags-find-next ...                                           */
;*---------------------------------------------------------------------*/
(defun bee-tags-find-next ()
  "Find a variable/class/module definition."
  (interactive)
  (if (not (bee-find-tagsfile-create-p))
      (ude-error "Find variable: no tags file loaded.")
    (condition-case err
	(let ((tags-file-name (concat ude-root-directory ude-tagsfile-name))
	      (tags-add-tables t))
	  (find-tag nil t))
      (error (apply 'ude-error "No more definition %S" (cdr err))))))

;*---------------------------------------------------------------------*/
;*    bee-tag-find ...                                                 */
;*---------------------------------------------------------------------*/
(defun bee-tag-find ()
  "Find a variable or class definition."
  (interactive)
  (if (not (bee-find-tagsfile-create-p))
      (ude-error "Find variable: no tags file loaded.")
    (let* ((ident (ude-fetch-then-request-identifier (point) "Binding: ")))
      (condition-case err
	  (let ((tags-file-name (concat ude-root-directory ude-tagsfile-name))
		(tags-add-tables t))
	    (find-tag ident))
	(error (apply 'ude-error "Can't find definition for %S" (cdr err)))))))


