;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bmacs/bee/bee-mode.el                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Mon May 25 07:49:23 1998                          */
;*    Last change :  Thu Jan 21 07:29:45 1999 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The Bee mode declaration.                                        */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(provide 'bee-mode)
(require 'font-lock)
(require 'ude-autoload)
(require 'ude-config)
(require 'ude-custom)
(require 'bee-autoload)
(require 'bee-config)
(require 'bee-indent)
(require 'bee-toolbar)
(require 'bee-flock)
(require 'bee-keymap)

;*---------------------------------------------------------------------*/
;*    bee-mode-syntax-table ...                                        */
;*---------------------------------------------------------------------*/
(defvar bee-mode-syntax-table (make-syntax-table) "")

;*---------------------------------------------------------------------*/
;*    bee-init-syntax-table ...                                        */
;*---------------------------------------------------------------------*/
(defun bee-init-syntax-table ()
  (let ((i 0)
	(local-syntax-table (syntax-table (current-buffer))))
    ;; Default is atom-constituent.
    (while (< i 256)
      (modify-syntax-entry i "_   " local-syntax-table)
      (setq i (1+ i)))

    ;; Word components.
    (setq i ?0)
    (while (<= i ?9)
      (modify-syntax-entry i "w   " local-syntax-table)
      (setq i (1+ i)))
    (setq i ?A)
    (while (<= i ?Z)
      (modify-syntax-entry i "w   " local-syntax-table)
      (setq i (1+ i)))
    (setq i ?a)
    (while (<= i ?z)
      (modify-syntax-entry i "w   " local-syntax-table)
      (setq i (1+ i)))
    (modify-syntax-entry ?* "w   " local-syntax-table)
    (modify-syntax-entry ?@ "w   " local-syntax-table)
    (modify-syntax-entry ?! "w   " local-syntax-table)
    (modify-syntax-entry ?? "w   " local-syntax-table)
    (modify-syntax-entry ?= "w   " local-syntax-table)
    (modify-syntax-entry ?< "w   " local-syntax-table)
    (modify-syntax-entry ?> "w   " local-syntax-table)
    (modify-syntax-entry ?+ "w   " local-syntax-table)
    (modify-syntax-entry ?* "w   " local-syntax-table)
    (modify-syntax-entry ?~ "w   " local-syntax-table)
    (modify-syntax-entry ?$ "w   " local-syntax-table)
    (modify-syntax-entry ?% "w   " local-syntax-table)
    (modify-syntax-entry ?^ "w   " local-syntax-table)
    (modify-syntax-entry ?\\ "w   " local-syntax-table)
    (modify-syntax-entry ?. "w   " local-syntax-table)

    ;; Whitespace
    (modify-syntax-entry ?\t "    " local-syntax-table)
    (modify-syntax-entry ?\n ">   " local-syntax-table)
    (modify-syntax-entry ?\f "    " local-syntax-table)
    (modify-syntax-entry ?\r "    " local-syntax-table)
    (modify-syntax-entry ?  "    " local-syntax-table)

    ;; These characters are delimiters but otherwise undefined.
    ;; Brackets and braces balance for editing convenience.
    (modify-syntax-entry ?[ "(]  " local-syntax-table)
    (modify-syntax-entry ?] ")[  " local-syntax-table)
    (modify-syntax-entry ?{ "(}  " local-syntax-table)
    (modify-syntax-entry ?} "){  " local-syntax-table)
    (modify-syntax-entry ?\| "    " local-syntax-table)

    ;; Other atom delimiters
    (modify-syntax-entry ?\( "()  " local-syntax-table)
    (modify-syntax-entry ?\) ")(  " local-syntax-table)
    (modify-syntax-entry ?\; "<   " local-syntax-table)
    (modify-syntax-entry ?\" "\"    " local-syntax-table)
    (modify-syntax-entry ?' "'   " local-syntax-table)
    (modify-syntax-entry ?` "'   " local-syntax-table)
    (modify-syntax-entry ?\: "'   " local-syntax-table)

    ;; Special characters
    (modify-syntax-entry ?, "'   " local-syntax-table)
    (modify-syntax-entry ?# "'   " local-syntax-table)
    (modify-syntax-entry ?\\ "\\   " local-syntax-table)

    ;; legal Bigloo identifier chars that are not recognized by the \w syntax
    (setq ude-extra-identifier-chars "[-_/]")))

;*---------------------------------------------------------------------*/
;*    bee-mode-abbrev-table ...                                        */
;*---------------------------------------------------------------------*/
(defvar bee-mode-abbrev-table nil "")
(define-abbrev-table 'bee-mode-abbrev-table ())

;*---------------------------------------------------------------------*/
;*    bee keymap ...                                                   */
;*    -------------------------------------------------------------    */
;*    For a reason that I don't know these variables cannot be         */
;*    defined inside BEE-KEYMAP otherwise emacs don't succeed at       */
;*    loading the present file!                                        */
;*---------------------------------------------------------------------*/
(defvar bee-mode-map (make-sparse-keymap))

;*---------------------------------------------------------------------*/
;*    bee-mode-variables ...                                           */
;*---------------------------------------------------------------------*/
(defun bee-mode-variables ()
  (setq local-abbrev-table bee-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'bee-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start ";")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip ";+[ \t]*")
  (make-local-variable 'comment-column)
  (setq comment-column 40)
  (make-local-variable 'comment-indent-hook)
  (setq comment-indent-hook 'bee-comment-indent))

;*---------------------------------------------------------------------*/
;*    bee-docline ...                                                  */
;*---------------------------------------------------------------------*/
(defun bee-docline ()
  (interactive)
  (setq ude-info-file-list bee-info-file-list)
  (ude-info-docline (bee-font-lock-get-info-keywords)))

;*---------------------------------------------------------------------*/
;*    bee-info-sexp ...                                                */
;*    -------------------------------------------------------------    */
;*    This function checks its string argument in order to             */
;*    discover what kind of documentation is required. For instance,   */
;*    this function may decide to print the number, string or          */
;*    application documentation instead of the documentation of a      */
;*    particular pre-defined function.                                 */
;*---------------------------------------------------------------------*/
(defun bee-info-sexp (beg end)
  "Popup an online documentation according to the active region."
  (interactive "r")
  (let ((string (buffer-substring beg end)))
    (cond 
     ((string-match "^[ \t]*;" string)
      ;; this is a comment
      (ude-info-section "Comments"))
     ((string-match "^[0-9.]+$" string)
      ;; this is a number
      (ude-info-section "Numbers"))
     ((string-match "^\"[^\"]*\"" string)
      ;; this is a string
      (ude-info-section "Strings"))
     ((string-match "^((" string)
      ;; this is a computed call
      (ude-info-section "procedure call"))
     ((string-match "['`,]" string)
      ;; this is a quotation
      (ude-info-section "quotation"))
     ((string-match "#[ftFT]$" string)
      ;; booleans
      (ude-info-section "Booleans"))
     ((string-match "#[\\]" string)
      ;; chars
      (ude-info-section "Characters"))
     ((string-match "(\\(\\w+\\)" string)
      ;; an application
      (ude-info-ref-internal (substring string
					(match-beginning 1)
					(match-end 1))))
     (t
      (ude-info-ref-internal string)))))

;*---------------------------------------------------------------------*/
;*    bee-docline-init ...                                             */
;*    -------------------------------------------------------------    */
;*    Initialize the docline system, that is, tell Ude to use          */
;*    info with the Bigloo page.                                       */
;*---------------------------------------------------------------------*/
(defun bee-docline-init ()
  (setq ude-info-region (function bee-info-sexp))
  (setq ude-info-file-list bee-info-file-list))
  
;*---------------------------------------------------------------------*/
;*    bee-compile ...                                                  */
;*---------------------------------------------------------------------*/
(defun bee-compile ()
  (interactive)
  (ude-mode-compile bee-compilation-error-regexp-alist
		    bee-compilation-font-lock-keywords))

;*---------------------------------------------------------------------*/
;*    bee-generate/update-makefile ...                                 */
;*---------------------------------------------------------------------*/
(defun bee-generate/update-makefile ()
  (interactive)
  (if (not (file-exists-p (concat ude-root-directory ude-makefile)))
      (ude-generate-makefile bee-bmake-application-option)
    (ude-update-makefile)))

;*---------------------------------------------------------------------*/
;*    bee-find-toplevel-sexp ...                                       */
;*    -------------------------------------------------------------    */
;*    Find the toplevel POS is in.                                     */
;*---------------------------------------------------------------------*/
(defun bee-find-toplevel-sexp (pos)
  (save-excursion
    (let* ((sexp nil)
	   (res  (id-select-sexp pos))
	   (old  pos))
      (while (and (consp res) (not (eq (car res) old)))
	(setq sexp res)
	(setq old (car res))
	(setq res (id-select-sexp (car res))))
      sexp)))

;*---------------------------------------------------------------------*/
;*    bee-mode ...                                                     */
;*---------------------------------------------------------------------*/
(defun bee-mode ()
  "Major mode for editing Bigloo code.

Commands:
Delete converts tabs to spaces as it moves back.
Blank lines separate paragraphs.  Semicolons start comments.
\\{bee-mode-map}
Entry to this mode calls the value of bee-mode-hook
if that value is non-nil."
  (interactive)
  (setq debug-on-error t)
  (kill-all-local-variables)
  ;; mode declaration
  (setq major-mode 'bee-mode)
  (setq mode-name "Bee")
  (use-local-map bee-mode-map)
  ;; we setup the project root directory
  (ude-auto-set-root-directory)
  ;; syntax table
  (set-syntax-table bee-mode-syntax-table)
  (bee-init-syntax-table)
  ;; global buffer local variables
  (bee-mode-variables)
  ;; docline initialization
  (bee-docline-init)
  ;; bmake initialization
  (setq ude-makemake bee-bmake)
  ;; compilation initialization
  (setq ude-mode-menu-compile 'bee-compile)
  ;; keymap bindings
  (bee-keymap-init)
  ;; parenthesis blinking init
  (ude-paren-init)
  ;; profile highlighting
  (setq ude-profile-highlight-buffer 'bee-profile-highlight-buffer)
  ;; starting font-lock
  (if ude-font-lock-p
      (progn
	(setq font-lock-keywords bee-font-lock-keywords)
	(font-lock-mode t)))
  ;; the doc source fontification
  (ude-fontify-doc-source (current-buffer))
  ;; the toolbar
  (bee-toolbar-init)
  ;; repl initialization
  (make-local-hook 'ude-repl-hooks)
  (add-hook 'ude-repl-hooks
	    '(lambda () (set-syntax-table bee-mode-syntax-table)))
  ;; the bee hook
  (run-hooks 'bee-mode-hook))


