;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bde/btags/btags.scm                  */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu May 28 07:24:34 1998                          */
;*    Last change :  Mon Sep 21 08:34:14 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The Bigloo tag generator.                                        */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module btags
   (include "include/args.sch")
   (main    main))

;*---------------------------------------------------------------------*/
;*    Global parameters                                                */
;*---------------------------------------------------------------------*/
(define *btags-version*    "0.0")
(define *btags-append*     #f)
(define *btags-language*   #f)
(define *btags-table-name* "TAGS")
(define *btags-etags*      "etags")
(define *btags-files*      '())
(define *btags-modules*    #f)

(define *oport*            #f)

;*---------------------------------------------------------------------*/
;*    main ...                                                         */
;*---------------------------------------------------------------------*/
(define (main argv)
   ;; we parse command line arguments
   (parse-args argv)
   ;; ew start erasing the old table if not in append mode
   (if (not *btags-append*)
       (delete-file *btags-table-name*))
   ;; we produce tags entrie regarding the source language
   (for-each btags-file (reverse! *btags-files*))
   ;; we close the output port if currently opened
   (if (output-port? *oport*)
       (begin
	  (close-output-port *oport*)
	  (set! *oport* #f))))

;*---------------------------------------------------------------------*/
;*    parse-args ...                                                   */
;*---------------------------------------------------------------------*/
(define (parse-args cmd-args)
   (define (usage args-parse-usage level)
      (print "usage: btags [options] file ...")
      (newline)
      (args-parse-usage)
      (newline))
   (args-parse (cdr cmd-args)
      (("?")
       (usage args-parse-usage 1))
      (("-help" (synopsis "This help message"))
       (usage args-parse-usage 1))
      (("-a" (synopsis "-a, --append" "Append to existing tag file."))
       (set! *btags-append* #t))
      (("--append")
       (set! *btags-append* #t))
      (("-l"
	?language
	(synopsis "-l language, --language-language"
		  "Parse the following files according to the given language."))
       (if (string=? language "none")
	   (set! *btags-language* #f)
	   (set! *btags-language* language)))
      (("-l=?language")
       (if (string=? language "none")
	   (set! *btags-language* #f)
	   (set! *btags-language* language)))
      (("-o"
	?tagfile
	(synopsis "-o tagfile, --output=tagfile"
		  "Explict name of file for tag table (default `TAGS')."))
       (set! *btags-table-name* tagfile))
      (("--output=?tagfile")
       (set! *btags-table-name* tagfile))
      (("-module" (synopsis "Generate tag entry for module identifier"))
       (set! *btags-modules* #t))
      (("-v" (synopsis "-v, --version"
		       "Print the current-version of the program"))
       (print *btags-version*)
       (exit 0))
      (("--version")
       (print *btags-version*)
       (exit 0))
      (else
       (if (string? *btags-language*)
	   (set! *btags-files* (cons (cons else *btags-language*)
				     *btags-files*))
	   (set! *btags-files* (cons else *btags-files*))))))

;*---------------------------------------------------------------------*/
;*    btags-file ...                                                   */
;*---------------------------------------------------------------------*/
(define (btags-file file)
   (if (pair? file)
       (let ((fname (car file))
	     (language (cdr file)))
	  (if (string=? language "bigloo")
	      (bigloo-btags fname)
	      (other-btags fname language)))
       (let ((suffix (suffix file)))
	  (cond
	     ((member suffix '("scm" "sch" "bgl"))
	      (bigloo-btags file))
	     (else
	      (other-btags file #f))))))

;*---------------------------------------------------------------------*/
;*    other-btags ...                                                  */
;*    -------------------------------------------------------------    */
;*    We produce tags entries for other languages. This is quiet       */
;*    easy because we just call `etags'                                */
;*---------------------------------------------------------------------*/
(define (other-btags file language)
   (if (output-port? *oport*)
       (begin
	  (close-output-port *oport*)
	  (set! *oport* #f)))
   (let ((command (string-append *btags-etags*
				 (if language
				     (string-append " --language=" language)
				     "")
				 " --append"
				 " --output=" *btags-table-name* " "
				 file)))
      (system command)))
      
;*---------------------------------------------------------------------*/
;*    bigloo-btags ...                                                 */
;*    -------------------------------------------------------------    */
;*    The true jobs, the Bigloo tags generator...                      */
;*---------------------------------------------------------------------*/
(define (bigloo-btags file)
   ;; first, we open the output file
   (if (not (output-port? *oport*))
       (begin
	  (set! *oport* (append-output-file *btags-table-name*))
	  (if (not (output-port? *oport*))
	      (error "btags"
		     "Can't open file for output"
		     *btags-table-name*))))
   (bigloo-btags-file *oport* file))

;*---------------------------------------------------------------------*/
;*    bigloo-btags-file ...                                            */
;*    -------------------------------------------------------------    */
;*    Because we want to avoid parsing source file without reading     */
;*    (with the regular Bigloo reader), we process two readings of     */
;*    a source file. The first one, we locate declarations and         */
;*    in the second one, we emit tags declarations.                    */
;*---------------------------------------------------------------------*/
(define (bigloo-btags-file oport file)
   (let ((decl-lines (find-declaration-lines file)))
      (let ((iport (and (file-exists? file)
			(open-input-file file))))
	 (if (not (input-port? iport))
	     (error "btags" "Can't open file for intput" file)
	     (unwind-protect
		;; we have to reset the reader other line numbering is
		;; erroneous
		(reader-reset!)
		;; we now scan the file in order to emit the TAGs decls
		(let loop ((line  (read-line iport))
			   (lnum  1)
			   (pos   1)
			   (lines decl-lines)
			   (tags  ""))
		   (if (and (pair? lines) (not (eof-object? line)))
		       (if (=fx lnum (car (car lines)))
			   ;; this is a declaration line
			   (let ((tag (tags-substring line
						      (cdr (car lines)))))
			      (loop (read-line iport)
				    (+fx lnum 1)
				    (+fx pos (string-length line))
				    (cdr lines)
				    (if (string? tag)
					(string-append tags
						       tag
						       (make-string 1 #a127)
						       (number->string lnum)
						       ","
						       (number->string pos)
						       #"\n")
					tags)))
			   (loop (read-line iport)
				 (+fx lnum 1)
				 (+fx pos (string-length line))
				 lines
				 tags))
		       (begin
			  ;; we emit the file header
			  (fprint oport #a012)
			  (let ((len (string-length tags)))
			     (fprint oport file "," len)
			     (display tags oport)))))
		(close-input-port iport))))))

;*---------------------------------------------------------------------*/
;*    tags-substring ...                                               */
;*---------------------------------------------------------------------*/
(define (tags-substring str1 str2)
   (let* ((len1 (string-length str1))
          (len2 (string-length str2))
          (stop (- len1 len2)))
      (define (substring-at? start)
         (let loop ((i1 start)
                    (i2 0))
            (cond
               ((=fx i2 len2)
                #t)
               ((char-ci=? (string-ref str1 i1) (string-ref str2 i2))
                (loop (+ i1 1) (+ i2 1)))
               (else
                #f))))
      (let loop ((i 0))
         (cond
            ((> i stop)
             #f)
            ((substring-at? i)
	     (substring str1 0 (+fx i len2)))
            (else
             (loop (+ i 1)))))))

;*---------------------------------------------------------------------*/
;*    find-declaration-lines ...                                       */
;*    -------------------------------------------------------------    */
;*    During this stage, we walk thru the code in order to find        */
;*    all the source lines that contains a declaration. For each       */
;*    of these lines, we allocate a pair containing the line number    */
;*    and a delimiter sentinel string.                                 */
;*---------------------------------------------------------------------*/
(define (find-declaration-lines file)
   (let ((iport (and (file-exists? file) (open-input-file file))))
      (if (not (input-port? iport))
	  (error "btags" "Can't open file for intput" file)
	  (unwind-protect
	     (let ((exp (read iport #t)))
		(if (not (eof-object? exp))
		    (let ((lines (if (module? exp)
				     (btags-module exp)
				     (btags-expression exp))))
		       (let loop ((exp   (read iport #t))
				  (lines lines))
			  (if (eof-object? exp)
			      (reverse! lines)
			      (begin
				 (loop (read iport #t)
				       (append (btags-expression exp)
					       lines))))))))
	     (close-input-port iport)))))
	 
;*---------------------------------------------------------------------*/
;*    module? ...                                                      */
;*    -------------------------------------------------------------    */
;*    Is an expression a module expression.                            */
;*---------------------------------------------------------------------*/
(define (module? expr)
   (match-case expr
      ((module (? symbol?) . ?-)
       #t)
      ((?directives . ?-)
       #t)
      (else
       #f)))

;*---------------------------------------------------------------------*/
;*    line-number ...                                                  */
;*---------------------------------------------------------------------*/
(define (line-number expr)
   (and (epair? expr)
	(match-case (cer expr)
	   ((at ?- ?pos ?line)
	    line))))

;*---------------------------------------------------------------------*/
;*    id->string ...                                                   */
;*---------------------------------------------------------------------*/
(define (id->string id decl)
   (cond
      ((and (pair? id) (symbol? (car id)))
       (symbol->string (car id)))
      ((string? id)
       id)
      ((not (symbol? id))
       (error "btags" "Illegal identifier" decl))
      (else
       (let* ((string (symbol->string id))
	      (len    (string-length string)))
	  (let loop ((walker  0))
	     (cond
		((=fx walker len)
		 (symbol->string id))
		((and (char=? (string-ref string walker) #\:)
		      (<fx walker (-fx len 1))
		      (char=? (string-ref string (+fx walker 1)) #\:))
		 (substring string 0 walker))
		(else
		 (loop (+fx walker 1)))))))))

;*---------------------------------------------------------------------*/
;*    btags-module ...                                                 */
;*---------------------------------------------------------------------*/
(define (btags-module expr)
   (define (tags-module-clause lines clauses)
      (let loop ((clauses clauses)
		 (lines   lines))
	 (if (pair? clauses)
	     (let ((clause (car clauses)))
		(loop (cdr clauses)
		      (if (pair? clause)
			  (case (car clause)
			     ((eval)
			      (append (btags-eval-clause clause) lines))
			     ((static export)
			      (append (btags-static+export-clause clause)
				      lines))
			     ((extern)
			      (append (btags-extern-clause clause) lines))
			     (else
			      lines))
			  lines)))
	     lines)))
   (match-case expr
      ((module (and ?name (? symbol?)) . ?clauses)
       (tags-module-clause
	(if (and (epair? expr) *btags-modules*)
	    (let ((lnum (line-number expr)))
	       (list (cons lnum (symbol->string name))))
	    '())
	clauses))
      ((directives . ?clauses)
       (tags-module-clause '() clauses))))

;*---------------------------------------------------------------------*/
;*    btags-clause-walker ...                                          */
;*---------------------------------------------------------------------*/
(define (btags-clause-walker parser clause)
   (let loop ((decls clause)
	      (lines '()))
      (if (pair? decls)
	  (let ((line (parser (car decls))))
	     (loop (cdr decls)
		   (if (pair? line)
		       (cons line lines)
		       lines)))
	  lines)))
   
;*---------------------------------------------------------------------*/
;*    btags-eval-clause ...                                            */
;*---------------------------------------------------------------------*/
(define (btags-eval-clause clause)
   (define (eval-parse decl)
      (match-case decl
	 ((import (and (? symbol?) ?var))
	  (let ((lnum (line-number decl)))
	     (if (number? lnum)
		 (cons lnum var))))))
   (btags-clause-walker eval-parse clause))

;*---------------------------------------------------------------------*/
;*    btags-static+export-clause ...                                   */
;*---------------------------------------------------------------------*/
(define (btags-static+export-clause clause)
   (define (static+export-parse decl)
      (match-case decl
	 (((and ?class (or class final-class wide-class)) ?name . ?-)
	  (let ((lnum (line-number decl)))
	     (if (number? lnum)
		 (cons lnum (id->string name decl)))))))
   (btags-clause-walker static+export-parse clause))

;*---------------------------------------------------------------------*/
;*    btags-extern-clause ...                                          */
;*---------------------------------------------------------------------*/
(define (btags-extern-clause clause)
   (define (extern-parse decl)
      (if (epair? decl)
	  (match-case decl
	     ((export (and (? symbol?) ?bname) (and (? string?) ?cname))
	      (let ((lnum (line-number decl)))
		 (if (number? lnum)
		     (cons lnum (string-append "\"" cname "\"")))))
	     ((or (macro ?l-name ?proto ?c-name)
		  (infix macro ?l-name ?proto ?c-name))
	      (let ((lnum (line-number decl)))
		 (if (number? lnum)
		     (cons lnum (id->string l-name decl)))))
	     ((macro ?l-name ?c-name)
	      (let ((lnum (line-number decl)))
		 (if (number? lnum)
		     (cons lnum (id->string l-name decl)))))
	     ((?l-name ?proto ?c-name)
	      (let ((lnum (line-number decl)))
		 (if (number? lnum)
		     (cons lnum (id->string l-name decl)))))
	     ((?l-name ?c-name)
	      (let ((lnum (line-number decl)))
		 (if (number? lnum)
		     (cons lnum (id->string l-name decl))))))))
   (btags-clause-walker extern-parse clause))

;*---------------------------------------------------------------------*/
;*    btags-expression ...                                             */
;*---------------------------------------------------------------------*/
(define (btags-expression expr)
   (match-case expr
      ((begin . ?rest)
       (apply append (map btags-expression rest)))
      (((or define define-inline define-generic
	    define-method define-macro define-expander)
	(?fun . ?-) . ?-)
       (let ((lnum (line-number expr)))
	  (if (number? lnum)
	      (list (cons lnum (id->string fun expr)))
	      '())))
      (((or define define-struct) (and (? symbol?) ?var) . ?-)
       (let ((lnum (line-number expr)))
	  (if (number? lnum)
	      (list (cons lnum (id->string var expr)))
	      '())))
      (else
       '())))

	  
       

