;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bde/bmake/bmake.scm                  */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri May 29 07:25:21 1998                          */
;*    Last change :  Fri Dec  4 12:45:05 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The tools that update (or build) Makefiles.                      */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module bmake
   (include "include/args.sch")
   (import  (bmake_template "bmake/template.scm"))
   (eval    (export *source-suffixes*)
	    (export *target-name*)
	    (export *project-name*)
	    (export *template-name*)
	    (export *makefile-name*)
	    (export *mco?*)
	    (export *verbose*)
	    (export *make*))
   (main    main))

;*---------------------------------------------------------------------*/
;*    Global user parameters ...                                       */
;*---------------------------------------------------------------------*/
(define *bmake-version*          "0.0")
(define *search-path*            '("."))
(define *exclude-path*           '())

(define *object-entry*           '())
(define *source-entry*           '())
(define *heap-entry*             #f)
(define *main-entry*             #f)

(define *source-suffixes*        '("scm" "sch" "bgl"))

(define *load-rc?*               #t)

;; rc parameters
(define *makefile-name*          #f)
(define *template-name*          'application)
(define *project-name*           (prefix (basename (pwd))))
(define *target-name*            (prefix (basename (pwd))))
(define *mco?*                   #t)
(define *verbose*                #f)
(define *make*                   "make")

;*---------------------------------------------------------------------*/
;*    parse-args ...                                                   */
;*---------------------------------------------------------------------*/
(define (parse-args cmd-args)
   (define (usage args-parse-usage level)
      (print "usage: bmake [options] [file]")
      (newline)
      (args-parse-usage)
      (newline)
      (exit 0))
   
   (args-parse (cdr cmd-args)
      
      ;; miscellaneous
      (section "Misc")
      (("?")
       (usage args-parse-usage 1))
      (("-help" (synopsis "This help message"))
       (usage args-parse-usage 1))
      (("-v" (synopsis "Be verbose"))
       (set! *verbose* #t))
      (("-q" (synopsis "Do not load the ~/.bmakerc file"))
       (set! *load-rc?* #f))
      (("-suffix" ?suf (synopsis "Add source suffixes [default: scm sch bgl]"))
       (set! *source-suffixes* (cons suf *source-suffixes*)))
      (("-mco" (synopsis "-[no_]mco" "Enable [disable] mco rules production"))
       (set! *mco?* #t))
      (("-no_mco")
       (set! *mco?* #f))

      ;; source search path
      (section "Search path")
      (("-I?dir" (synopsis "Add DIR to the search path [default: .]"))
       (if (directory? dir)
	   (set! *search-path* (cons dir *search-path*))
	   (warning "bmake" "Can't find directory -- " dir)))
      (("-I" ?dir)
       (if (directory? dir)
	   (set! *search-path* (cons dir *search-path*))
	   (warning "bmake" "Can't find directory -- " dir)))
      (("-X?dir" (synopsis "Prevent the source search process to enter DIR"))
       (set! *exclude-path* (cons dir *exclude-path*)))
      (("-X" ?dir)
       (set! *exclude-path* (cons dir *exclude-path*)))

      ;; target file and project name
      (section "Makefile, Target and Project name")
      (("-o" ?name (synopsis "Set makefile target name"))
       (set! *makefile-name* name))
      (("-p" ?name (synopsis "Set the project name"))
       (set! *project-name* name))
      (("-t" ?name (synopsis "Set the target name"))
       (set! *target-name* name))

      ;; template Makefile
      (section "Template")
      (("-f?tpl" (synopsis "-f<template>" "The template for Makefile creation"))
       (cond
	  ((string=? tpl "application")
	   (set! *template-name* 'application))
	  ((string=? tpl "library")
	   (set! *template-name* 'library))
	  (else
	   (if (not (file-exists? tpl))
	       (error "bmake" "Can't find Makefile template" tpl)
	       (set! *template-name* tpl)))))
	  
      ;; sources
      (section "Sources and Objects")
      (("-object" ?obj (synopsis "Add an entry to the object list"))
       (set! *object-entry* (cons obj *object-entry*)))
      (("-source" ?src (synopsis "Add an entry to the source list"))
       (set! *source-entry* (cons src *source-entry*)))
      (("-main" ?entry (synopsis "Set the main entry point"))
       (set! *main-entry* entry))
      (("-heap" ?entry (synopsis "Set the heap entry point"))
       (set! *heap-entry* entry))
      (else
       (set! *main-entry* else))))

;*---------------------------------------------------------------------*/
;*    main ...                                                         */
;*---------------------------------------------------------------------*/
(define (main argv)
   ;; we parse command line arguments
   (parse-args argv)
   ;; we now may enter the true processing
   (engine))

;*---------------------------------------------------------------------*/
;*    with-makefile-output ...                                         */
;*---------------------------------------------------------------------*/
(define-macro (with-makefile-output expr)
   `(if (string? *makefile-name*)
	(with-output-to-file *makefile-name*
	   (lambda () ,expr))
	(with-output-to-port (current-output-port)
	   (lambda () ,expr))))

;*---------------------------------------------------------------------*/
;*    engine ...                                                       */
;*---------------------------------------------------------------------*/
(define (engine) 
   (cond
      ((and (string? *main-entry*) (string? *heap-entry*))
       (error "bmake" "You can't set both main and heap entry" *main-entry*))
      ((and (not (string? *main-entry*))
	    (not (string? *heap-entry*))
	    (null? *source-entry*)
	    (null? *object-entry*))
       ;; this is just a simple template dump
       (let ((port (open-input-template)))
	  (if (not (input-port? port))
	      (error "bmake" "Can't open template" *template-name*)
	      (unwind-protect
		 (with-makefile-output (dump-template port))
		 (close-input-port port)))))
      ((and (not (string? *main-entry*)) (not (string? *heap-entry*)))
       ;; this is just an addition of some entries
       (let* ((old-objects (get-makefile-list "objects"))
	      (old-sources (get-makefile-list "sources")))
	  (update-makefile
	   (object-sort
	    (uniquify! (append *object-entry* old-objects)))
	   (uniquify! (append *source-entry* old-sources)))))
      ((and (string? *makefile-name*) (file-exists? *makefile-name*))
       ;; this is an udpate because the Makefile already exists
       ;; we start searching all the Bigloo source files
       (find-directory-sources)
       ;; we fetch old Makefile fields
       (let* ((old-objects (get-makefile-list "objects"))
	      (old-source (get-makefile-list "sources"))
	      (bigloo-sources (find-bigloo-entries))
	      (new-source (append *source-entry* bigloo-sources)))
	  (update-makefile
	   (object-sort
	    (uniquify! (append *object-entry*
			       (bigloo-entries->bigloo-object bigloo-sources)
			       old-objects)))
	   (uniquify! (append *source-entry* bigloo-sources old-source)))))
      (else
       ;; this is a pure Makefile creation
       ;; we start searching all the Bigloo source files
       (find-directory-sources)
       (let* ((bigloo-sources (find-bigloo-entries))
	      (new-source (append *source-entry* bigloo-sources)))
	  (with-makefile-output
	   (create-makefile
	    *project-name*
	    *target-name*
	    (object-sort
	     (uniquify!
	      (append *object-entry*
		      (bigloo-entries->bigloo-object bigloo-sources))))
	    (uniquify! (append *source-entry* bigloo-sources))))))))

;*---------------------------------------------------------------------*/
;*    open-input-template ...                                          */
;*---------------------------------------------------------------------*/
(define (open-input-template)
   (case *template-name*
      ((application)
       (open-input-string *application-template*))
      ((library)
       (open-input-string *library-template*))
      (else
       (open-input-file *template-name*))))

;*---------------------------------------------------------------------*/
;*    dump-template ...                                                */
;*---------------------------------------------------------------------*/
(define (dump-template port)
   (let loop ((line (read-line port)))
      (if (not (eof-object? line))
	  (begin
	     (print line)
	     (loop (read-line port))))))

;*---------------------------------------------------------------------*/
;*    get-makefile-list ...                                            */
;*---------------------------------------------------------------------*/
(define (get-makefile-list field)
   (let* ((cmd  (string-append "| make -s get" field))
	  (port (open-input-file cmd)))
      (let loop ((exp (read-of-strings port))
		 (lst '()))
	 (if (eof-object? exp)
	     (begin
		(close-input-port port)
		lst)
	     (loop (read-of-strings port)
		   (cons exp lst))))))
      
;*---------------------------------------------------------------------*/
;*    bigloo-source? ...                                               */
;*---------------------------------------------------------------------*/
(define (bigloo-source? fname)
   (member (suffix fname) *source-suffixes*))

;*---------------------------------------------------------------------*/
;*    read-module-name ...                                             */
;*    -------------------------------------------------------------    */
;*    Read from a Bigloo source file the module name.                  */
;*---------------------------------------------------------------------*/
(define (read-module-name fname)
   (let ((iport (open-input-file fname)))
      (if (input-port? iport)
	  (unwind-protect
	     (try (let ((module (read iport)))
		     (match-case module
			((module ?name . ?-)
			 name)
			(else
			 #f)))
		  (lambda (escape obj proc msg)
		     (escape #f)))
	     (close-input-port iport))
	  (error "bmake" "Can't open file for input" fname))))

;*---------------------------------------------------------------------*/
;*    find-directory-sources ...                                       */
;*    -------------------------------------------------------------    */
;*    This function walks trought the directory structure in order     */
;*    to find all Bigloo sources.                                      */
;*---------------------------------------------------------------------*/
(define (find-directory-sources)
   (if *verbose*
       (fprint (current-error-port) "Scanning for source files..."))
   (define (find-all-bigloo-sources/basename basename files)
      (for-each (lambda (file)
		   (let ((fname (string-append basename file)))
		      (if (file-exists? fname)
			  (cond
			     ((directory? fname)
			      (if (not (member fname *exclude-path*))
				  (find-all-bigloo-sources/basename
				   (string-append fname "/")
				   (directory->list fname)))) 
			     ((bigloo-source? fname)
			      (let ((module (read-module-name fname)))
				 (if (symbol? module)
				     (putprop! module 'source fname))))))))
		files))
   (for-each (lambda (dir)
		(find-all-bigloo-sources/basename (string-append dir "/")
						  (directory->list dir)))
	     *search-path*))

;*---------------------------------------------------------------------*/
;*    find-bigloo-entries ...                                          */
;*    -------------------------------------------------------------    */
;*    Make a transitive closure for Bigloo source files. this function */
;*    returns the list of Bigloo source files.                         */
;*---------------------------------------------------------------------*/
(define (find-bigloo-entries)
   (if *verbose*
       (fprint (current-error-port) "Searching for Bigloo source files..."))
   ;; we put each source file inside an hash table. this function makes
   ;; the transitive closure of Bigloo importations. that is, this function
   ;; open bigloo source files to fetch the importation clauses.
   (if (string? *main-entry*)
       ;; when producing an application makefile we add the main to the
       ;; object list
       (add-one-source! *main-entry*)
       ;; when producing a library makefile we do not add the heap file
       ;; to the object list unless one module imports it.
       (add-imported-modules! *heap-entry*))
   (let ((sources '()))
      (for-each-hash (lambda (src)
			(set! sources (cons src sources)))
		     *source-env*)
      (if (and (string? *heap-entry*) (not (member *heap-entry* sources)))
	  (cons *heap-entry* sources)
	  sources)))
   
;*---------------------------------------------------------------------*/
;*    *source-env* ...                                                 */
;*---------------------------------------------------------------------*/
(define *source-env*
   (make-hash-table 1024
		    (lambda (o) (string->0..2^x-1 o 10))
		    (lambda (x) x)
		    string=?
		    64))

;*---------------------------------------------------------------------*/
;*    add-one-source! ...                                              */
;*---------------------------------------------------------------------*/
(define (add-one-source! source)
   ;; we have to removing the head `./' otherwise etags.el will be
   ;; confused when searching for source files.
   (let ((source (if (substring=? source "./" 2)
		     (substring source 2 (string-length source))
		     source)))
      (if (not (get-hash source *source-env*))
	  (begin
	     (put-hash! source *source-env*)
	     (if *verbose*
		 (fprint (current-error-port) source ":"))
	     (add-imported-modules! source)))))

;*---------------------------------------------------------------------*/
;*    add-imported-modules! ...                                        */
;*---------------------------------------------------------------------*/
(define (add-imported-modules! fname)
   (if (file-exists? fname)
       (let ((port (open-input-file fname)))
	  (if (not (input-port? port))
	      (error "bmake" "Can't open file for input" fname)
	      (unwind-protect
		 (try (let ((module (read port)))
			 (match-case module
			    ((module (? symbol?) . ?clauses)
			     (add-imported-modules/clauses! clauses))
			    ((directives . ?clauses)
			     (add-imported-modules/clauses! clauses)
			     clauses)))
		      (lambda (escape obj proc msg)
			 (escape #f)))
		 (close-input-port port))))
       (warning "bmake" "Can't find file -- " fname)))

;*---------------------------------------------------------------------*/
;*    add-imported-modules/clauses! ...                                */
;*---------------------------------------------------------------------*/
(define (add-imported-modules/clauses! clauses)
   (define (add-imported-modules/import! import)
      (match-case import
	 (((and ?module (? symbol?)) (and ?fname (? string?)) . ?rest)
	  ;; (module-name "file-name" ...)
	  (add-one-source! fname))
	 (((and ?var (? symbol?)) (and ?module (? symbol?)))
	  ;; (variable module-name)
	  (let ((source (getprop module 'source)))
	     (if (string? source)
		 (add-one-source! source))))
	 (((? symbol?) (? symbol?) (and ?fname (? string?)) . ?rest)
	  (add-one-source! fname))
	 ((and ?module (? symbol?))
	  ;; module-name
	  (let ((source (getprop module 'source)))
	     (if (string? source)
		 (add-one-source! source))))))
   (define (add-imported-modules/clause! clause)
      (match-case clause
	 (((or use import) . ?imports)
	  (for-each add-imported-modules/import! imports))
	 ((include . ?fnames)
	  (for-each add-imported-modules! fnames))))
   (for-each add-imported-modules/clause! clauses))

;*---------------------------------------------------------------------*/
;*    uniquify! ...                                                    */
;*    -------------------------------------------------------------    */
;*    This function removes multiple occurrences of the string         */
;*    members of LST.                                                  */
;*---------------------------------------------------------------------*/
(define (uniquify! lst)
   (let ((mark (gensym)))
      ;; we mark the strings
      (for-each (lambda (str) (putprop! (string->symbol str) mark #t)) lst)
      ;; we collect them
      (let loop ((lst lst)
		 (res '()))
	 (if (null? lst)
	     res
	     (let ((sym (string->symbol (car lst))))
		(if (getprop sym mark)
		    (begin
		       (remprop! sym mark)
		       (loop (cdr lst)
			     (cons (car lst) res)))
		    (loop (cdr lst) res)))))))

;*---------------------------------------------------------------------*/
;*    bigloo-entries->bigloo-object ...                                */
;*    -------------------------------------------------------------    */
;*    From a source file list this function builds an object file      */
;*    list.                                                            */
;*---------------------------------------------------------------------*/
(define (bigloo-entries->bigloo-object sources)
   (if (not *mco?*)
       (map (lambda (src)
	       (string-append (prefix src) ".o"))
	    sources)
       (let loop ((sources sources)
		  (res     '()))
	  (if (null? sources)
	      res
	      (let* ((src (car sources))
		     (base (prefix src)))
		 (loop (cdr sources)
		       (cons (string-append base ".mco")
			     (cons (string-append base ".o")
				   res))))))))
   
;*---------------------------------------------------------------------*/
;*    object-sort ...                                                  */
;*    -------------------------------------------------------------    */
;*    If mco is enabled we sort the object list for mco object to      */
;*    be first.                                                        */
;*---------------------------------------------------------------------*/
(define (object-sort objects)
   (if (not *mco?*)
       objects
       (let loop ((objects objects)
		  (mco     '())
		  (o       '()))
	  (cond
	     ((null? objects)
	      (append mco o))
	     ((string=? (suffix (car objects)) "mco")
	      (loop (cdr objects)
		    (cons (car objects) mco)
		    o))
	     (else
	      (loop (cdr objects)
		    mco
		    (cons (car objects) o)))))))

;*---------------------------------------------------------------------*/
;*    object->objects_p ...                                            */
;*    -------------------------------------------------------------    */
;*    Builds the list of the profile objects.                          */
;*---------------------------------------------------------------------*/
(define (objects->objects_p objects)
   (map (lambda (obj)
	   (if (string=? (suffix obj) "mco")
	       (string-append (prefix obj) "_p.mco")
	       (string-append (prefix obj) "_p.o")))
	objects))

;*---------------------------------------------------------------------*/
;*    sources->sources_c ...                                           */
;*    -------------------------------------------------------------    */
;*    Builds the list of C source files.                               */
;*---------------------------------------------------------------------*/
(define (sources->sources_c sources)
   (let loop ((sources sources)
	      (res     '()))
      (cond
	 ((null? sources)
	  (reverse! res))
	 ((member (suffix (car sources)) *source-suffixes*)
	  (loop (cdr sources)
		(cons (string-append (prefix (car sources)) ".c") res)))
	 (else
	  (loop (cdr sources)
		res)))))

;*---------------------------------------------------------------------*/
;*    skip-until-eol ...                                               */
;*---------------------------------------------------------------------*/
(define (skip-until-eol port)
   (let ((grammar (regular-grammar ()
		     (#\\
		      (ignore))
		     ((: #\\ #\Newline)
		      (ignore))
		     (#\Newline
		      'done) 
		     ((+ (out #\Newline #\\))
		      (ignore)))))
      (read/rp grammar port)))

;*---------------------------------------------------------------------*/
;*    update-makefile ...                                              */
;*---------------------------------------------------------------------*/
(define (update-makefile objects source)
   ;; first we have to rename the old Makefile
   (let ((old-name (string-append *makefile-name* "~"))
	 (update-grammar (regular-grammar ((blank (out #\tab #\space #\Newline)))
			    ((: "OBJECTS" (+ (in #\tab #\space)) #\=)
			     (print-makefile-list (the-string) objects)
			     (skip-until-eol (current-input-port))
			     (ignore))
			    ((: "OBJECTS_P" (+ (in #\tab #\space)) #\=)
			     (print-makefile-list (the-string)
						  (objects->objects_p objects))
			     (skip-until-eol (current-input-port))
			     (ignore))
			    ((: "SOURCES" (+ (in #\tab #\space)) #\=)
			     (print-makefile-list (the-string) source)
			     (skip-until-eol (current-input-port))
			     (ignore))
			    ((: "SOURCES_C" (+ (in #\tab #\space)) #\=)
			     (print-makefile-list (the-string)
						  (sources->sources_c source))
			     (skip-until-eol (current-input-port))
			     (ignore))
			    ((+ blank)
			     (display (the-string))
			     (ignore))
			    ((+ (in #\Newline #\tab #\space))
			     (display (the-string))
			     (ignore))
			    (else
			     (let ((c (the-failure)))
				(if (eof-object? c)
				    c
				    (error "bmake"
					   "Illegal char"
					   (string #\[ c #\]))))))))
      (rename-file *makefile-name* old-name)
      (with-makefile-output
       (with-input-from-file old-name
	  (lambda ()
	     (read/rp update-grammar (current-input-port)))))))

;*---------------------------------------------------------------------*/
;*    print-makefile-list ...                                          */
;*---------------------------------------------------------------------*/
(define (print-makefile-list field lst)
   (wprin field #\space)
   (for-each (lambda (el)
		(let ((len (+fx 1 (string-length el))))
		   (if (>fx len 73)
		       (wprint el #" \\\n      ")
		       (begin
			  (if (> (+fx *wcol* len) 73)
			      (begin
				 (wfill-to-column 73 #\space)
				 (wprin #"\\\n      ")))
			  (wprin el " ")))))
	     lst)
   (wnewline))
  
;*---------------------------------------------------------------------*/
;*    create-makefile ...                                              */
;*---------------------------------------------------------------------*/
(define (create-makefile project target objects source)
   (let* ((port (open-input-template))
	  (create-grammar (regular-grammar ((blank (out #\$ #\tab #\space #\Newline)))
			     ((: "OBJECTS" (+ (in #\tab #\space)) #\=)
			      (print-makefile-list (the-string) objects)
			      (skip-until-eol port)
			      (ignore))
			     ((: "OBJECTS_P" (+ (in #\tab #\space)) #\=)
			      (print-makefile-list
			       (the-string)
			       (objects->objects_p objects))
			      (skip-until-eol port)
			      (ignore))
			     ((: "SOURCES" (+ (in #\tab #\space)) #\=)
			      (print-makefile-list
			       (the-string)
			       source)
			      (skip-until-eol port)
			      (ignore))
			     ((: "SOURCES_C" (+ (in #\tab #\space)) #\=)
			      (print-makefile-list (the-string)
						   (sources->sources_c source))
			      (skip-until-eol port)
			      (ignore))
			     ((+ blank)
			      (display (the-string))
			      (ignore))
			     ((: #\$ #\[ (+ (in (#\A #\Z) #\-)) #\])
			      (let ((str (the-string)))
				 (cond
				    ((string=? str "$[PROJECT]")
				     (display *project-name*))
				    ((string=? str "$[TARGET-NAME]")
				     (display *target-name*))
				    ((string=? str "$[MAKEFILE]")
				     (display *makefile-name*))
				    ((string=? str "$[ENTRY]")
				     (if (string? *heap-entry*)
					 (display *heap-entry*)
					 (display *main-entry*)))
				    (else
				     (display (the-string)))))
			      (ignore))
			     ((+ (in #\Newline #\tab #\space))
			      (display (the-string))
			      (ignore))
			     (#\$
			      (display (the-string))
			      (ignore))
			     (else
			      (let ((c (the-failure)))
				 (if (eof-object? c)
				     c
				     (error "bmake"
					    "Illegal char"
					    (string #\[ c #\]))))))))
      (if (not (input-port? port))
	  (error "bmake" "Can't open template" *template-name*)
	  (unwind-protect
	     (begin
		(emit-header)
		(read/rp create-grammar port))
	     (close-input-port port)))))

;*---------------------------------------------------------------------*/
;*    Writer variables ...                                             */
;*---------------------------------------------------------------------*/
(define *wcol*  1)

;*---------------------------------------------------------------------*/
;*    wnewline ...                                                     */
;*---------------------------------------------------------------------*/
(define (wnewline)
   (set! *wcol* 1)
   (newline))

;*---------------------------------------------------------------------*/
;*    wdisplay ...                                                     */
;*---------------------------------------------------------------------*/
(define (wdisplay obj)
   (cond
      ((string? obj)
       (let ((len (string-length obj)))
	  (let loop ((i 0))
	     (cond
		((=fx i len)
		 (display obj))
		((char=? (string-ref obj i) #\Newline)
		 (set! *wcol* 1)
		 (loop (+fx i 1)))
		((char=? (string-ref obj i) #\tab)
		 (set! *wcol* (+fx *wcol* (-fx 8 (modulo *wcol* 8))))
		 (loop (+fx i 1)))
		(else
		 (set! *wcol* (+fx *wcol* 1))
		 (loop (+fx i 1)))))))
      ((char? obj)
       (if (char=? obj #\Newline)
	   (set! *wcol* 1)
	   (set! *wcol* (+fx 1 *wcol*)))
       (display obj))
      (else
       (let ((port (open-output-string)))
	  (display obj port)
	  (wdisplay (close-output-port port))))))

;*---------------------------------------------------------------------*/
;*    wprin ...                                                        */
;*---------------------------------------------------------------------*/
(define (wprin . obj)
   (for-each wdisplay obj))
   
;*---------------------------------------------------------------------*/
;*    wprint ...                                                       */
;*---------------------------------------------------------------------*/
(define (wprint . obj)
   (for-each wdisplay obj)
   (wnewline))

;*---------------------------------------------------------------------*/
;*    wfill-to-column ...                                              */
;*---------------------------------------------------------------------*/
(define (wfill-to-column column motif)
   (let loop ()
      (if (<fx *wcol* column)
	  (begin
	     (wprin motif)
	     (loop)))))

;*---------------------------------------------------------------------*/
;*    wcomment ...                                                     */
;*---------------------------------------------------------------------*/
(define (wcomment string)
   (if (<fx (string-length string) 65)
       (begin
	  (wprin "#*    ")
	  (wprin string)
	  (wfill-to-column 72 #\space)
	  (wprint "*/"))
       (begin
	  (wcomment (substring string 0 69))
	  (wcomment (substring string 70 (string-length string))))))
   
;*---------------------------------------------------------------------*/
;*    emit-header ...                                                  */
;*---------------------------------------------------------------------*/
(define (emit-header)
   ;; we emit the Makefile header
   (wprin "#*") (wfill-to-column 72 #\=) (wprint "*/")
   (wcomment (if (string? *makefile-name*) *makefile-name* "stdout"))
   (wcomment "-------------------------------------------------------------")
   (let ((author (getenv "LOGNAME")))
      (wcomment (string-append "Author      :  "
			       (if (string? author)
				   author
				   ""))))
   (let ((date::bstring (date)))
      (string-set! date (-fx (string-length date) 1) #\space)
      (wcomment (string-append "Creation    :  " date)))
   (wcomment "Last change :  ")
   (wcomment "-------------------------------------------------------------")
   (wcomment "Automatically generated file:")
   (wcomment (string-append "   " (car (command-line))))
   (for-each (lambda (arg)
		(wcomment (string-append "      " arg)))
	     (cdr (command-line)))
   (wprin "#*") (wfill-to-column 72 #\=) (wprint "*/"))


	       
