;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bde/bdepend/bdepend.scm              */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun May 31 07:37:29 1998                          */
;*    Last change :  Fri Oct 30 07:29:13 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The Bigloo depend utility.                                       */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module bdepend
   (include "include/args.sch")
   (main    main))

;*---------------------------------------------------------------------*/
;*    Global parameters                                                */
;*---------------------------------------------------------------------*/
(define *bdepend-version*      "0.0")
(define *bdepend-path*         '())
(define *bdepend-exclude-path* '())
(define *bdepend-source-files* '())
(define *bdepend-makefile*     #f)
(define *bdepend-suffixes*     '("scm" "sch" "bgl"))
(define *bdepend-initial-suffixes* *bdepend-suffixes*)
(define *bdepend-verbose*      #f)

(define *bdepend-start-sentinel* "#bdepend start (don't edit)")
(define *bdepend-stop-sentinel*  "#bdepend stop")
(define *bdepend-iport*          #f)
(define *bdepend-mco?*           #t)

;*---------------------------------------------------------------------*/
;*    main ...                                                         */
;*---------------------------------------------------------------------*/
(define (main argv)
   ;; we parse command line arguments
   (parse-args argv)
   ;; we setup default value
   (default-setup)
   (unwind-protect
      (if (not (and (string? *bdepend-makefile*)
		    (file-exists? *bdepend-makefile*)))
	  (begin
	     (start-writer! #f)
	     (generate-depends))
	  (begin
	     (duplicate-makefile-prolog)
	     (generate-depends)
	     (duplicate-makefile-epilogue)))
      (stop-writer!)))

;*---------------------------------------------------------------------*/
;*    default-setup ...                                                */
;*---------------------------------------------------------------------*/
(define (default-setup)
   #f)

;*---------------------------------------------------------------------*/
;*    parse-args ...                                                   */
;*---------------------------------------------------------------------*/
(define (parse-args cmd-args)
   (define (usage args-parse-usage level)
      (print "usage: bdepend [options] file ...")
      (newline)
      (args-parse-usage)
      (newline))
   (args-parse (cdr cmd-args)
      (("?")
       (usage args-parse-usage 1))
      (("-help" (synopsis "This help message"))
       (usage args-parse-usage 1))
      (("-search-path" ?path (synopsis "Add search path"))
       (if (directory? path)
	   (set! *bdepend-path* (cons path *bdepend-path*))
	   (warning "bdepend" "Can't find search directory -- " path)))
      (("-exclude-path" ?path (synopsis "Exclude search path"))
       (if (directory? path)
	   (set! *bdepend-exclude-path* (cons path *bdepend-exclude-path*))
	   (warning "bdepend" "Can't find search directory -- " path)))
      (("-suffix" ?suf (synopsis "Add Bigloo source suffixes"))
       (set! *bdepend-suffixes* (cons suf *bdepend-suffixes*)))
      (("-o" ?name (synopsis "Set makefile name"))
       (set! *bdepend-makefile* name))
      (("-v" (synopsis "Be verbose"))
       (set! *bdepend-verbose* #t))
      (("-fno-mco" (synopsis "Don't produce mco dependences"))
       (set! *bdepend-mco?* #f))
      (("-fmco" (synopsis "Do produce mco dependences"))
       (set! *bdepend-mco?* #t))
      (else
       (set! *bdepend-source-files* (cons else *bdepend-source-files*)))))

;*---------------------------------------------------------------------*/
;*    duplicate-makefile-prolog ...                                    */
;*---------------------------------------------------------------------*/
(define (duplicate-makefile-prolog)
   (let ((svg-name (string-append *bdepend-makefile* "~")))
      (rename-file *bdepend-makefile* svg-name)
      (start-writer! *bdepend-makefile*)
      (let ((iport (open-input-file svg-name)))
	 (if (not (input-port? iport))
	     (error "bdepend" "Can't open file for input" svg-name)
	     (begin
		(let loop ((line (read-line iport)))
		   (cond
		      ((eof-object? line)
		       (close-input-port iport))
		      ((string=? line *bdepend-start-sentinel*)
		       (let loop ((line (read-line iport)))
			  (cond
			     ((eof-object? line)
			      (close-input-port iport))
			     ((string=? line *bdepend-stop-sentinel*)
			      (set! *bdepend-iport* iport))
			     (else
			      (loop (read-line iport))))))
		      (else
		       (wprint line)
		       (loop (read-line iport))))))))))

;*---------------------------------------------------------------------*/
;*    duplicate-makefile-epilogue ...                                  */
;*---------------------------------------------------------------------*/
(define (duplicate-makefile-epilogue)
   (if (input-port? *bdepend-iport*)
       (let loop ((line (read-line *bdepend-iport*)))
	  (if (eof-object? line)
	      (close-input-port *bdepend-iport*)
	      (begin
		 (wprint line)
		 (loop (read-line *bdepend-iport*)))))))

;*---------------------------------------------------------------------*/
;*    Writer variables ...                                             */
;*---------------------------------------------------------------------*/
(define *wport* #f)
(define *wcol*  1)

;*---------------------------------------------------------------------*/
;*    stop-writer! ...                                                 */
;*---------------------------------------------------------------------*/
(define (stop-writer!)
   (if (output-port? *wport*)
       (begin
	  (if (>fx *wcol* 1)
	      (wnewline))
	  (flush-output-port *wport*)
	  (if (not (eq? *wport* (current-output-port)))
	      (close-output-port *wport*)))))

;*---------------------------------------------------------------------*/
;*    wnewline ...                                                     */
;*---------------------------------------------------------------------*/
(define (wnewline)
   (set! *wcol* 1)
   (newline *wport*))

;*---------------------------------------------------------------------*/
;*    wdisplay ...                                                     */
;*---------------------------------------------------------------------*/
(define (wdisplay obj)
   (cond
      ((string? obj)
       (let ((len (string-length obj)))
	  (let loop ((i 0))
	     (cond
		((=fx i len)
		 (display obj *wport*))
		((char=? (string-ref obj i) #\Newline)
		 (set! *wcol* 1)
		 (loop (+fx i 1)))
		(else
		 (set! *wcol* (+fx *wcol* 1))
		 (loop (+fx i 1)))))))
      ((char? obj)
       (if (char=? obj #\Newline)
	   (set! *wcol* 1)
	   (set! *wcol* (+fx 1 *wcol*)))
       (display obj *wport*))
      (else
       (let ((port (open-output-string)))
	  (display obj port)
	  (wdisplay (close-output-port port))))))

;*---------------------------------------------------------------------*/
;*    wprin ...                                                        */
;*---------------------------------------------------------------------*/
(define (wprin . obj)
   (for-each wdisplay obj))
   
;*---------------------------------------------------------------------*/
;*    wprint ...                                                       */
;*---------------------------------------------------------------------*/
(define (wprint . obj)
   (for-each wdisplay obj)
   (wnewline))

;*---------------------------------------------------------------------*/
;*    wfill-to-column ...                                              */
;*---------------------------------------------------------------------*/
(define (wfill-to-column column motif)
   (let loop ()
      (if (<fx *wcol* column)
	  (begin
	     (wprin motif)
	     (loop)))))

;*---------------------------------------------------------------------*/
;*    wcomment ...                                                     */
;*---------------------------------------------------------------------*/
(define (wcomment string)
   (if (<fx (string-length string) 65)
       (begin
	  (wprin "#*    ")
	  (wprin string)
	  (wfill-to-column 72 #\space)
	  (wprint "*/"))
       (begin
	  (wcomment (substring string 0 69))
	  (wcomment (substring string 70 (string-length string))))))
   
;*---------------------------------------------------------------------*/
;*    start-writer! ...                                                */
;*---------------------------------------------------------------------*/
(define (start-writer! file)
   (if (string? file)
       (begin
	  (set! *wport* (open-output-file file))
	  (if (not (output-port? *wport*))
	      (error "start-writer!" "Can't open file for output" file)))
       (set! *wport* (current-output-port))))
   
;*---------------------------------------------------------------------*/
;*    generate-depends ...                                             */
;*---------------------------------------------------------------------*/
(define (generate-depends)
   ;; The object files
   (wprint *bdepend-start-sentinel*)
   ;; the dependencies entry
   (wprin "#*") (wfill-to-column 72 #\-) (wprint "*/")
   (wcomment "Dependencies ...")
   (wprin "#*") (wfill-to-column 72 #\-) (wprint "*/")
   ;; we load all source files
   (find-all-bigloo-sources)
   ;; and we start emitting objects
   (let ((objects (find-all-dependences *bdepend-source-files*)))
      (for-each (lambda (module)
		   (wprin (source->object (car module)) ": ")
		   (for-each (lambda (object)
				(let ((len (+fx 1 (string-length object))))
				   (if (>fx len 74)
				       (wprint object #" \\\n      ")
				       (begin
					  (if (> (+fx *wcol* len) 74)
					      (begin
						 (wfill-to-column 74 #\space)
						 (wprin #"\\\n      ")))
					  (wprin object " ")))))
			     (cdr module))
		   (wnewline))
		objects)
      (wnewline))
   ;; we are done
   (wprint *bdepend-stop-sentinel*))

;*---------------------------------------------------------------------*/
;*    source? ...                                                      */
;*---------------------------------------------------------------------*/
(define (source? fname)
   (member (suffix fname) *bdepend-suffixes*))

;*---------------------------------------------------------------------*/
;*    source->module ...                                               */
;*---------------------------------------------------------------------*/
(define (source->module fname)
   (let ((iport (open-input-file fname)))
      (if (input-port? iport)
	  (unwind-protect
	     (try (let ((module (read iport)))
		     (match-case module
			((module ?name . ?-)
			 name)
			(else
			 #f)))
		  (lambda (escape obj proc msg)
		     (escape '())))
	     (close-input-port iport))
	  (error "bdepend" "Can't open file for input" fname))))

;*---------------------------------------------------------------------*/
;*    find-all-bigloo-sources ...                                      */
;*---------------------------------------------------------------------*/
(define (find-all-bigloo-sources)
   (if *bdepend-verbose*
       (print "Scanning for source files..."))
   (define (find-all-bigloo-sources/basename basename files)
      (for-each (lambda (file)
		   (let ((fname (string-append basename file)))
		      (if (file-exists? fname)
			  (cond
			     ((directory? fname)
			      (if (not (memq fname *bdepend-exclude-path*))
				  (find-all-bigloo-sources/basename
				   (string-append fname "/")
				   (directory->list fname)))) 
			     ((source? fname)
			      (let ((module (source->module fname)))
				 (if (symbol? module)
				     (putprop! module 'source fname))))))))
		files))
   (find-all-bigloo-sources/basename "" (directory->list "."))
   (for-each (lambda (dir)
		(find-all-bigloo-sources/basename (string-append dir "/")
						  (directory->list dir)))
	     *bdepend-path*))

;*---------------------------------------------------------------------*/
;*    *object-env* ...                                                 */
;*---------------------------------------------------------------------*/
(define *object-env*
   (make-hash-table 1024
		    (lambda (o) (string->0..2^x-1 o 10))
		    (lambda (x) (car x))
		    string=?
		    64))

;*---------------------------------------------------------------------*/
;*    find-all-dependences ...                                         */
;*---------------------------------------------------------------------*/
(define (find-all-dependences source-files)
   (if *bdepend-verbose*
       (print "Generating object list..."))
   (for-each add-one-source! source-files)
   (let ((res '()))
      (for-each-hash (lambda (src)
			(if (pair? (cdr src))
			    (set! res (cons src res))))
		     *object-env*)
      res))

;*---------------------------------------------------------------------*/
;*    add-one-source! ...                                              */
;*---------------------------------------------------------------------*/
(define (add-one-source! source)
   (if (not (get-hash source *object-env*))
       (let ((object (list source)))
	  (if *bdepend-verbose* (print source ":"))
	  (put-hash! object *object-env*)
	  (set-cdr! object (find-imported-modules source))
	  source)))

;*---------------------------------------------------------------------*/
;*    find-imported-modules ...                                        */
;*---------------------------------------------------------------------*/
(define (find-imported-modules fname)
   (if (file-exists? fname)
       (let ((port (open-input-file fname)))
	  (if (not (input-port? port))
	      (error "bdepend" "Can't open file for input" fname)
	      (unwind-protect
		 (try (let ((module (read port)))
			 (match-case module
			    ((module (? symbol?) . ?clauses)
			     (find-imported-modules/clauses clauses))
			    ((directives . ?clauses)
			     (find-imported-modules/clauses clauses))
			    (else
			     '())))
		      (lambda (escape obj proc msg)
			 (escape '())))
		 (close-input-port port))))
       '()))

;*---------------------------------------------------------------------*/
;*    source->object ...                                               */
;*---------------------------------------------------------------------*/
(define (source->object src)
   (string-append (prefix src) ".o"))

;*---------------------------------------------------------------------*/
;*    source->mco ...                                                  */
;*---------------------------------------------------------------------*/
(define (source->mco src)
   (string-append (prefix src) ".mco"))

;*---------------------------------------------------------------------*/
;*    find-imported-modules/clauses ...                                */
;*---------------------------------------------------------------------*/
(define (find-imported-modules/clauses clauses)
   (define (find-imported-modules/import import)
      (match-case import
	 (((and ?module (? symbol?)) (and ?fname (? string?)) . ?rest)
	  ;; (module-name "file-name" ...)
	  (let ((source (add-one-source! fname)))
	     (if *bdepend-mco?*
		 (list (source->mco source))
		 '())))
	 (((and ?var (? symbol?)) (and ?module (? symbol?)))
	  ;; (variable module-name)
	  (let ((source (getprop module 'source)))
	     (if (string? source)
		 (begin
		    (add-one-source! source)
		    (if *bdepend-mco?*
			(list (source->mco source))
			'())))))
	 (((? symbol?) (? symbol?) (and ?fname (? string?)) . ?rest)
	  (add-one-source! fname)
	  (if *bdepend-mco?*
	      (list (source->object fname) (source->mco fname))
	      (list (source->object fname))))
	 ((and ?module (? symbol?))
	  ;; module-name
	  (let ((source (getprop module 'source)))
	     (if (string? source)
		 (begin
		    (add-one-source! source)
		    (if *bdepend-mco?*
			(list (source->mco source))
			'())))))
	 (else
	  '())))
   (define (find-imported-modules/clause clause)
      (match-case clause
	 (((or use import) . ?imports)
	  (let loop ((imports imports)
		     (res     '()))
	     (if (null? imports)
		 res
		 (let ((aux (find-imported-modules/import (car imports))))
		    (loop (cdr imports)
			  (if (pair? aux)
			      (append aux res)
			      res))))))
	 ((include . ?fnames)
	  (for-each add-one-source! fnames)
	  fnames)
	 (else
	  '())))
   (let loop ((clauses clauses)
	      (res     '()))
      (if (null? clauses)
	  res
	  (loop (cdr clauses)
		(append (find-imported-modules/clause (car clauses)) res)))))

   

