/*---------------------------------------------------------------------*/
/*   A pratical implementation for the Scheme programming language     */
/*                                                                     */
/*                                    ,--^,                            */
/*                              _ ___/ /|/                             */
/*                          ,;'( )__, ) '                              */
/*                         ;;  //   L__.                               */
/*                         '   \\   /  '                               */
/*                              ^   ^                                  */
/*                                                                     */
/*   Copyright (c) 1992-1999 Manuel Serrano                            */
/*                                                                     */
/*     Bug descriptions, use reports, comments or suggestions are      */
/*     welcome. Send them to                                           */
/*       bigloo-request@kaolin.unice.fr                                */
/*       http://kaolin.unice.fr/bigloo                                 */
/*                                                                     */
/*   This program is free software; you can redistribute it            */
/*   and/or modify it under the terms of the GNU General Public        */
/*   License as published by the Free Software Foundation; either      */
/*   version 2 of the License, or (at your option) any later version.  */
/*                                                                     */
/*   This program is distributed in the hope that it will be useful,   */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/*   GNU General Public License for more details.                      */
/*                                                                     */
/*   You should have received a copy of the GNU General Public         */
/*   License along with this program; if not, write to the Free        */
/*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
/*   MA 02111-1307, USA.                                               */
/*---------------------------------------------------------------------*/
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/bdb/Lib/cbdb.c                       */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Thu Apr  9 17:52:43 1998                          */
/*    Last change :  Thu Jan 28 11:00:25 1999 (serrano)                */
/*    -------------------------------------------------------------    */
/*    The small C interface moslty used because Bigloo produces an     */
/*    huge C array that it is hard to parse in Scheme.                 */
/*=====================================================================*/
#include <bigloo.h>
#include <stdarg.h>
#include <signal.h>
#include <bdb.h>

/*---------------------------------------------------------------------*/
/*    Global function defined inside __bbd module                      */
/*---------------------------------------------------------------------*/
extern obj_t bdb_send_env( obj_t );
extern obj_t bdb_output_eval( obj_t, obj_t );
extern obj_t bdb_no_output_eval( obj_t, obj_t );
extern obj_t bdb_no_output_bool_eval( obj_t, obj_t );
extern obj_t bdb_send_env();
extern obj_t the_c_failure( char *, char *, obj_t );

/*---------------------------------------------------------------------*/
/*    The type used by Bigloo for the bdb table.                       */
/*---------------------------------------------------------------------*/
struct bdb_fun_info { char *sname, *cname; };

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bdb_table_to_list ...                                            */
/*    -------------------------------------------------------------    */
/*    This function construct a list of binding declaration from       */
/*    the table that has been build by the compiler in -gbdb mode.     */
/*---------------------------------------------------------------------*/
obj_t
bdb_table_to_list( obj_t bdb ) {
   obj_t glo_info  = BNIL;
   obj_t mod_info  = BNIL;
   obj_t init_info = BNIL;
   obj_t lnum_info = BNIL;
   struct bdb_fun_info *table_entry = (struct bdb_fun_info *)bdb;
   obj_t src_info = BNIL;
   long mod_lnum;
   
   /* we start fetching the module name and its            */
   /* C initialization function.                           */
   mod_info  = string_to_bstring( table_entry->sname );
   init_info = string_to_bstring( table_entry->cname );
   table_entry++;

   /* and the source files implementing this module.       */
   while( ((int *)table_entry->sname) ) {
      obj_t pair = MAKE_PAIR( string_to_bstring( table_entry->sname ), src_info );
      src_info = pair;
      
      table_entry++;
   }
   
   mod_lnum  = (long)table_entry->cname;
   src_info  = MAKE_PAIR( init_info, src_info );
   lnum_info = MAKE_PAIR( BINT( mod_lnum ), src_info );
   mod_info  = MAKE_PAIR( mod_info, lnum_info );
   table_entry++;
   
   /* then we fetch global variables informations         */
   while( *((int *)table_entry) ) {
      char *fname, *sname, *cname;
      long lnum;
      obj_t pair = BNIL;
      obj_t entry = BNIL;

      /* we first fetch the source file name and line num */
      fname = table_entry->sname;
      lnum  = (long)table_entry->cname;
      table_entry++;

      /* now, we pickup the global scheme and C names     */
      sname = table_entry->sname;
      cname = table_entry->cname;

      /* is it a global function or a global variable ?   */
      if( !cname ) {
	 /* thie is a global function                     */
	 char *bp_cname;
	 obj_t pair2;

	 table_entry++;

	 cname    = table_entry->sname;
	 bp_cname = table_entry->cname;

	 pair2 = MAKE_PAIR( cname ? string_to_bstring( cname ) : BUNSPEC,
			    BINT( lnum ) );
	 pair2 = MAKE_PAIR( pair2, string_to_bstring( bp_cname ) );
	 table_entry++;
	 
	 /* this is a global function, we are now free    */
	 /* to parse the local variables                  */
	 while( table_entry->sname ) {
	    pair = MAKE_PAIR( string_to_bstring( table_entry->sname ),
			      string_to_bstring( table_entry->cname ) );

	    entry = MAKE_PAIR( pair, entry );

	    table_entry++;
	 }

	 pair2 = MAKE_PAIR( pair2, BNIL );
	 pair  = MAKE_PAIR( string_to_bstring( sname ), pair2 );
      } else {
	 /* this is a global variable.                    */
	 pair = MAKE_PAIR( string_to_bstring( sname ),
			   string_to_bstring( cname ) );
      }
      
      entry = MAKE_PAIR( pair, entry );
      entry = MAKE_PAIR( string_to_bstring( fname ), entry );
	 
      table_entry++;
      glo_info = MAKE_PAIR( entry, glo_info );
   }

   return MAKE_PAIR( mod_info, glo_info );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bdb_eval_for_output ...                                          */
/*    -------------------------------------------------------------    */
/*    This function evaluates its first argument in an environment     */
/*    where optional names are bound to optional values.               */
/*    -------------------------------------------------------------    */
/*    This function outputs its result on the running process          */
/*    standard output port.                                            */
/*---------------------------------------------------------------------*/
obj_t bdb_eval_for_output( char *cmd, ... ) {
   obj_t bindings = BNIL;
   char *opt;
   
   va_list argl;
   va_start( argl, cmd );

   while( (opt = va_arg( argl, char * )) ) {
      obj_t binding;
      obj_t value = va_arg( argl, obj_t );

      binding  = MAKE_PAIR( string_to_bstring( opt ), value ? value:BUNSPEC );
      bindings = MAKE_PAIR( binding, bindings );
   }
   
   va_end( argl );

   return bdb_output_eval( string_to_bstring( cmd ), bindings );
}
 
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bdb_eval_for_value ...                                           */
/*    -------------------------------------------------------------    */
/*    This function evaluates its first argument in an environment     */
/*    where optional names are bound to optional values. The result is */
/*    not print out on the string. It is send to the bdb process.      */
/*---------------------------------------------------------------------*/
obj_t bdb_eval_for_value( char *cmd, ... ) {
   obj_t bindings = BNIL;
   char *opt;
   
   va_list argl;
   va_start( argl, cmd );

   while( (opt = va_arg( argl, char * )) ) {
      obj_t binding;
      obj_t value = va_arg( argl, obj_t );

      binding  = MAKE_PAIR( string_to_bstring( opt ), value ? value:BUNSPEC );
      bindings = MAKE_PAIR( binding, bindings );
   }
   
   va_end( argl );

   return bdb_no_output_eval( string_to_bstring( cmd ), bindings );
}
 
/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    bdb_eval_for_bool ...                                            */
/*    -------------------------------------------------------------    */
/*    This function behaves as bdb_eval but it converts its result     */
/*    into a C integer. This function is called when by conditional    */
/*    breakpoints.                                                     */
/*---------------------------------------------------------------------*/
int bdb_eval_for_bool( char *cmd, ... ) {
   obj_t bindings = BNIL;
   char *opt;
   
   va_list argl;
   va_start( argl, cmd );

   while( (opt = va_arg( argl, char * )) ) {
      obj_t binding;
      obj_t value = va_arg( argl, obj_t );

      binding  = MAKE_PAIR( string_to_bstring( opt ), value ? value:BUNSPEC );
      bindings = MAKE_PAIR( binding, bindings );
   }
   
   va_end( argl );
 
   return CBOOL( bdb_no_output_bool_eval( string_to_bstring(cmd), bindings ) );
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    bdb_assert ...                                                   */
/*    -------------------------------------------------------------    */
/*    This function evaluates its result and raise an error if the     */
/*    result is not true.                                              */
/*---------------------------------------------------------------------*/
int bdb_assert( char *cmd, ... ) {
   obj_t bindings = BNIL;
   char *opt;
   
   va_list argl;
   va_start( argl, cmd );

   while( (opt = va_arg( argl, char * )) ) {
      obj_t binding;
      obj_t value = va_arg( argl, obj_t );

      binding  = MAKE_PAIR( string_to_bstring( opt ), value ? value:BUNSPEC );
      bindings = MAKE_PAIR( binding, bindings );
   }
   
   va_end( argl );

   if( CBOOL( bdb_no_output_bool_eval( string_to_bstring(cmd), bindings ) ) )
      return 0;
   else {
      fprintf( stderr, "*** BDB ASSERTION FAILED: %s\n", cmd );
      return 1;
   }
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bdb_procedure_entry ...                                          */
/*    -------------------------------------------------------------    */
/*    This function takes a bigloo object, checks if it is a procedure */
/*    and, if it is, prints its entry point address.                   */
/*---------------------------------------------------------------------*/
void bdb_procedure_entry( obj_t obj ) {
   if( PROCEDUREP( obj ) ) {
      char addr[ 80 ];
      if( PROCEDURE_ARITY( obj ) < 0 )
	 sprintf( addr, "*0x%x", PROCEDURE_VA_ENTRY( obj ) );
      else
	 sprintf( addr, "*0x%x", PROCEDURE_ENTRY( obj ) );

      bdb_send_to_server( string_to_bstring( addr ) );
   } else {
      bdb_send_to_server( BFALSE );
   }
}

/* {*---------------------------------------------------------------------*} */
/* {*    int                                                              *} */
/* {*    bdb_resume_from_record ...                                       *} */
/* {*    -------------------------------------------------------------    *} */
/* {*    In this function we have to wait a little bit for the gdb        *} */
/* {*    attach to be completed.                                          *} */
/* {*---------------------------------------------------------------------*} */
/* int bdb_resume_from_record() {                                      */
/*    printf( "bdb_resume_from_record...\n" );                         */
/*    sleep( 1 );                                                      */
/*    return 1;                                                        */
/* }                                                                   */
/*                                                                     */
/* {*---------------------------------------------------------------------*} */
/* {*    int                                                              *} */
/* {*    bdb_suspend_for_record ...                                       *} */
/* {*---------------------------------------------------------------------*} */
/* int                                                                 */
/* bdb_suspend_for_record() {                                          */
/*    bdb_record_process();                                            */
/* }                                                                   */
/*                                                                     */
/* {*---------------------------------------------------------------------*} */
/* {*    int                                                              *} */
/* {*    bdb_record_process ...                                           *} */
/* {*---------------------------------------------------------------------*} */
/* int bdb_record_process() {                                          */
/*    long n;                                                          */
/*                                                                     */
/*    {* we fork the current process to spawn a gdb *}                 */
/*    switch( n = fork() ) {                                           */
/*       case -1:                                                      */
/*          {* error *}                                                */
/*          fprintf( stderr, "*** ERROR:can't fork process -- %s\n",   */
/*                   strerror( errno ) );                              */
/*          exit( 1 );                                                 */
/*                                                                     */
/*       case 0:                                                       */
/*          {* The child process input/output   *}                     */
/* 	 signal( SIGUSR1, bdb_resume_from_record );                    */
/* 	 puts( "Pausing in child...");                                 */
/* 	 pause();                                                      */
/* 	 puts( "Apres le pause" );                                     */
/* 	 return 1;                                                     */
/*                                                                     */
/*       default:                                                      */
/*          {* The parent process *}                                   */
/*          {                                                          */
/* 	    char aux[ 80 ];                                            */
/*                                                                     */
/* 	    sprintf( aux, "%d %d\n", getpid(), n );                    */
/* 	    sleep( 1 );                                                */
/* 	    printf( "bdb_record_process: %d\n", n );                   */
/*                                                                     */
/* 	    bdb_send_to_server( string_to_bstring( aux ) );            */
/* 	 }                                                             */
/*                                                                     */
/*          return n;                                                  */
/*    }                                                                */
/* }                                                                   */
