;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bdb/Lib/bdb.scm                      */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Wed Apr  8 19:20:15 1998                          */
;*    Last change :  Fri Dec  4 15:29:03 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    This file implements the bdb library.                            */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __bdb
   
   (extern (export bdb-output-eval         "bdb_output_eval")
	   (export bdb-no-output-eval      "bdb_no_output_eval")
	   (export bdb-no-output-bool-eval "bdb_no_output_bool_eval")
	   (export bdb-send-client-env     "bdb_send_client_env")
	   (export bdb-show-client-env     "bdb_show_client_env")
	   (export bdb-send-to-server      "bdb_send_to_server")
	   (export bdb-init-client!        "bdb_init_client")
	   (export bdb-close-client!       "bdb_close_client")
	   (export bdb-display             "bdb_display")
	   (export bdb-display/width       "bdb_display_width")
	   (export bdb-initial-breakpoint  "bdb_initial_breakpoint")
	   (export bdb-failure             "bdb_failure")
	   (export bdb-failure-breakpoint  "bdb_failure_breakpoint")
	   
	   (bdb-table->list::obj (::obj)   "bdb_table_to_list"))
   
   (import (__write-circle "Lib/wcircle.scm"))
   
   (export (bdb-output-eval          ::bstring ::obj)
	   (bdb-no-output-eval       ::bstring ::obj)
	   (bdb-no-output-bool-eval  ::bstring ::obj)
	   (bdb-send-client-env)
	   (bdb-show-client-env)
	   (bdb-send-to-server       ::obj)
	   (bdb-set-module-info!     ::obj ::obj)
	   (bdb-display::int ::obj)
	   (bdb-display/width::int   ::obj ::int)
	   (bdb-failure::obj ::obj   ::obj ::obj)
	   (bdb-initial-breakpoint)
	   (bdb-failure-breakpoint   ::int)
	   (bdb-init-client!::int    ::int)
	   (bdb-close-client!))
   
   (static (class record
	      (pid::long (default 0))
	      (num::long read-only)))
   
   (option (set! *inlining?* #f)))

;* (define *out* (if (file-exists? "BDB.OUT")                          */
;* 		  (append-output-file "BDB.OUT")                       */
;* 		  (open-output-file "BDB.OUT")))                       */

;*---------------------------------------------------------------------*/
;*    *client* ...                                                     */
;*---------------------------------------------------------------------*/
(define *client* #unspecified)
(define (out . value)
   'nop)

;* (define (out . value)                                               */
;*    (for-each (lambda (x)                                            */
;* 		(write-char #\[ *out*)                                 */
;* 		(write x *out*)                                        */
;* 		(write-char #\] *out*))                                */
;* 	     value)                                                    */
;*    (newline *out*)                                                  */
;*    (flush-output-port *out*))                                       */

;*---------------------------------------------------------------------*/
;*    bdb-init-client! ...                                             */
;*    -------------------------------------------------------------    */
;*    This function is called by the server each time the server       */
;*    wants to communicate. There is no way to improve that because    */
;*    there is no way for the server to know that the client is        */
;*    already initialized. In consequence, we have to take care        */
;*    here not to initalize several time the same client.              */
;*---------------------------------------------------------------------*/
(define (bdb-init-client! port-number)
   (out "bdb-init-client!" port-number)
   (if (not (socket? *client*))
       (begin
	  ;; we setup the socket client
	  (set! *client* (make-client-socket "localhost" port-number))
	  (if (not (socket? *client*))
	      (begin
		 (out "CAN'T SETUP CLIENT...")
		 (error "bdb-init-client" "Can't setup client" *client*)))
	  ;; we register its termination
	  (pragma "atexit( (void (*)(void))bdb_close_client )" )))
   ;; we return from this setup
   0)

;*---------------------------------------------------------------------*/
;*    bdb-close-client! ...                                            */
;*---------------------------------------------------------------------*/
(define (bdb-close-client!)
   (out "bdb-close-client!")
   (if (socket? *client*)
       (begin
	  (socket-shutdown *client*)
	  (set! *client* #unspecified))))

;*---------------------------------------------------------------------*/
;*    bdb-initial-breakpoint ...                                       */
;*---------------------------------------------------------------------*/
(define (bdb-initial-breakpoint)
   #unspecified)

;*---------------------------------------------------------------------*/
;*    bdb-failure-breakpoint ...                                       */
;*---------------------------------------------------------------------*/
(define (bdb-failure-breakpoint val)
   (bigloo-exit val))

;*---------------------------------------------------------------------*/
;*    bdb-eval/env ...                                                 */
;*    -------------------------------------------------------------    */
;*    Eval the expression represented by `str' in the environment      */
;*    of `bindings'. If `err?' is #t error are notified.               */
;*---------------------------------------------------------------------*/
(define (bdb-eval/env str::bstring bindings err?)
   (let* ((port (open-input-string str))
	  (expr (read port))
	  (bdgs (map (lambda (binding)
			(list (string->symbol (car binding))
			      `',(cdr binding)))
		     bindings))
	  (form `(let ,bdgs ,expr)))
      (close-input-port port)
      (flush-output-port (current-output-port))
      (let ((value (try (eval form)
			(lambda (escape obj proc msg)
			   (if err? 
			       (notify-error obj proc msg))
			   (escape 'error...)))))
	 value)))
   
;*---------------------------------------------------------------------*/
;*    bdb-output-eval ...                                              */
;*    -------------------------------------------------------------    */
;*    Eval an expression in the environment of `bindings'. Output      */
;*    the (possibly recursive) result to the standard output.          */
;*---------------------------------------------------------------------*/
(define (bdb-output-eval str::bstring bindings)
   (write-circle (bdb-eval/env str bindings #t))
   (newline))

;*---------------------------------------------------------------------*/
;*    bdb-no-output-eval ...                                           */
;*    -------------------------------------------------------------    */
;*    Eval an expression in the environment of `bindings'. No output   */
;*    is performed.                                                    */
;*---------------------------------------------------------------------*/
(define (bdb-no-output-eval str::bstring bindings)
   (let ((value (bdb-eval/env str bindings #f))
	 (port  (open-output-string)))
      (write-circle value port)
      (bdb-send-to-server (close-output-port port))))

;*---------------------------------------------------------------------*/
;*    bdb-no-output-bool-eval ...                                      */
;*    -------------------------------------------------------------    */
;*    Eval an expression in the environment of `bindings'. No output   */
;*    is performed.                                                    */
;*---------------------------------------------------------------------*/
(define (bdb-no-output-bool-eval str::bstring bindings)
   (bdb-eval/env str bindings #f))

;*---------------------------------------------------------------------*/
;*    *bdb-module-info* ...                                            */
;*---------------------------------------------------------------------*/
(define *bdb-module-info* '())

;*---------------------------------------------------------------------*/
;*    bdb-set-module-info! ...                                         */
;*---------------------------------------------------------------------*/
(define (bdb-set-module-info! module-name table)
   (let ((lst (bdb-table->list table)))
      (out "LA TABLE: " lst)
      (set! *bdb-module-info* (cons lst *bdb-module-info*))))

;*---------------------------------------------------------------------*/
;*    bdb-display ...                                                  */
;*---------------------------------------------------------------------*/
(define (bdb-display obj)
   (write-circle obj)
   (newline)
   (flush-output-port (current-output-port))
   0)

;*---------------------------------------------------------------------*/
;*    bdb-display/width ...                                            */
;*---------------------------------------------------------------------*/
(define (bdb-display/width obj width)
   (let ((port (open-output-string)))
      (write-circle obj port)
      (let ((string (close-output-port port)))
	 (if (>fx (string-length string) width)
	     (print (substring string 0 width))
	     (print string))))
   (flush-output-port (current-output-port))
   0)

;*---------------------------------------------------------------------*/
;*    bdb-send-client-env ...                                          */
;*    -------------------------------------------------------------    */
;*    This function send to the running bdb process the complete       */
;*    environment.                                                     */
;*---------------------------------------------------------------------*/
(define (bdb-send-client-env)
   (out "bdb-send-client-env: ")
   (bdb-send-to-server (obj->string *bdb-module-info*)))

;*---------------------------------------------------------------------*/
;*    bdb-show-client-env ...                                          */
;*    -------------------------------------------------------------    */
;*    This function is just for Bdb debugging purposes. It can be      */
;*    called withing the gdb process with a                            */
;*    `call bdb_show_client_env'.                                      */
;*---------------------------------------------------------------------*/
(define (bdb-show-client-env)
   (print "Bdb client env:")
   (for-each print *bdb-module-info*))
   
;*---------------------------------------------------------------------*/
;*    bdb-send-to-server ...                                           */
;*---------------------------------------------------------------------*/
(define (bdb-send-to-server obj) 
   (out "bdb-send-to-server: " (string-length obj))
   (write obj (socket-output *client*))
   (newline (socket-output *client*))
   (flush-output-port (socket-output *client*))
   (bdb-close-client!))
   
;*---------------------------------------------------------------------*/
;*    bdb-failure                                                      */
;*---------------------------------------------------------------------*/
(define (bdb-failure obj proc msg)
   (notify-error obj proc msg))

;*---------------------------------------------------------------------*/
;*    bdb-initialize-error-handling! ...                               */
;*---------------------------------------------------------------------*/
(define (bdb-initialize-error-handling!)
   (add-error-handler! (lambda (escape obj proc msg)
			  (bdb-failure obj proc msg)
			  (escape 1))
		       (lambda (arg)
			  (bdb-failure-breakpoint -1))))
    
(bdb-initialize-error-handling!)

