/*---------------------------------------------------------------------*/
/*   A pratical implementation for the Scheme programming language     */
/*                                                                     */
/*                                    ,--^,                            */
/*                              _ ___/ /|/                             */
/*                          ,;'( )__, ) '                              */
/*                         ;;  //   L__.                               */
/*                         '   \\   /  '                               */
/*                              ^   ^                                  */
/*                                                                     */
/*   Copyright (c) 1992-1999 Manuel Serrano                            */
/*                                                                     */
/*     Bug descriptions, use reports, comments or suggestions are      */
/*     welcome. Send them to                                           */
/*       bigloo-request@kaolin.unice.fr                                */
/*       http://kaolin.unice.fr/bigloo                                 */
/*                                                                     */
/*   This program is free software; you can redistribute it            */
/*   and/or modify it under the terms of the GNU General Public        */
/*   License as published by the Free Software Foundation; either      */
/*   version 2 of the License, or (at your option) any later version.  */
/*                                                                     */
/*   This program is distributed in the hope that it will be useful,   */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
/*   GNU General Public License for more details.                      */
/*                                                                     */
/*   You should have received a copy of the GNU General Public         */
/*   License along with this program; if not, write to the Free        */
/*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
/*   MA 02111-1307, USA.                                               */
/*---------------------------------------------------------------------*/
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/bdb/Gdb/gdb.c                        */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Apr  8 12:03:35 1998                          */
/*    Last change :  Fri Jan  8 15:03:17 1999 (serrano)                */
/*    -------------------------------------------------------------    */
/*    This file implements functions that start and stop and           */
/*    communicate with a gdb child process.                            */
/*=====================================================================*/
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <signal.h>
#include <sys/types.h>
#include <bigloo.h>
#include <bdb.h>

/*---------------------------------------------------------------------*/
/*    Importations                                                     */
/*---------------------------------------------------------------------*/
extern obj_t make_string( long, char );
extern obj_t bdb_emacsp;

/*---------------------------------------------------------------------*/
/*    Control variables                                                */
/*---------------------------------------------------------------------*/
static int gdb_process = 0L;

/*---------------------------------------------------------------------*/
/*    Communication pipe between embedded debugger and parent          */
/*---------------------------------------------------------------------*/
static int to_par[ 2 ], to_child[ 2 ];

/*---------------------------------------------------------------------*/
/*    Global definitions                                               */
/*---------------------------------------------------------------------*/
extern void gdb_send_string( obj_t );
extern obj_t gdb_lisent_result();
extern void gdb_killed( int );

/*---------------------------------------------------------------------*/
/*    A global Bigloo `newline' string.                                */
/*---------------------------------------------------------------------*/
DEFINE_STRING( nl, nl_aux, "\n", 1 );

#define MAX_SENTINEL_LEN 100
#define DEFAULT_SENTINEL "(gdb) "

static char end_output_sentinel[ MAX_SENTINEL_LEN ];
static int  end_output_sentinel_len;

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    gdb_init_process ...                                             */
/*---------------------------------------------------------------------*/
int gdb_init_process( char *gdb, obj_t exec ) {
   int   n;
   char *tty;
   char *ttyarg  = 0;
   char *execarg = 0;
   
   /* we open the gdb connecting pipe  */
   if( !pipe( to_par ) < 0 ) {
      fprintf( stderr, "*** ERROR:can't open pipe -- %s\n",
	       strerror( errno ) );
      exit( 1 );
   }

   if( pipe( to_child ) < 0 ) {
      fprintf( stderr, "*** ERROR:can't open pipe -- %s\n",
	       strerror( errno ) );
      exit( 1 );
   }

   /* we set the gdb tty argument      */
   if( tty = ttyname( 0 ) ) {
      ttyarg = alloca( 5 + strlen( tty ) );
      sprintf( ttyarg, "-tty=%s", tty );
   }

   /* we now set the gdb exec argument */
   if( STRINGP( exec ) )
      execarg = BSTRING_TO_STRING( exec );

   /* we fork the current process to spawn a gdb */
   switch( n = fork() ) {
      case -1:
         /* error */
         fprintf( stderr, "*** ERROR:can't fork process -- %s\n",
		  strerror( errno ) );
	 exit( 1 );
	 
      case 0:
	 /* The child process input/output   */
	 close( 0 );
	 dup( to_child[ 0 ] );
	 close( 1 );
	 dup( to_par[ 1 ] );
	 close( 2 );
	 dup( to_par[ 1 ] );
	 close( to_par[ 0 ] );
	 close( to_par[ 1 ] );
	 close( to_child[ 0 ] );
	 close( to_child[ 1 ] );

	 /* we now start the gdb process     */
	 if( ttyarg ) {
	    if( CBOOL( bdb_emacsp ) )
	       execlp( gdb, gdb, "--quiet", "--fullname", ttyarg, execarg, 0 );
	    else
	       execlp( gdb, gdb, "--quiet", ttyarg, execarg, 0 );
	 } else {
	    if( CBOOL( bdb_emacsp ) )
	       execlp( gdb, gdb, "--quiet", "--fullname", execarg, 0 );
	    else
	       execlp( gdb, gdb, "--quiet", execarg, 0 );
	 }
	 
         /* error */
         fprintf( stderr, "*** ERROR:can't start gdb process\n%s -- %s\n",
		  gdb, strerror( errno ) );
	 exit( 2 );

      default:
	 /* The parent process */
	 gdb_process = n;

	 /* we close useless pipes */
	 close( to_par[ 1 ] );
	 close( to_child[ 0 ] );

	 /* We flush out the gdb output port while */
	 /* setting the gdb end output sentinel    */
	 end_output_sentinel_len = read( to_par[ 0 ],
					 end_output_sentinel,
					 MAX_SENTINEL_LEN );
	 end_output_sentinel[ end_output_sentinel_len ] = '\0';

	 /* we he have read as much caracter as the max allow, it is */
	 /* an error. we stop the execution.                         */
	 if( (end_output_sentinel_len == MAX_SENTINEL_LEN) ||
	     strcmp( end_output_sentinel, DEFAULT_SENTINEL ) ) {
	    fprintf( stderr, "*** ERROR:can't set end sentinel -- %s\n",
		     end_output_sentinel );
	    exit( -2 );
	 }

	 signal( SIGSTOP, gdb_killed );
	 signal( SIGTERM, gdb_killed );
	 signal( SIGABRT, gdb_killed );
	 signal( SIGCHLD, gdb_killed );
	 signal( SIGHUP, gdb_killed );
	 return 1;
   }
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    gdb_killed ...                                                   */
/*---------------------------------------------------------------------*/
void gdb_killed( int signal ) {
   fflush( stderr );
   fflush( stdout );
   puts( "" );
   exit( 0 );
}
   
/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    gdb_kill_process ...                                             */
/*---------------------------------------------------------------------*/
void gdb_kill_process() {
   kill( gdb_process, SIGKILL );
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    gdb_send_string ...                                              */
/*---------------------------------------------------------------------*/
void gdb_send_string( obj_t bstring ) {
   int len = STRING_LENGTH( bstring );
   write( to_child[ 1 ], BSTRING_TO_STRING( bstring ), len );
}

/*---------------------------------------------------------------------*/
/*    debug listen                                                     */
/*---------------------------------------------------------------------*/
#define DEBUG_LISTEN
#undef DEBUG_LISTEN

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    gdb_listen_result ...                                            */
/*---------------------------------------------------------------------*/
obj_t
gdb_listen_result() {
#define BUFSZ 255
   static char *buf   = 0;
   static int   bufsz = BUFSZ;
   
   /* first time, we initialize buf ...      */
   if( !buf ) {
      if( !(buf = malloc( bufsz + 1 )) ) {
	 fprintf( stderr, "*** ERROR:can't allocate buffer -- %s\n",
		  strerror( errno ) );
	 exit( -2 );
      }
   }
   buf[ 0 ] = '\0';

   /* and we listen to the gdb ouput pipe... */
   {
      int count;
      int write = 0;

      while( (count = read( to_par[ 0 ], &buf[ write ], BUFSZ )) >= 0 ) {
	 buf[ write + count ] = '\0';
#if DEBUG_LISTEN
	 printf( "count: %d\n", count );
	 printf( "buf: [%s]\n", buf );
	 printf( "&buf[ write + count ]: [%s]\n",
		 &buf[ write + count - end_output_sentinel_len ] );
	 printf( "sentinel: [%s]\n", end_output_sentinel );
#endif
	 if( ((write + count) >= end_output_sentinel_len) &&
	     !(strncmp( &buf[ write + count - end_output_sentinel_len ],
			end_output_sentinel,
			end_output_sentinel_len - 1)) ) {
#if DEBUG_LISTEN
	    printf( "listen (avant sentinel): [%s]\n", buf );
#endif
	    buf[ write + count - end_output_sentinel_len ] = '\0';
	    break;
	 }
	 write += count;
	 buf[ write ] = '\0';

	 if( (write - bufsz) < BUFSZ ) {
	    /* we have to reallocate the buffer. it is now too small */
	    bufsz += BUFSZ;
	    if( !(buf = realloc( buf, bufsz + 1 )) ) {
	       fprintf( stderr, "*** ERROR:can't allocate buffer -- %s\n",
			strerror( errno ) );
	       exit( -2 );
	    }
	 }
      }
   }

#if DEBUG_LISTEN
   printf( "--> gdb_listen[%s]\n", buf );
#endif
   return string_to_bstring( buf );
}
   
   


