;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bdb/Env/env.scm                      */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun Apr 12 07:46:22 1998                          */
;*    Last change :  Fri Oct 16 08:12:50 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The implementation of the Bdb environments. Bdb uses two         */
;*    environments. One for module and one for global variable. That   */
;*    is, it is possible to have the list of modules in the running    */
;*    process and for each module the list of its global bindings.     */
;*    Each global info contains, in turn, the list of local variables  */
;*    its defines.                                                     */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module env_env

   (export  (class module-info
	       ;; the name of the module
	       (name::bstring read-only)
	       ;; the list of the source file (strings)
	       (sources read-only)
	       ;; the list of global that this module defines
	       (globals (default '()))
	       ;; the name of the C function that initialize this module
	       (init-c-name::bstring read-only))

	    (class file-info
	       ;; the name of the file
	       (name::bstring read-only)
	       ;; the module (name) the file belongs to
	       (module::module-info read-only))

	    (class variable-info
	       ;; the Bigloo identifier for the variable
	       (scm-name::bstring read-only)
	       ;; the name for the C variable for the variable (for function
	       ;; this field contains the name of the C function).
	       (c-name::bstring read-only)
	       ;; the name for the C variable associated to this variable
	       ;; if the scheme variable holds a function.
	       (value-c-name (default #unspecified))
	       ;; is this variable a constant function?
	       (function?::bool (default #f)))

	    (class global-info::variable-info
	       ;; the module this global belongs to
	       (module::module-info read-only)
	       ;; the list of the local this global defines
	       (locals (default '())))

	    (class local-info::variable-info)

	    (class function-info
	       ;; the C name under which this function is know by gdb
	       (c-name::bstring read-only)
	       ;; the Scheme name under which Bigloo knows this function
	       (scm-name::bstring read-only))

	    (class breakpoint-info
	       ;; the hash key for the breakpoints
	       (key::bstring read-only)
	       ;; the function associated to that location
	       (c-name::bstring read-only))

	    (initialize-envs!)
	    
	    (find-module     ::bstring)
	    (find-file       ::bstring)
	    (find-scm-global ::bstring . module)
	    (find-c-global   ::bstring . module)
	    (find-scm-local  ::bstring ::global-info)
	    (find-c-local    ::bstring ::global-info)

	    (find-c-function     ::bstring)
	    (function-breakpoint ::variable-info)

	    (for-each-module ::procedure)
	    (for-each-file   ::procedure)
	    (for-each-global ::procedure)
	    (for-each-local  ::procedure ::global-info)

	    (bind-module-info! ::pair)

	    (get-globals list)

	    (find-breakpoint::bstring ::bstring))

   (import  tools_regexp
	    engine_param
	    command_bwhatis
	    (verbose tools_speek)
	    (gdb-send-for-output engine_engine)))

;*---------------------------------------------------------------------*/
;*    initialize-envs! ...                                             */
;*    -------------------------------------------------------------    */
;*    This function initialize (and reset) all the execution           */
;*    environment.                                                     */
;*---------------------------------------------------------------------*/
(define (initialize-envs!)
   (initialize-module-env!)
   (initialize-file-env!)
   (initialize-global-env!)
   (initialize-function-env!)
   (initialize-breakpoint-env!))

;*---------------------------------------------------------------------*/
;*    get-case-hash ...                                                */
;*---------------------------------------------------------------------*/
(define (get-case-hash name env)
   (get-hash (bdb-case name) env))

;*---------------------------------------------------------------------*/
;*    *module-env* ...                                                 */
;*    -------------------------------------------------------------    */
;*    The module environment.                                          */
;*---------------------------------------------------------------------*/
(define *module-env* #unspecified)

;*---------------------------------------------------------------------*/
;*    initialize-module-env! ...                                       */
;*---------------------------------------------------------------------*/
(define (initialize-module-env!)
   (set! *module-env* (make-hash-table 1024
				       (lambda (o) (string->0..2^x-1 o 10))
				       module-info-name
				       string=?
				       64)))

;*---------------------------------------------------------------------*/
;*    find-module ...                                                  */
;*---------------------------------------------------------------------*/
(define (find-module module::bstring)
   (get-hash (bdb-case module) *module-env*))

;*---------------------------------------------------------------------*/
;*    bind-module! ...                                                 */
;*    -------------------------------------------------------------    */
;*    This function binds a module with an empty list of global. The   */
;*    globals will be added later on.                                  */
;*---------------------------------------------------------------------*/
(define (bind-module! name::bstring cname::bstring file-list lnum)
   (let ((new (instantiate::module-info
		 (name        name)
		 (init-c-name cname)
		 (sources     file-list))))
      ;; we bind the module initialization function
      (bind-c-function! cname
			lnum
			(string-append "(@ " name ")")
			(car file-list))
      ;; and we bind the module itself
      (put-hash! new *module-env*)
      new))

;*---------------------------------------------------------------------*/
;*    for-each-module ...                                              */
;*---------------------------------------------------------------------*/
(define (for-each-module proc::procedure)
   (for-each-hash proc *module-env*))

;*---------------------------------------------------------------------*/
;*    *file-env* ...                                                   */
;*    -------------------------------------------------------------    */
;*    The file environment.                                            */
;*---------------------------------------------------------------------*/
(define *file-env* #unspecified)

;*---------------------------------------------------------------------*/
;*    initialize-file-env! ...                                         */
;*---------------------------------------------------------------------*/
(define (initialize-file-env!)
   (set! *file-env* (make-hash-table 1024
				       (lambda (o) (string->0..2^x-1 o 10))
				       file-info-name
				       string=?
				       64)))

;*---------------------------------------------------------------------*/
;*    find-file ...                                                    */
;*---------------------------------------------------------------------*/
(define (find-file file::bstring)
   (get-hash file *file-env*))

;*---------------------------------------------------------------------*/
;*    bind-file! ...                                                   */
;*    -------------------------------------------------------------    */
;*    This function binds a file with an empty list of global. The     */
;*    globals will be added later on.                                  */
;*---------------------------------------------------------------------*/
(define (bind-file! name::bstring module::module-info)
   (let ((new (instantiate::file-info
		 (name   name)
		 (module module))))
      (put-hash! new *file-env*)
      new))

;*---------------------------------------------------------------------*/
;*    for-each-file ...                                                */
;*---------------------------------------------------------------------*/
(define (for-each-file proc::procedure)
   (for-each-hash proc *file-env*))

;*---------------------------------------------------------------------*/
;*    *global-XXX-env* ...                                             */
;*    -------------------------------------------------------------    */
;*    The global environments. We use to disctincts hash tables. The   */
;*    first one `*global-scm-env*' is used to find a global variable   */
;*    from its Scheme name. The second `*global-c-env*' is used when   */
;*    we are looking for a global variable with its C name.            */
;*---------------------------------------------------------------------*/
(define *global-scm-env* #unspecified)
(define *global-c-env*   #unspecified)

;*---------------------------------------------------------------------*/
;*    initialize-global-env! ...                                       */
;*---------------------------------------------------------------------*/
(define (initialize-global-env!)
   (set! *global-scm-env*
	 (make-hash-table 1024
			  (lambda (o) (string->0..2^x-1 o 10))
			  car
			  string=?
			  64))
   (set! *global-c-env*
	 (make-hash-table 1024
			  (lambda (o) (string->0..2^x-1 o 10))
			  car
			  string=?
			  64)))

;*---------------------------------------------------------------------*/
;*    find-scm-global ...                                              */
;*    -------------------------------------------------------------    */
;*    Find a global variable from its Scheme name.                     */
;*---------------------------------------------------------------------*/
(define (find-scm-global scm-name . module)
   [assert (module) (or (null? module) (module-info? (car module)))]
   (let ((module (if (null? module)
		     '()
		     (car module))))
      (find-global/env get-case-hash *global-scm-env* scm-name module)))

;*---------------------------------------------------------------------*/
;*    find-c-global ...                                                */
;*    -------------------------------------------------------------    */
;*    Find a global from variable from its C name.                     */
;*---------------------------------------------------------------------*/
(define (find-c-global c-name . module)
   [assert (module) (or (null? module) (module-info? (car module)))]
   (let ((module (if (null? module)
		     '()
		     (car module))))
      (find-global/env get-hash *global-c-env* c-name module)))

;*---------------------------------------------------------------------*/
;*    find-global/env ...                                              */
;*---------------------------------------------------------------------*/
(define (find-global/env get-hash env name module)
   (let ((bucket (get-hash name env)))
      (cond
	 ((not (pair? bucket))
	  #f)
	 ((null? (cdr bucket))
	  #f)
	 ((null? module)
	  (cadr bucket))
	 ((not (module-info? module))
	  #f)
	 (else
	  (let loop ((globals (cdr bucket)))
	     (cond
		((null? globals)
		 #f)
		((eq? (global-info-module (car globals)) module)
		 (car globals))
		(else
		 (loop (cdr globals)))))))))

;*---------------------------------------------------------------------*/
;*    bind-global! ...                                                 */
;*---------------------------------------------------------------------*/
(define (bind-global! scm-name c-name::bstring val-c-name fun? module)
   (verbose 2 "   - bind variable: " scm-name "(" c-name ")" #\Newline)
   (let ((global (find-scm-global scm-name module)))
      (if (global-info? global)
	  (error "bdb" "Illegal global redefinition" scm-name)
	  (let ((new        (instantiate::global-info
			       (scm-name     scm-name)
			       (c-name       c-name)
			       (value-c-name val-c-name)
			       (function?    fun?)
			       (module       module)))
		(scm-bucket (get-hash (bdb-case scm-name) *global-scm-env*))
		(c-bucket   (get-hash (bdb-case scm-name) *global-c-env*)))
	     (module-info-globals-set! module (cons new
						    (module-info-globals module)))
	     (if (not (pair? scm-bucket))
		 (put-hash! (list scm-name new) *global-scm-env*)
		 (set-cdr! (cdr scm-bucket) (cons new (cddr scm-bucket))))
	     (if (not (pair? c-bucket))
		 (put-hash! (list c-name new) *global-c-env*)
		 (set-cdr! (cdr c-bucket) (cons new (cddr c-bucket))))
	     new))))

;*---------------------------------------------------------------------*/
;*    for-each-global ...                                              */
;*---------------------------------------------------------------------*/
(define (for-each-global proc::procedure)
   (for-each-hash (lambda (bucket) (for-each proc (cdr bucket)))
		  *global-scm-env*))

;*---------------------------------------------------------------------*/
;*    find-scm-local ...                                               */
;*---------------------------------------------------------------------*/
(define (find-scm-local scm-name::bstring global::global-info)
   (find-local/get-key local-info-scm-name scm-name global))

;*---------------------------------------------------------------------*/
;*    find-c-local ...                                                 */
;*---------------------------------------------------------------------*/
(define (find-c-local c-name::bstring global::global-info)
   (find-local/get-key local-info-c-name c-name global))

;*---------------------------------------------------------------------*/
;*    find-local/get-key ...                                           */
;*---------------------------------------------------------------------*/
(define (find-local/get-key get-key name::bstring global::global-info)
   (let loop ((locals (global-info-locals global)))
      (cond
	 ((null? locals)
	  #f)
	 ((string=? (get-key (car locals)) name)
	  (car locals))
	 (else
	  (loop (cdr locals))))))

;*---------------------------------------------------------------------*/
;*    for-each-local ...                                               */
;*---------------------------------------------------------------------*/
(define (for-each-local proc::procedure global::global-info)
   (for-each proc (global-info-locals global)))

;*---------------------------------------------------------------------*/
;*    bind-module-info! ...                                            */
;*---------------------------------------------------------------------*/
(define (bind-module-info! list)
   (define (match-global pattern mod-info)
      (match-case pattern
	 ;; the pattern that matches global functions
	 ((?fname (?scm-name ((?value-c-name . ?lnum) . ?bp-c-name)) . ?locals)
	  (let ((glo (bind-global! (string-upcase scm-name)
				   bp-c-name
				   value-c-name
				   #t
				   mod-info)))
	     ;; and we mark that this initialization function is a regular
	     ;; scheme function
	     (bind-c-function! bp-c-name lnum scm-name fname)
	     ;; now, in turn, the local variables
	     (global-info-locals-set! glo 
				      (map (lambda (local)
					      (instantiate::local-info
						 (scm-name
						  (string-upcase
						   (car local)))
						 (c-name
						  (cdr local))))
					   locals))))
	 ;; the pattern that matches global variables
	 ((?- (?scm-name . ?c-name))
	  (bind-global! (string-upcase scm-name) c-name c-name #f mod-info))
	 (else
	  (error "bdb" "Illegal info format" pattern))))
   (for-each (match-lambda
		((?module . ?globals)
		 (if (>fx *verbose* 0)
		     (print "- receiving bindings from module " module))
		 (match-case module
		    ((?name ?lnum ?init . ?src)
		     (let ((mod-info (bind-module! (string-upcase name)
						   init
						   src
						   lnum)))
			;; we now bind all the source files
			(for-each (lambda (file)
				     (bind-file! file mod-info))
 				  src)
			;; we now proceed the global variables. we use
			;; two encoding frameworks for global variables.
			;; the first one denotes variables and the second
			;; denotes functions.
			(for-each (lambda (pattern)
				     (match-global pattern mod-info))
				  globals)))
		    (else
		     (error "bdb" "Illegal info format" list))))
		(else
		 (error "bdb" "Illegal info format" list)))
	     list))

;*---------------------------------------------------------------------*/
;*    get-globals ...                                                  */
;*    -------------------------------------------------------------    */
;*    This function returns the list of global variables that are      */
;*    of type obj.                                                     */
;*    -------------------------------------------------------------    */
;*    List must be a symbol list (the list of the global scheme        */
;*    variable names).                                                 */
;*---------------------------------------------------------------------*/
(define (get-globals list)
   (let loop ((list list)
	      (res  '()))
      (if (null? list)
	  res
	  (let ((g-info (find-scm-global (symbol->string (car list)))))
	     (if (and (global-info? g-info)
		      (string? (global-info-value-c-name g-info))
		      (bigloo-object? (global-info-value-c-name g-info)))
		 (loop (cdr list) (cons g-info res))
		 (loop (cdr list) res))))))

;*---------------------------------------------------------------------*/
;*    *function-env* ...                                               */
;*    -------------------------------------------------------------    */
;*    The function environment.                                        */
;*---------------------------------------------------------------------*/
(define *function-env* #unspecified)

;*---------------------------------------------------------------------*/
;*    initialize-function-env! ...                                     */
;*---------------------------------------------------------------------*/
(define (initialize-function-env!)
   (set! *function-env* (make-hash-table 1024
					 (lambda (o) (string->0..2^x-1 o 10))
					 function-info-c-name
					 string=?
					 64)))

;*---------------------------------------------------------------------*/
;*    find-c-function ...                                              */
;*    -------------------------------------------------------------    */
;*    this function search for a Scheme function with C name. This     */
;*    function uses its own hash table.                                */
;*---------------------------------------------------------------------*/
(define (find-c-function c-name::bstring)
   (get-hash c-name *function-env*))

;*---------------------------------------------------------------------*/
;*    function-breakpoint ...                                          */
;*    -------------------------------------------------------------    */
;*    This function takes as input a variable-info and returns the     */
;*    C function associated to this function. The job is easy if the   */
;*    variable is a constant function. It is hard if the variable-info */
;*    is a true variable (non constant).                               */
;*---------------------------------------------------------------------*/
(define (function-breakpoint var-info::variable-info)
   (if (variable-info-function? var-info)
       (variable-info-c-name var-info)
       (bigloo-procedure-entry-point (variable-info-c-name var-info))))

;*---------------------------------------------------------------------*/
;*    bind-c-function! ...                                             */
;*    -------------------------------------------------------------    */
;*    This function binds a module with an empty list of global. The   */
;*    globals will be added later on.                                  */
;*    -------------------------------------------------------------    */
;*    It is mandatory to insert the extra ":" in between the file      */
;*    name and the line number because this help the parsing of        */
;*    the bbreak command (@ generic-bbreak-command command_brreak).    */
;*    -------------------------------------------------------------    */
;*    For the same reason we insert a "*" for address breakpoints.     */
;*    -------------------------------------------------------------    */
;*    When we bind a function we also add its address inside the       */
;*    breakpoint environment. That way breakpoint with address         */
;*    about Bigloo function will be correctly handled by bdb.          */
;*---------------------------------------------------------------------*/
(define (bind-c-function! c-name::bstring lnum::long scm-name::bstring fname)
   (verbose 2 "   - bind function: " scm-name "(" c-name ") "
	    fname ":" lnum #\Newline)
   (let ((new (instantiate::function-info
		 (c-name   c-name)
		 (scm-name scm-name)))
	 (bi  (instantiate::breakpoint-info
		 (key    (string-append fname ":" (integer->string lnum)))
		 (c-name c-name)))
	 (addr (gdb-send-for-output (string-append "info address " c-name))))
      (put-hash! new *function-env*)
      (put-hash! bi *breakpoint-env*)
      (string-case addr
	 ((: (+ all) "ess " (submatch (: "0x" (+ xdigit))) ".")
	  (let* ((address (the-submatch 1))
		 (bi      (instantiate::breakpoint-info
			     (key    (string-append "*" address))
			     (c-name c-name))))
	     (put-hash! bi *breakpoint-env*))))
      new))

;*---------------------------------------------------------------------*/
;*    *breakpoint-env* ...                                             */
;*    -------------------------------------------------------------    */
;*    The environment used to find if a breakpoint is set to a         */
;*    function definition location. Because of the definition of that  */
;*    hash table, a source location can either be a file:lnum or an    */
;*    address in the binary image.                                     */
;*    -------------------------------------------------------------    */
;*    The need for that environment is because of GDB's bugs. GDB      */
;*    appears not to be able to handle correctly breakpoint on         */
;*    line number when used with C # line nor it is able to handle     */
;*    correctly *0xaddress breakpoints. For that two situations GDB    */
;*    get confused and set the breakpoint before $sp and $fp are       */
;*    correctly setup. This is very annoying because the result of     */
;*    this is that arguments have random values when stopping on a     */
;*    function. This makes the Bigloo printer to possibly crashes...   */
;*---------------------------------------------------------------------*/
(define *breakpoint-env* #unspecified)

;*---------------------------------------------------------------------*/
;*    initialize-breakpoint-env! ...                                   */
;*---------------------------------------------------------------------*/
(define (initialize-breakpoint-env!)
   (set! *breakpoint-env* (make-hash-table 1024
					   (lambda (o) (string->0..2^x-1 o 10))
					   breakpoint-info-key
					   string=?
					   64)))

;*---------------------------------------------------------------------*/
;*    find-breakpoint ...                                              */
;*    -------------------------------------------------------------    */
;*    We find in the hash table if we have a function definition       */
;*    for that location. We have to search into the real position      */
;*    (that we get asking gdb), not the user one.                      */
;*---------------------------------------------------------------------*/
(define (find-breakpoint::bstring position::bstring)
   (let* ((pos (string-case position
		  ((: (+ all) #\: (submatch (+ digit)))
		   (let ((num (the-submatch 1)))
		      (let* ((cmd (string-append "info line " position))
			     (out (gdb-send-for-output cmd)))
			 (string-case out 
			    ((: "Line " (+ digit) " of \""
					(submatch (+ (out #\"))))
			     (string-append (the-submatch 1) ":" num))
			    (else
			     position)))))
		  (else
		   position)))
	  (bi (get-hash pos *breakpoint-env*)))
      (if (breakpoint-info? bi)
	  (breakpoint-info-c-name bi)
	  position)))
