;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/out/bigloo1.9e/bdb/Command/file.scm                      */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Apr 17 18:05:18 1998                          */
;*    Last change :  Fri Jan 15 09:59:06 1999 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The file overriden command                                       */
;*    -------------------------------------------------------------    */
;*    When opening a file, we load Bigloo symbols and we check for     */
;*    a file.ass file. If this exists, this file is a configuration    */
;*    file that contains assertions.                                   */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module command_file
   (import tools_read
	   tools_regexp
	   tools_echo
	   engine_param
	   engine_engine
	   engine_server
	   command_command
	   command_bbreak
	   env_env)
   (export (file-command         ::bstring ::obj)
	   (load-bigloo-symbols! ::bstring)
	   (load-assertions!     ::bstring)
	   (get-run-arguments)))

;*---------------------------------------------------------------------*/
;*    file-command ...                                                 */
;*---------------------------------------------------------------------*/
(define (file-command line line-list)
   (match-case line-list
      ((?-)
       (if (string? *exec*)
	   (begin
	      (gdb-send-for-output (string-append "file " *exec*))
	      (load-bigloo-symbols! *exec*))))
      ((?- ?fname)
       (dump-assertions!)
       (set! *exec* fname)
       (gdb-send-for-output line)
       (load-bigloo-symbols! *exec*)
       (load-assertions! *exec*))))

;*---------------------------------------------------------------------*/
;*    load-bigloo-symbols! ...                                         */
;*---------------------------------------------------------------------*/
(define (load-bigloo-symbols! name)
   (gdb-start-echo)
   (unwind-protect 
      (begin
	 (display* "Reading Bigloo symbols from " name "...")
	 (flush-output-port (current-output-port))
	 ;; before anything else we allocate rooms for the environments
	 (initialize-envs!)
	 ;; we now have to reset the server
	 (reset-server!)
	 ;; first we check that symbols have been loaded
	 (cond
	    ((not (regexp-match? "Symbols from" (gdb-send-for-output "info file")))
	     (print "Can't load symbols from \"" name "\"")
	     #f)
	    ((not (regexp-match? "Symbol \"bdb_initial_breakpoint\" is a"
				 (gdb-send-for-output
				  "info addre bdb_initial_breakpoint")))
	     (print "Can't find bdb initial breakpoint \"" name "\"")
	     #f)
	    (else
	     ;; then we set a temporary break point inside the debugger
	     (gdb-send-for-output "tbreak bdb_initial_breakpoint")
	     ;; and we start the dummy execution.
	     (let ((args (get-run-arguments)))
		(gdb-send-for-output "run > /dev/null")
		;; we now check that the process is running
		(let loop ((run (gdb-send-for-output "info stack")))
		   (cond
		      ((substring=? run "No stack" 7)
		       ;; there is a problem
		       (newline)
		       (print "Can't read symbols, aborting.")
		       (print "Link your program with -gbdb -static-bigloo")
		       (print "flags plus your machine dependent option for static linking")
		       #f)
		      ((regexp-match? "bdb_initial_breakpoint" run)
		       ;; since the process is now started we can
		       ;; ask the running process
		       ;; the list of its modules and bindings.
		       (fetch-client-environment)
		       ;; and we just have to kill the gdb
		       ;; child process by now.
		       (gdb-send-for-output "kill")
		       ;; we restore the gdb child tty
		       (gdb-send-for-output
			(string-append "set args " args))
		       ;; we are done for this time
		       (print #"done.")
		       #t)
		      (else
		       (gdb-send-for-output "continue")
		       (loop (gdb-send-for-output "info stack")))))))))
      (gdb-stop-echo)))

;*---------------------------------------------------------------------*/
;*    fetch-client-environment ...                                     */
;*---------------------------------------------------------------------*/
(define (fetch-client-environment)
   (let* ((env       (gdb-send-for-value "call bdb_send_client_env()"))
	  (bdbs-list (string->obj env)))
      (bind-module-info! bdbs-list)))

;*---------------------------------------------------------------------*/
;*    get-run-arguments ...                                            */
;*---------------------------------------------------------------------*/
(define (get-run-arguments)
   (let* ((cmd   "show args")
	  (args  (gdb-send-for-output cmd))
	  (start (string-length
		  "Arguments to give program being debugged when it is started is"))
	  (end   (-fx (string-length args) 2)))
      ;; we real arugment start is the first #\" char
      (let loop ((start start))
	 (cond
	    ((=fx start end)
	     "")
	    ((char=? (string-ref args start) #\")
	     (substring args (+fx 1 start) (-fx end 1)))
	    (else
	     (loop (+fx start 1)))))))

;*---------------------------------------------------------------------*/
;*    load-assertions! ...                                             */
;*---------------------------------------------------------------------*/
(define (load-assertions! ename)
   (delete-assertions!)
   (let ((aname (string-append ename ".ass")))
      (if (file-exists? aname)
	  (load-assertions-file! aname)))
   (let ((aname (string-append (basename ename) ".ass")))
      (if (file-exists? aname)
	  (load-assertions-file! aname))))

;*---------------------------------------------------------------------*/
;*    load-assertions-file! ...                                        */
;*    -------------------------------------------------------------    */
;*    This function checks if the assertion file aname exists.         */
;*    If it exists, it is loaded, that is each defined assertions      */
;*    is setup.                                                        */
;*---------------------------------------------------------------------*/
(define (load-assertions-file! aname)
   (if (file-exists? aname)
       (let ((port (open-input-file aname)))
	  (gdb-start-echo)
	  (print "Reading assertions from " aname "...done.")
	  (gdb-stop-echo)
	  (if (input-port? port)
	      (unwind-protect
		 (let loop ((ass (read port)))
		    (if (not (eof-object? ass))
			(begin
			   (match-case ass
			      (((and (? string?) ?fname)
				(and (? fixnum?) ?line)
				(and (? string?) ?expr))
			       (load-assertion! fname
						(integer->string line)
						expr)))
			   (loop (read port)))))
		 (close-input-port port))))))
