;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bdb/Command/display.scm              */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Wed May  6 14:27:11 1998                          */
;*    Last change :  Wed Aug 26 09:43:28 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    Display implementation.                                          */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module command_display

   (import tools_echo
	   engine_engine
	   command_eval)
   
   (export (bdisplay-command        ::bstring ::pair)
	   (display-command         ::bstring ::pair)
	   (undisplay-command       ::bstring ::pair)
	   (enable-display-command  ::bstring ::pair)
	   (disable-display-command ::bstring ::pair)
	   (info-display-command    ::bstring ::pair)
	   (delete-display-command  ::bstring ::pair)
	   (display-display-command))
   
   (static (class bdb-display
	      (number::int (default 0))
	      (enable?::bool (default #t))
	      (expr::bstring read-only)
	      (old-value (default #unspecified)))
	   
	   (class c-display::bdb-display
	      (c-expr::bstring read-only))
	   
	   (class scm-display::bdb-display
	      (scm-expr::bstring read-only)
	      (env::obj read-only))))

;*---------------------------------------------------------------------*/
;*    bdisplay-command ...                                             */
;*---------------------------------------------------------------------*/
(define (bdisplay-command line line-list)
   (match-case line-list
      ((?-)
       (display-display-command))
      ((?- . ?exprs)
       (let* ((expr (let loop ((exprs (reverse! exprs))
			       (res   ""))
		       (if (null? exprs)
			   res
			   (loop (cdr exprs)
				 (string-append (car exprs) " " res)))))
	      (env (bdb-eval-environment expr))
	      (cmd (string-append "call bdb_eval_for_value( \""
				  (string-for-read expr)
				  "\","
				  env
				  ")")))
	  (let ((bp (instantiate::scm-display
		       (env      (bdb-eval-env-list expr))
		       (scm-expr expr)
		       (expr     cmd))))
	     (display-display (add-bdb-display! bp)))))))

;*---------------------------------------------------------------------*/
;*    display-command ...                                              */
;*---------------------------------------------------------------------*/
(define (display-command line line-list)
   (match-case line-list
      ((?-)
       (info-display-command line line-list))
      ((?- ?expr)
       (let* ((env (bdb-eval-environment expr))
	      (bp  (instantiate::c-display
		      (c-expr expr)
		      (expr   (string-append "print " expr)))))
	  (display-display (add-bdb-display! bp))))
      (else
       (gdb-echo* "A parse error in expression, near `" line #"'.\n"))))

;*---------------------------------------------------------------------*/
;*    undisplay-command ...                                            */
;*---------------------------------------------------------------------*/
(define (undisplay-command line line-list)
   (match-case line-list
      ((?-)
       (set! *bdb-display* '()))
      ((?- . ?nums)
       (for-each (lambda (num)
		    (let ((number (string->integer num)))
		       (rem-bdb-display! number)))
		 nums))))

;*---------------------------------------------------------------------*/
;*    enable-display-command ...                                       */
;*---------------------------------------------------------------------*/
(define (enable-display-command line line-list)
   (enable/disable-display-command line line-list #t))

;*---------------------------------------------------------------------*/
;*    disable-display-command ...                                      */
;*---------------------------------------------------------------------*/
(define (disable-display-command line line-list)
   (enable/disable-display-command line line-list #f))

;*---------------------------------------------------------------------*/
;*    enable/disable-display-command ...                               */
;*---------------------------------------------------------------------*/
(define (enable/disable-display-command line line-list flag)
   (match-case line-list
      ((?-)
       (for-each (lambda (display)
		    (bdb-display-enable?-set! display flag))
		 *bdb-display*))
      ((?- . ?nums)
       (for-each (lambda (num)
		    (let* ((number (string->integer num))
			   (dp     (find-display number)))
		       (if (bdb-display? dp)
			   (bdb-display-enable?-set! dp flag))))
		 nums))))

;*---------------------------------------------------------------------*/
;*    *bdb-display* ...                                                */
;*    -------------------------------------------------------------    */
;*    The list of active display.                                      */
;*---------------------------------------------------------------------*/
(define *bdb-display* '())
(define *bdb-display-num* 0)

;*---------------------------------------------------------------------*/
;*    find-display ...                                                 */
;*---------------------------------------------------------------------*/
(define (find-display number::int)
   (let loop ((dp *bdb-display*))
      (cond
	 ((null? dp)
	  #f)
	 ((=fx (bdb-display-number (car dp)) number)
	  (car dp))
	 (else
	  (loop (cdr dp))))))

;*---------------------------------------------------------------------*/
;*    add-bdb-display! ...                                             */
;*---------------------------------------------------------------------*/
(define (add-bdb-display! display::bdb-display)
   (set! *bdb-display-num* (+fx 1 *bdb-display-num*))
   (bdb-display-number-set! display *bdb-display-num*)
   (set! *bdb-display* (cons display *bdb-display*))
   display)

;*---------------------------------------------------------------------*/
;*    rem-bdb-display! ...                                             */
;*---------------------------------------------------------------------*/
(define (rem-bdb-display! num::int)
   (cond
      ((null? *bdb-display*)
       '())
      ((=fx num (bdb-display-number (car *bdb-display*)))
       (set! *bdb-display* (cdr *bdb-display*)))
      (else
       (let loop ((prev *bdb-display*))
	  (cond ((null? (cdr prev))
		 *bdb-display*)
		((=fx (bdb-display-number (cadr prev)) num)
		 (set-cdr! prev (cddr prev)))
		((<fx (bdb-display-number (cadr prev)) num)
		 (loop prev)))))))
	 
;*---------------------------------------------------------------------*/
;*    object-display ::bdb-display ...                                 */
;*---------------------------------------------------------------------*/
(define-method (object-display dp::bdb-display . port)
   (with-access::bdb-display dp (number enable?)
      (display* number #\:)
      (if (<fx number 10)
	  (display "   ")
	  (if (<fx number 100)
	      (display "   ")
	      (if (<fx number 1000)
		  (display "  ")
		  (display " "))))
      (display (if enable? "y  " "n  "))))

;*---------------------------------------------------------------------*/
;*    object-display ::c-display ...                                   */
;*---------------------------------------------------------------------*/
(define-method (object-display dp::c-display . port)
   (call-next-method)
   (with-access::c-display dp (c-expr)
      (display c-expr)))
	     
;*---------------------------------------------------------------------*/
;*    object-display ::scm-display ...                                 */
;*---------------------------------------------------------------------*/
(define-method (object-display dp::scm-display . port)
   (call-next-method)
   (with-access::scm-display dp (scm-expr)
      (display scm-expr)))
	     
;*---------------------------------------------------------------------*/
;*    info-display-command ...                                         */
;*---------------------------------------------------------------------*/
(define (info-display-command line line-list)
   (if (null? *bdb-display*)
       (gdb-echo #"There are no auto-display expressions now.\n")
       (begin
	  (gdb-start-echo)
	  (print "Auto-display expressions now in effect:")
	  (print "Num Enb Expression")
	  (for-each print (reverse *bdb-display*))
	  (gdb-stop-echo))))
	  
;*---------------------------------------------------------------------*/
;*    display-display ...                                              */
;*    -------------------------------------------------------------    */
;*    This function print out one display                              */
;*---------------------------------------------------------------------*/
(define-generic (display-display disp::bdb-display))

;*---------------------------------------------------------------------*/
;*    display-display ::scm-display ...                                */
;*---------------------------------------------------------------------*/
(define-method (display-display disp::scm-display)
   (define (active? c-var)
      (let* ((cmd (string-append "info address " c-var))
	     (res (gdb-send-for-output cmd)))
	 (not (substring=? res "No" 2))))
   (with-access::scm-display disp (enable? number scm-expr expr old-value env)
      (if enable?
	  ;; before evaluating the Scheme expression value, we have to
	  ;; check that all needed bindings are active
	  (begin
	     (display* number ": " scm-expr " = ")
	     (flush-output-port (current-output-port))
	     (let ((new (let loop ((env env))
			   (cond
			      ((null? env)
			       ;; this ok, all bindings are active, we just
			       ;; have to test that an execution is currently
			       ;; running
			       (let ((run (gdb-send-for-output
					   "info stack")))
				  (if (not (substring=? run "No stack" 7))
				      (let* ((str (gdb-send-for-value expr))
					     (len (string-length str)))
					 (let loop ((i 1))
					    (cond
					       ((>=fx i len)
						str)
					       ((char=? (string-ref str i)
							#\Newline)
						(substring str 0 (-fx i 1)))
					       (else
						(loop (+fx i 1))))))
				      #unspecified)))
			      ((active? (car env))
			       (loop (cdr env)))
			      (else
			       ;; this is not ok, this binding is not active
			       #unspecified)))))
		(if (string? new)
		    (begin
		       (display new)
		       (if (not (equal? new old-value))
			   (begin
			      (if (string? old-value)
				  (print "  (!)")
				  (newline))
			      (set! old-value new))
			   (newline)))
		    (if (string? old-value)
			(print old-value "  (-)")
			(print "<no value in frame>"))))))))

;*---------------------------------------------------------------------*/
;*    display-display ::c-display ...                                  */
;*---------------------------------------------------------------------*/
(define-method (display-display disp::c-display)
   (with-access::c-display disp (enable? number c-expr expr old-value)
      (if enable?
	  (begin
	     (display* number ": " c-expr " = ")
	     (flush-output-port (current-output-port))
	     (let ((new (parse-gdb-print (gdb-send-for-output expr))))
		(if (string? new)
		    (begin
		       (display new)
		       (if (not (equal? new old-value))
			   (begin
			      (if (string? old-value)
				  (print "  (!)")
				  (newline))
			      (set! old-value new))
			   (newline)))
		    (if (string? old-value)
			(print old-value "  (-)")
			(print "<no value in frame>"))))))))

;*---------------------------------------------------------------------*/
;*    parse-gdb-print ...                                              */
;*---------------------------------------------------------------------*/
(define (parse-gdb-print string)
   (let ((len (string-length string)))
      (if (or (=fx len 0) (not (char=? (string-ref string 0) #\$)))
	  #unspecified
	  (let loop ((read 1))
	     ;; we skip the number
	     (if (>=fx read len)
		 ""
		 (if (char-numeric? (string-ref string read))
		     (loop (+fx read 1))
		     (let ((start (+fx read 3)))
			(if (=fx start len)
			    ""
			    (substring string start (-fx len 1))))))))))
		     
;*---------------------------------------------------------------------*/
;*    display-display-command ...                                      */
;*---------------------------------------------------------------------*/
(define (display-display-command)
   (for-each display-display (reverse *bdb-display*)))

;*---------------------------------------------------------------------*/
;*    delete-display-command ...                                       */
;*---------------------------------------------------------------------*/
(define (delete-display-command line line-list)
   (match-case line-list
      ((?- ?-)
       (undisplay-command "undisplay" '("undisplay")))
      ((?- ?- . ?nums)
       (undisplay-command "undisplay" (cons "undisplay" nums)))))
