;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bdb/Command/command.scm              */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun Apr 12 13:19:02 1998                          */
;*    Last change :  Sun Oct 25 07:10:50 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The definition of the bdb commands                               */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module command_command
   
   (import command_completion
	   command_bmodule
	   command_bstack
	   command_bbreak
	   command_bvariables
	   command_blocals
	   command_bprint
	   command_bwhatis
	   command_file
	   command_set
	   command_complete
	   command_commands
	   command_help
	   command_display
	   command_line
	   command_bfunction
	   command_bstep
	   command_replay
	   engine_engine
	   tools_read
	   tools_echo)
   
   (export (class command
	      ;; the full name of the command
	      (name::bstring read-only)
	      ;; the minimal command abbreviation length
	      (abbrev::int read-only)
	      ;; the action to be executed
	      (action::procedure read-only (default command-default-action))
	      ;; the completer for this command
	      (completer::procedure (default (lambda (cmd string) '())))
	      ;; the list of subcommands
	      (sub-commands (default '()))
	      ;; the help of the command
	      (help read-only (default #unspecified)))
			    
	   (initialize-bdb-commands!)
	   (get-command-list)
	   (top-level-line-command?::bool ::obj)
	   (find-one-command ::bstring ::obj)
   	   (find-command ::bstring)))

;*---------------------------------------------------------------------*/
;*    *commands* ...                                                   */
;*    -------------------------------------------------------------    */
;*    This list of all commands (initialy empty).                      */
;*---------------------------------------------------------------------*/
(define *commands* '())

;*---------------------------------------------------------------------*/
;*    initialize-bdb-commands! ...                                     */
;*---------------------------------------------------------------------*/
(define (initialize-bdb-commands!)
   ;; the brun command
   (let ((brun (instantiate::command
		   (name   "brun")
		   (abbrev 3)
		   (action brun-command)
		   (help   #"Start debugged program (see `help run')."))))
      (register-top-level-command! brun))
   ;; the bstep command
   (let ((bstep (instantiate::command
		   (name   "bstep")
		   (abbrev 2)
		   (action bstep-command)
		   (help   "Step program until it reaches a different Bigloo source line (see `help step')."))))
      (register-top-level-command! bstep))
   ;; the bcont command
   (let ((bcont (instantiate::command
		   (name   "bcontinue")
		   (abbrev 2)
		   (action bcont-command)
		   (help   "Continue program being debugged, after signal or breakpoint (see 'help cont')."))))
      (register-top-level-command! bcont))
   ;; the bnext command
   (let ((bnext (instantiate::command
		   (name   "bnext")
		   (abbrev 2)
		   (action bnext-command)
		   (help   "Step program, proceeding through subroutine calls (see `help next')."))))
      (register-top-level-command! bnext))
   ;; the buntil command
   (let ((buntil (instantiate::command
		     (name   "buntil")
		     (abbrev 2)
		     (action buntil-command)
		     (help   #"Execute until the program reaches a source line greater than the current\n
or a specified line or address or function (see `help until')."))))
      (register-top-level-command! buntil))
   ;; the bfinish command
   (let ((bfinish (instantiate::command
		     (name   "bfinish")
		     (abbrev 2)
		     (action bfinish-command)
		     (help   "Execute until selected stack frame returns (see `help finish')."))))
      (register-top-level-command! bfinish))
   ;; breturn command
   (let ((breturn (instantiate::command
		     (name   "breturn")
		     (abbrev 4)
		     (action breturn-command)
		     (help   "Make selected stack frame return to its caller"))))
      (register-top-level-command! breturn))
   ;; the set serie
   (let ((set (instantiate::command
		 (name      "set")
		 (abbrev    2)
		 (completer gdb-sub-command-completer))))
      (register-top-level-command! set)
      (let ((set-case-sensitivity (instantiate::command
				     (name   "case")
				     (abbrev 2)
				     (action set-case-sensitivity-command)
				     (help   "Set case sensitivity"))))
	 (register-sub-command! set-case-sensitivity set)))
   ;; the show serie
   (let ((show (instantiate::command
		  (name      "show")
		  (abbrev    2)
		  (action    show-command)
		  (completer gdb-sub-command-completer))))
      (register-top-level-command! show)
      (let ((show-case-sensitivity (instantiate::command
				      (name   "case")
				      (abbrev 2)
				      (action show-case-sensitivity-command)
				      (help   "Show case sensitivity"))))
	 (register-sub-command! show-case-sensitivity show)))
   ;; the info overriding
   (let ((info (instantiate::command
		  (name       "info")
		  (abbrev     1)
		  (action     info-help-command)
		  (completer  gdb-sub-command-completer))))
      (register-top-level-command! info)
      ;; info stack
      (let ((info-stack (instantiate::command
			   (name   "stack")
			   (abbrev 1)
			   (action info-stack-command))))
	 (register-sub-command! info-stack info))
      ;; info display
      (let ((info-display (instantiate::command
			     (name   "display")
			     (abbrev 1)
			     (action info-display-command))))
	 (register-sub-command! info-display info))
      ;; info bargs
      (let ((info-bargs (instantiate::command
			   (name "bargs")
			   (abbrev 2)
			   (action info-bargs-command))))
	 (register-sub-command! info-bargs info))
      ;; info blocals
      (let ((info-blocals (instantiate::command
			     (name "blocals")
			     (abbrev 2)
			     (action info-blocals-command))))
	 (register-sub-command! info-blocals info))
      ;; info assert
      (let ((info-assert (instantiate::command
			    (name "assert")
			    (abbrev 2)
			    (action info-assert-command))))
	 (register-sub-command! info-assert info)))
   ;; the binfo serie
   (let ((binfo (instantiate::command
		   (name      "binfo")
		   (abbrev    2)
		   (action    binfo-help-command)
		   (completer sub-command-completer)
		   (help      "Generic command for showing things about the program being debugged"))))
      (register-top-level-command! binfo)
      ;; binfo module
      (let ((binfo-module (instantiate::command
			     (name      "module")
			     (abbrev    1)
			     (action    binfo-module-command)
			     (completer bmodule-command-completer)
			     (help      "All modules names, or thos matching REGEXP"))))
	 (register-sub-command! binfo-module binfo))
      ;; binfo stack
      (let ((binfo-stack (instantiate::command
			    (name   "stack")
			    (abbrev 1)
			    (action binfo-stack-command)
			    (help   "Backtrace of the Scheme stack, or innermost COUNT frames"))))
	 (register-sub-command! binfo-stack binfo))
      ;; binfo variables
      (let ((binfo-variable (instantiate::command
			       (name      "variables")
			       (abbrev    1)
			       (action    binfo-variables-command)
			       (completer bvariables-command-completer)
			       (help      "All Scheme global and static variable names, or those matching REGEXP"))))
	 (register-sub-command! binfo-variable binfo))
      ;; binfo functions
      (let ((binfo-variable (instantiate::command
			       (name      "functions")
			       (abbrev    1)
			       (action    binfo-functions-command)
			       (completer bfunctions-command-completer)
			       (help      "All Scheme global and static variable names, or those matching REGEXP"))))
	 (register-sub-command! binfo-variable binfo))
      ;; binfo binding
      (let ((binfo-binding (instantiate::command
			      (name      "bindings")
			      (abbrev    7)
			      (action    binfo-binding-command)
			      (help      "All Scheme bindings at LINESPEC"))))
	 (register-sub-command! binfo-binding binfo))
      ;; binfo args
      (let ((binfo-args (instantiate::command
			   (name      "args")
			   (abbrev    1)
			   (action    binfo-args-command)
			   (help      "Scheme argument variables of current stack frame"))))
	 (register-sub-command! binfo-args binfo))
      ;; binfo locals
      (let ((binfo-locals (instantiate::command
			     (name      "locals")
			     (abbrev    1)
			     (action    binfo-locals-command)
			     (help      "Scheme local variables of current stack frame"))))
	 (register-sub-command! binfo-locals binfo))
      ;; binfo line
      (let ((binfo-line (instantiate::command
			   (name      "line")
			   (abbrev    2)
			   (action    binfo-line-command)
			   (help      "Core addresses of the code for a source line"))))
	 (register-sub-command! binfo-line binfo)))
   ;; the break series
   (let ((break (instantiate::command
		   (name      "break")
		   (abbrev    1)
		   (action    break-command)
		   (help      "Set breakpoint"))))
      (register-top-level-command! break))
   (let ((tbreak (instantiate::command
		    (name      "tbreak")
		    (abbrev    2)
		    (action    break-command)
		    (help      "Set temporary breakpoint"))))
      (register-top-level-command! tbreak))
   ;; the bbreak series
   (let ((bbreak (instantiate::command
		    (name      "bbreak")
		    (abbrev    2)
		    (action    bbreak-command)
		    (completer bbreak-command-completer)
		    (help      "Set breakpoint at specified Scheme function"))))
      (register-top-level-command! bbreak))
   (let ((bbreak (instantiate::command
		    (name      "btbreak")
		    (abbrev    3)
		    (action    btbreak-command)
		    (completer bbreak-command-completer)
		    (help      "Set temporary breakpoint at specified Scheme function"))))
      (register-top-level-command! bbreak))
   ;; bwhatis
   (let ((bbreak (instantiate::command
		    (name   "bwhatis")
		    (abbrev 2)
		    (action bwhatis-command)
		    (help   "Print Scheme data type of Scheme expression"))))
      (register-top-level-command! bbreak))
   ;; file
   (let ((file (instantiate::command
		  (name   "file")
		  (abbrev 3)
		  (action file-command))))
      (register-top-level-command! file))
   ;; complete
   (let ((complete (instantiate::command
		      (name   "complete")
		      (abbrev 3)
		      (action complete-command))))
      (register-top-level-command! complete))
   ;; bprint command
   (let ((bprint (instantiate::command
		    (name   "bprint")
		    (abbrev 3)
		    (action bprint-command)
		    (help   "Print Scheme value"))))
      (register-top-level-command! bprint))
   ;; dprint command
   (let ((dprint (instantiate::command
		    (name   "dprint")
		    (abbrev 3)
		    (action dprint-command)
		    (help   "Print Scheme value of C expression"))))
      (register-top-level-command! dprint))
   ;; cprint command
   (let ((cprint (instantiate::command
		    (name   "cprint")
		    (abbrev 3)
		    (action cprint-command)
		    (help   "Print C value"))))
      (register-top-level-command! cprint))
   ;; commands command
   (let ((commands (instantiate::command
		      (name   "commands")
		      (abbrev 2)
		      (action commands-command))))
      (register-top-level-command! commands))
   ;; quit command
   (let ((quit (instantiate::command
		  (name   "quit")
		  (abbrev 4)
		  (action (lambda (line line-list)
			     (gdb-echo #"bdb quit.\n")
			     (bdb-exit 0)))
		  (help   "Exit bdb"))))
      (register-top-level-command! quit))
   ;; help command
   (let ((help (instantiate::command
		  (name   "help")
		  (abbrev 1)
		  (action help-command))))
      (register-top-level-command! help))
   ;; bcond command
   (let ((bcond (instantiate::command
		   (name   "bcond")
		   (abbrev 2)
		   (action bcond-command)
		   (help   "Specify breakpoint number N to break only if Scheme expression is true"))))
      (register-top-level-command! bcond))
   ;; assert command
   (let ((bassert (instantiate::command
		     (name   "assert")
		     (abbrev 1)
		     (action assert-command)
		     (help   "Specify breakpoint number N to be an assertion"))))
      (register-top-level-command! bassert))
   ;; bdisplay command
   (let ((bdisplay (instantiate::command
		      (name   "bdisplay")
		      (abbrev 2)
		      (action bdisplay-command)
		      (help   "Print Scheme value of expression EXP each time the program stops"))))
      (register-top-level-command! bdisplay))
   ;; display command
   (let ((display (instantiate::command
		     (name   "display")
		     (abbrev 4)
		     (action display-command)
		     (help   "Print value of expression EXP each time the program stops"))))
      (register-top-level-command! display))
   ;; undisplay command
   (let ((undisplay (instantiate::command
		       (name   "undisplay")
		       (abbrev 3)
		       (action undisplay-command)
		       (help   "Cancel some expressions to be displayed when program stops"))))
      (register-top-level-command! undisplay))
   ;; enable
   (let ((enable (instantiate::command
		    (name      "enable")
		    (abbrev    3)
		    (completer sub-command-completer)
		    (help      "Enable some breakpoints"))))
      (register-top-level-command! enable)
      (let ((enable-display (instantiate::command
			       (name   "display")
			       (abbrev 1)
			       (action enable-display-command)
			       (help   "Enable some expressions to be displayed when program stops"))))
	 (register-sub-command! enable-display enable)))
   ;; disable
   (let ((disable (instantiate::command
		     (name      "disable")
		     (completer sub-command-completer)
		     (abbrev    5)
		     (help      "Disable some breakpoints"))))
      (register-top-level-command! disable)
      (let ((disable-display (instantiate::command
				(name   "display")
				(abbrev 1)
				(action disable-display-command)
				(help   "Disable some expressions to be displayed when program stops"))))
	 (register-sub-command! disable-display disable)))
   ;; delete
   (let ((delete (instantiate::command
		    (name      "delete")
		    (abbrev    1)
		    (help      "Delete some breakpoints, assertions or auto-display expressions")
		    (completer sub-command-completer)
		    (action    delete-command))))
      (register-top-level-command! delete)
      (let ((delete-breakpoints (instantiate::command
				   (name   "breakpoints")
				   (abbrev 1)
				   (action delete-breakpoints-command)
				   (help   "Delete some breakpoints"))))
	 (register-sub-command! delete-breakpoints delete))
      (let ((delete-display (instantiate::command
			       (name   "display")
			       (abbrev 1)
			       (action delete-display-command)
			       (help   "Delete some display"))))
	 (register-sub-command! delete-display delete))
      (let ((delete-assertion (instantiate::command
				 (name   "assertion")
				 (abbrev 1)
				 (action delete-assertions-command)
				 (help   "Delete some assertion"))))
	 (register-sub-command! delete-assertion delete)))
   ;; unassert
   (let ((unassert (instantiate::command
		      (name      "unassert")
		      (abbrev    3)
		      (help      "Delete some assertions")
		      (action    unassert-command))))
      (register-top-level-command! unassert)))
;*    ;; record                                                        */
;*    (let ((record (instantiate::command                              */
;* 		    (name   "record")                                  */
;* 		    (abbrev 3)                                         */
;* 		    (help   "Record an execution")                     */
;* 		    (action record-command))))                         */
;*       (register-top-level-command! record))                         */
;*    ;; replay                                                        */
;*    (let ((replay (instantiate::command                              */
;* 		    (name   "replay")                                  */
;* 		    (abbrev 3)                                         */
;* 		    (help   "Replay a recorded execution")             */
;* 		    (action replay-command))))                         */
;*       (register-top-level-command! replay)))                        */
   
;*---------------------------------------------------------------------*/
;*    get-command-list ...                                             */
;*---------------------------------------------------------------------*/
(define (get-command-list)
   *commands*)

;*---------------------------------------------------------------------*/
;*    top-level-line-command? ...                                      */
;*    -------------------------------------------------------------    */
;*    A top level command is a command with only one word.             */
;*---------------------------------------------------------------------*/
(define (top-level-line-command? list)
   (or (null? list) (and (pair? list) (null? (cdr list)))))

;*---------------------------------------------------------------------*/
;*    register-top-level-command! ...                                  */
;*---------------------------------------------------------------------*/
(define (register-top-level-command! command)
   (set! *commands* (cons command *commands*)))

;*---------------------------------------------------------------------*/
;*    register-sub-command! ...                                        */
;*---------------------------------------------------------------------*/
(define (register-sub-command! sub-command command)
   (with-access::command command (sub-commands)
      (set! sub-commands (cons sub-command sub-commands))))

;*---------------------------------------------------------------------*/
;*    find-one-command ...                                             */
;*    -------------------------------------------------------------    */
;*    This function scans the list of command in order to find         */
;*    exaclty one that matches the prefix. If no command is found, we  */
;*    return #f.                                                       */
;*---------------------------------------------------------------------*/
(define (find-one-command cmd::bstring list-of-cmds)
   (let ((len (string-length cmd)))
      (if (and (>fx len 0) (char=? (string-ref cmd (-fx len 1)) #\space))
	  (set! len (-fx len 1)))
      (let loop ((cmds list-of-cmds))
	 (cond
	    ((null? cmds)
	     #f)
	    ((and (substring=? (command-name (car cmds)) cmd len)
		  (>=fx len (command-abbrev (car cmds))))
	     (car cmds))
	    (else
	     (loop (cdr cmds)))))))

;*---------------------------------------------------------------------*/
;*    find-command ...                                                 */
;*    -------------------------------------------------------------    */
;*    This function takes has input a string (the command line) and    */
;*    it computes the command that is applied. If no command is        */
;*    found, this function returns #f.                                 */
;*---------------------------------------------------------------------*/
(define (find-command string)
   (let ((list (string->word-list string)))
      (if (top-level-line-command? list)
	  (values (find-one-command string (get-command-list)) list)
	  (values (find-nested-command string list) list))))

;*---------------------------------------------------------------------*/
;*    find-nested-command ...                                          */
;*---------------------------------------------------------------------*/
(define (find-nested-command string cmd-list)
   (let loop ((cmd-list cmd-list)
	      (cmds     (get-command-list))
	      (cmd      #f))
      (if (null? cmd-list)
	  ;; we have found the last word, the one we complete
	  cmd
	  ;; we have to decode the current command to find the next
	  ;; completer
	  (let ((new-cmd (find-one-command (car cmd-list) cmds)))
	     (if (command? new-cmd)
		 (loop (cdr cmd-list)
		       (command-sub-commands new-cmd)
		       new-cmd)
		 cmd)))))

;*---------------------------------------------------------------------*/
;*    command-default-action ...                                       */
;*---------------------------------------------------------------------*/
(define (command-default-action line line-list)
   (gdb-echo (gdb-send-for-output line)))
