;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bdb/Command/bstep.scm                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sat Oct 10 17:49:34 1998                          */
;*    Last change :  Thu Feb 11 17:36:02 1999 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The bstep command implementation                                 */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module command_bstep
   (export (bstep-command ::bstring ::pair)
	   (bfinish-command line line-list)
	   (buntil-command line line-list)
	   (bnext-command line line-list)
	   (breturn-command line line-list)
	   (bcont-command line line-list)
	   (brun-command line line-list))
   (import command_command
	   env_env
	   (gdb-send-for-output engine_engine)
	   (get-current-global command_bfunction)
	   (gdb-echo tools_echo)
	   (gdb-echo* tools_echo)
	   (bdb-eval-environment command_eval)
	   engine_param
	   (regexp-match? tools_regexp)))

;*---------------------------------------------------------------------*/
;*    brun-command ...                                                 */
;*---------------------------------------------------------------------*/
(define (brun-command line line-list)
   (let* ((cmd (substring line 1 (string-length line)))
	  (out (gdb-send-for-output cmd)))
      (let loop ((stack (gdb-send-for-output "info stack"))
		 (out   out))
	 (cond
	    ((regexp-match? "GC_find_limit" stack)
	     (let ((out (gdb-send-for-output "continue")))
		(loop (gdb-send-for-output "info stack") out)))
	    (else
	     (gdb-echo out))))))

;*---------------------------------------------------------------------*/
;*    bstep-command ...                                                */
;*---------------------------------------------------------------------*/
(define (bstep-command line line-list)
   (match-case line-list
      ((?-)
       (bdb-cmd-then-bigloo "step"))
      ((?- ?val)
       (bdb-cmd-then-bigloo (string-append "step " val)))
      (else
       (gdb-echo (string-append
		  "A parse error in expression, near `"
		  line
		  #"'.\n")))))

;*---------------------------------------------------------------------*/
;*    bcont-command ...                                                */
;*---------------------------------------------------------------------*/
(define (bcont-command line line-list)
   (let ((cmd (substring line 1 (string-length line))))
      (gdb-echo (gdb-send-for-output cmd))))

;*---------------------------------------------------------------------*/
;*    bnext-command ...                                                */
;*---------------------------------------------------------------------*/
(define (bnext-command line line-list)
   (match-case line-list
      ((?-)
       (bdb-cmd-then-bigloo "next"))
      ((?- ?val)
       (bdb-cmd-then-bigloo (string-append "next " val)))
      (else
       (gdb-echo (string-append
		  "A parse error in expression, near `"
		  line
		  #"'.\n")))))

;*---------------------------------------------------------------------*/
;*    buntil-command ...                                               */
;*---------------------------------------------------------------------*/
(define (buntil-command line line-list)
   (match-case line-list
      ((?-)
       (bdb-cmd-then-bigloo "until"))
      ((?- ?val)
       (bdb-cmd-then-bigloo (string-append "until " val)))
      (else
       (gdb-echo (string-append
		  "A parse error in expression, near `"
		  line
		  #"'.\n")))))

;*---------------------------------------------------------------------*/
;*    bfinish-command ...                                              */
;*---------------------------------------------------------------------*/
(define (bfinish-command line line-list)
   (match-case line-list
      ((?-)
       (bdb-cmd-then-bigloo "finish"))
      ((?- ?val)
       (bdb-cmd-then-bigloo (string-append "finish " val)))
      (else
       (gdb-echo (string-append
		  "A parse error in expression, near `"
		  line
		  #"'.\n")))))

;*---------------------------------------------------------------------*/
;*    breturn-command ...                                              */
;*---------------------------------------------------------------------*/
(define (breturn-command line line-list)
   (match-case line-list
      ((?-)
       (bdb-cmd-then-bigloo "return"))
      ((?- ?expr)
       (let* ((env (bdb-eval-environment expr))
	      (cmd (string-append "return bdb_eval( \""
				  (string-for-read expr)
				  "\","
				  env
				  ")")))
	  (bdb-cmd-then-bigloo cmd)))
      (else
       (gdb-echo* "Illegal breturn expression \"" line #"\".\n"))))

;*---------------------------------------------------------------------*/
;*    set-stack-breakpoints! ...                                       */
;*---------------------------------------------------------------------*/
(define (set-stack-breakpoints! add-breakpoint!)
   (define *backtrace-grammar*
      (regular-grammar ((frame-number (: #\# (+ digit)))
			(address      (: #\0 (in #\x #\X) (+ xdigit)))
			(ident        (: (or #\_ alpha)
					 (* (or #\_ alpha digit))))
			(blank        (in #\space #\tab))
			(otherchars   (in #\( #\) #\, #\= #\" #\. #\:
					  #\/ #\- #\+ #\/ #\* #\@ #\< #\>)))
	 
	 ;; first line shape: #XXX id (args-list) at file
	 ((: frame-number (+ blank) (out #\0 blank) (* all))
	  (ignore))
	 
	 ;; following ones: #XXX address in id (args-list) at file:line
	 ((: frame-number (+ blank) address (+ blank) "in" (+ blank)
			  (submatch (+ (out #\space)))
			  (+ blank)
			  #\( (* (out "()")) #\)
			  (? #\Newline)
			  (* blank)
			  "at"
			  (* blank)
			  (? #\Newline)
			  (submatch (+ (out #\:)))
			  ":"
			  (submatch (+ digit)))
	  (let* ((cname (the-submatch 1))
		 (fun-info (find-c-function cname)))
	     (if (function-info? fun-info)
		 (let* ((file (the-submatch 2))
			(line (the-submatch 3))
			(bp   (string-append "break " file ":"
					     (number->string
					      (+ 1 (string->number line))))))
		    ;; when we find a Bigloo in the stack frame we
		    ;; set a breakpoint on the line that follows the
		    ;; call.
		    (add-breakpoint! (gdb-send-for-output bp)))))
	  (ignore))
	 
	 ;; just skip extra newlines or any other char (but not repeated)
	 ((or (+ #\Newline) all)
	  (ignore))
	 
	 (else 'done)))

   ;; first, we build the gdb command that print out the stack trace
   ;; and execute it
   (let* ((gdb-cmd "info stack")
	  (stack   (gdb-send-for-output gdb-cmd)))
      ;; now we parse it
      (if (not (substring=? stack "No stack." 8))
	  (let ((port (open-input-string stack)))
	     (read/rp *backtrace-grammar* port)
	     (close-input-port port)))))

;*---------------------------------------------------------------------*/
;*    bdb-cmd-then-bigloo/breakpoint-policy ...                        */
;*    -------------------------------------------------------------    */
;*    For that function, we set breakpoint inside every Scheme file    */
;*    an we continue the execution.                                    */
;*---------------------------------------------------------------------*/
(define (bdb-cmd-then-bigloo cmd)
   (define (parse-line-number str)
      (string-case str
	 ((: "Line" (+ (or #\Newline #\Tab #\Space))
		    (submatch (+ digit))
		    (+ (or #\Newline #\Tab #\Space))
		    (* (or all #\Newline)))
	  (string->integer (the-submatch 1)))))
   (define (fetch-global-define-line ginfo)
      (parse-line-number
       (gdb-send-for-output (string-append "info line "
					   (global-info-c-name ginfo)))))
   (define (fetch-current-line)
      (parse-line-number (gdb-send-for-output "info line")))
   (let* ((bps      "")
	  (add-breakpoint! (lambda (bp)
			      (string-case bp
				 ((: (* (or all #\Newline))
				     "Breakpoint "
				     (submatch (+ digit))
				     " at")
				  (set! bps
					(string-append (the-submatch 1)
						       " "
						       bps)))))))
      ;; we set a breakpoint to the next sources lines. I have not been
      ;; able to find a correct way to do that. I set several breakpoints
      ;; on the next 4 lines. This is totally heuristic. It has to been
      ;; checked to see if it is sufficient. I think 4 is enough because the
      ;; real problem is when stepping inside an if construction where
      ;; both arms are one expression long...
      (let loop ((i 5))
	 (if (>fx i 0)
	     (let ((bp (string-append "break +" (integer->string i))))
		(add-breakpoint! (gdb-send-for-output bp))
		(loop (-fx i 1)))))
      ;; in addition we have to set breakpoint from the beginning of the
      ;; current to the current source location. If we do not set that
      ;; breakpoint backward goto won't be stepped which is rather annoying
      ;; because backward gotos are used to implement Scheme loops.
      ;; then we executed the requested command
      (let ((ginfo (get-current-global)))
	 (if (global-info? ginfo)
	     (let* ((current-line (fetch-current-line))
		   (define-line  (fetch-global-define-line ginfo)))
		(if (and (integer? define-line)
			 (integer? current-line)
			 (> current-line define-line))
		    (let loop ((start define-line))
		       (if (<fx start current-line)
			   (let* ((delta (integer->string
					  (-fx start current-line)))
				  (bp    (string-append "break " delta)))
			      (add-breakpoint! (gdb-send-for-output bp))
			      (loop (+fx start 1)))))))))
      ;; now we have to set breakpoints into all location in the stack
      ;; that correspond to Bigloo activations
      (set-stack-breakpoints! add-breakpoint!)
      ;; now we emit the gdb commant associated to the user command
      (let* ((out      (gdb-send-for-output cmd))
	     (ginfo    (get-current-global))
	     (stop     "The program is not being run.")
	     (stop-len (string-length stop)))
	 (if (or (global-info? ginfo)
		 (substring=? out stop stop-len))
	     (begin
		(gdb-send-for-output (string-append "delete " bps))
		(gdb-echo out))
	     (begin
		;; we set a breakpoint on each global Scheme function
		(for-each-global (lambda (ginfo)
				    (let* ((c-name (global-info-c-name ginfo))
					   (cmd    (string-append "break "
								  c-name))
					   (bp     (gdb-send-for-output cmd)))
				       (add-breakpoint! bp))))
		;; and we start continuing
		(let ((out (gdb-send-for-output "continue")))
		   (gdb-send-for-output (string-append "delete " bps))
		   ;; we parse the output seeking the echo marker
		   (let ((len (string-length out)))
		      (let loop ((i 0))
			 (cond
			    ((=fx i len)
			     (gdb-echo ""))
			    ((char=? (string-ref out i)
				     *bdb-emacs-echo-marker*)
			     (gdb-echo (substring out i len)))
			    (else
			     (loop (+fx i 1))))))))))))
	  

      
