;*---------------------------------------------------------------------*/
;*   A pratical implementation for the Scheme programming language     */
;*                                                                     */
;*                                    ,--^,                            */
;*                              _ ___/ /|/                             */
;*                          ,;'( )__, ) '                              */
;*                         ;;  //   L__.                               */
;*                         '   \\   /  '                               */
;*                              ^   ^                                  */
;*                                                                     */
;*   Copyright (c) 1992-1999 Manuel Serrano                            */
;*                                                                     */
;*     Bug descriptions, use reports, comments or suggestions are      */
;*     welcome. Send them to                                           */
;*       bigloo-request@kaolin.unice.fr                                */
;*       http://kaolin.unice.fr/bigloo                                 */
;*                                                                     */
;*   This program is free software; you can redistribute it            */
;*   and/or modify it under the terms of the GNU General Public        */
;*   License as published by the Free Software Foundation; either      */
;*   version 2 of the License, or (at your option) any later version.  */
;*                                                                     */
;*   This program is distributed in the hope that it will be useful,   */
;*   but WITHOUT ANY WARRANTY; without even the implied warranty of    */
;*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     */
;*   GNU General Public License for more details.                      */
;*                                                                     */
;*   You should have received a copy of the GNU General Public         */
;*   License along with this program; if not, write to the Free        */
;*   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,   */
;*   MA 02111-1307, USA.                                               */
;*---------------------------------------------------------------------*/
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/bdb/Command/bbreak.scm               */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Apr 10 17:34:09 1998                          */
;*    Last change :  Wed Nov 25 15:15:48 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The bbreak and assert commands implementation                    */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module command_bbreak
   (import tools_read
	   tools_echo
	   tools_regexp
	   tools_file
	   engine_param
	   engine_engine
	   engine_error
	   env_env
	   command_blocals
	   command_completion
	   command_command
	   command_eval)
   (export (break-command              ::bstring ::pair)
	   (bbreak-command             ::bstring ::pair)
	   (btbreak-command            ::bstring ::pair)
	   (bbreak-command-completer   ::command ::bstring)
	   (bcond-command              ::bstring ::pair)
	   (assert-command             ::bstring ::pair)
	   (generic-bbreak-command     ::bstring ::bstring ::pair)
	   (delete-command             ::bstring ::pair)
	   (delete-breakpoints-command ::bstring ::pair)
	   (delete-assertions-command  ::bstring ::pair)
	   (delete-assertions!)
	   (unassert-command           ::bstring ::pair)
	   (info-assert-command        ::bstring ::pair)
	   (load-assertion!            ::bstring ::bstring ::bstring)
	   (dump-assertions!))
   (static (final-class breakpoint
	      (num::int read-only)
	      (file::bstring read-only)
	      (line::int read-only)
	      (addr::bstring read-only)
	      (deleted?::bool (default #f)))
	   (wide-class breakpoint-conditional::breakpoint
	      (condition::bstring read-only))
	   (wide-class assertion::breakpoint
	      (assertion::bstring read-only))))

;*---------------------------------------------------------------------*/
;*    break-command ...                                                */
;*---------------------------------------------------------------------*/
(define (break-command line line-list)
   (let* ((out (gdb-send-for-output line))
	  (bp  (parse-gdb-breakpoint-output out)))
      (if (breakpoint? bp)
	  (register-breakpoint! bp))
      (gdb-echo out)))
   
;*---------------------------------------------------------------------*/
;*    bbreak-command ...                                               */
;*    -------------------------------------------------------------    */
;*    The bbreak command.                                              */
;*---------------------------------------------------------------------*/
(define (bbreak-command line line-list)
   (generic-bbreak-command "break " line line-list))

;*---------------------------------------------------------------------*/
;*    btbreak-command ...                                              */
;*    -------------------------------------------------------------    */
;*    The btbreak command.                                             */
;*---------------------------------------------------------------------*/
(define (btbreak-command line line-list)
   (generic-bbreak-command "tbreak " line line-list))

;*---------------------------------------------------------------------*/
;*    generic-bbreak-command ...                                       */
;*---------------------------------------------------------------------*/
(define (generic-bbreak-command break line line-list)
   (define (not-a-function scm-name)
      (print "\"" scm-name "\" is not a function."))
   (define (native-break bp::pair)
      (let* ((cmd (apply string-append (cons break bp)))
	     (out (gdb-send-for-output cmd))
	     (bp  (parse-gdb-breakpoint-output out)))
	 (if (breakpoint? bp)
	     (register-breakpoint! bp))
	 (gdb-echo out)))
   (define (break-function var-info)
      (let ((bp-c-name (function-breakpoint var-info)))
	 (if (not (string? bp-c-name))
	     (not-a-function (variable-info-scm-name var-info))
	     (string-case bp-c-name
		((: (uncase "*0x") (+ xdigit))
		 (native-break (list (find-breakpoint (the-string)))))
		(else
		 (let* ((cmd (string-append break bp-c-name))
			(out (gdb-send-for-output cmd))
			(bp  (parse-gdb-breakpoint-output out)))
		    (if (breakpoint? bp)
			(register-breakpoint! bp))
		    (gdb-echo out)))))))
   (define (bbreak/w-module scm-name)
      (let ((scm-name (bdb-case scm-name)))
	 (let loop ((locals (get-locals)))
	    (cond
	       ((null? locals)
		(let ((g-info (find-scm-global scm-name)))
		   (if (or (not (global-info? g-info))
			   (not (string? (global-info-c-name g-info))))
		       (native-break (cdr line-list))
		       (break-function g-info))))
	       ((string=? (variable-info-scm-name (car locals)) scm-name)
		(break-function (car locals)))
	       (else
		(loop (cdr locals)))))))
   (define (bbreak scm-name m-info)
      (let ((g-info (find-scm-global scm-name m-info)))
	 (if (or (not (global-info? g-info))
		 (not (string? (global-info-c-name g-info))))
	     (print "Function \"" scm-name "\" not defined.")
	     (break-function g-info))))
   (define (module-bbreak m-info)
      (let* ((bp-c-name (module-info-init-c-name m-info))
	     (cmd (string-append break bp-c-name))
	     (out (gdb-send-for-output cmd))
	     (bp  (parse-gdb-breakpoint-output out)))
	 (if (breakpoint? bp)
	     (register-breakpoint! bp))
	 (gdb-echo out)))
   (match-case line-list
      ((?-)
       (let* ((out (gdb-send-for-output break))
	      (bp  (parse-gdb-breakpoint-output out)))
	  (if (breakpoint? bp)
	      (register-breakpoint! bp))
	  (gdb-echo out)))
      ((?- ?id)
       ;; we have to look at the shape of id.
       ;;   is it file:lnum ?
       ;;   is it a file:id ?
       ;;   is it an identifier ?
       (string-case id
	  ((: (uncase "*0x") (+ xdigit))
	   (native-break (list (find-breakpoint (the-string)))))
	  ((+ digit)
	   ;; this is a lnum expression we have to fetch the
	   ;; current file name
	   (let ((source (gdb-send-for-output "info source"))
		 (lnum   (the-string)))
	      (string-case source
		 ((eol (: (+ all) "is " (+ all)))
		  (let* ((fname  (the-substring 4 (the-length)))
			 (bpname (string-append fname ":" lnum)))
		     (native-break (list (find-breakpoint bpname)))))
		 (else
		  ;; we are not able to find the current file name
		  (native-break (cdr line-list))))))
	  ((: (+ (out #\:)) #\: (+ digit))
	   ;; this is a file:lnum expression
	   (native-break (list (find-breakpoint (the-string)))))
	  ((: (submatch (+ (out #\:))) #\: (submatch (+ all)))
	   ;; this is a module:id
	   (let ((fun-scm-name (the-submatch 2))
		 (file-name    (the-submatch 1)))
	      (let ((f-info (find-file file-name)))
		 (if (not (file-info? f-info))
		     (native-break (cdr line-list))
		     (let ((m-info (file-info-module f-info)))
			(bbreak fun-scm-name m-info))))))
	  ((+ (out #\:))
	   ;; this is an identifier breakpoint
	   (bbreak/w-module id))
	  (else
	   ;; we don't know that form we delegate to gdb break command
	   (native-break (cdr line-list)))))
      ((?- ?mark ?scm-name ?mod-name)
       (let ((mod-len (string-length mod-name)))
	  (if (and (string=? mark "(@")
		   (char=? (string-ref mod-name (-fx mod-len 1)) #\)))
	      ;; ok, we are requesting a specific global variable
	      (let* ((m-name (substring mod-name 0 (-fx mod-len 1)))
		     (m-info (find-module m-name)))
		 (if (not (module-info? m-info))
		     (print "Module \"" m-name "\" not defined.")
		     (bbreak scm-name m-info)))
	      (print "Illegal " break "command \"" line "\"."))))
      ((?- ?mark ?mod-name)
       ;; this a module breakpoint
       (let ((mod-len (string-length mod-name)))
	  (if (and (string=? mark "(@")
		   (char=? (string-ref mod-name (-fx mod-len 1)) #\)))
	      ;; ok, we are requesting a specific global variable
	      (let* ((m-name (substring mod-name 0 (-fx mod-len 1)))
		     (m-info (find-module m-name)))
		 (if (not (module-info? m-info))
		     (print "Module \"" m-name "\" not defined.")
		     (module-bbreak m-info)))
	      (print "Illegal " break "command \"" line "\"."))))))
			    
;*---------------------------------------------------------------------*/
;*    bbreak-command-completer ...                                     */
;*---------------------------------------------------------------------*/
(define (bbreak-command-completer cmd::command string::bstring)
   (let ((len (string-length string)))
      (if (=fx len 0)
	  (find-global-names (lambda (name) #t))
	  (let ((line-list (string->word-list string)))
	     (match-case line-list
		((?id)
		 ;; we complete the identifier
		 (let ((prefix-len (string-length string)))
		    (find-global-names
		     (lambda (name)
			(if *bdb-case-sensitive?*
			    (substring=? name string len)
			    (substring-ci=? name string len))))))
		(else
		 '()))))))

;*---------------------------------------------------------------------*/
;*    find-global-names ...                                            */
;*    -------------------------------------------------------------    */
;*    This function build the list of all global variables for whom    */
;*    global filter returns #t.                                        */
;*---------------------------------------------------------------------*/
(define (find-global-names filter)
   (let ((res '()))
      (for-each-global (lambda (global)
			  (let ((name (global-info-scm-name global)))
			     (if (filter name)
				 (set! res (cons name res))))))
      res))

;*---------------------------------------------------------------------*/
;*    breakpoint-eval-command ...                                      */
;*---------------------------------------------------------------------*/
(define (breakpoint-eval-command number expr evaluator)
   (let* ((num  (string->integer number))
	  (expr (multiple-value-bind (id type)
		   (if (and (pair? expr) (null? (cdr expr)))
		       (parse-id (car expr))
		       (values #f #f))
		   (if (and (string? id) (string? type))
		       (string-append "(" type "? " id ")")
		       (let loop ((expr (reverse expr))
				  (res  ""))
			  (if (null? expr)
			      res
			      (loop (cdr expr)
				    (string-append (car expr)
						   " "
						   res)))))))
	  (bp   (if (and (>fx num 0) (<fx num (vector-length *breakpoints*)))
		    (vector-ref *breakpoints* num)
		    #f)))
      (cond
	 ((=fx num 0)
	  (gdb-echo #"Argument required (breakpoint number).\n"))
	 ((not (breakpoint? bp))
	  (gdb-echo (string-append "No breakpoint number "
				   number
				   #".\n")))
	 (else
	  (with-access::breakpoint bp (line file)
	     ;; we have to widen the breakpoint
	     (if (string=? evaluator "bdb_eval_for_bool")
		 (widen!::breakpoint-conditional bp
		    (condition expr))
		 (widen!::assertion bp
		    (assertion expr)))
	     ;; we now compute the eval environment for
	     ;; the breakpoint expression
	     (let ((env (bdb-eval-environment expr
					      file
					      (integer->string line))))
		;; env is set to the local enviroment (the one
		;; of the breakpoint)
		(let ((cmd (string-append "cond "
					  number
					  " "
					  evaluator
					  "(\""
					  (string-for-read expr)
					  "\","
					  env
					  ")")))
		   (gdb-echo (gdb-send-for-output cmd)))))))))

;*---------------------------------------------------------------------*/
;*    bcond-command ...                                                */
;*---------------------------------------------------------------------*/
(define (bcond-command line line-list)
   (match-case line-list
      ((?-)
       (gdb-echo #"Argument required (breakpoint number).\n"))
      ((?- ?number . ?expr)
       (breakpoint-eval-command number expr "bdb_eval_for_bool"))
      (else
       (gdb-echo #"Argument required (breakpoint number).\n"))))

;*---------------------------------------------------------------------*/
;*    assert-command ...                                               */
;*---------------------------------------------------------------------*/
(define (assert-command line line-list)
   (match-case line-list
      ((?-)
       (gdb-echo #"Argument required.\n"))
      ((?- ?number . ?expr)
       (breakpoint-eval-command number expr "bdb_assert"))
      (else
       (gdb-echo #"Illegal assert expression.\n"))))
			    
;*---------------------------------------------------------------------*/
;*    parse-id ...                                                     */
;*---------------------------------------------------------------------*/
(define (parse-id string::bstring)
   (let ((len (string-length string)))
      (let loop ((walker     0)
		 (id-stop    0)
		 (type-start 0))
	 (cond
	    ((=fx walker len)
	     (cond
		((or (and (=fx id-stop 0) (>fx type-start 0))
		     (=fx id-stop 0)
		     (=fx type-start len))
		 (values #unspecified #unspecified))
		(else
		 (values (substring string 0 id-stop)
			 (substring string type-start len)))))
	    ((and (char=? (string-ref string walker) #\:)
		  (<fx walker (-fx len 1))
		  (char=? (string-ref string (+fx walker 1)) #\:))
	     (if (>fx type-start 0)
		 (values #unspecified #unspecified)
		 (loop (+fx walker 2) walker (+fx walker 2))))
	    (else
	     (loop (+fx walker 1) id-stop type-start))))))

;*---------------------------------------------------------------------*/
;*    parse-gdb-breakpoint-output ...                                  */
;*    -------------------------------------------------------------    */
;*    This function parse the output of a gdb breakpoint setting       */
;*    in order to fetch the breakpoint number, the file and the        */
;*    line and the address of the breakpoint. This function performs   */
;*    no output but it returns a breakpoint that must be registered.   */
;*    The strings we have to parse look like:                          */
;*       Breakpoint 1 at 0x8049fa3: file queens.c, line 344.           */
;*---------------------------------------------------------------------*/
(define (parse-gdb-breakpoint-output output)
   (bind-exit (exit)
      (let ((len (string-length output)))
	 (define (get-next-word start . delims)
	    ;; this function returns then index of the first char
	    ;; that is in the delims set
	    (let ((delims (if (null? delims) '(#\space) delims)))
	       (if (=fx start len)
		   (exit #f)
		   (let loop ((i       start)
			      (started #f))
		      (cond
			 ((=fx i len)
			  len)
			 ((memq (string-ref output i) delims)
			  (if started
			      i
			      (loop (+fx i 1) #f)))
			 (else
			  (loop (+fx i 1) #t)))))))
	 ;; we skip the first word
	 (let* ((num-start (get-next-word 0))
		(num-stop (get-next-word num-start))
		(num (string->integer (substring output
						    (+fx num-start 1)
						    num-stop)))
		(addr-start (get-next-word num-stop))
		(addr-stop (get-next-word addr-start #\:))
		(addr (substring output (+fx addr-start 1) addr-stop))
		(file-start (get-next-word addr-stop #\space #\:))
		(file-stop (get-next-word file-start #\,))
		(file (substring output (+fx file-start 1) file-stop))
		(line-start (get-next-word file-stop #\, #\space))
		(line-stop (get-next-word line-start #\.))
		(line (string->integer (substring output
						  (+fx line-start 1)
						  line-stop))))
	    (instantiate::breakpoint
	       (num num)
	       (file file)
	       (line line)
	       (addr addr))))))
		       
;*---------------------------------------------------------------------*/
;*    *breakpoints* ...                                                */
;*    -------------------------------------------------------------    */
;*    This vector holds all breakpoints that are setup. Even deleted   */
;*    breakpoints are registered in that vector.                       */
;*---------------------------------------------------------------------*/
(define *breakpoints* (make-vector 30))

;*---------------------------------------------------------------------*/
;*    register-breakpoint! ...                                         */
;*---------------------------------------------------------------------*/
(define (register-breakpoint! bp)
   (with-access::breakpoint bp (num)
      (let ((bp-len (vector-length *breakpoints*)))
	 ;; we may have to enlarge the breakpoint vector...
	 (if (>=fx num bp-len)
	     (let* ((new-len (*fx num 2))
		    (new-vec (make-vector new-len)))
		(let loop ((i 0))
		   (cond
		      ((=fx i bp-len)
		       (set! *breakpoints* new-vec))
		      (else
		       (vector-set! new-vec i (vector-ref *breakpoints* i))
		       (loop (+fx i 1)))))))
	 ;; we store the breakpoint
	 (vector-set! *breakpoints* num bp))))

;*---------------------------------------------------------------------*/
;*    for-each-delete-on-condition ...                                 */
;*---------------------------------------------------------------------*/
(define (for-each-delete-on-condition verb? pred? kind)
   (let loop ((i (-fx (vector-length *breakpoints*) 1))
	      (num 0))
      (cond
	 ((=fx i 0)
	  (if verb?
	      (gdb-echo (string-append (if (>fx num 0)
					   (integer->string num)
					   "no")
				       " " kind
				       (if (>fx num 1) "s" "")
				       #" deleted.\n"))))
	 (else
	  (let ((bp (vector-ref *breakpoints* i)))
	     (if (pred? bp)
		 (let ((cmd (string-append "delete "
					   (integer->string
					    (breakpoint-num bp)))))
		    (gdb-send-for-output cmd)
		    (loop (-fx i 1) (+fx 1 num)))
		 (loop (-fx i 1) num)))))))
    
;*---------------------------------------------------------------------*/
;*    delete-command ...                                               */
;*    -------------------------------------------------------------    */
;*    This function delete all breakpoint that are not assertion.      */
;*    The breakpoint on failure is not registrated thus there is no    */
;*    need to re-install it.                                           */
;*---------------------------------------------------------------------*/
(define (delete-command line line-list)
   (match-case line-list
      ((?-)
       (for-each-delete-on-condition #t
				     (lambda (bp)
					(and (breakpoint? bp)
					     (not (assertion? bp))))
				     "breakpoint"))
      ((?- . ?what)
       (cond
	  ((string=? (car what) "breakpoints")
	   (for-each-delete-on-condition #t
					 (lambda (bp)
					    (and (breakpoint? bp)
						 (not (assertion? bp))))
					 "breakpoint"))
	  ((string=? (car what) "assertions")
	   (for-each-delete-on-condition #t assertion? "assertion"))
	  (else
	   (let ((num (string->integer (car what))))
	      (if (and (>fx num 0) (<fx num (vector-length *breakpoints*)))
		  (let ((bp (vector-ref *breakpoints* num)))
		     (cond
			((assertion? bp)
			 (gdb-echo (string-append (car what)
						  #" is an assertion!\n")))
			((not (breakpoint? bp))
			 (gdb-echo (string-append "No such breakpoint "
						  (car what)
						  #".\n")))
			(else
			 (gdb-echo (gdb-send-for-output line)))))
		  (gdb-echo (gdb-send-for-output line)))))))))

;*---------------------------------------------------------------------*/
;*    delete-breakpoints-command ...                                   */
;*---------------------------------------------------------------------*/
(define (delete-breakpoints-command line line-list)
   (match-case line-list
      ((?- ?-)
       (for-each-delete-on-condition #t
				     (lambda (bp)
					(and (breakpoint? bp)
					     (not (assertion? bp))))
				     "breakpoint"))
      ((?- ?- . ?nums)
       (for-each (lambda (str)
		    (let ((num (string->integer str)))
		       (if (and (>fx num 0)
				(<fx num (vector-length *breakpoints*))
				(not (breakpoint? (vector-ref *breakpoints*
							      num))))
			   (gdb-echo (string-append "No such breakpoint "
						    str
						    #".\n"))
			   (let ((bp (vector-ref *breakpoints* num)))
			      (if (assertion? bp)
				  (gdb-echo
				   (string-append str #" is an assertion!\n"))
				  (let ((cmd (string-append
					      "delete breakpoint "
					      str)))
				     (gdb-echo (gdb-send-for-output cmd))))))))
		 nums))))

;*---------------------------------------------------------------------*/
;*    delete-assertions! ...                                           */
;*---------------------------------------------------------------------*/
(define (delete-assertions!)
   (for-each-delete-on-condition #f assertion? "assertion"))
   
;*---------------------------------------------------------------------*/
;*    delete-assertions-command ...                                    */
;*---------------------------------------------------------------------*/
(define (delete-assertions-command line line-list)
   (match-case line-list
      ((?- ?-)
       (for-each-delete-on-condition #t assertion? "assertion"))
      ((?- ?- . ?nums)
       (for-each (lambda (str)
		    (let ((num (string->integer str)))
		       (if (and (>fx num 0)
				(<fx num (vector-length *breakpoints*))
				(not (assertion? (vector-ref *breakpoints*
							     num))))
			   (gdb-echo (string-append "No such assertion "
						    str
						    #".\n"))
			   (let ((bp (vector-ref *breakpoints* num)))
			      (if (not (assertion? bp))
				  (gdb-echo
				   (string-append str #" is a breakpoint!\n"))
				  (let ((cmd (string-append
					      "delete breakpoint "
					      str)))
				     (vector-set! *breakpoints*
						  num
						  #unspecified)
				     (gdb-echo (gdb-send-for-output cmd))))))))
		 nums))))

;*---------------------------------------------------------------------*/
;*    unassert-command ...                                             */
;*---------------------------------------------------------------------*/
(define (unassert-command line line-list)
   (match-case line-list
      ((?- . ?nums)
       (for-each (lambda (str)
		    (let ((num (string->integer str)))
		       (if (and (>fx num 0)
				(<fx num (vector-length *breakpoints*))
				(not (assertion? (vector-ref *breakpoints*
							     num))))
			   (gdb-echo (string-append "No such assertion "
						    str
						    #".\n"))
			   (let ((bp (vector-ref *breakpoints* num)))
			      (if (not (assertion? bp))
				  (gdb-echo
				   (string-append str #" is a breakpoint!\n"))
				  (let ((cmd (string-append
					      "delete breakpoint "
					      str)))
				     (vector-set! *breakpoints*
						  num
						  #unspecified)
				     (gdb-echo (gdb-send-for-output cmd))))))))
		 nums))))

;*---------------------------------------------------------------------*/
;*    info-assert-command ...                                          */
;*    -------------------------------------------------------------    */
;*    This command pretty prints the expression of an assertion.       */
;*---------------------------------------------------------------------*/
(define (info-assert-command line line-list)
   (match-case line-list
      ((?- ?-)
       (gdb-echo (gdb-send-for-output "info breakpoints")))
      ((?- ?- . ?str)
       (let ((num (string->integer (car str))))
	  (if (and (>fx num 0)
		   (<fx num (vector-length *breakpoints*))
		   (assertion? (vector-ref *breakpoints* num)))
	      (let* ((in (open-input-string
			  (assertion-assertion
			   (vector-ref *breakpoints* num))))
		     (expr (let ((start (read in)))
			      (let loop ((expr (read in))
					 (res '()))
				 (cond
				    ((eof-object? expr)
				     (if (null? res)
					 start
					 `(begin ,start ,@res)))
				    (else
				     (loop (read in) (cons expr res)))))))
		     (out (open-output-string))
		     (old-width *pp-width*))
		 (close-input-port in)
		 (set! *pp-width* 30)
		 (pp expr out)
		 (set! *pp-width* old-width)
		 (gdb-echo (close-output-port out)))
	      (gdb-echo (string-append "No assertion " (car str) #".\n")))))
      (else
       (gdb-echo (string-append "Illegal info assert command \""
				line
				"\".\n")))))
	      
;*---------------------------------------------------------------------*/
;*    load-assertion! ...                                              */
;*---------------------------------------------------------------------*/
(define (load-assertion! file line expr)
   (let* ((cmd (string-append "break " file ":" line))
	  (out (gdb-send-for-output cmd))
	  (bp  (parse-gdb-breakpoint-output out)))
      (if (breakpoint? bp)
	  (begin
	     (register-breakpoint! bp)
	     (assert-command "assert"
			     (list "assert"
				   (integer->string (breakpoint-num bp))
				   expr))))))

;*---------------------------------------------------------------------*/
;*    dump-assertions! ...                                             */
;*---------------------------------------------------------------------*/
(define (dump-assertions!)
   (let ((aname (string-append (basename *exec*) ".ass")))
      (if (file-exists? aname)
	  (delete-file aname))
      ;; we check if there is at least one assertion before opening
      ;; a file for output.
      (let loop ((i (-fx (vector-length *breakpoints*) 1)))
	 (cond
	    ((=fx i 0)
	     ;; no assertion at all
	     'nop)
	    ((assertion? (vector-ref *breakpoints* i))
	     ;; yep, we have found one
	     (let ((port (open-output-file aname)))
		(if (output-port? port)
		    (unwind-protect
		       (let loop ((i (-fx (vector-length *breakpoints*) 1)))
			  (if (>fx i 0)
			      (let ((bp (vector-ref *breakpoints* i)))
				 (if (assertion? bp)
				     (with-access::assertion bp (file line assertion)
					(display #\( port)
					(write file port)
					(display " " port)
					(write line port)
					(display " " port)
					(write assertion port)
					(fprint port #\))))
				 (loop (-fx i 1)))))
		       (close-output-port port)))))
	    (else
	     (loop (-fx i 1)))))))
