/* alloc.h
 * (c) Alexandre Frey 1996
 */

/*
 * This is a home-brewed memory allocator
 *
 * It is designed to allow :
 *  . memory protection without too much space loss
 *  . allocation of small segments (within a 16 bits chunk)
 *    (for example alpha .lita sections )
 *  . use of garbage collector : the code is parametrisez by the
 *    basic allocation routines 
 */

#ifndef __DLINK_ALLOC_H__
#define __DLINK_ALLOC_H__

#include <stdlib.h>

/* The error code returned when a call has failed */
extern int dlink_memory_errno;

#define DLINK_MEMORY_ENO_ERROR 0
#define DLINK_MEMORY_EILLEGAL_ARGUMENT 1
#define DLINK_MEMORY_EOUT_OF_MEMORY 2
#define DLINK_MEMORY_EOUT_OF_SMALL 3

typedef struct dlink_memory dlink_memory;

/* Create a dlink_memory structure
 * INTERNAL_ALLOC and INTERNAL_FREE should have the semantic of
 * malloc / free and are used to allocate and free the internal 
 * structures of dlink_memory (including the structure returned
 * by dlink_memory_create
 * SECTION_ALLOC / SECTION_FREE are used to allocate the sections
 * returns NULL when no memory is available
 */
dlink_memory *dlink_memory_create (void* (*internal_alloc) (size_t),
				   void  (*internal_free ) (void*),
				   void* (*section_alloc ) (size_t),
				   void  (*section_free  ) (void*) );

/* Allocate a section of size SIZE in MEMORY with class CLASS */
void* dlink_section_allocate (dlink_memory *memory,
			      size_t size,
			      int class);

/* Defines the classes of sections 
 * some combination assertions are implicit
 * CODE is automatically READ_ONLY and can't be SMALL
 * ( XXX : could small sections be code ???)
 * returns NULL when the internal or section
 * allocator returns NULL or when the small
 * segment is full
 */
#define DLINK_SEC_READ_WRITE   0x00
#define DLINK_SEC_READ_ONLY    0x01
#define DLINK_SEC_CODE         0x02
#define DLINK_SEC_SMALL        0x04

/* returns a pointer from which all the small segment
 * of MEMORY is accessible with a 16 bit signed offset
 * (used to fill the gp_value for alpha)
 * returns NULL if no small sections are allocated
 */
void* dlink_memory_get_gp (dlink_memory *memory);

/* allocate the small segment */
int dlink_memory_allocate_small_segment (dlink_memory* memory);

/* Free all the sections and the internal machinery 
 * associated with MEMORY
 */
void dlink_memory_free (dlink_memory *memory);

/* Apply the memory protections to MEMORY*/
void dlink_memory_protect (dlink_memory *memory);

/* Relax the protection : all sections are again read-write */
void  dlink_memory_unprotect (dlink_memory *memory);

#endif  /* __DLINK_ALLOC_H__ */

/* EOF : alloc.h */

			      
