-- Facilities for controling exception handling

class EXCEPTIONS

feature

   show_except is
      -- Print a trace message when an exception occurs
      -- (This is not the default: see no_show_except.)
	 external "C"
	   alias "_ert_show_trace"
   end -- show_except

   no_show_except is
      -- Don't print a trace message when an exception occurs
      -- (This is not the default: see show_except.)
	 external "C"
	   alias "_ert_no_show_trace"
   end -- no_show_except

   assertion_violation:BOOLEAN is
		-- Was last exception due to a violated assertion or
		-- non-decreasing variant?
	external "C"
		alias "_ert_assertion_violation"
   end -- assertion_violation

   catch ( code:INTEGER) is
		-- Make sure that any exception of code `code' will be caught.
		-- This is the default. (See `continue', `ignore'.)
	do
		except_mode ( code, Caught)

     ensure
		status ( code) = Caught
   end -- catch

   Check_instruction	:INTEGER is 105	-- See EXCEPT_CHECK_INSTRUCTION
		-- Exception code for violated check

   Class_invariant	:INTEGER is 102	-- See EXCEPT_INVARIANT
		-- Exception code for violated class invariant

   class_name: STRING is
         -- Name of the class containing the routine which was the
         -- recipient of the oldest exceptions not leading to a retry.
         -- Is void initially and after a retry.
       external "C"
		alias "_ert_last_exception_name"
   end -- class_name

   continue(code:INTEGER) is
		-- Make sure that any exception of code `code' will cause
		-- exceution to resume after a call to `continue_action(code)'
		-- This is not the default. (See `catch', `ignore')
      require
         must_be_a_signal: is_signal(code)
      do
		except_mode ( code, Continued)
      ensure
          status(code) = Continued
      end -- continue

   continue_action(code:INTEGER) is
      do
	 end

   developer_exception_name:STRING
      -- Name of last developer-raise exception (See `raise')
     
   exception:INTEGER is
      -- Code of last exception that occured
      external "C"
         alias "_ert_last_exception"
   end -- exception

   ignore(code:INTEGER) is
		-- Make sure that any exception of code `code' will br ignored
		-- This is not the default. (See `catch', `ignore')
      do
		except_mode ( code, Ignored)
      ensure
          status(code) = Ignored
      end -- ignore

   Incorrect_inspection_value:INTEGER is 112	-- See EXCEPT_INSPECTION_VALUE

   is_developer_exception_code(code:INTEGER):BOOLEAN is
      -- is `code' the code of a developer-defined exception (See `raise')?
      do
         if
           code < 0
         then
           Result := true
         else
           Result := false
         end
   end -- is_developer_exception_code

   is_assertion_violation(code:INTEGER):BOOLEAN is
      -- is `code' the code of an exception resulting from the violation
      -- of an assertion (precondition,poscondition,invariant,check)?
      do
         if
            code >= Precondition and code <= Check_instruction
         then
            Result := true
         else
            Result := false
	    end
   end -- is_assertion_violation

   is_signal(code:INTEGER):BOOLEAN is
      -- is `code' the code of an exception due to hardware
      -- or operating system?
      do
         if
            code > 0 and code < Precondition
         then
            Result := true
         else
            Result := false
         end
   end -- is_signal_code 
   
   Loop_invariant:INTEGER is 103 	-- See EXCEPT_LOOP_INVARIANT
      -- Exception code for violated loop invariant

   Loop_variant:INTEGER is 104	-- See EXCEPT_LOOP_VARIANT
      -- Exception code for non-decreased loop variant

   meaning(code:INTEGER):STRING is
      -- Nature of exception of code `code', expressed in plain English
      external "C"
		alias "_ert_except_mesg"
   end -- meaning

   message_on_failure is
      -- Print an Exception History Table in case of failure.
      -- (This is the default: see no_message_on_failure.)
	 external "C"
	   alias "_ert_show_trace_on_failure"
   end -- message_on_failure

   no_message_on_failure is
      -- Do not print an Exception History Table in case of failure.
      -- (This is not the default: see message_on_failure.)
	 external "C"
	   alias "_ert_no_show_trace_on_failure"
   end -- no_message_on_failure

   No_more_memory:INTEGER is  108 	-- See EXCEPT_NO_MORE_MEMORY
      -- Exception code for failed memory allocation

   original_exception:INTEGER is
      -- Code of oldest exception not leading to a Retry
      external "C"
         alias "_ert_original_exception"
   end -- original_exception

   Postcondition:INTEGER is 101	-- See EXCEPT_POSTCONDITION
      -- Exception code for violated postcondition

   Precondition:INTEGER is 100	-- See EXCEPT_PRECONDITION
      -- Exception code for violated precondition

   Rescue_failure:INTEGER is 100	-- See EXCEPT_RESCUE_FAILED
      -- Exception code for violated precondition

   raise(code:INTEGER; name:STRING) is
      -- Raise a developer exception of `code' and name `name'
      require
		negative_code: code < 0 and code > Developer_5
      do
	    developer_exception_name := name
         raise_developer_exception ( code)
   end -- raise

   reset_all_default is
      -- Make sure that all exceptions will lead to their default handling
   external "C"
      alias "_ert_reset_all_default"
   end -- reset_all_default

   reset_default(code:INTEGER) is
      -- Make sure that exception of code `code' will lead 
      -- to its default action.
   external "C"
      alias "_ert_reset_default"
   end -- reset_all_default

   Routine_failure:INTEGER is 106	-- See EXCEPT_ROUTINE_FAILURE
      -- Exception code for failed routine

   routine_name:STRING is
      -- Code of oldest exception not leading to a Retry
      external "C"
         alias "_ert_routine_name"
   end -- routine_name

   status(code:INTEGER):INTEGER is
   		-- Status currently set for exception `code' (default: `Caught')
   external "C"
		alias "_ert_except_status"
      ensure
          Result = Caught or Result = Continued or Result = Ignored
   end -- status
	 
   Ignored:INTEGER   is 1 	-- See EXCEPT_IGNORE
   Caught:INTEGER    is 2 	-- See EXCEPT_CATCH
   Continued:INTEGER is 3 	-- See EXCEPT_CONTINUE	
      -- Possible status for exception codes 

   tag_name:STRING is
      -- Tag of last violated assertion
      external "C"
         alias "_ert_tag_name"
   end -- tag_name

   Void_assigned_to_expanded:INTEGER is 113	-- See EXCEPT_ASSIGNED_TO_EXPANDED
      -- Exception code for assignement of void value to expanded entity

   Void_target_call	:INTEGER is 107 	-- See EXCEPT_VOID_TARGET_CALL
      -- Exception code for feature called on void reference

   void_call_target:STRING is
      -- Name of feature that was called on a void reference
      external "C"
         alias "_ert_void_call_target"
   end -- void_call_target

	call_to_deferred	:INTEGER is 109 -- See EXCEPT_CALL_TO_DEFERRED

   Bad_signal    		:INTEGER is 110	-- See EXCEPT_BAD_SIGNAL
      -- An unexpected signal was received - can't handle

   Nested_funcions	:INTEGER is 111 	-- See EXCEPT_NESTED_FUNCTIONS
      -- Run time limit exceeded. Due to too many function calls.
      -- Limit can be increased in the runtime module source code
	
   Developer_1		:INTEGER is -1 	-- See EXCEPT_DEVELOPER_1
   Developer_2		:INTEGER is -2 	-- See EXCEPT_DEVELOPER_2
   Developer_3		:INTEGER is -3 	-- See EXCEPT_DEVELOPER_3
   Developer_4		:INTEGER is -4 	-- See EXCEPT_DEVELOPER_4
   Developer_5		:INTEGER is -5 	-- See EXCEPT_DEVELOPER_5

   set_exitcode ( code:INTEGER) is
      -- Set the termination code used upon next abnormal termination.
      -- Default is 1
      external "C"
	    alias "_ert_set_exitcode"

   end -- set_exitcode

   except_mode(code, mode:INTEGER) is
      -- Set signal/raise mode for `code'
      require
          exception_mode: code = Continued or code = Caught or code = Ignored
      external "C"
		alias "_ert_except_mode"

   end -- except_mode

   raise_developer_exception(code:INTEGER) is
      external "C"
         alias "_ert_raise_developer_exception"
   end -- raise_developer_exception


end -- class EXCEPTIONS

