/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/

/*
** tokentable.c
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "osd.h"
# include "tokentable.h"

static long unsigned MaxToken;	
static /*@null@*/ /*@only@*/ o_ltoken *TokenTable;       

static void AllocTokenTable (void) /*@modifies TokenTable, MaxToken@*/ ;

ltoken
  LSLInsertToken (ltokenCode cod, lsymbol sym, lsymbol rTxt, bool def)
{
  while (sym >= MaxToken)
    {
      AllocTokenTable ();
    }

  llassert (TokenTable != NULL);

  if (ltoken_isUndefined (TokenTable[sym]))
    {
      TokenTable[sym] = ltoken_create (cod, sym);        
      ltoken_setRawText (TokenTable[sym], rTxt);
      ltoken_setDefined (TokenTable[sym], def);
    }
  
  return TokenTable[sym];
}

void
LSLUpdateToken (ltokenCode cod, lsymbol sym, bool def)
{
  llassert (TokenTable != NULL);

  if (!ltoken_isUndefined (TokenTable[sym]))
    {
      ltoken_setCode (TokenTable[sym], cod);
      ltoken_setDefined (TokenTable[sym], def);
    }
  else
    {
      llfatalbug (message ("LSLUpdateToken: token not in table: %d, text: %s", 
			   (int) cod, cstring_fromChars (lsymbol_toChars (sym))));
    }
}

void
LSLSetTokenHasSyn (lsymbol sym, bool syn)
{
  llassert (TokenTable != NULL);
    
  if (!ltoken_isUndefined (TokenTable[sym]))
    {
      ltoken_setHasSyn (TokenTable[sym], syn);
    }
  else
    {
      llbuglit ("LSLSetTokenHasSyn: null token");
    }
}

ltoken LSLGetToken (lsymbol sym)
{
  llassert (TokenTable != NULL);

  if (!((sym < MaxToken) || (!ltoken_isUndefined (TokenTable[sym]))))
    {
      llcontbuglit ("LSLGetToken: bad argument");
      return TokenTable[0];
    }

  return TokenTable[sym];
}

/*@exposed@*/ ltoken
LSLReserveToken (ltokenCode cod, char *txt)
{
  lsymbol sym;
  
  sym = lsymbol_fromChars (txt);

  /* 
  ** Reserved tokens never have raw text like synonyms.
  */

  return LSLInsertToken (cod, sym, lsymbol_undefined, TRUE);
}

static void
AllocTokenTable (void)
{
  long unsigned oldSize, newSize;
  long unsigned int i;

  oldSize = MaxToken;

  if (oldSize == 0)
    {
      newSize = INITTOKENTABLE;
      llassert (TokenTable == NULL);
      TokenTable = (ltoken *) dmalloc (newSize * sizeof (*TokenTable)); 
    }
  else
    {
      o_ltoken *oldTokenTable = TokenTable;

      newSize = (long unsigned) (DELTATOKENTABLE * oldSize);
      TokenTable = (ltoken *) dmalloc (newSize * sizeof (*TokenTable));

      llassert (oldSize > 0);
      llassert (oldTokenTable != NULL);
      
      for (i = 0; i < oldSize; i++)
	{
	  TokenTable[i] = oldTokenTable[i];
	}

      sfree (oldTokenTable);
    }

  /*@+loopexec@*/
  for (i = oldSize; i < newSize; i++)
    {
      TokenTable[i] = ltoken_undefined;
    }
  /*@=loopexec@*/

  MaxToken = newSize;
/*@-compdef@*/ } /*=compdef@*/

void
ltokenTableInit (void)
{
  MaxToken = 0;
}

void
ltokenTableCleanup (void)
{
  if (TokenTable != NULL)
    {
      long unsigned i;
      
      for (i = 0; i < MaxToken; i++)
	{
	  ltoken_free (TokenTable[i]);
	}

      sfree (TokenTable); 
      TokenTable = NULL;
    }
}
