/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** pcpp.c
**
** pre-c-pre-processor (gasp!)
**
** <filename> --> .lltmp.<filename>
**
**
** the character @ is reserved as a special token in lclint
**
** # include "X" --> # include ".lltmp.X"
**     check if .lltmp.X exists, and if timestamp is later or same
**           as X (if not call pcpp on X)
**     if later, we still need to check in it #includes anything
**
**  UNLESS X = Y.lh
**
**
** # include <X> unchanged
**
** # define X -->>
**
*/

# include "lclintMacros.nf"
# include "basic.h"
# include "osd.h"
# include "fileStack.h"
# include "fileIdList.h"
# include "pcpp.h"
# include "lcllib.h"
# include "cgrammar_tokens.h"

/*@constant observer char *READFILE; @*/
# define READFILE "r"

/*@constant observer char *WRITEFILE; @*/
# define WRITEFILE "w"

/*@constant int NOTFOUND; @*/
# define NOTFOUND -1

/*
** each indx map is used to convert actual fileId's associated with the
** original file, to fileId's associated with a derived file.  
*/

static /*@only@*/ indxMap ppMap;    /* pre-processed files */
static /*@only@*/ indxMap defMap;   /* defines files (for use with +singleinclude) */

static fileId definesMap (fileId fid)
{
  return (fileId_fromInt (indxMap_mapSafe (defMap, fileId_toInt (fid))));
}

static fileId preMap (fileId fid) /*@*/
{
  return (fileId_fromInt (indxMap_mapSafe (ppMap, fileId_toInt (fid))));
}

static fileId addMap (fileId fid1)
{
  fileId fid2 = fileTable_addTempFile (context_fileTable (), fid1);
  indxMap_addpair (ppMap, fileId_toInt (fid1), fileId_toInt (fid2));
  return fid2;
}

static fileId addDefines (fileId fid1)
{
  fileId fid2 = fileTable_addTempFile (context_fileTable (), fid1);
  indxMap_addpair (defMap, fileId_toInt (fid1), fileId_toInt (fid2));
  return fid2;
}

static bool expectmacro = FALSE;
static bool expectmacronext = FALSE;
static bool expectenditer = FALSE;
static /*@observer@*/ cstring lastControl = cstring_undefined;
static bool in_comment = FALSE;
static bool last_in_comment = FALSE;
static bool notfunction = FALSE;
static bool notparseable = FALSE;
static bool noexpand = FALSE;
static bool pcpp_inner_file (fileId p_fid, fileId p_ofid, fileId p_dfid);
static /*@exposed@*/ /*@null@*/ char *
  dofix_line (/*@returned@*/ /*@dependent@*/ char *p_s);

static /*@dependent@*/ /*@null@*/ FILE *infile;
static /*@dependent@*/ /*@null@*/ FILE *outfile;
static /*@dependent@*/ /*@null@*/ FILE *deffile;
static bool dodefs = FALSE;
static bool definecontinuation = FALSE;

static bool open_file_w (fileId p_of);

static fileId curfile;
static int    curline;
static /*@only@*/ fileStack fstack;

static int sbuf_size = 256; 

/*@notfunction@*/
# define NEXT_CHAR() (incColumn(), fgetc (infile))

static /*@null@*/ /*@observer@*/ char *
  identifyLintComment (cstring lintcomment)
{
  if (cstring_equalLit (lintcomment, "FALLTHROUGH"))
    {
      return "@fallthrou@";
    }
  else if (cstring_equalLit (lintcomment, "FALLTHRU"))
    {
      return "@fallth@";
    }
  else if (cstring_equalLit (lintcomment, "NOTREACHED"))
    {
      return "@notreach@";
    }
  else if (cstring_equalLit (lintcomment, "PRINTFLIKE"))
    {
      return "@printfli@";
    }
  else if (cstring_equalLit (lintcomment, "ARGSUSED"))
    {
      return "@argsus@";
    }
  else 
    {
      return NULL;
    }
}

static flagcode identifyMacroFlag (char *rest)
{			      
  if (mstring_equalPrefix (rest, "allmacros"))
    {
      return FLG_ALLMACROS;
    }
  else if (mstring_equalPrefix (rest, "fcnmacros"))
    {
      return FLG_FCNMACROS;
    }
  else if (mstring_equalPrefix (rest, "constmacros"))
    {
      return FLG_CONSTMACROS;
    }
  else
    {
      BADEXIT;
    }
}

/*
** don't put null annotation here!
*/

/*
** sbuf cannot be static since it gets freed (wish there were some way to
** control scoping better so it is not linkable externally.)
**
** Yikes...sure a lot of syntactic comments for such a simple declaration...
*/ 

/*@-exportheadervar@*/ /*@-exportlocal@*/ 
/*@relnull@*/ /*@reldef@*/ /*@owned@*/ char *sbuf = NULL;
/*@=exportheadervar@*/ /*@-exportlocal@*/ 

static void ppNextLine (void)
{
  incLine ();
  context_processedPPLine ();
}

static bool continuationLine (/*@null@*/ char *s)
{
  bool lastslash = FALSE;
  bool inquote = FALSE;
  bool inchar = FALSE;
  bool inescape = FALSE;
  char *here = s;
  char c;

  if (s == NULL)
    {
      return FALSE;
    }

  while ((c = *here) != '\0')
    {
      if (inquote || inchar)
	{
	  if (c == '\\')
	    {
	      inescape = TRUE;
	    }
	  else 
	    {
	      if (!inescape)
		{
		  if (inquote && c == '\"')
		    {
		      inquote = FALSE;
		    }
		  else 
		    {
		      if (inchar && c == '\'')
			{
			  inchar = FALSE;
			}
		    }
		}
	    }
	}
      else
	{
	  if (lastslash && !isspace (c))
	    {
	      llerror 
		(FLG_PREPROC,
		 message ("Non-space character %h follows continuation "
			  "marker", c));
	      break;
	    }
	  
	  if (c == '\\')
	    {
	      lastslash = TRUE;
	    }
	  else if (c == '\'')
	    {
	      inchar = TRUE;
	    }
	  else if (c == '\"')
	    {
	      inquote = TRUE;
	    }
	  else
	    {
	      ;
	    }
	}

      here++;
    }

  return lastslash;
}

static
void defout (cstring cs)
{
  char *s = cstring_toCharsSafe (cs);
  char *slash = strstr (s, "/*");

  llassert (deffile != NULL);

  while (slash != NULL)
    {
      char *end = slash + 2;

      while ((*end != '\0') && !((*end == '*') && (*(end + 1) == '/')))
	{
	  end++;
	}

      if (*end == '\0')
	{
	  *slash = '\0';
	  fprintf (deffile, "%s\n", s);	  
	  *slash = '/';
	  return;
	}
      else
	{
	  end++;
	  slash = strstr (end, "/*");
	}
    }

  fprintf (deffile, "%s\n", s);	
}

static 
void defoutFree (/*@only@*/ cstring cs)
{
  defout (cs);
  cstring_free (cs);
}

static
void defoutc (char *cs)
{
  defout (cstring_fromChars (cs));
}

/*
** returns TRUE is f is readable
*/

static bool
  file_isReadable (char *f)
{
  FILE *fl = fopen (f, READFILE);

  if (fl != (FILE *) 0)
    {
      check (fclose (fl) == 0);
      return (TRUE);
    }
  else
    {
      return (FALSE);
    }
}

static void out_line (cstring l)
{
  if (outfile != NULL)
    {
      check (fputs (cstring_toCharsSafe (l), outfile) != EOF);
      check (putc ('\n', outfile) == (int) '\n');
    }
  else
    {
      llbug (message ("outfile NULL: %s", l));
    }
}

static bool
open_file_rw (fileId f, fileId of)
{
  /* returns TRUE if f can be opened for reading, and
  **         assigns infile to file f.
  */

  curfile = f;

  outfile = fopen (cstring_toCharsSafe (fileName (of)), WRITEFILE);
  infile  = fopen (cstring_toCharsSafe (fileName (f)), READFILE);

  if (infile != NULL && outfile != NULL)
    {
      return TRUE;
    }
  else
    {
      if (infile == NULL)
	{
	 ; /* why no error here? */
	}
      else
	{
	  ppllerror (message ("Cannot open tmp file %s for writing", fileName (of)));
	}
      
      return FALSE;
    }
}

static 
bool open_file_w (fileId of)
{
 /* returns TRUE if of can be opened for writing, and
 **         assigns infile to file of.
 */

  outfile = fopen (cstring_toCharsSafe (fileName (of)), WRITEFILE);

  if (outfile != NULL) 
    {
      return TRUE;
    }
  else
    {
      ppllerror (message ("Cannot open tmp file %s for writing", fileName (of)));
      
      return FALSE;
    }
}

static void
open_definesFile (fileId f)
{
  deffile = fopen (cstring_toCharsSafe (fileName (f)), WRITEFILE);

  if (deffile == NULL)
    {
      llfatalbug (message ("Cannot open tmp file for preprocessing: %s", fileName (f)));
    }
  
  dodefs = TRUE;
}

static bool
open_file_r (fileId f)
{
  /* returns TRUE if f can be opened for reading, and
  **         assigns infile to file f.
  */
  
  infile = fopen (cstring_toCharsSafe (fileName (f)), READFILE);
  outfile = (FILE *) NULL;
  curfile = f;
  
  return (!(infile == (FILE *) NULL));
}

static char
peekc (FILE *f)
{
  int ic;
  
  ic = getc (f);
  checkUngetc (ic, f);
  return (char_fromInt (ic));
}

/*@notfunction@*/
# define NEWCHAR(c)  \
   if (--i < 1) \
   { \
     char *tmp = sbuf; char *tmp2 = tmp; \
     i = sbuf_size; sbuf_size *= 2; \
     *s = '\0';\
     /*@ignore@*/ sbuf = (char *) dmalloc (sizeof (char) * sbuf_size); \
     s = sbuf; /*@end@*/ \
     while /*@-usedef@*/ (*tmp2 != '\0') /*@=usedef@*/ \
       { *s++ = *tmp2++; } \
     sfree (tmp);\
   } \
   *s++ = c;

/*@notfunction@*/
# define OUTCHARS(q) \
  while (*q != '\0') { NEWCHAR (*q); q++; }

typedef struct {
  /*@null@*/ /*@observer@*/ char *name;
  /*@null@*/ /*@observer@*/ char *tag;
  /*@null@*/ /*@observer@*/ char *endtag;
  int   tagtok;
  int   endtok;
} scomment;

static scomment scomments[] = 
{
  { "modifies", "MODIFIS",  "EM", QMODIFIES,   QENDMODIFIES },
  { "globals",  "GLOBLS",   "EG", QGLOBALS,    QENDGLOBALS },
  { "alt",      "AL",       "EA", QALT,        QENDALT },
  { "constant", "CONSTNT",  "EC", QCONSTANT,   QENDCONSTANT },
  { "iter",     "ITR",      "EI", QITER,       QENDITER },
  { "defines",  "DEFNES",   "ED", QDEFINES,    QENDDEFINES },
  { "uses",     "USS",      "EU", QUSES,       QENDUSES },
  { "allocates","ALLOCATS", "EL", QALLOCATES,  QENDALLOCATES },
  { "sets",     "STS",      "ES", QSETS,       QENDSETS },
  { "releases", "RELEASS",  "ER", QRELEASES,   QENDRELEASES } ,
  { "pre",      "PR",       "EP", QPRECLAUSE,  QENDPRECLAUSE } ,
  { "post",     "POS",      "EO", QPOSTCLAUSE, QENDPOSTCLAUSE } ,
  { NULL, NULL, NULL, BADTOK, BADTOK },
} ;

# if 0
void checkMarkers (void)
{
  int i = 0;
  
  while (scomments[i].name != NULL)  
    {
      llassert (strlen (scomments[i].name) == strlen (scomments[i].tag) + 1);
      llassert (strlen (scomments[i].endtag) == 2);
      i++;
    }
}
# endif
	  
int commentMarkerToken (mstring s, bool *isstart)
{
  int i = 0;

  while (scomments[i].name != NULL)
    {
      if (mstring_equal (s, scomments[i].tag))
	{
	  *isstart = TRUE;
	  return scomments[i].tagtok;
	}
      else 
	{
	  if (mstring_equal (s, scomments[i].endtag))
	    {
	      *isstart = FALSE;
	      return scomments[i].endtok;
	    }
	}

      i++;
    }  

  *isstart = FALSE;
  return BADTOK;
}


static int specialCommentIndex (mstring s)
{
  char *thisname;
  int i = 0;
  
  while ((thisname = scomments[i].name) != NULL)
    {
      if (mstring_equalPrefix (s, thisname))
	{
	  return i;
	}
      i++;
    }

  return NOTFOUND;
}

static /*@observer@*/ mstring specialCommentTag (mstring s)
{
  int index = specialCommentIndex (s);
  
  if (index != NOTFOUND)
    {
      llassert (scomments[index].tag != NULL);

      return scomments[index].tag;
    }
  else
    {
      llfatalbug (message ("specialCommentTag: bad: %s", cstring_fromChars (s)));
    }
}

static bool isStateComment (mstring s)
{
  return (mstring_equalPrefix (s, "pre")
	  || mstring_equalPrefix (s, "post"));
}

static bool isSpecialComment (mstring s)
{
  int index = specialCommentIndex (s);
  
  return (index != NOTFOUND);
}


static /*@observer@*/ mstring specialCommentEndTag (mstring s)
{
  int index = specialCommentIndex (s);
  
  if (index != NOTFOUND)
    {
      llassert (scomments[index].endtag != NULL);

      return scomments[index].endtag;
    }
  else
    {
      llfatalbug (message ("specialCommentEndTag: bad: %s", cstring_fromChars (s)));
    }
}

static char followingSpecialComment (mstring s)
{
  int index = specialCommentIndex (s);
  
  if (index != NOTFOUND)
    {
      int len = mstring_length (scomments[index].tag) + 1;
      
      llassert (mstring_length (s) > len);
      return (s[len]);
    }
  else
    {
      llfatalbug (message ("followingSpecialComment: bad: %s", cstring_fromChars (s)));
    }
}

static /*@null@*/ /*@exposed@*/ char *get_line (bool inner)
  /*@modifies infile@*/
{
  /*
  ** returns  string containing next line in file
  */

  char *s = sbuf; /* get some space to work with... */
  int ic;
  char c;
  register int i = sbuf_size; /* number of characters to play with */
  bool inCommentCommand = FALSE;
  bool firstChar = TRUE;
  bool inMacroParams = FALSE;
  bool pastMacroParams = FALSE;
  bool inQuote = FALSE;
  bool inEscape = FALSE;
  static bool inDefine = FALSE;
  static char commentMarkerChar = '\0';
  static bool setComment = FALSE;

  llassert (infile != NULL);

  if (commentMarkerChar == '\0' || setComment)
    {
      commentMarkerChar = context_getCommentMarkerChar ();
      DPRINTF (("marker = %c", commentMarkerChar));
    }

 toploop:
  while (TRUE)
    {
      ic = NEXT_CHAR ();

      if (ic == EOF)
	break;

      c = (char)ic;

      if (c == '\n')
	{
	  inDefine = FALSE;

	  if (inCommentCommand)
	    {
	      /* okay --- need to keep scanning in comment commands */
	      ppNextLine ();
	    }
	  else
	    {
	      break;
	    }
	}

      if (!inQuote)
	{
	  if (firstChar && c == '#') 
	    {
	      inDefine = TRUE;
	      DPRINTF (("inDefine TRUE"));
	    }

	  firstChar = FALSE;

	  if (c == '(' && !pastMacroParams && inner)
	    {
	      inMacroParams = inDefine;
	      DPRINTF (("inMacroParams"));
	    }
	  else if (c == ')' && inMacroParams)
	    {
	      inMacroParams = FALSE;
	      DPRINTF (("out of macroparams"));
	      pastMacroParams = TRUE;
	    }
	  else
	    {
	      ;
	    }

	  if (inCommentCommand && (c == '*') && (peekc (infile) == '/'))
	    {
	      char *comment;
	      char saves = *s;
	      *s = '\0';

	      DPRINTF (("sbuf = %s, saves = %c", sbuf, saves));

	      /*
	      ** Note: must be strrchr to handle lines with multiple
	      **       comments.  
              **       
	      */

	      comment = strrchr (sbuf, commentMarkerChar); /* find start of comment */
	      DPRINTF (("Comment = %s", comment));
	      *s = saves;
	      
	      DPRINTF (("comment = :%s:", comment));
	     
	      if (comment == NULL)
		{
		  llbug (message ("Checking comment: no @: %s", 
				  cstring_fromChars (sbuf)));
		}
	      else if (*comment == commentMarkerChar)
		{
		  char *scomment = comment + 1;
		  
		  *comment = DEFAULT_COMMENTCHAR;

		  DPRINTF (("comment: %s / %s", comment, scomment));
		  
		  if (isSpecialComment (scomment))
		    {
		      if (isStateComment (scomment))
			{
			  char next = followingSpecialComment (scomment);
			  mstring endtag = specialCommentEndTag (scomment);
			  int len;
			  
			  DPRINTF (("specialComment: %s, %s / %c / %s",
				    scomment, comment, next, endtag));
			  
			  if (next != ':')
			    {
			      ppllerror
				(message ("Syntactic comment %s not followed by colon",
					  cstring_fromChars (scomment)));
			    }
			  
			  strcpy (scomment, specialCommentTag (scomment));
			  len = size_toInt (strlen(comment));

			  comment[len] = DEFAULT_COMMENTCHAR;
			  comment[len + 1] = ' ';
			  
			  DPRINTF (("-> %s", comment));
			  NEWCHAR (DEFAULT_COMMENTCHAR);
			  OUTCHARS (endtag);
			  NEWCHAR (DEFAULT_COMMENTCHAR);

			}
		      else
			{
			  char next = followingSpecialComment (scomment);
			  mstring endtag = specialCommentEndTag (scomment);
			  int len;
			  
			  DPRINTF (("specialComment: %s, %s / %c / %s",
				    scomment, comment, next, endtag));
			  
			  if (next != ' ' && next != '\t' && next != '\n')
			    {
			      ppllerror
				(message 
				 ("Syntactic comment %s not followed by whitespace",
				  cstring_fromChars (scomment)));
			    }
			  
			  strcpy (scomment, specialCommentTag (scomment));
			  len = size_toInt (strlen(comment));

			  comment[len] = DEFAULT_COMMENTCHAR;
			  comment[len + 1] = next;
			  
			  if (*(endtag + 1) == 'C' || *(endtag + 1) == 'I')
			    {
			      /* don't expand after constant or iter */
			      
			      noexpand = TRUE;
			      expectmacro = TRUE;			      

			      if (expectenditer)
				{
				  decLine ();
				  ppllerror (message ("Macro expected (to define end_<iter> "
						      "after %s syntactic comment: %s",
						      lastControl, cstring_fromChars (s)));
				  incLine ();
				  expectenditer = FALSE;
				}
			      
			      
			      lastControl = cstring_makeLiteralTemp 
				((*(endtag + 1) == 'I') ? "iter" : "constant");
			      
			      DPRINTF (("set noexpand!"));
			    }
			  
			  DPRINTF (("-> %s", comment));
			  NEWCHAR (DEFAULT_COMMENTCHAR);
			  OUTCHARS (endtag);
			  NEWCHAR (DEFAULT_COMMENTCHAR);
			}
		    }
		  else if (mstring_equalPrefix (scomment, "ignore"))
		    {
		      if (!context_getFlag (FLG_NOCOMMENTS))
			{
			  context_enterSuppressRegion ();
			}

		      NEWCHAR ('@');
		      NEWCHAR (' ');
		    }
		  else if (mstring_equalPrefix (scomment, "end"))
		    {
		      if (!context_getFlag (FLG_NOCOMMENTS))
			{
			  context_exitSuppressRegion ();
			}
		      
		      NEWCHAR ('@');
		      NEWCHAR (' ');
		    }
		  else if (mstring_equalPrefix (scomment, "notparseable"))
		    {
		      notparseable = TRUE;
		      expectmacro = TRUE;

		      *comment = '/'; /* remove marker */
		      *(comment + 1) = '*';
		      lastControl = cstring_makeLiteralTemp ("notparseable");

		      NEWCHAR ('*');
		      NEWCHAR ('/');
		    }
		  else if (mstring_equalPrefix (scomment, "notfunction"))
		    {
		      notfunction = TRUE;
		      expectmacro = TRUE;

		      *comment = '/'; /* remove marker */
		      *(comment + 1) = '*';
		      lastControl = cstring_makeLiteralTemp ("notfunction");

		      NEWCHAR ('*');
		      NEWCHAR ('/');
		    }
		  else
		    {
		      char sChar = *scomment;

		      if (sChar == '=' 
			  || sChar == '-'
			  || sChar == '+')
			{
			  char *rest = scomment + 1;

			  if (mstring_equalPrefix (rest, "commentchar"))
			    {
			      if (sChar == '=')
				{
				  *comment = '/'; /* remove marker */
				  *(comment + 1) = '*';
				  NEWCHAR ('*');
				  NEWCHAR ('/');

				  ppllerror (cstring_makeLiteral
					     ("Cannot restore commentchar"));
				}
			      else
				{
				  char *next = scomment + 12; /* strlen commentchar = 12 */
				  DPRINTF (("next: :%s:", next));
				  
				  if (*next != ' ' && *next != '\t' && *next != '\n')
				    {
				      ppllerror
					(message
					 ("Syntactic commentchar comment is not followed by a "
					  "whitespace character: %s", cstring_fromChars (comment)));
				    }
				  else
				    {
				      char cchar = *(next + 1);
				      
				      if (cchar == '\0')
					{
					  ppllerror
					    (cstring_makeLiteral
					     ("Cannot set commentchar to NUL"));
					}
				      else
					{
					  context_setCommentMarkerChar (cchar);
					  setComment = TRUE;
					}
				    }
				  
				  *comment = '/'; /* remove marker */
				  *(comment + 1) = '*';
				  NEWCHAR ('*');
				  NEWCHAR ('/');
				}
			    }
			  else if (mstring_equalPrefix (rest, "namechecks"))
			    {
			      context_fileSetFlag (FLG_NAMECHECKS,
						   ynm_fromCodeChar (sChar));
				  
			      NEWCHAR ('@');
			      NEWCHAR (' ');
			    }
			  else if (mstring_equalPrefix (rest, "usevarargs"))
			    {
			      context_fileSetFlag (FLG_USEVARARGS,
						   ynm_fromCodeChar (sChar));
				  
			      NEWCHAR ('@');
			      NEWCHAR (' ');
			    }
			  else if (mstring_equalPrefix (rest, "nextlinemacros"))
			    {
			      context_fileSetFlag (FLG_MACRONEXTLINE,
						   ynm_fromCodeChar (sChar));

			      NEWCHAR ('@');
			      NEWCHAR (' ');				  
			    }
			  else if (mstring_equalPrefix (rest, "allmacros")
				   || mstring_equalPrefix (rest, "fcnmacros")
				   || mstring_equalPrefix (rest, "constmacros"))
			    {
			      flagcode fl = identifyMacroFlag (rest);
			      
			      context_fileSetFlag (fl, ynm_fromCodeChar (sChar));
			      notfunction = FALSE;
			      
			      *comment = '/'; /* remove marker */
			      *(comment + 1) = '*';
			      NEWCHAR ('*');
			      NEWCHAR ('/');
			    }
			  else
			    {
			      NEWCHAR ('@');
			      NEWCHAR (' ');
			    }
			}
		      else
			{
			  NEWCHAR ('@');
			  NEWCHAR (' ');
			}
		    }
		 
		  ic = NEXT_CHAR ();
		  llassert (ic == (int)'/');
		  
		  inCommentCommand = FALSE;
		  goto toploop;
		}
	      else
		{
		  ; /* not comment */
		}
	    }

	  if (inCommentCommand && (c == commentMarkerChar)) 
	    {
	      c = ' ';
	    }
	  
	  if (c == '/' && peekc (infile) == '*')
	    {
	      bool hascontinue = FALSE;

	      (void) NEXT_CHAR ();
	      ic = NEXT_CHAR ();
	      c = (char) ic;

	      NEWCHAR (' '); /* fill up the '/''*' */
	      NEWCHAR (' ');

	      if ((c == commentMarkerChar)
		  && !(inMacroParams
		       && (!(context_getFlag (FLG_FCNMACROS) || notfunction))))
		{
		  inCommentCommand = TRUE;
		}
	      else
		{
		  bool gotstar = FALSE;
		  bool gotslash = FALSE;
		  bool couldbelintcomment = context_getFlag (FLG_LINTCOMMENTS);
		  cstring lintcomment = cstring_undefined;

		  if ((c == commentMarkerChar) && inMacroParams)
		    {  /* notfunction && inMacroParams */
		      llerrorlit 
			(FLG_PREPROC,
			 "Stylized comment in parameter list for "
			 "expanded macro.  "
			 "Comment is ignored.  Use +allmacros to "
			 "prevent macro expansion.");
		    }
		  
		  while (ic != EOF)
		    {
		      NEWCHAR (' ');
		      
		      switch (c)
			{
			case '\\':
			  {
			    /* Why was this here? */
			    /* NEWCHAR ('\\'); */
			    hascontinue = TRUE;
			    /*@switchbreak@*/ break;
			  }
			case '*': 
			  gotstar = TRUE; 
			  hascontinue = FALSE;

			  if (gotslash) 
			    {
			      genppllerror
				(FLG_NESTCOMMENT,
				 cstring_makeLiteral 
				 ("Open comment inside comment (likely missing close comment)"));			 
			    }
			  /*@switchbreak@*/ break;
			case '/': 
			  if (gotstar) 
			    {
			      if (hascontinue && !isspace (c))
				{
				  genppllerror 
				    (FLG_CONTINUECOMMENT,
				     cstring_makeLiteral
				     ("Comment close follows continuation marker"));
				}
			      
			      if (couldbelintcomment)
				{
				  char *lcomment = identifyLintComment (lintcomment);

				  if (lcomment != NULL)
				    {
				      OUTCHARS (lcomment);
				    }
				  else
				    {
				      ; /* okay, unrecognize comment */
				    }
				  
				}
			      
			      cstring_free (lintcomment);
			      lintcomment = cstring_undefined;
			      hascontinue = FALSE;
			      goto toploop;
			    }
			  else
			    {
			      gotslash = TRUE;
			    }

			  /*@switchbreak@*/ break;
			case '\n':
			  *(s-1) = '\n';

			  if (hascontinue) 
			    {
			      NEWCHAR ('\\');
			    }

			  hascontinue = FALSE;
			  couldbelintcomment = FALSE;
			  gotstar = FALSE;
			  gotslash = FALSE;
			  ppNextLine (); 
			  /*@switchbreak@*/ break;
			default:
			  if (couldbelintcomment)
			    {
			      if (isupper (c))
				{
				  lintcomment = cstring_appendChar (lintcomment, c);
				}
			      else
				{
				  couldbelintcomment = FALSE;
				}
			    }

			  if (!isspace (c))
			    {
			      hascontinue = FALSE;
			    }

			  gotstar = FALSE;
			  gotslash = FALSE;
			  /*@switchbreak@*/ break;
			}
		      
		      ic = NEXT_CHAR ();
		      c = (char) ic;
		    }
		  
		  ppllerror (cstring_makeLiteral ("End of file inside comment"));
		  *s = '\0';
		  llassert (cstring_isUndefined (lintcomment));

		  /*@-usedef@*/ 
		  if (sbuf[0] == '\0')  /*@=usedef@*/
		    {
		      return NULL;
		    }
		  else
		    {
		      DPRINTF (("comment returns: %s", sbuf));
		      return (sbuf);
		    }
		}
	    }
	}
      else
	{
	  if (c == '\\' && !inEscape)
	    inEscape = TRUE;
	  else
	    inEscape = FALSE;
	}
      
      if (!inEscape)
	if (c == '\"')
	  inQuote = !inQuote;

      NEWCHAR (c);
    }
  
  *s = '\0';

  ppNextLine ();

  /* ic must be defined */
  if (/*@-usedef@*/ic/*@=usedef@*/ == EOF && sbuf[0] == '\0') 
    {
      if (inCommentCommand)
	{
	  ppllerror (cstring_makeLiteral ("End of file inside comment"));
	}
      return (NULL);
    }
  /*@=usedef@*/

  llassert (!inCommentCommand);
  DPRINTF (("getline: %s", sbuf));
  return (sbuf);
}

static /*@dependent@*/ /*@null@*/ char *
  fix_line (/*@returned@*/ /*@dependent@*/ char *s)
{
  DPRINTF (("fix_line: %s / %d %d", s, expectmacro, expectmacronext));

  if (*(s) != '#')
    {
      if (expectmacronext)
	{
	  decLine ();

	  if (context_getFlag (FLG_MACRONEXTLINE))
	    {
	      voptgenerror 
		(FLG_MACRONEXTLINE,
		 message 
		 ("Macro expected in line after %s syntactic comment",
		  lastControl),
		 currentloc);
	    }

	  incLine ();

	  expectmacronext = FALSE;
	}

      if (expectmacro)
	{
	  DPRINTF (("expectmacro: %s", s));
	  expectmacronext = TRUE;
	  expectmacro = FALSE;
	}
    }
  else
    {
      s = dofix_line (s);
    }

  return s;
}

/*@=macroparams@*/

static /*@owned@*/ cstring retbuffer = cstring_undefined;

static /*@exposed@*/ /*@null@*/ char *
  dofix_line (/*@returned@*/ /*@dependent@*/ char *s)
{
  char *os = s;
  static bool reportmac = FALSE;
  char c;

  if (cstring_isDefined (retbuffer))
    {
      cstring_free (retbuffer);
      retbuffer = cstring_undefined;
    }

  s++;			/* skip '#' */

  while (*s == ' ' || *s == '\t')
    {
      s++;
    }
  
  /* starts with command character, is next token define? */

  DPRINTF (("fix: *%s*", s));

  if ((strstr (s, "endif ") == s)
       || (strstr (s, "endif\t") == s)
       || (strcmp (s, "endif") == 0)
       || (strstr (s, "ifdef ") == s)
       || (strstr (s, "ifdef\t") == s)
       || (strstr (s, "ifndef ") == s)
       || (strstr (s, "ifndef\t") == s)
       || (strstr (s, "if ") == s)
       || (strstr (s, "if\t") == s)
       || (strstr (s, "else ") == s)
       || (strcmp (s, "else") == 0)
       || (strstr (s, "else\t") == s))
    {
      if (strchr (s, '\\') != NULL)
	{
	  definecontinuation = TRUE;
	  return (os);
	}
      else
	{
	  if (dodefs)
	    {
	      defoutc (os);
	    }

	  retbuffer = message ("%q\n%s\n%q",
			       fileloc_previousLineMarker (currentloc),
			       cstring_fromChars (os), 
			       fileloc_lineMarker (currentloc));

	  return (cstring_toCharsSafe (retbuffer));
	}
    }
  else if (strstr (s, "include") == s && (isspace (*(s + 7))))
    {
      s += 7;			/* skip past include */

      while (*s == ' ' || *s == '\t')
	{
	  s++;
	}

      if (*s == '\"' || *s == '<')
	{
	  /*
           ** difference between # include "xxxx"  (1)
           **                and # include <xxxx>  (2)
           **
           **    (1) searches home directory/local include path, then
           **        /usr/include,
           **    (2) searches /usr/include.
           **
           ** for pcpp, just convert <xxxx> --> "/usr/include/xxxx"
           ** I believe this is semantically identical.
           */
	  bool local = (*s == '\"');
	  char *searchpath;
	  char *sf;
	  char *fname;
	  char *ofname;
	  	  
	  s++;
	  sf = s;
	  
	  while (((c = *s) != ' ') && c != '>'
		 && c != '\"' && c != '\t' && c != '\n')
	    {
	      s++;
	    }
	  
	  if ((local && (c != '\"')) ||
	      (!local && (c != '>')))
	    {
	      ppllerror (message ("Bad include name: %s (ignored)\n", 
				  cstring_fromChars (os)));

	      return NULL;
	    }
	  
	  *s = '\0';

	  if (context_getFlag (FLG_NEVERINCLUDE))
	    {
	      char *ext = filenameExtension (sf);

	      if (mstring_equal (ext, ".h")
		  || mstring_equal (ext, ".lh"))
		{
		  retbuffer = message ("/*-%s*/", 
				       cstring_fromChars (sf));
		  return cstring_toCharsSafe (retbuffer);
		}
	    }
	      
	  if (*sf == '/') /* complete path was given */
	    {
	      fname = mstring_copy (sf);
	      ofname = fname;
	    }
	  else
	    {
	      /* this is needed to avoid bug in gcc -O2 ??? */
	      /* more experiments later, eh? */
	      
	      fname = (char *) dmalloc (sizeof (*fname));
	      ofname = fname;

	      if (local)
		{
		  searchpath = cstring_toCharsSafe (localIncludePath);
		}
	      else
		{
		  searchpath = cstring_toCharsSafe (includePath);
		}

	      while (*sf == '.' && *sf != '\0' && *(sf + 1) == '/') 
		{
		  sf += 2;
		}
	      
	      if (mstring_equalPrefix (sf, "../"))
		{
		  char *curpath = cstring_toCharsSafe (fileName (currentFile ()));
		  char *last;
		  
		  llassert (curpath != NULL);

		  last = strrchr (curpath, '/');
		  
		  if (last == (char *) 0)
		    {
		      ;
		    }
		  else
		    {
		      static /*@owned@*/ cstring nline = cstring_undefined;

		      if (cstring_isDefined (nline))
			{
			  cstring_free (nline);
			}

		      /*@-modobserver@*/
		      *last = '\0';
		      nline = message ("%s/%s", cstring_fromChars (curpath), 
				       cstring_fromChars (sf));
		      sf = cstring_toCharsSafe (nline);
		      *last = '/';
		      /*@=modobserver@*/
		      /* The modification is undone. */
		    }
		}
	      else
		{
		  if (local)
		    {
		      char *curpath = cstring_toCharsSafe (fileName (currentFile ()));
		      char *last = strrchr (curpath, '/');
		      
		      if (curpath != NULL && last != NULL)
			{
			  static /*@owned@*/ cstring npath = cstring_undefined;
			  
			  if (cstring_isDefined (npath))
			    {
			      cstring_free (npath);
			    }

			  /*@-modobserver@*/
			  *last = '\0';
			  npath = message ("%s:%s", cstring_fromChars (curpath), 
					   cstring_fromChars (searchpath));
			  searchpath = cstring_toCharsSafe (npath);
			  *last = '/';
			  /*@=modobserver@*/
			}
		    }
		}

	      if (osd_getPath (searchpath, sf, &fname) == OSD_FILENOTFOUND)
		{
		  if (fileloc_isLib (currentloc))
		    {
		      sfree (ofname); 
		      return NULL;
		    }
		  else
		    {
		      /*
		      ** Return the include so C pre-processor will
		      ** produce message if it really can't be found.
		      */

		      if (lcllib_isSkipHeader (sf)
			  || (!local && lcllib_isSkipHeader (fname)))
			{
			  DPRINTF (("standard library: %s", fname));
			  
			  sfree (ofname);
			  return NULL;
			}
		      else
			{
			  retbuffer = 
			    message ("%q\n#include \"%s\"\n%q",
				     fileloc_previousLineMarker (currentloc),
				     cstring_fromChars (sf),
				     fileloc_lineMarker (currentloc));

			  return cstring_toCharsSafe (retbuffer);
			}
		    }
		}
	    }
	  
	  *s = c;
	  s++;
	  
	  if (context_isSystemDir (cstring_fromChars (fname))
	      && lcllib_isSkipHeader (fname))
	    {
	      DPRINTF (("standard library: %s", fname));

	      sfree (ofname);
	      return NULL;
	    }

	  if (context_getFlag (FLG_SKIPSYSHEADERS))
	    {
	      if (context_isSystemDir (cstring_fromChars (fname)))
		{
		  sfree (ofname);
		  return NULL;
		}
	    }

	  fname = removePreDirs (fname);

	  if (fileTable_exists (context_fileTable (), 
				cstring_fromChars (fname)))
	    {
	      fileId fid = fileTable_lookup (context_fileTable (), 
					     cstring_fromChars (fname));
	      
	      if (context_getFlag (FLG_SINGLEINCLUDE))
		{
		  fileId defId = definesMap (fid);
		  
		  if (fileId_isValid (defId))
		    {
		      char *defname = cstring_toCharsSafe (fileName (defId));
		      
		      if (dodefs)
			{
			  defoutFree (message ("%q\n#include \"%s\"\n%q",
					       fileloc_previousLineMarker (currentloc),
					       cstring_fromChars (defname),
					       fileloc_lineMarker (currentloc)));
			}

		      sfree (ofname);

		      retbuffer = message ("%q\n#include \"%s\"\n%q", 
					   fileloc_previousLineMarker (currentloc),
					   cstring_fromChars (defname),
					   fileloc_lineMarker (currentloc));  
		      return (cstring_toCharsSafe (retbuffer));
		    }
		  else
		    {
		      ppllerror (message ("Non-header file #include'd: %s", 
					  cstring_fromChars (fname)));
		      
		      defId = preMap (fid);
		      
		      if (fileId_isValid (defId))
			{
			  char *defname = cstring_toCharsSafe (fileName (defId));
			  
			  if (dodefs)
			    {
			      defoutFree
				(message ("%q\n#include \"%s\"\n%q", 
					  fileloc_previousLineMarker (currentloc),
					  cstring_fromChars (defname),
					  fileloc_lineMarker (currentloc)));
			    }

			  sfree (ofname);
			  
			  retbuffer = 
			    message ("%q\n#include \"%s\"\n%q",
				     fileloc_previousLineMarker (currentloc),
				     cstring_fromChars (defname),
				     fileloc_lineMarker (currentloc));

			  return (cstring_toCharsSafe (retbuffer));
			}
		    }
		}
	      else
		{
		  sfree (ofname);

		  retbuffer = 
		    message ("%q\n#include \"%s\"\n%q",
			     fileloc_previousLineMarker (currentloc),
			     fileTable_getName (context_fileTable (), 
						preMap (fid)),
			     fileloc_lineMarker (currentloc));

		  return (cstring_toCharsSafe (retbuffer));
		}
	    }
	  else
	    {
	      if (file_isReadable (fname))
		{
		  fileId fid  = fileTable_addHeaderFile
		    (context_fileTable (), cstring_fromChars (fname));
		  fileId tfid = addMap (fid);
		  fileloc origloc = fileloc_copy (currentloc);

		  if (context_getFlag (FLG_SINGLEINCLUDE))
		    {
		      fileId dfid = addDefines (fid);

		      (void) pcpp_inner_file (fid, tfid, dfid);
		      
		      if (dodefs)
			{
			  defoutFree 
			    (message ("%q\n#include \"%s\"\n%q",
				      fileloc_previousLineMarker (origloc),
				      fileName (dfid),
				      fileloc_lineMarker (origloc)));
			}
		    }
		  else
		    {
		      (void) pcpp_inner_file (fid, tfid, fileId_invalid);
		    }
		  
		  fileloc_reallyFree (currentloc);
		  currentloc = origloc;

		  retbuffer = 
		    message ("%q\n#include \"%s\"\n%q",
			     fileloc_previousLineMarker (origloc),
			     fileName (tfid),
			     fileloc_lineMarker (origloc));

		  sfree (ofname);
		  return cstring_toCharsSafe (retbuffer);
		}
	      else
		{
		  if (fileloc_isLib (currentloc))
		    {
		     ;
		    }
		  else
		    {
		      if (*(sf) == '/')
			{
			  ppllerror (message ("Cannot find include file %s",
					      sf - 1));
			}
		      else
			{
			  /*@-usedef  searchpath must be defined @*/
			  ppllerror 
			    (message ("Cannot find include file %s on path: %s", 
				      sf - 1,
				      cstring_fromChars (searchpath)));
			  /*@=usedef@*/
			}
		    }
		  return (os);
		}
	    }
	  sfree (ofname);
	}
      else
	{
	  if (!reportmac)
	    {
	      reportmac = TRUE;
	      
	      decLine ();
	      
	      llerrorlit
		(FLG_PREPROC,
		 "Warning: #include syntax using macro (specified functions "
		 "in included file may not "
		 "be handled correctly, further instances unreported)");
	      
	      context_decError ();
	      incLine ();
	    }
	  
	  if (dodefs)
	    {
	      defoutc (os);
	    }

	  return (os);
	}
    }
  else if (strstr (s, "define") == s && (isspace (*(s + 6))))
    {
      bool checkmacro = FALSE;
      bool hasParams = FALSE;
      char *sdef = s;
      cstring sname;

      if (expectmacronext && cstring_equalLit (lastControl, "iter"))
	{
	  expectenditer = TRUE;
	  noexpand = TRUE;
	  DPRINTF (("expect end iter!"));
	}
      else
	{
	  expectenditer = FALSE;
	}

      expectmacro = FALSE;
      expectmacronext = FALSE;
      
      s += 6;		/* skip past define */
      
      while (*s == ' ' || *s == '\t')
	{
	  s++;
	}
      
      sname = cstring_fromChars (s);
      
      while (((c = *s) != ' ')
	     && c != '\0' && c != '(' 
	     && c != '\t' && c != '\\' && c != '\n'
	     && !iscntrl (c))
	{
	  s++;
	}
      
      hasParams = (c == '(');
      *s = '\0';

      if (notparseable)
	{
	  ;
	}
      else if (notfunction || fileloc_isStandardLib (currentloc))
	{
	  ;
	}
      else
	{
	  if (noexpand)
	    {
	      DPRINTF (("check macro: %s", sname));
	      checkmacro = TRUE;

	      if (!expectenditer)
		{
		  noexpand = FALSE;
		}
	    }
	  else
	    {
	      if (usymtab_existsReal (sname))
		{
		  uentry ue = usymtab_lookup (sname);
		  
		  if (uentry_isSpecified (ue))
		    {
		      checkmacro = context_getFlag (FLG_SPECMACROS);
		    }
		  else
		    {
		      if (hasParams)
			{
			  checkmacro = context_getFlag (FLG_LIBMACROS)
			    || context_getFlag (FLG_FCNMACROS);
			  DPRINTF (("checkmacro -> %d", checkmacro));
			}
		    }
		}
	      else
		{
		  if (fileloc_isSystemFile (currentloc)
		      && context_getFlag (FLG_SYSTEMDIREXPAND))
		    {
		      ; /* don't check this macro */
		    }
		  else
		    {
		      uentry le;
		      
		      if (hasParams)
			{
			  if (context_getFlag (FLG_FCNMACROS))
			    {
			      fileloc loc = fileloc_makePreproc (currentloc);
			      
			      DPRINTF (("function macro: %s", sname));
			      
			      /* the line is off-by-one, since the newline was already read */
			      decLine ();
			      
			      le = uentry_makeForwardFunction (sname, typeId_invalid, loc);
			      fileloc_free (loc);
			      
			      incLine ();
			      
			      /* Do not define here! */
			      
			      (void) usymtab_addEntry (le);
			      DPRINTF (("forward function: %s", uentry_unparseFull (le)));
			      checkmacro = TRUE;
			    }
			}
		      else
			{
			  if (context_getFlag (FLG_CONSTMACROS))
			    {
			      fileloc loc = fileloc_makePreproc (currentloc);
			      bool nocontent = FALSE;
			      
			      if (c == '\0')
				{
				  nocontent = TRUE;
				}
			      else
				{
				  if (isspace (c))
				    {
				      char *rest = s + 1;
				      
				      /*
				       ** Check if there is nothing after the define.
				       */
				      
				      while ((*rest) != '\0' && isspace (*rest))
					{
					  rest++;
					}
				      
				      if (*rest == '\0')
					{
					  nocontent = TRUE; /* empty macro, don't check */
					}
				    }
				}
			      
			      /* the line is off-by-one, since newline was already read */
			      decLine ();
			      le = uentry_makeConstant (sname, ctype_unknown, loc);
			      incLine ();
			      
			      (void) usymtab_addEntry (le);
			      checkmacro = !nocontent;
			    }
			}
		    }
		  
		  if (checkmacro && usymtab_existsType (sname))
		    {
		      decLine (); 
		      ppllerror (message ("Specified type implemented as macro: %s", sname));
		      checkmacro = FALSE;
		      incLine ();
		    }
		}
	    }
	}
      
      if (checkmacro)
	{
	  *sdef++ = LLMRCODE[0];	/* d */
	  *sdef++ = LLMRCODE[1];	/* e */
	  *sdef++ = LLMRCODE[2];	/* f */
	  *sdef++ = LLMRCODE[3];	/* i */
	  *sdef++ = LLMRCODE[4];	/* n */
	  *sdef++ = LLMRCODE[5];	/* e */
	  *os = ' ';	/* replace '#' */
	  *s = c;       /* restore char */
	}
      else
	{
	  DPRINTF (("don't check: %s", sname));

	  if (usymtab_exists (sname))
	    {
	      uentry ue = usymtab_lookupExpose (sname);
	      fileloc tloc = fileloc_makePreproc (currentloc);
	      uentry_setDefined (ue, tloc);
	      fileloc_free (tloc);
	      uentry_setUsed (ue, fileloc_undefined);
	    }
	  else
	    {
	      fileloc tloc = fileloc_makePreprocPrevious (currentloc);
	      uentry ue = uentry_makeExpandedMacro (sname, tloc);
	      cstring line = message ("#pragma %s %s\n%q", 
				      cstring_fromChars (PRAGMA_EXPAND),
				      sname, 
				      fileloc_lineMarker (tloc));

	      /*
	      ** Need to make expand a pragma, to avoid messing up
	      ** some pre-processors.
	      */

	      out_line (line);
	      cstring_free (line);

	      fileloc_free (tloc);
	      DPRINTF (("add expanded: %s", uentry_unparse (ue)));
	      (void) usymtab_addGlobalEntry (ue);	      
	    }

	  *s = c; /* restore char */

	  if (notparseable)
	    {
	      cstring ret = cstring_makeLiteral ("/* [not parseable] ");

	      if (continuationLine (os))
		{
		  ret = cstring_appendChar (ret, '\n');

		  while (continuationLine (get_line (TRUE)))
		    {
		      ret = cstring_appendChar (ret, '\n');
		    }
		}

	      ret = cstring_appendChar (ret, '*');
	      ret = cstring_appendChar (ret, '/');

	      out_line (ret);
	      cstring_free (ret);

	      *os = '\0';
	    }
	  else
	    {
	      if (dodefs)
		{
		  if (continuationLine (os))
		    {
		      definecontinuation = TRUE;
		    }
		  else
		    {
		      defoutc (os);
		    }
		}
	    }
	}
      
      notfunction = FALSE;
      notparseable = FALSE;
    }
  else
    {
      if (dodefs)
	{
	  if (continuationLine (os))
	    {
	      definecontinuation = TRUE;
	    }
	  else
	    {
	      defoutc (os);
	    }
	}
    }
  
  return (os);
}

static void
do_pcpp (bool inner)
{
  char *s;

  curline = 0;

  while ((s = get_line (inner)) != NULL)
    {
      curline++;

      if (!last_in_comment) 
	{
	  s = fix_line (s);
	}

      if (dodefs && definecontinuation)
	{
	  if (s != NULL)
	    {
	      defoutc (s);
	      
	      if (strrchr (s, '\\') == NULL)
		{
		  definecontinuation = FALSE;
		}
	    }
	  else
	    {
	      definecontinuation = FALSE;
	    }
	}

      out_line (cstring_fromChars (s));
      last_in_comment = in_comment;
    }

  if (expectmacro || expectmacronext)
    {
      ppllerror (message ("File ends while a macro is expected in "
			  "line after %s syntactic comment",
			  lastControl));
      expectmacro = FALSE;
      expectmacronext = FALSE;
    }
  else if (expectenditer)
    {
      ppllerror
	(message ("File ends while a macro (to define end_<iter>) is expected in "
		  "line after %s syntactic comment",
		  lastControl));
      expectmacro = FALSE;
      expectmacronext = FALSE;
      expectenditer = FALSE;
    }
  else
    {
      ; /* okay */
    }

}

static fileId
pcpp_file (fileId fid)
{
  fileId ofid = fileId_fromInt (indxMap_mapSafe (ppMap, fileId_toInt (fid)));

  if (!fileId_equal (ofid, fid)) /* already pre-processed */
    {
     ;
    }
  else
    {
      if (open_file_r (fid))
	{
	  llassert (infile != NULL);
	  ofid = fileTable_addTempFile (context_fileTable (), fid);
	  indxMap_addpair (ppMap, fileId_toInt (fid), fileId_toInt (ofid));

	  if (open_file_w (ofid))
	    {
	      cstring line;
	      llassert (outfile != NULL);
	      context_setFileId (fid);
	      line = message ("#line 1 \"%s\"", fileName (fid));
	      out_line (line);
	      cstring_free (line);
	      do_pcpp (FALSE);
	      check (fclose (outfile) == 0);
	    }
	  
	  check (fclose (infile) == 0);
	}
      else
	{
	  ppllmsg (message ("Source file not found: %s", fileName (fid)));
	  ofid = fileId_invalid;
	  indxMap_addpair (ppMap, fileId_toInt (fid), fileId_toInt (ofid));
	}
    }

  context_exitPPFile ();
  return ofid;
}

fileIdList
  pcpp_fileIdList (fileIdList fl, cstring cppcmd)
  /*@globals undef killed ppMap, undef killed defMap, 
             undef killed fstack; @*/
  /*@modifies fileSystem@*/
{
  bool msg = (context_getFlag (FLG_SHOWSCAN) && fileIdList_size (fl) > 10);
  int skip = (fileIdList_size (fl) / 5);
  int filesprocessed = 0;
  fileId ppid;
  fileIdList ppfiles = fileIdList_create ();
  fileIdList dfiles = fileIdList_create ();

  ppMap = indxMap_create (); 

  if (context_getFlag (FLG_SINGLEINCLUDE))
    {
      defMap = indxMap_create ();
    }

  if (sbuf == NULL)
    {
      sbuf = (char *) dmalloc (sizeof (*sbuf) * sbuf_size);
    }
  
  fstack = fileStack_create ();

  /*
  ** create the temp files
  */
  
  fileIdList_elements (fl, fid)
    {
      if (msg)
	{
	  if ((filesprocessed % skip) == 0) 
	    {
	      if (filesprocessed == 0) fprintf (stderr, " ");
	      else fprintf (stderr, ".");
	    }
	  filesprocessed++;
	}
      
      ppid = pcpp_file (fid);

      if (fileId_isValid (ppid))
	{
	  fileIdList_add (ppfiles, ppid);
	}

    } end_fileIdList_elements; 
  
  /*
  ** use CPP_COMMAND to process # includes and # defines
  */
  
  fileIdList_elements (ppfiles, fid)
    {
      fileId  dfile = fileTable_addCTempFile (context_fileTable (), fid);
      char *ppfname = cstring_toCharsSafe (fileName (fid));

      llassert (!mstring_isEmpty (ppfname));

      if (!(file_isReadable (ppfname)))
	{
	  llbug (message ("pcpp: missing file %s", cstring_fromChars (ppfname)));
	}
      else
	{
	  cstring syscal = message ("%s %s > %s", cppcmd, 
				    cstring_fromChars (ppfname),
				    fileName (dfile));
	  
	  DPRINTF (("syscall: %s", syscal));
	  (void) system (cstring_toCharsSafe (syscal));
	  
	  cstring_free (syscal);
	  fileIdList_add (dfiles, dfile);
	}
    } end_fileIdList_elements; 
  
  fileIdList_free (ppfiles);
  indxMap_free (ppMap);
  fileStack_annihilate (fstack);

  if (context_getFlag (FLG_SINGLEINCLUDE))
    {
      /*@-usedef@*/ indxMap_free (defMap); /*@=usedef@*/
    }

  if (cstring_isDefined (retbuffer))
    {
      cstring_free (retbuffer);
      retbuffer = cstring_undefined;
    }

  return dfiles;
}

static 
bool pcpp_inner_file (fileId fid, fileId ofid, fileId dfid)
{
  bool ret = TRUE;
  int restoreline = curline;

  fileStack_push (fstack, infile, outfile, curfile, curline, deffile);
  
  definecontinuation = FALSE;

  if (!fileStack_contains (fstack, fid))
    {
      bool lastdodefs = dodefs;
      
      if (!open_file_rw (fid, ofid))
	{
	  if (context_getFlag (FLG_SHOWSCAN))
	    {
	      fprintf (stderr, " >\n");
	    }

	  llfatalerror (message ("Cannot open %s\n", fileName (fid)));
	}

      if (fileId_isValid (dfid))
	{
	  open_definesFile (dfid);
	}
      
      /* get it onto old stack! */

      fileStack_pushOld (fstack, curfile);
      
      context_setFileId (fid);
      do_pcpp (TRUE);

      llassert (infile != NULL && outfile != NULL);
      
      check (fclose (infile) == 0);
      check (fclose (outfile) == 0);

      if (dodefs)
	{
	  llassert (deffile != NULL);
	  check (fclose (deffile) == 0);
	}
      dodefs = lastdodefs;
    }
  else
    {
      ret = FALSE;
    }
  
  curline = restoreline;
  curfile = fileStack_pop (fstack, &infile, &outfile, &deffile);

  /*
  ** warning:  DON'T context_exitFile () here
  ** so control comments behave as expected across includes
  */

  return ret;
}


