/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** messageLog.c (from slist_template.c)
*/

# include "lclintMacros.nf"
# include "basic.h"

/*@only@*/ messageLog
messageLog_new ()
{
  messageLog s = (messageLog) dmalloc (sizeof (*s));
  
  s->nelements = 0;
  s->nspace = messageLogBASESIZE;
  s->elements = (msgentry *) dmalloc (sizeof (*s->elements) * messageLogBASESIZE);

  return (s);
}

static /*@only@*/ msgentry
msgentry_create (fileloc loc, cstring mess)
{
  msgentry msg = (msgentry) dmalloc (sizeof (*msg));

  msg->loc = fileloc_copy (loc);
  msg->msg = cstring_copy (mess);

  return msg;
}

static void msgentry_free (/*@only@*/ msgentry msg)
{
  fileloc_free (msg->loc);
  cstring_free (msg->msg);
  sfree (msg);
}

/*
** returns TRUE if m1 < m2
*/

static bool
msgentry_lessthan (msgentry m1, msgentry m2)
{
  return (fileloc_lessthan (m1->loc, m2->loc)
	  || (fileloc_equal (m1->loc, m2->loc) 
	      && (cstring_lessthan (m1->msg, m2->msg))));
}

static bool
msgentry_equal (msgentry m1, msgentry m2)
{
  return (fileloc_equal (m1->loc, m2->loc) &&
	  cstring_equal (m1->msg, m2->msg));
}

/*
** returns highest index of element less than msg
*/

static int
messageLog_index (messageLog s, msgentry msg)
{
  int high;
  int low  = 0;

  llassert (messageLog_isDefined (s));

  high = s->nelements - 1;

  for (low = high; low >= 0; low--)
    {
      if (msgentry_lessthan (s->elements[low], msg))
	{
	  return low;
	}
    }

  return -1;
# if 0      
  while (low < high)
    {
      int mid = (low + high + 1) / 2;

      if (msgentry_lessthan (s->elements[mid], msg)) /* mid < msg */
	{
	  if (high == mid) break;
	  high = mid;
	}
      else
	{
	  if (low == mid) break;
	  low = mid;
	}
    }

  return low - 1;
# endif
}

static void
messageLog_grow (/*@notnull@*/ messageLog s)
{
  int i;
  msgentry *newelements;
  
  s->nspace += messageLogBASESIZE; 
  newelements = (msgentry *) dmalloc (sizeof (*newelements) * (s->nelements + messageLogBASESIZE));
  
  for (i = 0; i < s->nelements; i++)
    {
      newelements[i] = s->elements[i];
    }
  
  sfree (s->elements);
  s->elements = newelements;
}

bool messageLog_add (messageLog s, fileloc fl, cstring mess)
{
  msgentry msg = msgentry_create (fl, mess);
  int ind, i;

  llassert (messageLog_isDefined (s));

  ind = messageLog_index (s, msg);

  if (ind + 1 < s->nelements)
    {
      if (msgentry_equal (msg, s->elements[ind + 1]))
	{
	  msgentry_free (msg);
	  return FALSE;
	}
    }

  if (s->nspace <= 0)
    messageLog_grow (s);

  for (i = s->nelements; i > ind + 1; i--)
    {
      s->elements[i] = s->elements[i-1];
    }
  
  s->elements[ind + 1] = msg;
  s->nspace--;
  s->nelements++;

  return TRUE;
}

/*@only@*/ cstring
messageLog_unparse (messageLog s)
{
   int i;
   cstring st = cstring_makeLiteral ("[");

   if (messageLog_isDefined (s))
     {
       for (i = 0; i < s->nelements; i++)
	 {
	   if (i == 0)
	     {
	       st = message ("%q %q", st, fileloc_unparseDirect (s->elements[i]->loc));
	     }
	   else
	     st = message ("%q, %q", st, fileloc_unparseDirect (s->elements[i]->loc));
	 }
     }

   st = message ("%q ]", st);
   return st;
}

void
messageLog_free (messageLog s)
{
  if (s != NULL)
    {
      int i;

      for (i = 0; i < s->nelements; i++)
	{
	  msgentry_free (s->elements[i]);
	}
      
      sfree (s->elements); 
      sfree (s);
    }
}
