/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/

/*
** lslOpSet.c
**
** based on set_template.c
**
** where T has T_equal (or change this) and T_unparse
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "checking.h"		/* for lslOp_equal */

static bool lslOpSet_member (lslOpSet p_s, lslOp p_el);

lslOpSet lslOpSet_new ()
{
  lslOpSet s = (lslOpSet) dmalloc (sizeof (*s));

  s->entries = 0;
  s->nspace = lslOpSetBASESIZE;
  s->elements = (lslOp *)
    dmalloc (sizeof (*s->elements) * lslOpSetBASESIZE);
  
  return (s);
}

static /*@only@*/ lslOpSet
lslOpSet_predict (int size)
{
  lslOpSet s = (lslOpSet) dmalloc (sizeof (*s));
  
  s->entries = 0;

  if (size > 0)
    {
      s->nspace = size;
      s->elements = (lslOp *) dmalloc (sizeof (*s->elements) * size);
    }
  else
    {
      s->nspace = 0;
      s->elements = NULL;
    }

  return (s);
}

static void
lslOpSet_grow (/*@notnull@*/ lslOpSet s)
{
  int i;
  lslOp *newelements;

  s->nspace = lslOpSetBASESIZE;
  newelements = (lslOp *) dmalloc (sizeof (*newelements)
					 * (s->entries + lslOpSetBASESIZE));

  if (newelements == (lslOp *) 0)
    {
      llfatalerror (cstring_makeLiteral ("lslOpSet_grow: out of memory!"));
    }

  for (i = 0; i < s->entries; i++)
    {
      newelements[i] = s->elements[i];
    }

  sfree (s->elements); 
  s->elements = newelements;
}

/*
** Ensures: if *e \in *s
**          then unchanged (*s) & result = false
**          else *s' = insert (*s, *e) & result = true
** Modifies: *s
*/

bool
lslOpSet_insert (lslOpSet s, /*@only@*/ lslOp el)
{
  llassert (lslOpSet_isDefined (s));

  if (lslOpSet_member (s, el))
    {
      lslOp_free (el);
      return FALSE;
    }
  else
    {
      if (s->nspace <= 0)
	lslOpSet_grow (s);
      s->nspace--;
      s->elements[s->entries] = el;
      s->entries++;
      return TRUE;
    }
}

static bool
lslOpSet_member (lslOpSet s, lslOp el)
{
  if (lslOpSet_isDefined (s))
    {
      int i;
      
      for (i = 0; i < lslOpSet_size (s); i++)
	{
	  if (lslOp_equal (el, s->elements[i]))
	    return TRUE;
	}
    }

  return FALSE;
}

/*@only@*/ cstring
lslOpSet_unparse (lslOpSet s)
{
  if (lslOpSet_isDefined (s))
    {
      int i;
      cstring st = cstring_makeLiteral ("{");
      
      for (i = 0; i < lslOpSet_size (s); i++)
	{
	  st = message ("%q   %q", st, lslOp_unparse (s->elements[i]));
	}
      
      st = message ("%q}", st);
      return st;
    }
  else
    {
      return (cstring_makeLiteral ("{ }"));
    }
}

/*@only@*/ lslOpSet
lslOpSet_copy (lslOpSet s)
{
  if (lslOpSet_isDefined (s))
    {
      lslOpSet t = lslOpSet_predict (lslOpSet_size (s));
      int i;
      
      for (i = 0; i < lslOpSet_size (s); i++)
	{
	  (void) lslOpSet_insert (t, lslOp_copy (s->elements[i])); 
	}
      
      return t;
    }
  else
    {
      return lslOpSet_undefined;
    }
}

void
lslOpSet_free (lslOpSet s)
{
  if (lslOpSet_isDefined (s))
    {
      int i;
      for (i = 0; i < s->entries; i++)
	{
	  lslOp_free (s->elements[i]); 
	}
      
      sfree (s->elements); 
      sfree (s);
    }
}
