/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** llmain.c
**
** Main module for LCLint checker
*/

# include <time.h>
# include <signal.h>
# include "lclintMacros.nf"
# include "llbasic.h"
# include "osd.h"
# include "gram.h"
# include "lclscan.h"
# include "scanline.h"
# include "lclscanline.h"
# include "lclsyntable.h"
# include "lcltokentable.h"
# include "lslparse.h"
# include "scan.h"
# include "syntable.h"
# include "tokentable.h"
# include "lslinit.h"
# include "lclinit.h"
# include "lh.h"
# include "imports.h"
# include "version.h"
# include "herald.h"
# include "fileIdList.h"
# include "pcpp.h"
# include "lcllib.h"
# include "cgrammar.h"
# include "llglobals.h"
# include "llmain.h"

extern /*@external@*/ int yydebug;

static /*@only@*/ cstring cppcmd;
static void printMail (void);
static void printMaintainer (void);
static void printReferences (void);
static void printFlags (void);
static void printAnnotations (void);
static void printComments (void);
static void describePrefixCodes (void);
static void cleanupFiles (void);
static void showHelp (void);
static void interrupt (int p_i);
static void loadrc (FILE *);
static void describeVars (void);
static bool specialFlagsHelp (char *p_next);
static bool hasShownHerald = FALSE;

static bool anylcl = FALSE;
static /*@only@*/ /*@null@*/ tsource *initFile = (tsource *) 0;

static
void lslCleanup (void)
   /*@globals killed symtab@*/
   /*@modifies internalState, symtab@*/
{
  /*
  ** Cleanup all the LCL/LSL.
  */

  static bool didCleanup = FALSE;

  llassert (!didCleanup);
  llassert (anylcl);

  didCleanup = TRUE;

  lsymbol_destroyMod ();
  LCLSynTableCleanup ();
  LCLTokenTableCleanup ();
  LCLScanLineCleanup ();
  LCLScanCleanup ();

  /* clean up LSL parsing */

  lsynTableCleanup ();
  ltokenTableCleanup ();
  lscanLineCleanup ();
  LSLScanCleanup ();

  symtable_free (symtab);
  sort_destroyMod (); 
}

static clock_t inittime;

static
  void lslInit (void)
  /*@globals undef symtab; @*/
  /*@modifies symtab, internalState, fileSystem; @*/
{
  /*
  ** Open init file provided by user, or use the default LCL init file 
  */
  
  char *larchpath = NULL;
  tsource *LSLinitFile = (tsource *) 0;

  setCodePoint ();
  larchpath = getLarchPath ();

  if (initFile == (tsource *) 0)
    {
      initFile = tsource_create (INITFILENAME, LCLINIT_SUFFIX, FALSE);
      
      if (!tsource_getPath (larchpath, initFile, ltoken_undefined))
	{
	  llmsg (message ("Continuing without LCL init file: %s",
			  cstring_fromChars (tsource_fileName (initFile))));
	}
      else 
	{
	  if (!tsource_open (initFile))
	    {
	      llmsg (message ("Continuing without LCL init file: %s",
			      cstring_fromChars (tsource_fileName (initFile))));
	    }
	}
    }
  else 
    {
      if (!tsource_open (initFile))
	{
	  llmsg (message ("Continuing without LCL init file: %s",
			  cstring_fromChars (tsource_fileName (initFile))));
	}
    }
  
  setCodePoint ();
  
  /* Initialize checker */

  lsymbol_initMod ();
  LCLSynTableInit ();

  setCodePoint ();

  LCLSynTableReset ();
  LCLTokenTableInit ();

  setCodePoint ();

  LCLScanLineInit ();
  setCodePoint ();
  LCLScanLineReset ();
  setCodePoint ();
  LCLScanInit ();

  setCodePoint ();

  /* need this to initialize LCL checker */
  llassert (initFile != NULL);
      
  if (tsource_isOpen (initFile))
    {
      setCodePoint ();

      LCLScanReset (initFile);
      LCLProcessInitFileInit ();
      LCLProcessInitFileReset ();

      setCodePoint ();
      LCLProcessInitFile ();
      LCLProcessInitFileCleanup ();

      setCodePoint ();
    }
  
  /* Initialize LSL init files, for parsing LSL signatures from LSL */
  
  LSLinitFile = tsource_create ("lslinit.lsi", ".lsi", FALSE);
  
  if (!tsource_getPath (larchpath, LSLinitFile, ltoken_undefined))
    {
      llmsg (message ("Continuing without LSL init file: %s",
		      cstring_fromChars (tsource_fileName (LSLinitFile))));
    }
  else 
    {
      if (!tsource_open (LSLinitFile))
	{
	  llmsg (message ("Continuing without LSL init file: %s",
			  cstring_fromChars (tsource_fileName (LSLinitFile))));
	}
    }
      
  setCodePoint ();
  lsynTableInit ();
  lsynTableReset ();

  setCodePoint ();
  ltokenTableInit ();

  setCodePoint ();
  lscanLineInit ();
  lscanLineReset ();
  LSLScanInit ();

  if (tsource_isOpen (LSLinitFile))
    {
      setCodePoint ();
      LSLScanReset (LSLinitFile);
      LSLProcessInitFileInit ();
      setCodePoint ();
      LSLProcessInitFile ();
      setCodePoint ();
    }
      
  (void) tsource_close (LSLinitFile);
  
  if (lclHadError ())
    {
      lclplainerror 
	(cstring_makeLiteral ("LSL init file error.  Attempting to continue."));
    }
  
  setCodePoint ();
  symtab = symtable_new ();
  
  /* 
  ** sort_init must come after symtab has been initialized 
  */
  sort_init ();
  abstract_init ();
  setCodePoint ();
  
  inittime = clock ();
  
  /* 
  ** Equivalent to importing old spec_csupport.lcl
  ** define immutable LCL type "bool" and bool constants TRUE and FALSE
  ** and initialized them to be equal to LSL's "true" and "false".
  **
  ** Reads in CTrait.syms (derived from CTrait.lsl) on LARCH_PATH.
  */
      
  LCLBuiltins (); 
  LCLReportEolTokens (FALSE);
}

static void
lslProcess (fileIdList lclfiles)
   /*@globals undef currentSpec, undef currentSpecName, currentloc,
              undef killed symtab; @*/
   /*@modifies currentSpec, currentSpecName, currentloc, internalState, fileSystem; @*/
{
  char *path = NULL;
  bool parser_status = FALSE;
  bool overallStatus = FALSE;

  lslInit ();

  context_resetSpecLines ();

  fileIdList_elements (lclfiles, fid)
    {
      char *actualName = (char *) dmalloc (sizeof (*actualName));
      char *oactualName = actualName;
      char *fname = cstring_toCharsSafe (fileName (fid));
      
      if (osd_getPath (localSpecPath, fname, &actualName) == OSD_FILENOTFOUND)
	{
	  if (mstring_equal (localSpecPath, "."))
	    {
	      llmsg (message ("Spec file not found: %s",
			      cstring_fromChars (fname)));
	    }
	  else
	    {
	      llmsg (message ("Spec file not found: %s (on %s)", 
			      cstring_fromChars (fname), 
			      cstring_fromChars (localSpecPath)));
	    }
	}
      else
	{
	  tsource *specFile;
	  
	  while (*actualName == '.' && *(actualName + 1) == '/') 
	    {
	      actualName += 2;
	    }
	  
	  specFile = tsource_create (actualName, LCL_SUFFIX, TRUE);
	  llassert (specFile != (tsource *) 0);
	  
	  currentSpec = cstring_fromChars (mstring_copy (actualName));
	  currentSpecName = specFullName (cstring_toCharsSafe (currentSpec),
					  &path);
	  setSpecFileId (fid);
	  	  
	  if (context_getFlag (FLG_SHOWSCAN))
	    {
	      llmsg (message ("< reading spec %s >", currentSpec));
	    }
	  
	  /* Open source file */
	  
	  if (!tsource_open (specFile))
	    {
	      llmsg (message ("Cannot open file: %s",
			      cstring_fromChars (tsource_fileName (specFile))));
	      tsource_free (specFile);
	    }
	  else
	    {
	      scopeInfo dummy_scope = (scopeInfo) dmalloc (sizeof (*dummy_scope));
	      dummy_scope->kind = SPE_INVALID;
	      
	      lhInit (specFile);
	      LCLScanReset (specFile);
	      
	      /* 
              ** Minor hacks to allow more than one LCL file to
	      ** be scanned, while keeping initializations
	      */
	      
	      symtable_enterScope (symtab, dummy_scope);
	      resetImports (cstring_fromChars (currentSpecName));
	      context_enterLCLfile ();
	      (void) lclHadNewError ();
	      
	      parser_status = (ylparse () != 0);
	      context_exitLCLfile ();
	      lhCleanup ();
	      overallStatus = parser_status || lclHadNewError (); 

	      if (context_getFlag (FLG_DOLCS))
		{
		  if (overallStatus)
		    {
		      outputLCSFile (path, "%%FAILED Output from ",
				     currentSpecName);
		    }
		  else
		    {
		      outputLCSFile (path, "%%PASSED Output from ", 
				     currentSpecName);
		    }
		}

	      (void) tsource_close (specFile);
	      symtable_exitScope (symtab);
	    }
	}
      
      sfree (oactualName);
    } end_fileIdList_elements; 
  
  /* Can cleanup lsl stuff right away */

  lslCleanup ();

  currentSpec = cstring_undefined;
  currentSpecName = NULL;
}

static void handlePassThroughFlag (char *arg)
{
  char *curarg = arg;
  char *quotechar = strchr (curarg, '\"');
  int offset = 0;
  bool open = FALSE;

  while (quotechar != NULL)
    {
      if (*(quotechar - 1) == '\\')
	{
	  char *tp = quotechar - 2;
	  bool escape = TRUE;

	  while (*tp == '\\')
	    {
	      escape = !escape;
	      tp--;
	    }
	  
	  if (escape)
	    {
	      curarg = quotechar + 1;
	      quotechar = strchr (curarg, '\"');
	      continue;
	    }
	}
      
      *quotechar = '\0';
      offset = (quotechar - arg) + 2;
      
      if (open)
	{
	  arg = cstring_toCharsSafe
	    (message ("%s\"\'%s", 
		      cstring_fromChars (arg), 
		      quotechar + 1));
	  open = FALSE;
	}
      else
	{
	  arg = cstring_toCharsSafe
	    (message ("%s\'\"%s", 
		      cstring_fromChars (arg), 
		      quotechar + 1));
	  open = TRUE;
	}
      
      curarg = arg + offset;
      quotechar = strchr (curarg, '\"');
    }

  if (open)
    {
      showHerald ();
      llerror (FLG_BADFLAG,
	       message ("Unclosed quote in flag: %s",
			cstring_fromChars (arg)));
    }
  else
    {
      cppcmd = message ("%q -%s", cppcmd, cstring_fromChars (arg));
    }
}

void showHerald (void)
{
  if (hasShownHerald || context_getFlag (FLG_QUIET)) return;

  else
    {
      fprintf (stderr, "%s\n\n", LCL_VERSION);
      hasShownHerald = TRUE;
    }
}

static void addFile (fileIdList files, /*@only@*/ cstring s)
{
  DPRINTF (("addFile: *%s*", s));

  if (fileTable_exists (context_fileTable (), s))
    {
      showHerald ();
      llmsg (message ("File listed multiple times: %s", s));
      cstring_free (s);
    }
  else
    {
      fileIdList_add (files, fileTable_addFileOnly (context_fileTable (), s));
      DPRINTF (("lclfiles: %s", s));
    }
}

int main (int argc, char *argv[])
  /*@globals killed undef currentloc,
             killed       localIncludePath,
	     killed undef initFile,
             killed       includePath, 
	     killed       localSpecPath,  
	     killed undef currentSpec,
	     killed undef currentSpecName,
	     killed undef cppcmd,
	     killed undef yyin,
	     yydebug;
   @*/
  /*@modifies currentloc, localIncludePath, initFile, includePath,
              localSpecPath, currentSpec, currentSpecName, fileSystem,
	      yyin, yydebug; 
  @*/
{
  bool first_time = TRUE;
  bool showhelp = FALSE;
  bool allhelp = TRUE;
  tsource *sourceFile = (tsource *) 0;
  fileIdList dercfiles;
  cstringSList fl = cstringSList_undefined;
  fileIdList cfiles = fileIdList_create ();
  fileIdList lclfiles = fileIdList_create ();
  clock_t before, lcltime, libtime, pptime, cptime, rstime;
  int i = 0;
  cstring ocppcmd;

  flags_initMod ();
  typeIdSet_initMod ();

  setCodePoint ();

  currentloc = fileloc_createBuiltin ();
  ocppcmd = cstring_fromChars (getenv ("LCLINT_CPPCMD"));

  if (!cstring_isDefined (ocppcmd))
    {
      ocppcmd = cstring_makeLiteralTemp (DEFAULT_CPPCMD);
    }

# ifdef NEED_STDC
  cppcmd = cstring_concat (ocppcmd, cstring_makeLiteralTemp (" -D__LCLINT__ -D__STDC__ "));
# else
  cppcmd = cstring_concat (ocppcmd, cstring_makeLiteralTemp (" -D__LCLINT__ "));
# endif

  signal (SIGINT, interrupt);
  signal (SIGSEGV, interrupt); 
  
  before = clock ();

  context_initMod ();

  if (argc <= 1)
    {
      showHelp ();
      llexit (LLGIVEUP);
    }

  setCodePoint ();
  yydebug = 0;

  /*
  ** check RCFILE for default flags
  */

  {
    cstring home = cstring_fromChars (osd_getHomeDir ());
    char *fname  = NULL;
    FILE *rcfile;
    bool defaultf = TRUE;
    bool nof = FALSE;

    for (i = 1; i < argc; i++)
      {
	char *thisarg;
	thisarg = argv[i];
	
	if (*thisarg == '-' || *thisarg == '+')
	  {
	    thisarg++;

	    if (mstring_equal (thisarg, "nof"))
	      {
		nof = TRUE;
	      }
	    else if (mstring_equal (thisarg, "f"))
	      {
		if (++i < argc)
		  {
		    defaultf = FALSE;
		    fname = argv[i];
		    rcfile = fopen (fname, "r");

		    if (rcfile != NULL)
		      {
			fileloc oloc = currentloc;
			
			currentloc = fileloc_createRc (cstring_fromChars (fname));
			loadrc (rcfile);

			fileloc_reallyFree (currentloc);
			currentloc = oloc;
		      }
		    else 
		      {
			showHerald ();
			llmsg (message ("Options file not found: %s", 
					cstring_fromChars (fname)));
		      }
		  }
		else
		  llfatalerror
		    (cstring_makeLiteral ("Flag f to select options file "
					  "requires an argument"));
	      }
	    else
	      {
		; /* wait to process later */
	      }
	  }
      }
    
    if (fname == NULL)
      {
	fname = cstring_toCharsSafe (message ("%s/%s", home, 
					      cstring_fromChars (RCFILE)));
	mstring_markFree (fname);
      }

    setCodePoint ();

    if (!nof && defaultf)
      {
	rcfile = fopen (fname, "r");
	
	if (rcfile != NULL)
	  {
	    fileloc oloc = currentloc;

	    currentloc = fileloc_createRc (cstring_fromChars (fname));
	    loadrc (rcfile);
	    fileloc_reallyFree (currentloc);
	    currentloc = oloc;
	  }

	fname = cstring_toCharsSafe (message ("./%s", 
					      cstring_fromChars (RCFILE)));

	rcfile = fopen (fname, "r");

	if (rcfile != NULL)
	  {
	    fileloc oloc = currentloc;

	    currentloc = fileloc_createRc (cstring_fromChars (fname));
	    loadrc (rcfile);
	    fileloc_reallyFree (currentloc);
	    currentloc = oloc;
	  }
	sfree (fname); 
      }
  }

  setCodePoint ();

  for (i = 1; i < argc; i++)
    {
      char *thisarg;
      flagcode opt;

      thisarg = argv[i];

      if (showhelp)
	{
	  if (allhelp)
	    {
	      showHerald ();
	    }

	  allhelp = FALSE;

	  if (*thisarg == '-' || *thisarg == '+')
	    {
	      thisarg++;	/* skip '-' */
	    }
	  if (mstring_equal (thisarg, "modes"))
	    {
	      llmsg (describeModes ());
	    }
	  else if (mstring_equal (thisarg, "vars")
		   || mstring_equal (thisarg, "env"))
	    {
	      describeVars ();
	    }
	  else if (mstring_equal (thisarg, "annotations"))
	    {
	      printAnnotations ();
	    }
	  else if (mstring_equal (thisarg, "comments"))
	    {
	      printComments ();
	    }
	  else if (mstring_equal (thisarg, "prefixcodes"))
	    {
	      describePrefixCodes ();
	    }
	  else if (mstring_equal (thisarg, "references") 
		   || mstring_equal (thisarg, "refs"))
	    {
	      printReferences ();
	    }
	  else if (mstring_equal (thisarg, "mail"))
	    {
	      printMail ();
	    }
	  else if (mstring_equal (thisarg, "maintainer")
		   || mstring_equal (thisarg, "version"))
	    {
	      printMaintainer ();
	    }
	  else if (mstring_equal (thisarg, "flags"))
	    {
	      if (i + 1 < argc)
		{
		  char *next = argv[i + 1];

		  if (specialFlagsHelp (next))
		    {
		      i++;
		    }
		  else
		    {
		      flagkind k = identifyCategory (cstring_fromChars (next));
		      
		      if (k != FK_NONE)
			{
			  printCategory (k);
			  i++;
			}
		    }
		}
	      else
		{
		  printFlags ();
		}
	    }
	  else
	    {
	      cstring s = describeFlag (cstring_fromChars (thisarg));

	      if (cstring_isDefined (s))
		{
		  llmsg (s);
		}
	    }
	}
      else
	{
	  if (*thisarg == '-' || *thisarg == '+')
	    {
	      bool set = (*thisarg == '+');
	      cstring flagname;

	      thisarg++;	/* skip '-' */
	      flagname = cstring_fromChars (thisarg);

	      opt = identifyFlag (flagname);

	      if (flagcode_isSkip (opt))
		{
		  ;
		}
	      else if (flagcode_isInvalid (opt))
		{
		  if (isMode (flagname))
		    {
		      context_setMode (flagname);
		    }
		  else
		    {
		      llgloberror (message ("Unrecognized option: %s", 
					    cstring_fromChars (thisarg)));
		    }
		}
	      else
		{
		  context_userSetFlag (opt, set);
		  
		  if (flagcode_hasArgument (opt))
		    {
		      if (opt == FLG_HELP)
			{
			  showhelp = TRUE;
			}
		      else if (flagcode_isPassThrough (opt)) /* -D or -U */
			{ 
			  handlePassThroughFlag (thisarg);
			}
		      else if (flagcode_hasValue (opt))
			{
			  if (++i < argc)
			    {
			      setValueFlag (opt, cstring_fromChars (argv[i]));
			    }
			  else
			    {
			      llfatalerror 
				(message
				 ("Flag %s must be followed by a number",
				  flagcode_unparse (opt)));
			    }
			} 
		      else if (opt == FLG_INCLUDEPATH 
			       || opt == FLG_SPECPATH)
			{
			  cstring dir = cstring_suffix (cstring_fromChars (thisarg), 1); /* skip over I */
			  
			  switch (opt)
			    {
			    case FLG_INCLUDEPATH:
			      localIncludePath = 
				message ("%q%s:", 
					 localIncludePath, dir);
			      cppcmd = message ("%q -I%s", cppcmd, dir);
			      /*@switchbreak@*/ break;
			    case FLG_SPECPATH:
			      /*@-mustfree@*/
			      localSpecPath = cstring_toCharsSafe
				(message ("%s:%s", cstring_fromChars (localSpecPath), dir));
			      /*@=mustfree@*/
			      /*@switchbreak@*/ break;
			      BADDEFAULT;
			    }
			}
		      else if (flagcode_hasString (opt)
			       || opt == FLG_INIT || opt == FLG_OPTF)
			{
			  if (++i < argc)
			    {
			      cstring arg = cstring_fromChars (argv[i]);
			      
			      if (opt == FLG_OPTF)
				{
				  ; /* -f already processed */
				}
			      else if (opt == FLG_INIT)
				{
				  initFile = tsource_create 
				    (cstring_toCharsSafe (arg), 
				     LCLINIT_SUFFIX, FALSE);
				  break;
				}
			      else
				{
				  setStringFlag (opt, arg);
				}
			    }
			  else
			    {
			      llfatalerror 
				(message
				 ("Flag %s must be followed by a string",
				  flagcode_unparse (opt)));
			    }
			}
		      else
			{
			  /* no argument */
			}
		    }
		}
	    }
	  else /* its a filename */
	    {
	      fl = cstringSList_add (fl, cstring_fromChars (thisarg));
	    }
	}
    }

  setCodePoint ();  

  /*
  ** create lists of C and LCL files
  */

  cstringSList_elements (fl, current)
    {
      char *fname = cstring_toCharsSafe (current);
      char *ext = strrchr (fname, '.');

      if (ext == NULL)
	{
	  /* no extension --- both C and LCL with default extensions */
	  
	  addFile (cfiles, message ("%s.c", cstring_fromChars (fname)));
	  addFile (lclfiles, message ("%s.lcl", cstring_fromChars (fname)));
	}
      else if (isCext (ext))
	{
	  addFile (cfiles, cstring_fromCharsNew (fname));
	}
      else 
	{
	  if (!mstring_equal (ext, ".lcl"))
	    {
	      llmsg (message ("Unrecognized file extension: %s (assuming lcl)", 
			      cstring_fromChars (ext)));
	    }

	  addFile (lclfiles, cstring_fromCharsNew (fname));
	}
    } end_cstringSList_elements;
  
  DPRINTF (("exportheader: %d", context_getFlag (FLG_EXPORTHEADER)));

  showHerald ();

  if (showhelp)
    {
      if (allhelp)
	{
	  showHelp ();
	}
      fprintf (stderr, "\n");

      fileIdList_free (cfiles);
      fileIdList_free (lclfiles);
      
      llexit (LLSUCCESS);
    }

  /* 
  ** append localIncludePath with standard include directory
  */

  llassert (cstring_isUndefined (includePath));
  includePath = message ("%s%s", localIncludePath,
			 cstring_fromChars (BASEPATH));
  localIncludePath = message (".:%q%s", localIncludePath, 
			      cstring_fromChars (BASEPATH));

# ifdef DOANNOTS
  initAnnots ();
# endif

  inittime = clock ();

  context_resetErrors ();

  anylcl = !fileIdList_isEmpty (lclfiles);
  DPRINTF (("anylcl = %d", anylcl));

  if (context_doMerge ())
    {
      cstring m = context_getMerge ();

      if (context_getFlag (FLG_SHOWSCAN))
	fprintf (stderr, "< loading %s ", cstring_toCharsSafe (m));
      loadState (m);
      if (context_getFlag (FLG_SHOWSCAN))
	fprintf (stderr, " >\n");

      if (!usymtab_existsType (context_getBoolName ()))
	{
	  usymtab_initBool (); 
	}
    }
  else
    {
      if (!context_getFlag (FLG_NOLIB) && loadStandardState ())
	{
	  ;
	}
      else
	{
	  ctype_initTable ();
	}

      /* setup bool type and constants */
      usymtab_initBool (); 
    }

  libtime = clock ();
  
  if (anylcl)
    {
      lslProcess (lclfiles);
    }

  /*
  ** pre-processing
  **
  ** call the pre-preprocessor and /lib/cpp to generate appropriate
  ** files
  **
  */
  
  cleanupMessages ();
  
  if (!context_getFlag (FLG_NOPP))
    {
      if (context_getFlag (FLG_SHOWSCAN))
	{
	  fprintf (stderr, "< preprocessing");
	}
      
      lcltime = clock ();

      context_setPreprocessing ();
      dercfiles = pcpp_fileIdList (cfiles, cppcmd);
      context_clearPreprocessing ();

      fileIdList_free (cfiles);

      if (context_getFlag (FLG_SHOWSCAN))
	{
	  fprintf (stderr, " >\n");
	}
      
      pptime = clock ();
    }
  else
    {
      lcltime = clock ();
      dercfiles = cfiles;
      pptime = clock ();
    }

  /*
  ** now, check all the corresponding C files
  **
  ** (for now these are just <file>.c, but after pre-processing
  **  will be <tmpprefix>.<file>.c)
  */
  
  exprNode_initMod ();

  fileIdList_elements (dercfiles, fid)
    {
      sourceFile = tsource_create (cstring_toCharsSafe (fileName (fid)),
				   C_SUFFIX, TRUE);
      context_setFileId (fid);
      
      /* Open source file  */
      
      if (sourceFile == (tsource *) 0 || (!tsource_open (sourceFile)))
	{
	  /* previously, this was ignored  ?! */
	  llbug (message ("Could not open temp file: %s", fileName (fid)));
	}
      else
	{
	  yyin = sourceFile->file; /*< shared <- only */
	
	  llassert (yyin != NULL);

	  if (context_getFlag (FLG_SHOWSCAN))
	    {
	      llmsg (message ("< checking %s >", rootFileName (fid)));
	    }
	  
	  /*
	  ** Every time, except the first time, through the loop,
	  ** need to call yyrestart to clean up the parse buffer.
	  */

	  if (!first_time)
	    {
	      (void) yyrestart (yyin);	
	    }
	  else
	    {
	      first_time = FALSE;
	    }
	  
	  context_enterFile ();
	  (void) yyparse ();
	  context_exitFile ();
		    
	  (void) tsource_close (sourceFile);
	}
      
    } end_fileIdList_elements;

  cptime = clock ();
  
  /* process any leftover macros */

  context_processAllMacros ();
  
  /* check everything that was specified was defined */
  
  /* don't check if no c files were processed ?
  **   is this correct behaviour?
  */
  
  if (context_getFlag (FLG_SHOWSCAN))
    {
      llmsglit ("< global checks >");
    }

  cleanupMessages ();
  
  if (context_getLinesProcessed () > 0)
    {
      usymtab_allDefined ();
    }

  if (context_maybeSet (FLG_TOPUNUSED))
    {
      uentry ue = usymtab_lookupSafe (cstring_makeLiteralTemp ("main"));

      if (uentry_isValid (ue))
	{
	  uentry_setUsed (ue, fileloc_observeBuiltin ());
	}

      usymtab_allUsed ();
    }

  if (context_maybeSet (FLG_EXPORTLOCAL))
    {
      usymtab_exportLocal ();
    }

  DPRINTF (("export header: %d", context_getFlag (FLG_EXPORTHEADER)));

  if (context_maybeSet (FLG_EXPORTHEADER))
    {
      usymtab_exportHeader ();
    }

  if (context_getFlag (FLG_SHOWUSES))
    {
      usymtab_displayAllUses ();
    }

  context_checkSuppressCounts ();

  if (context_doDump ())
    {
      cstring dump = context_getDump ();

      dumpState (dump);
    }

# ifdef DOANNOTS
  printAnnots ();
# endif

  cleanupFiles ();

  if (context_getFlag (FLG_SHOWSUMMARY))
    {
      summarizeErrors (); 
    }

  if (!context_getFlag (FLG_QUIET))
    {
      cstring specErrors = cstring_undefined;
      int nspecErrors = lclNumberErrors ();

      if (context_neednl ())
	fprintf (stderr, "\n");
      
      if (nspecErrors > 0)
	{
	  if (nspecErrors == context_getLCLExpect ())
	    {
	      specErrors = 
		message ("%d spec error%p found, as expected\n       ", 
			 nspecErrors);
	    }
	  else
	    {
	      if (context_getLCLExpect () > 0)
		{
		  specErrors = 
		    message ("%d spec error%p found, expected %d\n       ", 
			     nspecErrors,
			     (int) context_getLCLExpect ());
		}
	      else
		{
		  specErrors = message ("%d spec error%p found\n       ",
					nspecErrors);
		}
	    }
	}
      else
	{
	  if (context_getLCLExpect () > 0)
	    {
	      specErrors = message ("No spec errors found, expected %d\n       ", 
				    (int) context_getLCLExpect ());
	    }
	}

      if (context_anyErrors ())
	{
	  if (context_numErrors () == context_getExpect ())
	    {
	      llmsg (message ("Finished LCLint checking --- "
			      "%s%d code error%p found, as expected",
			      specErrors, context_numErrors ()));
	    }
	  else
	    {
	      if (context_getExpect () > 0)
		{
		  llmsg (message 
			 ("Finished LCLint checking --- "
			  "%s%d code error%p found, expected %d",
			  specErrors, context_numErrors (), 
			  (int) context_getExpect ()));
		}
	      else
		{
		  llmsg (message ("Finished LCLint checking --- "
				  "%s%d code error%p found", 
				  specErrors, context_numErrors ()));
		}
	    }
	}
      else
	{
	  if (context_getExpect () > 0)
	    {
	      llmsg (message
		     ("Finished LCLint checking --- "
		      "%sno code errors found, expected %d", 
		      specErrors,
		      (int) context_getExpect ()));
	    }
	  else
	    {
	      if (context_getLinesProcessed () > 0)
		{
		  llmsg (message ("Finished LCLint checking --- %sno code errors found", 
				  specErrors));
		}
	      else
		{
		  llmsg (message ("Finished LCLint checking --- %sno code processed", 
				  specErrors));
		}
	    }
	}

      cstring_free (specErrors);
    }

  if (context_getFlag (FLG_STATS))
    {
      clock_t ttime = clock () - before;
      int specLines = context_getSpecLinesProcessed ();

      rstime = clock ();

      if (specLines > 0)
	{
	  fprintf (stderr, "%d spec, ", specLines);
	}

# ifndef CLOCKS_PER_SEC
      fprintf (stderr, "%d source (%d before pre-processing) lines in %ld time steps (steps/sec unknown)\n", 
	       context_getLinesProcessed (), 
	       context_getBPPLines (),
	       (long) ttime);
# else
      fprintf (stderr, "%d source (%d before pre-processing) lines in %.2f s.\n", 
	       context_getLinesProcessed (), 
	       context_getBPPLines (),
	       (double) ttime / CLOCKS_PER_SEC);
# endif
    }
  else
    {
      rstime = clock ();
    }
  
  if (context_getFlag (FLG_TIMEDIST))
    {
      clock_t ttime = clock () - before;

      if (ttime > 0)
	{
	  char *msg = (char *) dmalloc (256 * sizeof (*msg));

	  if (anylcl)
	    {
	      sprintf (msg, 
		       "Time distribution (percent): initialize %.2f / lcl %.2f / "
		       "pre-process %.2f / c check %.2f / finalize %.2f \n", 
		       (100.0 * (double) (libtime - before) / ttime),
		       (100.0 * (double) (lcltime - libtime) / ttime),
		       (100.0 * (double) (pptime - lcltime) / ttime),
		       (100.0 * (double) (cptime - pptime) / ttime),
		       (100.0 * (double) (rstime - cptime) / ttime));
	    }
	  else
	    {
	      sprintf (msg, 
		       "Time distribution (percent): initialize %.2f / "
		       "pre-process %.2f / c check %.2f / finalize %.2f \n", 
		       (100.0 * (double) (libtime - before) / ttime),
		       (100.0 * (double) (pptime - libtime) / ttime),
		       (100.0 * (double) (cptime - pptime) / ttime),
		       (100.0 * (double) (rstime - cptime) / ttime));
	    }

	  llgenindentmsgnoloc (cstring_fromCharsO (msg));
	}
    }

  llexit (LLSUCCESS);
}

void
showHelp (void)
{
  showHerald ();
  
  llmsglit ("Source files are .c, .h and .lcl files.  If there is no suffix,");
  llmsglit ("   LCLint will look for <file>.c and <file>.lcl.");
  llmsglit ("");
  llmsglit ("Use lclint -help <topic or flag name> for more information");
  llmsglit ("");
  llmsglit ("Topics:");
  llmsglit ("");
  llmsglit ("   annotations (describes source-code annotations)");
  llmsglit ("   comments (describes control comments)");
  llmsglit ("   flags (describes flag categories)");
  llmsglit ("   flags <category> (describes flags in category)");
  llmsglit ("   flags all (short description of all flags)");
  llmsglit ("   flags alpha (list all flags alphabetically)");
  llmsglit ("   flags full (full description of all flags)");
  llmsglit ("   mail (information on mailing lists)");
  llmsglit ("   modes (show mode settings)");
  llmsglit ("   prefixcodes (character codes in namespace prefixes)");
  llmsglit ("   references (sources for more information)");
  llmsglit ("   vars (environment variables)"); 
  llmsglit ("   version (information on compilation, maintainer)");
  llmsglit ("");
}

static bool
specialFlagsHelp (char *next)
{
  if ((next != NULL) && (*next != '-') && (*next != '+'))
    {
      if (mstring_equal (next, "alpha"))
	{
	  printAlphaFlags ();
	  return TRUE;
	}
      else if (mstring_equal (next, "all"))
	{
	  printAllFlags (TRUE, FALSE);
	  return TRUE;
	}
      else if (mstring_equal (next, "categories")
	       || mstring_equal (next, "cats"))
	{
	  listAllCategories ();
	  return TRUE;
	}
      else if (mstring_equal (next, "full"))
	{
	  printAllFlags (FALSE, TRUE);
	  return TRUE;
	}
      else
	{
	  return FALSE;
	}
    }
  else
    {
      return FALSE;
    }
}

void
printAnnotations (void)
{
  llmsglit ("Annotations");
  llmsglit ("-----------");
  llmsglit ("");
  llmsglit ("Annotations are stylized comments that document certain "
	    "assumptions about functions, variables, parameters, and types.  ");
  llmsglit ("");
  llmsglit ("They may be used to indicate where the representation of a "
	    "user-defined type is hidden, to limit where a global variable may "
	    "be used or modified, to constrain what a function implementation "
            "may do to its parameters, and to express checked assumptions about "
	    "variables, types, structure fields, function parameters, and "
	    "function results.");
  llmsglit ("");
  llmsglit ("Annotations are introduced by \"/*@\".  The role of the @ may be "
	    "played by any printable character, selected using -commentchar <char>.");
  llmsglit ("");
  llmsglit ("Consult the User's Guide for descriptions of checking associated with each annotation.");
  llmsglit ("");
  llmsglit ("Globals: (in function declarations)");
  llmsglit ("   /*@globals <globitem>,+ @*/");
  llmsglit ("      globitem is an identifier, internalState or systemState");
  llmsglit ("");
  llmsglit ("Modifies: (in function declarations)");
  llmsglit ("   /*@modifies <moditem>,+ @*/");
  llmsglit ("      moditem is an lvalue");
  llmsglit ("   /*@modifies nothing @*/");
  llmsglit ("   /*@*/   (Abbreviation for no globals and modifies nothing.)");
  llmsglit ("");
  llmsglit ("Iterators:");
  llmsglit ("   /*@iter <identifier> (<parameter-type-list>) @*/ - declare an iterator");
  llmsglit ("");
  llmsglit ("Constants:");
  llmsglit ("   /*@constant <declaration> @*/ - declares a constant");
  llmsglit ("");
  llmsglit ("Alternate Types:");
  llmsglit ("   /*@alt <basic-type>,+ @*/");
  llmsglit ("   (e.g., int /*@alt char@*/ is a type matching either int or char)");
  llmsglit ("");
  llmsglit ("Declarator Annotations");
  llmsglit ("");
  llmsglit ("Type Definitions:");
  llmsglit ("   /*@abstract@*/ - representation is hidden from clients");
  llmsglit ("   /*@concrete@*/ - representation is visible to clients");
  llmsglit ("   /*@immutable@*/ - instances of the type cannot change value");
  llmsglit ("   /*@mutable@*/ - instances of the type can change value");
  llmsglit ("   /*@refcounted@*/ - reference counted type");
  llmsglit ("");
  llmsglit ("Global Variables:");
  llmsglit ("   /*@unchecked@*/ - weakest checking for global use");
  llmsglit ("   /*@checkmod@*/ - check modification by not use of global");
  llmsglit ("   /*@checked@*/ - check use and modification of global");
  llmsglit ("   /*@checkedstrict@*/ - check use of global strictly");
  llmsglit ("");
  llmsglit ("Memory Management:");
  llmsglit ("   /*@dependent@*/ - a reference to externally-owned storage");
  llmsglit ("   /*@keep@*/ - a parameter that is kept by the called function");
  llmsglit ("   /*@killref@*/ - a refcounted parameter, killed by the call");
  llmsglit ("   /*@only@*/ - an unshared reference");
  llmsglit ("   /*@owned@*/ - owner of storage that may be shared by /*@dependent@*/ references");
  llmsglit ("   /*@shared@*/ - shared reference that is never deallocated");
  llmsglit ("   /*@temp@*/ - temporary parameter");
  llmsglit ("");
  llmsglit ("Aliasing:");
  llmsglit ("   /*@unique@*/ - may not be aliased by any other visible reference");
  llmsglit ("   /*@returned@*/ - may be aliased by the return value");
  llmsglit ("");
  llmsglit ("Exposure:");
  llmsglit ("   /*@observer@*/ - reference that cannot be modified");
  llmsglit ("   /*@exposed@*/ - exposed reference to storage in another object");
  llmsglit ("");
  llmsglit ("Definition State:");
  llmsglit ("   /*@out@*/ - storage reachable from reference need not be defined");
  llmsglit ("   /*@in@*/ - all storage reachable from reference must be defined");
  llmsglit ("   /*@partial@*/ - partially defined, may have undefined fields");
  llmsglit ("   /*@reldef@*/ - relax definition checking");
  llmsglit ("");
  llmsglit ("Global State: (for globals lists, no /*@, since list is already in /*@\'s)");
  llmsglit ("   undef - variable is undefined before the call");
  llmsglit ("   killed - variable is undefined after the call");
  llmsglit ("");
  llmsglit ("Null State:");
  llmsglit ("   /*@null@*/ - possibly null pointer");
  llmsglit ("   /*@notnull@*/ - non-null pointer");
  llmsglit ("   /*@relnull@*/ - relax null checking");
  llmsglit ("");
  llmsglit ("Null Predicates:");
  llmsglit ("   /*@truenull@*/ - if result is TRUE, first parameter is NULL");
  llmsglit ("   /*@falsenull@*/ - if result is TRUE, first parameter is not NULL");
  llmsglit ("");
  llmsglit ("Execution:");
  llmsglit ("   /*@exits@*/ - function never returns");
  llmsglit ("   /*@mayexit@*/ - function may or may not return");
  llmsglit ("   /*@trueexit@*/ - function does not return if first parameter is TRUE");
  llmsglit ("   /*@falseexit@*/ - function does not return if first parameter if FALSE");
  llmsglit ("   /*@neverexit@*/ - function always returns");
  llmsglit ("");
  llmsglit ("Side-Effects:");
  llmsglit ("   /*@sef@*/ - corresponding actual parameter has no side effects");
  llmsglit ("");
  llmsglit ("Declaration:");
  llmsglit ("   /*@unused@*/ - need not be used (no unused errors reported)");
  llmsglit ("   /*@external@*/ - defined externally (no undefined error reported)");
  llmsglit ("");
  llmsglit ("Case:");
  llmsglit ("   /*@fallthrough@*/ - fall-through case");
  llmsglit ("");
  llmsglit ("Break:");
  llmsglit ("   /*@innerbreak@*/ - break is breaking an inner loop or switch");
  llmsglit ("   /*@loopbreak@*/ - break is breaking a loop");
  llmsglit ("   /*@switchbreak@*/ - break is breaking a switch");
  llmsglit ("   /*@innercontinue@*/ - continue is continuing an inner loop");
  llmsglit ("");
  llmsglit ("Unreachable Code:");
  llmsglit ("   /*@notreached@*/ - statement may be unreachable.");
  llmsglit ("");
  llmsglit ("Special Functions:");
  llmsglit ("   /*@printflike@*/ - check variable arguments like printf");
  llmsglit ("   /*@scanflike@*/ - check variable arguments like scanf");
}

void
printComments (void)
{
  llmsglit ("Control Comments");
  llmsglit ("----------------");
  llmsglit ("");
  llmsglit ("Setting Flags");
  llmsglit ("");
  llmsglit ("Most flags (all except those characterized as \"globally-settable only\") can be set locally using control comments.  A control comment can set flags locally to override the command line settings.  The original flag settings are restored before processing the next file.");
  llmsglit ("");
  llmsglit ("The syntax for setting flags in control comments is the same as that of the command line, except that flags may also be preceded by = to restore their setting to the original command-line value.  For instance,");
  llmsglit ("   /*@+boolint -modifies =showfunc@*/");
  llmsglit ("sets boolint on (this makes bool and int indistinguishable types), sets modifies off (this prevents reporting of modification errors), and sets showfunc to its original setting (this controls  whether or not the name of a function is displayed before a message).");
  llmsglit ("");
  llmsglit ("Error Suppression");
  llmsglit ("");
  llmsglit ("Several comments are provided for suppressing messages.  In general, it is usually better to use specific flags to suppress a particular error permanently, but the general error suppression flags may be more convenient for quickly suppressing messages for code that will be corrected or documented later.");
  llmsglit ("");
  llmsglit ("/*@ignore@*/ ... /*@end@*/");
  llgenindentmsgnoloc
    (cstring_makeLiteral 
     ("No errors will be reported in code regions between /*@ignore@*/ and /*@end@*/.  These comments can be used to easily suppress an unlimited number of messages."));
  llmsglit ("/*@i@*/");
    llgenindentmsgnoloc
    (cstring_makeLiteral 
     ("No errors will be reported from an /*@i@*/ comment to the end of the line."));
  llmsglit ("/*@i<n>@*/");
  llgenindentmsgnoloc
    (cstring_makeLiteral 
     ("No errors will be reported from an /*@i<n>@*/ (e.g., /*@i3@*/) comment to the end of the line.  If there are not exactly n errors suppressed from the comment point to the end of the line, LCLint will report an error."));
  llmsglit ("/*@t@*/, /*@t<n>@*/");
  llgenindentmsgnoloc
    (cstring_makeLiteral 
     ("Like i and i<n>, except controlled by +tmpcomments flag.  These can be used to temporarily suppress certain errors.  Then, -tmpcomments can be set to find them again."));
  llmsglit ("");
  llmsglit ("Type Access");
  llmsglit ("");
  llmsglit ("/*@access <type>@*/"); 
  llmsglit ("   Allows the following code to access the representation of <type>");
  llmsglit ("/*@noaccess <type>@*/");
  llmsglit ("   Hides the representation of <type>");
  llmsglit ("");
  llmsglit ("Macro Expansion");
  llmsglit ("");
  llmsglit ("/*@notfunction@*/");
  llgenindentmsgnoloc 
    (cstring_makeLiteral
     ("Indicates that the next macro definition is not intended to be a "
      "function, and should be expanded in line instead of checked as a "
      "macro function definition."));
}

  
void
printFlags (void)
{
  llmsglit ("Flag Categories");
  llmsglit ("---------------");
  listAllCategories ();
  llmsglit ("\nTo see the flags in a flag category, do\n   lclint -help flags <category>");
  llmsglit ("To see a list of all flags in alphabetical order, do\n   lclint -help flags alpha");
  llmsglit ("To see a full description of all flags, do\n   lclint -help flags full");
}

void
printMaintainer (void)
{
  llmsg (message ("Maintainer: %s", cstring_makeLiteralTemp (LCLINT_MAINTAINER)));
  llmsglit (LCL_COMPILE);
}

void
printMail (void)
{
  llmsglit ("Mailing Lists");
  llmsglit ("-------------");
  llmsglit ("");
  llmsglit ("There are two mailing lists associated with LCLint.  Send a "
	    "(human-readable) message to lclint-request@larch.lcs.mit.edu to "
	    "subscribe to a list.");
  llmsglit ("");
  llmsglit ("   lclint-announce@larch.lcs.mit.edu");
  llmsglit ("");
  llmsglit ("      Reserved for announcements of new releases and bug fixes.");
  llmsglit ("");
  llmsglit ("   lclint-interest@larch.lcs.mit.edu");
  llmsglit ("");
  llmsglit ("      Informal discussions on the use and development of lclint.");
}

void
printReferences (void)
{
  llmsglit ("References");
  llmsglit ("----------");
  llmsglit ("");
  llmsglit ("A web page for LCLint is available at");
  llmsglit ("   http://larch-www.lcs.mit.edu:8001/larch/lclint/index.html");
  llmsglit ("");
  llmsglit ("A web page on the Larch project is available at URL");
  llmsglit ("    http://larch-www.lcs.mit.edu:8001/larch/");
  llmsglit ("");
  llmsglit ("Technical papers relating to LCLint include:");
  llmsglit ("");
  llmsglit ("   David Evans. \"Static Detection of Dynamic Memory Errors\".");  
  llmsglit ("   To appear in SIGPLAN Conference on Programming Language Design ");
  llmsglit ("   and Implementation (PLDI '96), Philadelphia, PA, May 1996.");
  llmsglit ("      http://larch-www.lcs.mit.:8001/~evs/pldi96-abstract.html");
  llmsglit ("");
  llmsglit ("   David Evans, John Guttag, Jim Horning and Yang Meng Tan. ");
  llmsglit ("   \"LCLint: A Tool for Using Specifications to Check Code\".");
  llmsglit ("   SIGSOFT Symposium on the Foundations of Software Engineering,");
  llmsglit ("   December 1994.");
  llmsglit ("");
  llmsglit ("A general book on Larch is:");
  llmsglit ("");
  llmsglit ("   Guttag, John V., Horning, James J., (with Garland, S. J., Jones, ");
  llmsglit ("   K. D., Modet, A., and Wing, J. M.), \"Larch: Languages and Tools ");
  llmsglit ("   for Formal Specification\", Springer-Verlag, 1993.");
}

void
describePrefixCodes (void)
{
  llmsglit ("Prefix Codes");
  llmsglit ("------------");
  llmsglit ("");
  llmsglit ("These characters have special meaning in name prefixes:");
  llmsglit ("");
  llmsg (message ("   %h  Any uppercase letter [A-Z]", PFX_UPPERCASE));
  llmsg (message ("   %h  Any lowercase letter [a-z]", PFX_LOWERCASE));
  llmsg (message ("   %h  Any character (valid in a C identifier)", PFX_ANY));
  llmsg (message ("   %h  Any digit [0-9]", PFX_DIGIT));
  llmsg (message ("   %h  Any non-uppercase letter [a-z0-9_]", PFX_NOTUPPER));
  llmsg (message ("   %h  Any non-lowercase letter [A-Z0-9_]", PFX_NOTLOWER));
  llmsg (message ("   %h  Any letter [A-Za-z]", PFX_ANYLETTER));
  llmsg (message ("   %h  Any letter or digit [A-Za-z0-9]", PFX_ANYLETTERDIGIT));
  llmsglit ("   *  Zero or more repetitions of the previous character class until the end of the name");
}

void
describeVars (void)
{
  char *eval;

  eval = getenv ("LARCH_PATH");

  if (eval != NULL)
    {
      llmsg (message ("LARCH_PATH = %s", cstring_fromChars (eval)));
    }
  else
    {
      llmsg (message ("LARCH_PATH = <not set> (default = %s)",
		      cstring_fromChars (DEFAULT_LARCHPATH)));
    }
  
  llmsglit ("   --- path used to find larch initialization files and LSL traits");

  eval = getenv ("LCLINT_CPPCMD");

  if (eval != NULL)
    {
      llmsg (message ("LCLINT_CPPCMD = %s", cstring_fromChars (eval)));
    }
  else
    {
      llmsg (message ("LCLINT_CPPCMD = <not set, default: %s>", 
		      cstring_makeLiteralTemp (DEFAULT_CPPCMD)));
    }
  
  llmsglit ("   --- command used to invoke C preprocessor "
	    "(recommend \"gcc -E\" or \"/lib/cpp\")");

  eval = getenv (LCLIMPORTDIR);

  if (eval != NULL)
    {
      llmsg (message ("%q = %s", cstring_makeLiteral (LCLIMPORTDIR), 
		      cstring_fromChars (eval)));
    }
  else
    {
      llmsg (message ("%s = <not set, default: %s>", cstring_makeLiteralTemp (LCLIMPORTDIR), 
		      cstring_makeLiteralTemp (DEFAULT_LCLIMPORTDIR)));
    }
  
  llmsglit ("   --- directory containing lcl standard library files "
	    "(import with < ... >)");;
}

void
interrupt (int i)
{
  switch (i)
    {
    case SIGINT:
      llmsglit ("*** Interrupt\n");
      llexit (LLINTERRUPT);
    case SIGSEGV:
      {
	cstring loc;
	
	/* cheat when there are parse errors */
	checkParseError (); 
	
	fprintf (stderr, "*** Segmentation Violation\n");
	
	/* don't catch it if fileloc_unparse causes a signal */
	signal (SIGSEGV, NULL); 

	loc = fileloc_unparse (currentloc);
	
	fprintf (stderr, "*** Location (not trusted): %s\n", 
		 cstring_toCharsSafe (loc));
	cstring_free (loc);
	printCodePoint ();
	fprintf (stderr, "*** Please report bug to %s\n", LCLINT_MAINTAINER);
	exit (LLGIVEUP);
      }
    default:
      fprintf (stderr, "Signal: %d\n", i);
      /*@-mustfree@*/
      fprintf (stderr, "*** Location (not trusted): %s\n", 
	       cstring_toCharsSafe (fileloc_unparse (currentloc)));
      /*@=mustfree@*/
      printCodePoint ();
      fprintf (stderr, "*** Please report bug to %s ***\n", LCLINT_MAINTAINER);
      exit (LLGIVEUP);
    }
}

void
cleanupFiles (void)
{
  static bool doneCleanup = FALSE;

  /* make sure this is only called once! */

  if (doneCleanup) return;

  setCodePoint ();

  if (context_getFlag (FLG_KEEP))
    {
      check (fputs ("Temporary files kept:\n", stderr) != EOF);
      fileTable_printTemps (context_fileTable ());
    }
  else
    {
      fileTable_cleanup (context_fileTable ());
    }

  doneCleanup = TRUE;
}

/*
** cleans up temp files (if necessary)
** exits lclint
*/

/*@exits@*/ void
llexit (int status)
{
  cleanupFiles ();

  if (status != LLFAILURE)
    {
      context_destroyMod ();
      exprNode_destroyMod ();
      
      sRef_destroyMod ();
      uentry_destroyMod ();
      typeIdSet_destroyMod ();
      
# ifdef USEDMALLOC
      dmalloc_shutdown ();
# endif
    }

  exit ((status == LLSUCCESS) ? EXIT_SUCCESS : EXIT_FAILURE);
}

void
loadrc (FILE *rcfile)
{
  char *s = mstring_create (MAX_LINE_LENGTH);
  char *os = s;
  
  s = os;

  while (fgets (s, MAX_LINE_LENGTH, rcfile) != NULL)
    {
      char c;
      bool set = FALSE;	    
      char *thisflag;
      flagcode opt;
      
      incLine ();

      DPRINTF (("line: %s", s));

      while (*s == ' ' || *s == '\t' || *s == '\n') 
	{
	  s++;
	  incColumn ();
	}
      
      while (*s != '\0')
	{
	  bool escaped = FALSE;
	  bool quoted = FALSE;
	  c = *s;
	  
	  /* comment characters */
	  if (c == '#' || c == ';' || c == '\n') 
	    {
	      /*@innerbreak@*/
	      break;
	    }
	  
	  if (c == '-' || c == '+')
	    {
	      set = (c == '+');
	    }
	  else
	    {
	      showHerald ();
	      llerror (FLG_SYNTAX, 
		       message ("Bad flag syntax (+ or - expected, "
				"+ is assumed): %s", 
				cstring_fromChars (s)));
	      s--;
	      set = TRUE;
	    }
	  
	  s++;
	  incColumn ();
	  
	  thisflag = s;
	  
	  while ((c = *s) != '\0')
	    { /* remember to handle spaces and quotes in -D and -U ... */
	      if (escaped)
		{
		  escaped = FALSE;
		}
	      else if (quoted)
		{
		  if (c == '\\')
		    {
		      escaped = TRUE;
		    }
		  else if (c == '\"')
		    {
		      quoted = FALSE;
		    }
		  else
		    {
		      ;
		    }
		}
	      else if (c == '\"')
		{
		  quoted = TRUE;
		}
	      else
		{
		 if (c == ' ' || c == '\t' || c == '\n')
		   {
		     /*@innerbreak@*/ break;
		   }
	       }
		  
	      s++; 
	      incColumn ();
	    }
	  
	  *s = '\0';

	  if (mstring_isEmpty (thisflag))
	    {
	      llfatalerror (message ("Missing flag: %s",
				     cstring_fromChars (os)));
	    }

	  opt = identifyFlag (cstring_fromChars (thisflag));
	  
	  if (flagcode_isSkip (opt))
	    {
	      ;
	    }
	  else if (flagcode_isInvalid (opt))
	    {
	      if (isMode (cstring_fromChars (thisflag)))
		{
		  context_setMode (cstring_fromChars (thisflag));
		}
	      else
		{
		  llerror (FLG_BADFLAG,
			   message ("Unrecognized option: %s", 
				    cstring_fromChars (thisflag)));
		}
	    }
	  else
	    {
	      context_userSetFlag (opt, set);

	      if (flagcode_hasArgument (opt))
		{
		  DPRINTF (("hasArgument: %s", flagcode_unparse (opt)));
		  
		  if (opt == FLG_HELP)
		    {
		      showHerald ();
		      llerror (FLG_BADFLAG,
			       message ("Cannot use help in rc files"));
		    }
		  else if (flagcode_isPassThrough (opt)) /* -D or -U */
		    {
		      handlePassThroughFlag (thisflag);
		    }
		  else if (opt == FLG_INCLUDEPATH 
			   || opt == FLG_SPECPATH)
		    {
		      cstring dir = cstring_suffix (cstring_fromChars (thisflag), 1); /* skip over I/S */
		      DPRINTF (("include path: %s", dir));
		      
		      switch (opt)
			{
			case FLG_INCLUDEPATH:
			  localIncludePath = message ("%q%s:", localIncludePath, dir);
			  cppcmd = message ("%q -I%s", cppcmd, dir);
			  /*@switchbreak@*/ break;
			case FLG_SPECPATH:
			  /*@-mustfree@*/
			  localSpecPath = cstring_toCharsSafe
			    (message ("%s:%s", cstring_fromChars (localSpecPath), dir));
			  /*@=mustfree@*/
			  /*@switchbreak@*/ break;
			  BADDEFAULT;
			}
		    }
		  else if (flagcode_hasString (opt)
			   || flagcode_hasValue (opt)
			   || opt == FLG_INIT || opt == FLG_OPTF)
		    {
		      cstring extra = cstring_undefined;
		      char *rest, *orest;
		      char rchar;
		      
		      *s = c;
		      rest = mstring_copy (s);
		      orest = rest;
		      *s = '\0';
		      
		      while ((rchar = *rest) != '\0'
			     && (isspace (rchar)))
			{
			  rest++;
			  s++;
			}
		      
		      while ((rchar = *rest) != '\0'
			     && !isspace (rchar))
			{
			  extra = cstring_appendChar (extra, rchar);
			  rest++; 
			  s++;
			}
		      
		      sfree (orest);

		      if (cstring_isUndefined (extra))
			{
			  showHerald ();
			  llerror 
			    (FLG_BADFLAG,
			     message
			     ("Flag %s must be followed by an argument",
			      flagcode_unparse (opt)));
			}
		      else
			{
			  s--;
			  
			  if (flagcode_hasValue (opt))
			    {
			      setValueFlag (opt, extra);
			      cstring_free (extra);
			    }
			  else if (opt == FLG_OPTF)
			    {
			      FILE *innerf = fopen (cstring_toCharsSafe (extra), "r");

			      cstring_markOwned (extra);
			      
			      if (innerf != NULL)
				{
				  fileloc fc = currentloc;
				  currentloc = fileloc_createRc (extra);
				  loadrc (innerf);
				  fileloc_reallyFree (currentloc);
				  currentloc = fc;
				}
			      else 
				{
				  showHerald ();
				  llerror
				    (FLG_SYNTAX, 
				     message ("Options file not found: %s", 
					      extra));
				}
			    }
			  else if (opt == FLG_INIT)
			    {
			      llassert (initFile == NULL);

			      initFile = tsource_create 
				(cstring_toCharsSafe (extra), 
				 LCLINIT_SUFFIX, FALSE);
			      cstring_markOwned (extra);
			    }
			  else if (flagcode_hasString (opt))
			    {
			      setStringFlag (opt, extra);
			    }
			  else
			    {
			      cstring_free (extra);
			      BADEXIT;
			    }
			}
		    }
		  else
		    {
		      BADEXIT;
		    }
		}
	    }
	  
	  *s = c;
	  while ((c == ' ') || (c == '\t'))
	    {
	      c = *(++s);
	      incColumn ();
	    } 
	}
    }

  /* sfree (os); */
  check (fclose (rcfile) == 0);
}
