/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** cprim.c
*/

# include "lclintMacros.nf"
# include "basic.h"

static bool cprim_isReal (cprim c)
{
  return (cprim_isAnyReal (c));
}

static bool cprim_isNumeric (cprim c)
{
  return (cprim_isReal (c) || cprim_isInt (c));
}

cprim
cprim_fromInt (int i)
{
  if (i < CTX_UNKNOWN || i > CTX_LAST)
    {
      llcontbug (message ("cprim_fromInt: out of range: %d", i));
      return CTX_UNKNOWN;
    }
  return (cprim) i;
}


/*
** not symmetric:  c1 := c2 or c2 is passed as c1
**    (if RELAXQUALS, c1 must be "bigger" than c2)
*/

bool
cprim_closeEnough (cprim c1, cprim c2)
{
  if (c1 == c2) return TRUE;

  DPRINTF (("close enough: %s / %s (pass %s as %s)",
	    cprim_unparse (c1),
	    cprim_unparse (c2),
	    cprim_unparse (c2),
	    cprim_unparse (c1)));

  if (c1 == CTX_ANYINTEGRAL)
    {
      if (context_getFlag (FLG_MATCHANYINTEGRAL))
	{
	  return (cprim_isAnyInt (c2) || (cprim_isAnyChar (c2) && context_msgCharInt ()));
	}
      else if (context_getFlag (FLG_LONGINTEGRAL))
	{
	  return (cprim_closeEnough (CTX_LINT, c2));
	}
      else if (context_getFlag (FLG_LONGUNSIGNEDINTEGRAL))
	{
	  return (cprim_closeEnough (CTX_ULINT, c2));
	}
      else
	{
	  return FALSE;
	}
    }

  if (c1 == CTX_UNSIGNEDINTEGRAL)
    {
      if (context_getFlag (FLG_MATCHANYINTEGRAL))
	{
	  return (cprim_isAnyInt (c2) || (cprim_isAnyChar (c2) && context_msgCharInt ()));
	}
      else if (context_getFlag (FLG_LONGUNSIGNEDUNSIGNEDINTEGRAL))
	{
	  return (cprim_closeEnough (CTX_ULINT, c2));
	}
      else
	{
	  return FALSE;
	}
    }

  if (c1 == CTX_SIGNEDINTEGRAL)
    {
      if (context_getFlag (FLG_MATCHANYINTEGRAL))
	{
	  return (cprim_isAnyInt (c2) || (cprim_isAnyChar (c2) && context_msgCharInt ()));
	}
      else if (context_getFlag (FLG_LONGSIGNEDINTEGRAL))
	{
	  return (cprim_closeEnough (CTX_LINT, c2));
	}
      else
	{
	  return FALSE;
	}
    }

  if (c2 == CTX_ANYINTEGRAL)
    {
      if (context_getFlag (FLG_MATCHANYINTEGRAL))
	{
	  return (cprim_isAnyInt (c1) || (cprim_isAnyChar (c1) && context_msgCharInt ()));
	}
      else if (context_getFlag (FLG_LONGINTEGRAL))
	{
	  return (cprim_closeEnough (c1, CTX_LINT));
	}
      else if (context_getFlag (FLG_LONGUNSIGNEDINTEGRAL))
	{
	  return (cprim_closeEnough (c1, CTX_ULINT));
	}
      else
	{
	  return FALSE;
	}
    }

  if (c2 == CTX_UNSIGNEDINTEGRAL)
    {
      if (context_getFlag (FLG_MATCHANYINTEGRAL))
	{
	  return (cprim_isAnyInt (c1) || (cprim_isAnyChar (c1) && context_msgCharInt ()));
	}
      else if (context_getFlag (FLG_LONGUNSIGNEDUNSIGNEDINTEGRAL))
	{
	  return (cprim_closeEnough (c1, CTX_ULINT));
	}
      else
	{
	  return FALSE;
	}
    }

  if (c2 == CTX_SIGNEDINTEGRAL)
    {
      if (context_getFlag (FLG_MATCHANYINTEGRAL))
	{
	  return (cprim_isAnyInt (c2) || (cprim_isAnyChar (c2) && context_msgCharInt ()));
	}
      else if (context_getFlag (FLG_LONGSIGNEDINTEGRAL))
	{
	  return (cprim_closeEnough (c1, CTX_LINT));
	}
      else
	{
	  return FALSE;
	}
    }

  if (context_getFlag (FLG_RELAXTYPES))
    {
      if (cprim_isNumeric (c1) && cprim_isNumeric (c2)) return TRUE;
    }

  if (context_getFlag (FLG_IGNOREQUALS))
    {
      switch (c1)
	{
	case CTX_CHAR:
	case CTX_UCHAR:
	  return (cprim_isAnyChar (c2) 
		  || (cprim_isAnyInt (c2) && (context_msgCharInt ())));
	case CTX_DOUBLE:
	case CTX_FLOAT:
	case CTX_LDOUBLE:
	  return (c2 == CTX_DOUBLE || c2 == CTX_FLOAT || c2 == CTX_LDOUBLE);
	case CTX_INT:
	case CTX_LINT:
	case CTX_SINT:
	case CTX_UINT:
	case CTX_ULINT:
	case CTX_USINT:
	  return (cprim_isAnyInt (c2) || (cprim_isAnyChar (c2) && context_msgCharInt ()));
	default:
	  return FALSE;
	}
    }
  else 
    {
      if (context_getFlag (FLG_IGNORESIGNS))
	{
	  if (c1 == CTX_UCHAR)  
	    {
	      c1 = CTX_CHAR;
	    }
	  else if (c1 == CTX_UINT)  
	    {
	      c1 = CTX_INT;
	    }
	  else if (c1 == CTX_ULINT) 
	    {
	      c1 = CTX_LINT;
	    }
	  else if (c1 == CTX_USINT)  
	    {
	      c1 = CTX_SINT;
	    }
	  else
	    {
	      ;
	    }

	  if (c2 == CTX_UCHAR)  
	    {
	      c2 = CTX_CHAR;
	    }
	  else if (c2 == CTX_UINT)   
	    {
	      c2 = CTX_INT;
	    }
	  else if (c2 == CTX_ULINT) 
	    {
	      c2 = CTX_LINT;
	    }
	  else if (c2 == CTX_USINT)  
	    {
	      c2 = CTX_SINT;
	    }
	  else
	    {
	      ;
	    }
	}

      if (c1 == c2) return TRUE;

      if (context_getFlag (FLG_RELAXQUALS))
	{
	  switch (c1)
	    {
	    case CTX_DOUBLE:
	      return (c2 == CTX_FLOAT);
	    case CTX_LDOUBLE:
	      return (c2 == CTX_DOUBLE || c2 == CTX_FLOAT);
	    case CTX_SINT:
	      return (c2 == CTX_CHAR && context_msgCharInt ());
	    case CTX_INT:
	      return (c2 == CTX_SINT
		      || (cprim_isAnyChar (c2) && context_msgCharInt ()));
	    case CTX_LINT:
	      return (c2 == CTX_SINT
		      || c2 == CTX_INT 
		      || (cprim_isAnyChar (c2) && context_msgCharInt ()));
	    case CTX_UINT:
	      return (c2 == CTX_USINT 
		      || (c2 == CTX_UCHAR && context_msgCharInt ()));
	    case CTX_USINT:
	      return (c2 == CTX_UCHAR && context_msgCharInt ());
	    case CTX_ULINT:
	      return (c2 == CTX_UINT || c2 == CTX_USINT);
	    case CTX_UCHAR:
	      return (c2 == CTX_UINT && context_msgCharInt ());
	    case CTX_CHAR:
	      return ((c2 == CTX_INT || c2 == CTX_SINT)
		      && context_msgCharInt ());
	    default:
	      return FALSE;
	    }
	}
      else
	{
	  switch (c1)
	    {
	    case CTX_DOUBLE:
	    case CTX_LDOUBLE:
	      return FALSE;
	    case CTX_SINT:
	    case CTX_INT:
	    case CTX_LINT:
	      return (c2 == CTX_CHAR && context_msgCharInt ());
	    case CTX_UINT:
	    case CTX_USINT:
	    case CTX_ULINT:
	      return (c2 == CTX_UCHAR && context_msgCharInt ());
	    case CTX_UCHAR:
	      return (c2 == CTX_UINT && context_msgCharInt ());
	    case CTX_CHAR:
	      return ((c2 == CTX_INT || c2 == CTX_SINT)
		      && context_msgCharInt ());
	    default:
	      return FALSE;
	    }
	}
    }
}

/*@only@*/ cstring
cprim_unparse (cprim c)
{
  switch (c)
    {
    case CTX_UNKNOWN:
      return cstring_makeLiteral ("-");
    case CTX_VOID:
      return cstring_makeLiteral ("void");
    case CTX_CHAR:
      return cstring_makeLiteral ("char");
    case CTX_UCHAR:
      return cstring_makeLiteral ("unsigned char");
   case CTX_DOUBLE:
      return cstring_makeLiteral ("double");
    case CTX_LDOUBLE:
      return cstring_makeLiteral ("long double");
    case CTX_FLOAT:
      return cstring_makeLiteral ("float");
    case CTX_INT:
      return cstring_makeLiteral ("int");
    case CTX_LINT:
      return cstring_makeLiteral ("long int");
    case CTX_SINT:
      return cstring_makeLiteral ("short int");
    case CTX_UINT:
      return cstring_makeLiteral ("unsigned int");
    case CTX_ULINT:
      return cstring_makeLiteral ("unsigned long int");
    case CTX_USINT:
      return cstring_makeLiteral ("unsigned short int");
    case CTX_UNSIGNEDINTEGRAL:
      return cstring_makeLiteral ("arbitrary unsigned integral type");
    case CTX_SIGNEDINTEGRAL:
      return cstring_makeLiteral ("arbitrary signed integral type");
    case CTX_ANYINTEGRAL:
      return cstring_makeLiteral ("arbitrary integral type");
    default:
      return cstring_makeLiteral ("unknown prim");
    }
}

bool cprim_isInt (cprim c) 
{
  return (cprim_isAnyInt (c) || (cprim_isAnyChar (c) && context_msgCharInt ()));
}
    
