/*  menue.c       with SRGP     V.4                  12.07.1996  */
/* Routines for stbasic  Structured-BASIC-Interpreter            */
/*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation (any version).

* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.

* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*   MENU array()         read text for menu-header from array() */
/*                        the string-array contains text         */
/*                        for menu-line and menu-rows            */
/*   MENU n,x             change menu-point #n with value x      */
/*                        x=0 ' '  normal, reset marker          */
/*                        x=1 '^'  set marker                    */
/*                        x=2 '='  set non selectable            */
/*                        x=3 ' '  set selectable                */
/*                            '-'  permanent non selectable      */
/*   MENU STOP            switch off the menu                    */
/*   ON MENU GOSUB p1     define p1 procedure for menu-action    */
/*   ON MENU              switch on the menu and                 */
/*                        wait for menue-events                  */

#include <stdio.h>
#include <string.h>
#include "srgp.h"

#define GRF_XMAX 640
#define GRF_XMIN 160
#define GRF_YMAX 400
#define GRF_YMIN 100
#define GRF_MAXW 10
#define GRF_MAXP 500

#define ME_TXLEN 200  /* array of string */
#define ME_CHARS 16   /* max. # of char. in menue-point */
#define ME_MXCOL 10
#define ME_MXROW 20
#define ME_YTBOX 16
#define ME_WINDOW 8

/* ------------------------------------------------------------ */
/* coordinate-system on the screen canvas :   */
/*  SRGP:  (0,0) is lower left point          */
/*  Atari: (0,0) is upper left point          */
/*    Ynew = GRF_YMAX - Yold                  */

static char menutext[ME_TXLEN][ME_CHARS];  /* copy from mtxt[]     */
static int menlcol[ME_MXCOL];  /*  nr. of menu-points in column */
static int menuxcc[ME_MXCOL];  /*  x-coord. of colums           */
static int mencwid[ME_MXCOL];  /*  width of column              */
static int mencidx[ME_MXCOL];  /*  index to column-start        */
static int mentwid[ME_MXCOL];  /*  width of title-line          */

static int mebutx[22], mebuty[22], mebutw[22], mebuth[22];
static int menu_xcanv, menu_ycanv, menu_texi;

/* ------------------------------------------------------------ */
/* check, if locator is in one of the itmax boxes               */
/* position of boxes in: xb[i], yb[i]                           */
/* size of boxes in:     xw[i], xh[i]                           */ 
/* wait for event: box is selected and button is pressed        */
/* if itmax<0 then exit with vr=-1 if only button is pressed    */
void
grfsevent (int itmax, int *vr, int *xb, int *yb, int *xw, int *yh);

/*   make new canvas #win, size (xmax,ymax) at (xcnv,ycnv)      */
void
grfnewcanvas (int win, int xmax, int ymax, int xcnv, int ycnv);

/* restore the original window                                  */
void
grfoldcanvas (int win, int xmax, int ymax, int xcnv, int ycnv);

/* ------------------------------------------------------------ */
/*  copy the text "s" to "t" with max. "l" characters           */
void
textcopy (char *t, char *s, int l)
{
  if (strlen(s) < l) { strcpy (t, s); }
  else { strncpy (t, s, l); }
}

/*  copy the text "s" to "t" with max. "l" characters           */
/*  and add 2 control-characters                                */
void
texucopy (char *t, char *s, int l)
{ 
  if (s[0] == '-') { strcpy (t, "- "); }
  else { strcpy (t, "  "); }
  if (strlen(s) < l-2) { strcat (t, s); }
  else { strncat (t, s, l-2); }
}

/* ------------------------------------------------------------ */
/* display text-boxes and set select-box-coord.  */
void
menu_xbox (int x, int y, int xw, int dx, char *btxt)
{
  int bx, by;
  bx = menu_xcanv+x;
  by = menu_ycanv+y;
  /*   clear box at (x0,y0, x1,y1)                 */
  SRGP_setColor (0) ;  /* COLOR_WHITE */
  SRGP_setFillStyle (SOLID);
  SRGP_setWriteMode (WRITE_REPLACE);
  SRGP_fillRectangleCoord (bx, by, bx+xw, by+ME_YTBOX);
  SRGP_setColor (1);  /* BLACK */
  /*  draw box and write text into it  */
  SRGP_rectangleCoord (bx, by, bx+xw, by+ME_YTBOX); 
  SRGP_text (SRGP_defPoint (bx+2, by+2), btxt);
  if ((btxt[0] == '-') || (btxt[0] == '='))  bx = -1; 
  mebutx[dx] = bx;
  mebuty[dx] = by;
  mebutw[dx] = xw;
  mebuth[dx] = ME_YTBOX;
  /* printf("XBOX %d %d %d %d %d \n", \
     dx,mebutx[dx],mebuty[dx],mebutw[dx],mebuth[dx]); */
} 

/*  stop the MENU            */
int
menustop (void)
{
  int cntcols;
  /*  preset index-table  */
  for (cntcols=0; cntcols<ME_MXCOL; cntcols++) 
    mencidx[cntcols] = -1;
  return 0;
}

/* --------------------------------------------- */
/*   read menutext[][] from mtxt[]               */
/*   set menlcol []    nr. of menu-points column */
/*   set menuxcc []    x-coord. of colums        */
/*   set mencwid []    width of column           */
/*   set mentwid []    width of title-line       */
/*   set mencidx []    index to column-start     */
int
menrdtext (char **mtxt)
{
  int textindex, colindex, txtleng, mtxcoord;
  int maxleng, cntcols;
  /*  preset index-table  */
  textindex = menustop ();
  mtxcoord = 1;
  colindex = 0;
  /*  scan MENU-text-field, end of row by "", end of field by "",""  */
  while (mtxt[textindex][0] != '\0') {
    textcopy (menutext[textindex], mtxt[textindex], ME_CHARS);
    txtleng = (strlen(menutext[textindex]) + 1) * 8;
    menuxcc [colindex] = mtxcoord;
    mentwid [colindex] = txtleng;
    mencidx [colindex] = textindex;
    mtxcoord += txtleng;
    textindex++;
    maxleng = 0;
    cntcols = 0;
    while (mtxt[textindex][0] != '\0') {
      texucopy (menutext[textindex], mtxt[textindex], ME_CHARS);
      txtleng = (strlen(menutext[textindex]) + 1) * 8;
      if (maxleng < txtleng) maxleng = txtleng;
      textindex++;
      cntcols++;
    }
    textindex++;
    mencwid [colindex] = maxleng; 
    menlcol [colindex] = cntcols;
    colindex++;
  }
  return textindex;
}

/*  display header-line of MENU  */
int
dispmenrow (void)
{ 
  int cx, tx;
  menu_xcanv = 0;
  menu_ycanv = GRF_YMAX-ME_YTBOX;
  cx = 0;
  while (mencidx[cx] >= 0) {
    tx = mencidx[cx];
    menu_xbox (menuxcc[cx], 0, mentwid[cx], cx, menutext[tx]);
    /* printf("MENU-Disp  %d %d %d %s \n",cx, menuxcc[cx],mentwid[cx],\
           menutext[tx]); */
    cx++;
  }
  SRGP_refresh();
  return cx;
}

/*  wait for selection of an element in header-line of MENU  */
/*  display selected column and wait for further selection   */
int
dispmencol (void)
{ 
  int cx, tx, texit, dx, dy, cmax, ci, ck, cy, bcmax;
  /*  loop for MENU-box selection  */
  texit = -1;
  while (texit < 0) {
    cx = dispmenrow ();
    menu_xcanv = 0;
    menu_ycanv = GRF_YMAX-ME_YTBOX;
    grfsevent (cx, &tx, mebutx, mebuty, mebutw, mebuth);
    /* printf("MENU-EVENT %d %d %s \n",cx,tx,menutext[mencidx[tx]]); */
    /*  display selected column #tx  */
    dx = mencwid[tx];
    dy = menlcol[tx]*ME_YTBOX;
    menu_xcanv = menuxcc[tx]+1;
    menu_ycanv = GRF_YMAX - ME_YTBOX - dy;
    cy = dy - ME_YTBOX;
    /* printf("MENU-E1 xy= %d %d canv= %d %d \n", \
              dx,dy,menu_xcanv,menu_ycanv); */
    /*  make new canvas for the selected row  */
    grfnewcanvas (ME_WINDOW, dx, dy, menu_xcanv,menu_ycanv);
    cmax = menlcol[tx];
    for (ci=0; ci<cmax; ci++) {
      ck = mencidx[tx] + ci + 1;
      menu_xbox (0, cy, dx, ci, menutext[ck]);
      cy -= ME_YTBOX;
    }
    /*  wait for event, even if no menu-point is selected */
    bcmax = -cmax;
    grfsevent (bcmax, &texit, mebutx, mebuty, mebutw, mebuth);
    grfoldcanvas (ME_WINDOW, dx, dy, menu_xcanv,menu_ycanv);
  }
  texit += mencidx[tx] + 1;
  /* printf("MENU-EVENT %d %d %s \n",tx,texit,menutext[texit]); */
return texit;
}

/*  set values for menue-points                */
/*      x=0 ' '  normal, reset marker          */
/*      x=1 '^'  set marker                    */
/*      x=2 '='  set non selectable            */
/*      x=3 ' '  set selectable                */
/*          '-'  permanent non selectable      */
void
setmpoint (int index, int mode)
{
  int ri, si, ti, fi;
  char ct, dt;
  /*  find out the "forbidden" index-values  */
  /* printf("MENU-setmpoint %d %d \n",index,mode); */
  ri = 0;
  if ((index > 0) && (index < menu_texi )) {
    ti = mencidx[ri];
    fi = 0;
    while (mencidx[ri] >= 0) {
      if ((mencidx[ri] == index) || \
          (mencidx[ri] + menlcol[ri]+1 == index)) fi = index;
      ri++;
    }
  }
  /*  set value into the menu-point  */
  if (fi == 0) {
    ct = menutext[index] [0];
    if (ct != '-') {
      if ((mode == 0) || (ct != '=')) dt = ' ';
      if ((mode == 1) || (ct != '=')) dt = '^';
      if (mode == 2) dt = '=';
      if (mode == 3) dt = ' ';
    }
    menutext[index] [0] = dt;
    /* printf("MENU-setmpoint %c %c \n",ct,dt); */
  }
}

/* --------------------------------------------------------------- */
void grf_menu (long mode, long *index, char **stringarr)
{ 
  int mx, mi;
  /* printf("grf_menu  m= %d i= %d \n",mode,*index); */
  /*   MENU array()         read text for menu-header from array() */
  if (mode == 0) {
    menu_texi = menrdtext (stringarr);
    mx = dispmenrow ();
  }

  /*   ON MENU              switch on the menu and                 */
  /*                        wait for menue-events                  */
  if (mode == 1) {  /*  ONMENU  */
    /* printf ("ONMENU\n"); */
    *index = (long) dispmencol ();
  }

  /*   MENU STOP            switch off the menu              */
  if (mode == 2) { 
    /* printf ("MENU STOP\n"); */
    menu_texi = menustop ();
  }

  /*   MENU n,x             change menu-point #n with value x      */
  if (mode >= 4) {
    /* printf ("MENU %d,%d\n",*index,mode); */
    mx = (mode-4) & 3;
    mi = *index;
    setmpoint (mi, mx);
  }
}

