/*  graphic.c       with SRGP               20.06.1996  */
/*  last mod.                               10.12.1996  */
/* Routines for stbasic  Structured-BASIC-Interpreter   */
/*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation (any version).

* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.

* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*   TITLEW n,title$      set title in window n         */
/*   OPENW n,x,y          open window n at (x,y)        */
/*   CLEARW n             clear window n                */
/*   CLOSEW n             close window n                */
/*   COLOR cdraw,cback    set colour for drawing/backgnd*/
/*   INFOW n,info$        info-line for window          */
/*   SETCOLOR i,r,g,b     set colours for display       */
/*   GRAPHMODE n          set graphic-mode              */
/*   DEFFILL c,a,b        define colour and texture     */
/*   DEFLINE s,b          line stile, width, type       */
/*   DEFMARK c,a,g        colour, size, type (POLYMARK) */
/*   DEFTEXT c,s,r,g      colour,type,stile,rot. (TEXT) */
/*   PLOT x,y             draw a point on the screen    */
/*   LINE x0,y0,x1,y1     line from (x0,y0) to (x1,y1)  */
/*   TEXT x,y,s$          draw string s$ at (x,y)       */
/*   FILL x,y             fill pattern into an areal    */
/*   BOX x0,y0,x1,y1      draw rectangle                */
/*   PBOX x0,y0,x1,y1     draw filled box               */
/*   RBOX x0,y0,x1,y1     draw rounded box              */
/*   PRBOX x0,y0,x1,y1    draw rounded filled box       */
/*   CIRCLE x,y,r,p0,p1   draw circle at (x,y) radius r */
/*   PCIRCLE x,y,r,p0,p1  draw filled circle            */
/*   ELLIPSE x,y,rx,ry    draw ellipse at (x,y)         */
/*   PELLIPSE x,y,rx,ry   draw filled ellipse           */
/*   DEFMOUSE n           define mouse symbol           */
/*   HIDEM                hide mouse symbol             */
/*   SHOWM                show mouse symbol             */
/*   MOUSE x,y,k          mouse-position (x,y) and key  */
/*   POLYLINE n,x(),y()   draw polygon in (x(),y())     */
/*   POLYFILL n,x(),y()   draw filled polygon           */
/*   POLYMARK n,x(),y()   draw polygon points           */

#include <stdio.h>
#include <string.h>
#include "srgp.h"

#define GRF_XMAX 640
#define GRF_XMIN 160
#define GRF_YMAX 400
#define GRF_YMIN 100
#define GRF_MAXW 10   /*  max. nr. of windows */
#define GRF_MAXP 500
#define GRF_RS1 2     /*  parameters for rounded box */
#define GRF_RS2 8
#define GRF_RS3 16
#define GRF_HTXT 16

/*  common area for the window-parameter */
struct {
  int grfwindx[GRF_MAXW];
  int grfwindy[GRF_MAXW];
  attribute_group grfclwbund, grfalrbund;
  canvasID grfwindow[GRF_MAXW];
} grf_comm1;

static locator_measure grfevent;
static rectangle largerect;
static char grftitle[80]="stbasic - graphic-screen\0";
static char grftext[80];
static int grfpolyx[GRF_MAXP];
static int grfpolyy[GRF_MAXP];
static unsigned short scolr[32];
static unsigned short scolg[32];
static unsigned short scolb[32];
static int colordraw, colorback, colormark, colorfill;

/* coordinate-system on the screen canvas :   */
/*  SRGP:  (0,0) is lower left point          */
/*  Atari: (0,0) is upper left point          */
/*    Ynew = GRF_YMAX - Yold                  */

/* ------------------------------------------------------------ */
/* switch on LOCATOR(1) with CURSOR(0) / LEFT_BUTTON(0)         */
/*  SAMPLE(3) mode                                              */
void
grfsamploc (void);

/* switch off LOCATOR(1) with CURSOR(0) / LEFT_BUTTON(0)        */
void
grfsampoff (void);

/*  read locator values x, y , key                             */
void
grfsamples (int *x, int *y, int *k);

void
grfsetdefaults (void);

/* ------------------------------------------------------------ */
void 
grfcompare (int *gv, int lo, int hi)
{
  if(*gv < lo) {*gv = lo;}
  if(*gv > hi) {*gv = hi;}
}

void 
grftest (int *a, int *b)
{
  int h;
  if (*a > *b) {
    h = *a; *a = *b; *b = h;
  }
}

/* load AtariST-colours          */
/* colours defined in /usr/lib/X11/rgb.txt  */
void 
grfloadcolor ()
{
  SRGP_loadCommonColor(1,"white");
  SRGP_loadCommonColor(0,"black");
  SRGP_loadCommonColor(2,"red");
  SRGP_loadCommonColor(3,"green");
  SRGP_loadCommonColor(4,"blue");
  SRGP_loadCommonColor(5,"MediumBlue");
  SRGP_loadCommonColor(6,"brown");
  SRGP_loadCommonColor(7,"DarkGreen");
  SRGP_loadCommonColor(8,"grey");
  SRGP_loadCommonColor(9,"DimGrey");
  SRGP_loadCommonColor(10,"cyan");
  SRGP_loadCommonColor(11,"DeepSkyBlue");
  SRGP_loadCommonColor(12,"magenta");
  SRGP_loadCommonColor(13,"DarkViolet");
  SRGP_loadCommonColor(14,"yellow3");
  SRGP_loadCommonColor(15,"yellow1");
  SRGP_loadCommonColor(16,"white");
  SRGP_loadCommonColor(17,"black");
  SRGP_loadCommonColor(18,"red");
  SRGP_loadCommonColor(19,"green");
  SRGP_loadCommonColor(20,"blue");
  SRGP_loadCommonColor(21,"MediumBlue");
  SRGP_loadCommonColor(22,"brown");
  SRGP_loadCommonColor(23,"DarkGreen");
  SRGP_loadCommonColor(24,"grey");
  SRGP_loadCommonColor(25,"DimGrey");
  SRGP_loadCommonColor(26,"cyan");
  SRGP_loadCommonColor(27,"DeepSkyBlue");
  SRGP_loadCommonColor(28,"magenta");
  SRGP_loadCommonColor(29,"DarkViolet");
  SRGP_loadCommonColor(30,"yellow3");
  SRGP_loadCommonColor(31,"yellow1");
}

/*  load locator shapes, defined in /usr/include/X11/cursorfont.h  */
void
grfloadcursor (void)
{
  SRGP_loadCursor (1, 2);
  SRGP_loadCursor (2, 4);
  SRGP_loadCursor (3, 6);
  SRGP_loadCursor (4, 8);
  SRGP_loadCursor (5, 10);
  SRGP_loadCursor (6, 12);
  SRGP_loadCursor (7, 14);
}

/*  clear window # n    */
void 
grfclear(int n)
{ 
  int x,y;
  x = grf_comm1.grfwindx[n] - 1;
  y = grf_comm1.grfwindy[n] - 1;
  if ((x > 0) && (y > 0)) {
    SRGP_useCanvas(grf_comm1.grfwindow[n]);
    SRGP_inquireAttributes (&grf_comm1.grfclwbund);
    SRGP_setColor (0);
    SRGP_setFillStyle (SOLID);
    SRGP_setWriteMode (WRITE_REPLACE);
    SRGP_setClipRectangle (SRGP_defRectangle(0,0,x,y));
    SRGP_fillRectangleCoord (0,0, x,y);
    SRGP_setBackgroundColor (0);
    SRGP_setAttributes (&grf_comm1.grfclwbund);
  }
  SRGP_refresh ();
}

/* for POLYLINE, POLYFILL, POLYMARK  */
void 
grfpcopy(int n, long *x, long *y)
{
  int k;
  for (k=0; k<n; k++) {
    grfpolyx[k] = x[k];
    grfpolyy[k] = GRF_YMAX - y[k];
  }
} 

/* calculate Coordinates for rounded box    */ 
void 
grffrbox (int x0, int y0, int x1, int y1)
{ 
  grfpolyx[0] = x0;
  grfpolyy[0] = y0+GRF_RS3;
  grfpolyx[1] = x0;
  grfpolyy[1] = y1-GRF_RS3;
  grfpolyx[2] = x0+GRF_RS1;
  grfpolyy[2] = y1-GRF_RS2;
  grfpolyx[3] = x0+GRF_RS2;
  grfpolyy[3] = y1-GRF_RS1;
  grfpolyx[4] = x0+GRF_RS3;
  grfpolyy[4] = y1;
  grfpolyx[5] = x1-GRF_RS3;
  grfpolyy[5] = y1;
  grfpolyx[6] = x1-GRF_RS2;
  grfpolyy[6] = y1-GRF_RS1;
  grfpolyx[7] = x1-GRF_RS1;
  grfpolyy[7] = y1-GRF_RS2;
  grfpolyx[8] = x1;
  grfpolyy[8] = y1-GRF_RS3;
  grfpolyx[9] = x1;
  grfpolyy[9] = y0+GRF_RS3;
  grfpolyx[10] = x1-GRF_RS1;
  grfpolyy[10] = y0+GRF_RS2;
  grfpolyx[11] = x1-GRF_RS2;
  grfpolyy[11] = y0+GRF_RS1;
  grfpolyx[12] = x1-GRF_RS3;
  grfpolyy[12] = y0;
  grfpolyx[13] = x0+GRF_RS3;
  grfpolyy[13] = y0;
  grfpolyx[14] = x0+GRF_RS2;
  grfpolyy[14] = y0+GRF_RS1;
  grfpolyx[15] = x0+GRF_RS1;
  grfpolyy[15] = y0+GRF_RS2;
  grfpolyx[16] = x0;
  grfpolyy[16] = y0+GRF_RS3;
}

/*   TITLEW n,title$      set title in window n                */
/*                        n        # of window */
/*                        title$   text in title-line */
void 
grf_titlew (int n, char *title)
{
  /* printf("grf_titlew  %d  %s \n",n,title); */
  strcpy (grftitle,title);
}

/*   OPENW n,x,y          open window n at (x,y)                */
/*                        n        # of window  */
/*                        (x,y)    placement of window */
/*                        OPENW 0,x,y     set origin o f*/
/*                        graphic-instrucions        */
void 
grf_openw (int n, int x, int y)
{
  /* printf("grf_openw  %d  %d  %d \n",n,x,y); */
  if (n == 0) {
    SRGP_begin (grftitle, GRF_XMAX, GRF_YMAX, 8, FALSE);
    grfsetdefaults ();
    grf_comm1.grfwindx[0] = GRF_XMAX;    
    grf_comm1.grfwindy[0] = GRF_YMAX;
    grf_comm1.grfwindow[0] = 0;     
  }
  else
  {
    if(n < GRF_MAXW) {
      grfcompare (&x, GRF_XMIN, GRF_XMAX);
      grfcompare (&y, GRF_YMIN, GRF_YMAX);
      grf_comm1.grfwindow[n]=SRGP_createCanvas(x,y);
      /* printf("grf_openw  %d return %d \n",n,grf_comm1.grfwindow[n]); */ 
      grf_comm1.grfwindx[n] = x;    
      grf_comm1.grfwindy[n] = y; 
    }
  }
  grfclear (n);
}

/*   CLEARW n             clear window n       */
/*                        n        # of window */
void 
grf_clearw (int n)
{
  /* printf("grf_clearw  %d  \n",n); */
  grfclear(n);
}

/*   CLOSEW n             close window n       */
/*                        n        # of window */
void 
grf_closew (int n)
{
  int k, kw;
  /* printf("grf_closew  %d w %d \n",n,grf_comm1.grfwindow[n]); */
  if (n == 0) {
    /*  close all other windows  */
    for (k=1; k<GRF_MAXW; k++) {
      kw = grf_comm1.grfwindow[k]; 
      if (kw > 0) SRGP_deleteCanvas(kw);
    }
    SRGP_end ();
  }
  else
  {
    SRGP_deleteCanvas(grf_comm1.grfwindow[n]);
    SRGP_refresh();
  }  
}

/*   COLOR cdraw,cback    set colour for drawing and background     */
/*                        cdraw: colour for BOX,CIRCLE,DRAW,ELLIPSE,*/
/*                                      LINE,PLOT,RBOX              */
/*                        cback: colour for background in pattern   */
void 
grf_color (int cd, int cb)
{
  /* printf("grf_color  %d %d \n",cd,cb); */
  colordraw = cd & 31;
  SRGP_setColor(colordraw);
  colorback = cb & 31;
  SRGP_setBackgroundColor (colorback);
}

/*   INFOW n,info$        info-line for window                */
/*                        n        # of window */
/*                        info$   text in info line */
void 
grf_infow (int n, char *info)
{
  int x, y;  
  if (grf_comm1.grfwindow[n] > 0) {
    x = grf_comm1.grfwindx[n] - strlen(info)*8;
    y = grf_comm1.grfwindy[n] - GRF_HTXT;
    SRGP_text (SRGP_defPoint (x, y), info);
    SRGP_refresh();
  }
    /* printf("grf_infow  %d  %s \n",n,info); */
}

/*   SETCOLOR i,r,g,b     set colours for display                */
/*                        i        # of colour-register*/
/*                        r,g,b    red-green-blue value (0..7)*/
void 
grf_setcolor (int i, int r, int g, int b)
{ 
  int incol;
  incol = i;
  grfcompare (&incol, 0, 31);
  /* printf("grf_setcolor  %d  %d  %d  %d \n",i,r,g,b); */
  scolr[incol] = (r & 7)*32;
  scolg[incol] = (g & 7)*32;
  scolb[incol] = (b & 7)*32;
  SRGP_loadColorTable (0, 32, scolr+2, scolg+2, scolb+2);
}

/*   GRAPHMODE n          set graphic-mode                */
/*                        n       mode: 1=replace 2=transparent */
/*                                3=xor 4=reverse transparent */
void 
grf_graphmode (int n)
{
  writeMode grfmode;
  /* printf("grf_graphmode  %d  \n",n); */
  if(n > 0) {
    grfmode= (n-1) % 4;
    SRGP_setWriteMode (grfmode);
  }
}

/*   DEFFILL c,a,b        define colour and texture             */
/*                        c       colour for filling            */
/*                        a,b     select pattern for FILL,PBOX, */
/*                                PCIRCLE,PELLIPSE,PRBOX        */
void 
grf_deffill (int c,int a,int b)
{ 
  int p;
  /* printf("grf_deffill  %d  %d  %d  \n",c,a,b); */
  SRGP_setFillStyle (BITMAP_PATTERN_OPAQUE);
  /* SRGP_setColor (c); */
  colorfill = c & 31;
  p = (a & 7) + b * 8;
  grfcompare (&p, 0, 103);
  SRGP_setFillBitmapPattern (p);
}

/*   DEFLINE s,b          line stile, width, type                */
/*                        s       linestile                      */
/*                        1=cont. / 2=dash / 3=dot 4=dot-dash    */
/*                        b       width of line                  */
/*                                for BOX,CIRCLE,DRAW,ELLIPSE,LINE,RBOX*/
void 
grf_defline (int s,int b)
{ 
  drawStyle grfstyle;
  /* printf("grf_defline  %d  %d  \n",s,b); */
  if (s > 0) {
    grfstyle= (s-1) & 3;
    SRGP_setLineStyle (grfstyle);
  }
  SRGP_setLineWidth(b);
}

/*   DEFMARK c,a,g        colour, size, type (POLYMARK)        */
/*                        c       colour*/
/*                        a       type of marker   */
/*                                (1=o 2=[] 3=+)   */
/*                        g       size of marker (0,20,40,60,80,100)*/
/*                                for POLYMARK*/
void 
grf_defmark (int c,int a,int g)
{ 
  markerStyle grfm;
  /* printf("grf_defmark  %d  %d  %d  \n",c,a,g); */
  if (a > 0) {
    grfm=(a-1) & 3;
    SRGP_setMarkerStyle(grfm);
    SRGP_setMarkerSize(g);
    /* SRGP_setColor(c); */
    colormark = c & 31;
  }
}

/*   DEFTEXT c,s,r,g        colour,type,stile,rot. (TEXT)        */
/*                        c       colour    */
/*                        s       stile (1=bold 2=light 4=cursiv */
/*                                8=underline 16=outlined)       */
/*                        r       rotation angle in 1/10 degree  */
/*                        g       size of text (4,6,13,32)       */
/**/
void 
grf_deftext (int c,int s,int r,int g)
{
  /* printf("grf_deftext  %d  %d  %d  %d \n",c,s,r,g); */

}

/*   PLOT x,y             draw a point on the screen        */
/*                        (x,y)        point-coordinate*/
/*                        equivalent to DRAW x,y*/
void 
grf_plot (int x, int y)
{
  /* printf("grf_plot  %d  %d  \n",x,y); */
  SRGP_pointCoord(x,GRF_YMAX-y);
  SRGP_refresh();
}

/*   LINE x0,y0,x1,y1     line from (x0,y0) to (x1,y1)  */
/*                        (x0,y0)     start-coordinate  */
/*                        (x1,y1)     end-coordinate    */
void 
grf_line (int x0, int y0, int x1, int y1)
{
  /* printf("grf_line  %d  %d  %d  %d \n",x0,y0,x1,y1); */
  SRGP_lineCoord (x0, GRF_YMAX-y0, x1, GRF_YMAX-y1);
  SRGP_refresh();
}

/*   TEXT x,y,s$          draw string s$ at (x,y)       */
/*                        (x,y)        text-coordinate  */
/*                        s$        text-string         */
void 
grf_text (int x, int y, char *s)
{
  /* printf("grf_text  %d  %d  %s \n",x,y,s); */
  SRGP_text (SRGP_defPoint (x, GRF_YMAX-y), s);
  SRGP_refresh();
}

/*   FILL x,y             fill pattern into an areal        */
/*                        (x,y)        coordinate to start filling*/
void 
grf_fill (int x, int y)
{
  /* printf("grf_fill  %d  %d \n",x,y); */
}

/*   BOX x0,y0,x1,y1      draw rectangle                        */
/*                        (x0,y0)        lower left corner*/
/*                        (x1,y1)        upper right corner*/
void 
grf_box (int x0, int y0, int x1, int y1)
{ int y2, y3;
  /* printf("grf_box  %d  %d  %d  %d \n",x0,y0,x1,y1); */
  grftest (&x0, &x1);
  y2 = GRF_YMAX-y0;
  y3 = GRF_YMAX-y1;
  grftest (&y2, &y3);
  /* printf("grf_box  %d  %d  %d  %d \n",x0,y2,x1,y3); */
  SRGP_rectangleCoord (x0, y2, x1, y3);
  SRGP_refresh();
}

/*   PBOX x0,y0,x1,y1     draw filled box                   */
/*                        (x0,y0)        lower left corner  */
/*                        (x1,y1)        upper right corner */
/*   Atari-GEM: a filled box has the outline visible!       */
void 
grf_pbox (int x0, int y0, int x1, int y1)
{ int y2, y3;
  /* printf("grf_pbox  %d  %d  %d  %d \n",x0,y0,x1,y1); */
  grftest (&x0, &x1);
  y2 = GRF_YMAX-y0;
  y3 = GRF_YMAX-y1;
  grftest (&y2, &y3);
  /* printf("grf_pbox  %d  %d  %d  %d \n",x0,y2,x1,y3); */
  SRGP_setColor (colorfill);
  SRGP_fillRectangleCoord (x0, y2, x1, y3);
  SRGP_setColor (colordraw);
  SRGP_rectangleCoord (x0, y2, x1, y3);
  SRGP_refresh();
}

/*   RBOX x0,y0,x1,y1     draw rounded box                 */
/*                        (x0,y0)        lower left corner */
/*                        (x1,y1)        upper right corner*/
void 
grf_rbox (int x0, int y0, int x1, int y1)
{ int y2, y3; 
  /* printf("grf_rbox  %d  %d  %d  %d \n",x0,y0,x1,y1); */
  if ((abs(x1-x0) < 2*GRF_RS3) || (abs(y1-y0) < 2*GRF_RS3))
  {
    grf_box (x0, y0, x1, y1);
  }
  else
  {  
  grftest (&x0, &x1);
  y2 = GRF_YMAX-y0;
  y3 = GRF_YMAX-y1;
  grftest (&y2, &y3);
  /* printf("grf_rbox  %d  %d  %d  %d \n",x0,y2,x1,y3); */
  grffrbox (x0, y2, x1, y3);
  SRGP_polygonCoord (17, grfpolyx, grfpolyy);
  }
  SRGP_refresh();
}

/*   PRBOX x0,y0,x1,y1    draw rounded filled box            */
/*                        (x0,y0)        lower left corner   */
/*                        (x1,y1)        upper right corner  */
/*   Atari-GEM: a filled box has the outline visible!        */
void 
grf_prbox (int x0, int y0, int x1, int y1)
{ int y2, y3;
  /* printf("grf_prbox  %d  %d  %d  %d \n",x0,y0,x1,y1); */
  if ((abs(x1-x0) < 2*GRF_RS3) || (abs(y1-y0) < 2*GRF_RS3))
  {
     grf_pbox (x0, y0, x1, y1);
  }
  else
  {
  grftest (&x0, &x1);
  y2 = GRF_YMAX-y0;
  y3 = GRF_YMAX-y1;
  grftest (&y2, &y3);
  /* printf("grf_prbox  %d  %d  %d  %d \n",x0,y2,x1,y3); */
  grffrbox (x0, y2, x1, y3);
  SRGP_setColor (colorfill);
  SRGP_fillPolygonCoord (17, grfpolyx, grfpolyy);
  SRGP_setColor (colordraw);
  SRGP_polygonCoord (17, grfpolyx, grfpolyy);
  }
  SRGP_refresh();
}

void 
grfxellipse (int mx, int x0, int y0, int x1, int y1, int p0, int p1)
{ int y2, y3;
  double ang1, ang2;
  y2 = GRF_YMAX-y0;
  y3 = GRF_YMAX-y1;
  grftest (&y2, &y3);
  largerect = SRGP_defRectangle (x0, y2, x1, y3);
  /* printf("grfxellipse %d  %d  %d  %d | %d  %d\n",x0,y2,x1,y3,p0,p1); */
  if ((p0 == 0) && (p1 == 0)) { ang2 = 360.0; }
  else { ang2 = (double) p1 / 10.0; }
  ang1 = (double) p0 / 10.0;
  if (mx == 0) {
    SRGP_ellipseArc (largerect, ang1, ang2);
  }
  else {
    SRGP_setColor (colorfill);
    SRGP_fillEllipseArc (largerect, ang1, ang2);
    SRGP_setColor (colordraw);
  }
  SRGP_refresh();
}

/*   CIRCLE x,y,r,p0,p1   draw circle at (x,y) radius r        */
/*                        (x,y)   center*/
/*                        r       radius*/
/*                        p0,p1   arc angle p0..p1 in 1/10 degree*/
void 
grf_circle (int x, int y, int r, int p0, int p1)
{
 /*  printf("grf_circle  %d  %d  %d  %d  %d \n",x,y,r,p0,p1); */
  grfxellipse (0, x-r, y-r, x+r, y+r, p0, p1);
}

/*   PCIRCLE x,y,r,p0,p1  draw filled circle                */
/*                        (x,y)   center*/
/*                        r       radius*/
/*                        p0,p1   arc angle p0..p1 in 1/10 degree*/
void 
grf_pcircle (int x, int y, int r, int p0, int p1)
{
  /* printf("grf_pcircle  %d  %d  %d  %d  %d \n",x,y,r,p0,p1); */
  grfxellipse (1, x-r, y-r, x+r, y+r, p0, p1);
}

/*   ELLIPSE x,y,rx,ry    draw ellipse at (x,y)                */
/*                        (x,y)   center coordinate*/
/*                        rx,ry   axis*/
/*                        p0,p1   arc angle p0..p1 in 1/10 degree*/
void 
grf_ellipse (int x, int y, int rx, int ry, int p0, int p1)
{
  /* printf("grf_ellipse   %d  %d  %d  %d  %d  %d \n",x,y,rx,ry,p0,p1); */
  grfxellipse (0, x-rx, y-ry, x+rx, y+ry, p0, p1);
}

/*   PELLIPSE x,y,rx,ry   draw filled ellipse                */
/*                        (x,y)   center coordinate*/
/*                        rx,ry   axis*/
/*                        p0,p1   arc angle p0..p1 in 1/10 degree*/
/**/
void 
grf_pellipse (int x, int y, int rx, int ry, int p0 ,int p1)
{
  /* printf("grf_pellipse  %d  %d  %d  %d  %d  %d \n",x,y,rx,ry,p0,p1); */
  grfxellipse (1, x-rx, y-ry, x+rx, y+ry, p0, p1);
}

/*   DEFMOUSE n           define mouse symbol                */
/*                        n      mouse-sysmbol (0..7)*/
/*   SHOWM                show mouse symbol                 */
/*   HIDEM                hide mouse symbol                */
void 
grf_defmouse (long *mode, long n)
{
  /* printf("grf_defmouse  %d %d  \n",*mode,n); */
  if ((n == 0) && (*mode != 0)) {
    grfsampoff(); 
    *mode = 0;
  }
  if ((n == 1) && (*mode == 0)) {
    grfsamploc(); 
    *mode = 1;
  }
  if (n >= 4) {
    SRGP_setLocatorEchoCursorShape ((n-4) & 7);
  }
}

/*   MOUSE x,y,k          mouse-position (x,y) and key k        */
/*                        (x,y)   mouse-coordinate*/
/*                        k       status of mouse keys 1=left 2=right*/
/**/
void 
grf_mouse (long *x, long *y, long *k)
{
  int sx, sy, sk;  
  grfsamples (&sx, &sy, &sk);
  *x = (long) sx;
  *y = (long) (GRF_YMAX - sy);
  *k = (long) sk; 
  /* printf("grf_mouse  %d  %d  %d  \n", *x, *y, *k); */
}

/*   POLYLINE n,x(),y()   draw polygon in (x(),y())        */
/*                        n       # of points*/
/*                        x(),y()   arrays with (x,y)-values of points*/
/*                        the first point = last point to draw*/
/*                        a closed polygon*/
void 
grf_polyline (int n, long *x, long *y)
{ 
  grfpcopy (n, x, y);
  SRGP_polygonCoord (n-1, grfpolyx, grfpolyy);
  SRGP_refresh();
}

/*   POLYFILL n,x(),y()   draw filled polygon with outline           */
/*                        n       # of points                        */
/*                        x(),y()  arrays with (x,y)-values of points*/
/*                        connects first point with last point       */
/*                        and fills the resulting polygon with       */
/*                        pattern defined in DEFFILL                 */
void 
grf_polyfill (int n, long *x, long *y)
{
  grfpcopy (n, x, y);
  SRGP_setColor (colorfill);
  SRGP_fillPolygonCoord (n-1, grfpolyx, grfpolyy);
  SRGP_setColor (colordraw);
  SRGP_polygonCoord (n-1, grfpolyx, grfpolyy);
  SRGP_refresh();
}

/*   POLYMARK n,x(),y()   draw polygon points                         */
/*                        n       # of points                         */
/*                        x(),y()   arrays with (x,y)-values of points*/
/*                        type of points and colour defined in DEFMARK*/
void 
grf_polymark (int n, long *x, long *y)
{
  grfpcopy (n, x, y);
  SRGP_setColor (colormark);
  SRGP_polyMarkerCoord (n-1, grfpolyx, grfpolyy);
  SRGP_setColor (colordraw);
  SRGP_refresh();
}

void
grfsetdefaults (void)
{
  long mode;
  grfloadcolor();
  grfloadcursor();
  grf_color (1, 0);       /* set: colordraw, colorback */ 
  grf_graphmode (1);      /* mode: replace */
  grf_deffill (1, 0, 0);  /* set: colorfill,  pattern 0,0 */
  grf_defline (0, 1);     /* line, w=1 */
  grf_defmark (1, 1, 20); /* set: colormark */
  grf_deftext (1, 0, 0, 6); /* not yet implemented  */
  grf_defmouse (&mode, 4); /* mouse-symbol=0 */
}

