/*	comps19 - compare customer s-records against verification file
	This program reads two S-record files into memory (customer code and RVU
	verification file), and compares the contents of selected address
	ranges as defined by a parameter file.

command line: comps19 prmfile ctmfile [ctmfile] ... -v verfile [verfile] ...

where
	prmfile		is a text file containing parameters for the ROM device
	ctmfile		is the customer's S-record file
	verfile		is the verification file received from the RVU center

prmfile format:

line 1:		device name		map start address (in hex)		size
line 2:		range name		start address					end address
line 3:		range name		start address					end address
.
.
.
and so on.  Here's an example:

MC68HC05C4	0		2000
Data_ROM	20		4F
Code_ROM	100		10FF
Vectors		1FF4	1FFF
 
  $Log: comps19.c,v $
 * Revision 1.3  1995/09/07  17:17:27  wadeh
 * fixed mismatch printing of address
 *
 * Revision 1.2  1995/09/07  17:07:22  wadeh
 * Wade Hampton
 * - changes per notes (added PRINT_MISMATCH, range checking, notes,
 * etc.)
 *

  Modifications by Wade Hampton (tasi029@tmn.com) (7/7/95)
  - added notes (search for WWH)
  - added rcs stuff and rcs_id
  - expanded Usage function (more help added)
  - added range check in comp_buff
  - added PRINT_MISMATCH compile option to be very verbose and print all mismatches

  This has been tested on Linux using GCC:

      gcc -g -o comps19 comps19.c

    or for debug:

      gcc -DDEBUG -g -o comps19 comps19.c
  
    to print mismatches:

      gcc -DPRINT_MISMATCH -g -o comps19 comps19.c
*/
static char *rcs_id = "$Id: comps19.c,v 1.3 1995/09/07 17:17:27 wadeh Exp $";

#include	<stdio.h>

#ifdef	DEBUG
#define	DB(x)	x
#else
#define	DB(x)
#endif

#define		MAX_RANGE_CNT	8		/* max no of ranges in a map */
#define		RANGE_NAME_SIZE	33
#define		FN_MAX		128			/* max size of filename */

typedef unsigned ADDRESS;

struct range {
char name [RANGE_NAME_SIZE];
ADDRESS start, end;
} ranges [MAX_RANGE_CNT];

int range_cnt;						/* no. of ranges active in this run */
char chipname [RANGE_NAME_SIZE];	/* what device are we using? */
char allok, reading, *ctm_buff, *ver_buff;
ADDRESS address, offset, rom_size;
int checksum, count;
FILE *infile;
int fileline;						/* line number of input file */
char getbuff [128];
char *dashes = "===========================================================\n";

extern char *malloc ();

main (argc, argv)
int argc;
char **argv;
{
	char *cur_buff;
	int ctm_cnt, ver_cnt, cur_arg;

	if (argc < 5) usage ();			/* check for valid command line */
	read_parms (argv [1]);			/* get parameter info */
	if ((ctm_buff = malloc (rom_size)) == (char *) NULL)
		error ("out of memory");
	if ((ver_buff = malloc (rom_size)) == (char *) NULL)
		error ("out of memory");
	memset (ctm_buff,0,rom_size);
	memset (ver_buff,0,rom_size);
	ctm_cnt = ver_cnt = 0;
	for (cur_buff = ctm_buff, cur_arg = 2; cur_arg < argc; cur_arg++)
	{
		if (!strcmp (argv [cur_arg], "-v") || !strcmp (argv [cur_arg], "-V"))
		{
			if (!ctm_cnt) error ("no customer filenames specified");
			cur_buff = ver_buff;
			continue;
		}
		(cur_buff == ctm_buff) ? ctm_cnt++ : ver_cnt++;
		read_s19 (argv [cur_arg], cur_buff);
	}
	if (!ver_cnt) error ("no verify filenames specified");

	allok = 1;
	for (cur_arg = 0; cur_arg < range_cnt; cur_arg++)
	{
		printf ("Checking %s range\n",ranges[cur_arg].name);
		comp_buff (cur_arg);
	}
	if (allok) printf ("Files check OK!\n");
	exit (0);
}

/* read_s19 reads data from file specified into buff */

read_s19 (filename, buff)
char *filename, *buff;
{
	int c;
	ADDRESS writeat;

	printf ("Reading %s file %s\n",
		(buff == ctm_buff) ? "customer" : "verify", filename);
	if ((infile = fopen (filename,"r")) == (FILE *) NULL)
		error ("could not open input file");
	fileline = reading = 0;
	while ((c = getbyte ()) != EOF)
	{
		DB (printf ("writing %02x to %04x\n",c,(address-1)-offset);)
		if ((writeat = (address - 1) - offset) < rom_size) buff [writeat] = c;
	}
	if (fclose (infile)) error ("error closing input file");
}

/* skipwhite moves pointer past white space to next non-white */

char *skipwhite (s)
char *s;
{
	while (isspace (*s)) s++;
	return s;
}

/* cvthex converts string of hex chars at ptr into 16-bit unsigned at dest */
/* skips white space, returns updated pointer */

char *cvthex (ptr, dest)
char *ptr;
ADDRESS *dest;
{
/*	DB (printf ("cvthex: starting buffer %s\n",ptr);) */
	ptr = skipwhite (ptr);
/*	DB (printf ("after skipwhite: %s\n",ptr);) */
	*dest = 0;
	while (isxdigit (*ptr))
	{
		*dest = (*dest << 4) + htoi (*ptr++);
/*		DB (printf ("dest is %04x buffer is %s\n",*dest,ptr);) */
	}
	return ptr;
}

/* read_parms reads parameter info from filename given */

read_parms (filename)
char *filename;
{
	char linebuff [128], workbuff [16], *dest, *cptr;
	int c;

	if ((infile = fopen (filename,"r")) == (FILE *) NULL)
		error ("could not open input file");

	if (fgets (linebuff,128, infile) == (char *) NULL)
		error ("parameter file error");

	getptoke (linebuff, chipname, &offset, &rom_size);
	printf ("\n%-33s start = %04x end = %04x\n%s",
		chipname, offset, offset + rom_size - 1,dashes);

	for (range_cnt = 0; range_cnt < MAX_RANGE_CNT; range_cnt++)
	{
		if (fgets (linebuff,128, infile) == (char *) NULL) break;
		getptoke (linebuff,
			ranges [range_cnt].name,
			&ranges [range_cnt].start,
			&ranges [range_cnt].end);
		printf ("%-33s start = %04x end = %04x\n",
			ranges [range_cnt].name,
			ranges [range_cnt].start,
			ranges [range_cnt].end);
	}
	printf ("%s", dashes);
	if (fclose (infile)) error ("error closing parameter file");
}

/* comp_buff compares customer data vs verify data for range given */
/* displays mismatches on screen and terminates if descrepancy found */

comp_buff (which)
int which;

{
    char *ctm, *ver, *last;
    int iTemp;
	ADDRESS where;
	char ok;

        /* WWH - added range checking and error messages */
        iTemp = ranges [which].start - offset;
        if (iTemp >= rom_size)
	   error ("ctm out of memory");
	ctm = iTemp + ctm_buff;

        /* WWH - added range checking and error messages */
	iTemp = ranges [which].start - offset;
        if (iTemp >= rom_size)
	   error ("ver out of memory");
	ver = iTemp + ver_buff;

        /* WWH - added range checking and error messages */
	iTemp = ranges [which].end - offset;
        if (iTemp >= rom_size)
	   error ("ver out of memory");
	last = iTemp + ctm_buff;

	for (ok = 1; ctm <= last; ctm++,ver++)
	if (ok)
	{
		if (*ctm == *ver) continue;
		else
		{
			allok = ok = 0;
			where = (ADDRESS) ctm;
#ifdef PRINT_MISMATCH
			printf ("Mismatch at %04x ctm=0x%02x ver=0x%02x\n",
				where - (ADDRESS) ctm_buff + offset,
				(0xFF&(*ctm)),(0xFF&(*ver))); 
#endif
		}
	}
	else
	{
#ifndef PRINT_MISMATCH
		if (*ctm != *ver) continue; 
#else
		if (*ctm != *ver) 
		{
			printf ("Mismatch at %04x ctm=0x%02x ver=0x%02x\n",
				(ADDRESS)ctm - (ADDRESS) ctm_buff + offset,
				(0xFF&(*ctm)),(0xFF&(*ver))); 
		}
#endif                                                
		else
		{
			ok = 1;
#ifndef PRINT_MISMATCH
			printf ("Mismatch at %04x length %04x\n",
				where - (ADDRESS) ctm_buff + offset,
				ctm - where);
#endif
		}
	}
	if (!ok)
		printf ("Mismatch at %04x length %04x\n",
			where - (ADDRESS) ctm_buff + offset,
			ctm - where);
}

/* getptoke parses up parameter line into string and two hex numbers */

getptoke (buffer, string, hex1, hex2)
char *buffer, *string;
ADDRESS *hex1, *hex2;
{
	buffer = skipwhite (buffer);
	while (isprint (*buffer)) *string++ = *buffer++;
	buffer = cvthex (buffer, hex1);
	cvthex (buffer, hex2);
}

char *next;

 /* getbyte returns next byte from input file, or EOF if end of file */

getbyte ()
{
    int c;

	for (;;)
	{
    if (!reading)
    {
		DB (printf ("scanning\n");)
        for (;;)
        {
			if (fgets (getbuff,128,infile) == (char *) NULL) return (EOF);
			fileline++;
			if (*getbuff != 'S') continue;
            if ((c = *(getbuff+1)) == '9') return (EOF);
            if (c == '1') break;
        }
		DB (printf ("start of line\n");)
		next = getbuff + 2;
        checksum = 0;
		if ((count = _getbyte ()) == EOF) return (EOF);
		if ((address = _getbyte ()) == EOF) return (EOF);
		if ((c = _getbyte ()) == EOF) return (EOF);
        address = (address << 8) + c;
        reading++;
    }
	if ((c = _getbyte ()) == EOF) return (EOF);
    address++;
    if (!count)
    {
        if (~checksum & 0xFF)
		{
			printf ("Checksum error on input file line %d\n",fileline);
			exit (1);
		}
		DB (printf ("end of line\n");)
        reading = 0;
    }
    else return (c);
	}
}

/* _getbyte returns the binary value of the next two ASCII characters read */
/* returns EOF if end of file occurs - also updates checksum */

_getbyte ()
{
    int c1,c2;

    count--;
	if ((c1 = *next++) == '\0')
	{
		DB (printf ("_getbyte got an EOF\n");)
		return (EOF);
	}
	if ((c2 = *next++) == '\0')
	{
		DB (printf ("_getbyte got an EOF\n");)
		return (EOF);
	}
    c1 = htoi (c1);
    checksum += c1 = (c1 << 4) + htoi (c2);
    return (c1);
}

/* htoi converts ASCII character to binary */

htoi (c)
char c;
{
	c = toupper (c) - '0';
    if (c > 9) c -= ('A' - '9' - 1);
    return c;
}


/* usage shows command format and quits to operating system */

usage ()
{
/* 	error ("usage: comps19 ctmfile [ctmfile] ... -v verfile [verfile] ..."); */
  fprintf (stderr, "comps19:  compare two Motorola Hex files\n");
  fprintf (stderr, "  Usage:  prmfile ctmfile [ctmfile] ... -v verfile [verfile] ...\n");
  fprintf (stderr, "  Options:  -v verfild\n");
  fprintf (stderr, "  Prmfile format:\n");
  fprintf (stderr, "    line 1:  device name    map start address (in hex)   size\n");
  fprintf (stderr, "    line 2:  range name     start address                end address\n");
  fprintf (stderr, "    line 3:  range name     start address                end address\n");
  fprintf (stderr, "      e.g.:\n");
  fprintf (stderr, "    MC68HC05C4  0               2000\n");
  fprintf (stderr, "    Data_ROM    20              4F\n");
  fprintf (stderr, "    Code_ROM    100             10FF\n");
  fprintf (stderr, "    Vectors     1FF4            1FFF\n");
  fprintf (stderr, "      to compare vs an EPROM:\n");
  fprintf (stderr, "    74C64        E000            2000\n");
  fprintf (stderr, "    Code_ROM     E000            FFFF\n");
  fprintf (stderr, "  Version:  %s\n", rcs_id);
}

error (problem)
char *problem;

{
	printf ("%s\n",problem);
	exit (1);
}

/* WWH added the ifdef below */
#ifdef NEED_ISXDIGIT
/* isxdigit - to fix problem with the one in Datalight C */

isxdigit (c)
char c;
{
	return ((c >= '0' && c <= '9')
         || (c >= 'A' && c <= 'F')
		 || (c >= 'a' && c <= 'f'));
}
#endif
