#! /usr/bin/igawk -f
# Use igawk instead of gawk to process include files

#=========================================================================
#    structured_pgm.awk
#
#    Copyright (C) 1998  Jacques Pelletier
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#=========================================================================

@include elemental.awk

# THESE MACROS SHOULD BE COMPATIBLE WITH ALL 8051 COMPILERS
# HOW TO USE:
#
# igawk -f structured_pgm.awk myprog.asm >myprog.result
#
# The file myprog.result will contain the resulting assembler source with
# the macros expanded (code and labels generated). 

# ALL MACROS BLOCK TYPES CAN BE FREELY MIXED TOGETHER
# SUPPORTS NESTED IF BLOCKS, DO-WHILE BLOCKS,SWITCHES AND WHILE-END BLOCKS
#
# LEVEL NESTING IS NOT CHECKED, IT'S ASSUMED THAT THE AWK'S STACK SHOULD BE ENOUGH
# FOR MOST APPLICATIONS WITHOUT BUSTING 

# For these 3 macros, here's the implementation:
# M_DO_WHILE(REGISTER,CONDITION,VALUE)
# M_WHILE(REGISTER,CONDITION,VALUE)
# M_IF_CMP(REGISTER,CONDITION,VALUE)

# Condition	   Compare		TRUE IF:
#
# R0 <  R1	R0-R1 <  0			CY
# R0 <= R1 		 0 <= R1-R0 		NC
# R0 >  R1		 0 >  R1-R0		CY
# R0 >= R1	R0-R1 >= 0			NC
#
# R0 <  A		 0 <  A-R0		NC and NZ
# R0 <= A		 0 <= A-R0		NC
# R0 >  A		 0 >  A-R0		CY
# R0 >= A		 0 >= A-R0		CY or Z
#
# A <  R1	 A-R1 <	 0			CY
# A <= R1	 A-R1 <= 0			CY or Z
# A >  R1	 A-R1 >  0			NC and NZ
# A >= R1	 A-R1 >= 0			NC
#
#-----------------------------------------
#					TRUE IF: 				
# 0 >  X or X <  0				CY
# 0 <= X or X >= 0				NC
# 0 >= X or X <= 0				CY or Z
# 0 <  X or X >  0				NC and NZ
#
#

#----------------------------------------------------------
function PUSH(VALUE)
{
	_sp_stack[_sp_stack_ptr++] = VALUE
}

function POP(   i)
{
	i = --_sp_stack_ptr
	if (i >= 0)
		return (_sp_stack[i])
	else
		ERR_MSG("PUSH AND POP ARE NOT BALANCED")
}

#----------------------------------------------------------
# register AND value CAN BE ALL AVAILABLE ADDRESSING MODE
# register, DIRECT, IMMEDIATE,INDIRECT
function M_COMPARE_JMP(register,condition,value,label,ctr1,ctr2,ctr3)
{
	# < OR >=
	if (condition ~ /^<$|^>=$|LT|HE/)
	{ 
		if (REG_NOT_ACC(value))
		{
			SUB(register,value)

			# register - value < 0
			if (condition ~ /<|LT/)
			{
				INST3("JNC",label,ctr1,ctr2,ctr3)
			}
			# register - value >= 0
			else
			{
				INST3("JC",label,ctr1,ctr2,ctr3)
			}
		}
		else
		{
			SUB_A(register)

			# 0 >= value - register
			if (condition ~ />=|HE/)
			{
				IF_Z_OR_C_ELSE2(label,ctr1,ctr2,ctr3)
			}
			# 0 < value - register
			else
			{
				IF_NC_AND_NZ_ELSE2(label,ctr1,ctr2,ctr3)
			}
		}
	}
	else
	{
		# > OR <=
		if (condition ~ /^>$|^<=$|HT|LE/)
		{ 
			if (REG_NOT_ACC(register))
			{
				SUB(register,value)

				# 0 <= value - register
				if (condition ~ /<=|LE/)
				{
					INST3("JC",label,ctr1,ctr2,ctr3)
				}
				# 0 > value - register
				else
				{
					INST3("JNC",label,ctr1,ctr2,ctr3)
				}
			}
			else
			{
				SUB_A(value)

				# register - value <= 0
				if (condition ~ /<=|LE/)
				{
					IF_Z_OR_C_ELSE2(label,ctr1,ctr2,ctr3)
			  	}
				# register - value > 0
			  	else
		  		{
					IF_NC_AND_NZ_ELSE2(label,ctr1,ctr2,ctr3)
			  	}
	  		}
	  	}
	  	else
  		{
			# = OR ==
			if (condition ~ /^=$|^==$|EQ/)
			{
		        MOV_A(register)
	   			JNE_A3(value,label,ctr1,ctr2,ctr3)
			}
			else
			{
				# != OR <>
				if (condition ~ /^!=$|^<>$|NE/)
				{
			        MOV_A(register)
			        JEQ_A3(value,label,ctr1,ctr2,ctr3)
				}
				else
				{
					ERR_MSG("M_CMP_JMP -- UNKNOWN condition")
				}
			}
		}
	}
}

# register AND value CAN BE ALL AVAILABLE ADDRESSING MODE
# register, DIRECT, IMMEDIATE,INDIRECT
function M_COMPARE_NJMP(register,condition,value,label,ctr1,ctr2,ctr3)
{
	# < OR >=
	if (condition ~ /^<$|^>=$|LT|HE/)
	{ 
		if (REG_NOT_ACC(value))
		{
			SUB(register,value)

			# register - value < 0
			if (condition ~ /<|LT/)
			{
				INST2("JC",label,ctr1,ctr2)
			}
			# register - value >= 0
			else
			{
				INST2("JNC",label,ctr1,ctr2)
			}
		}
		else
		{
			SUB_A(register)

			# 0 >= value - register
			if (condition ~ />=|HE/)
			{
				IF_NC_AND_NZ_ELSE(label,ctr1,ctr2)
			}
			# 0 < value - register
			else
			{
				IF_Z_OR_C_ELSE(label,ctr1,ctr2)
			}
		}
	}
	else
	{
		# > OR <=
		if (condition ~ /^>$|^<=$|HT|LE/)
		{ 
			if (REG_NOT_ACC(register))
			{
				SUB(register,value)

				# 0 <= value - register
				if (condition ~ /<=|LE/)
				{
					INST2("JNC",label,ctr1,ctr2)
				}
				# 0 > value - register
				else
				{
					INST2("JC",label,ctr1,ctr2)
				}
			}
			else
			{
				SUB_A(value)

				# register - value <= 0
				if (condition ~ /<=|LE/)
				{
					IF_NC_AND_NZ_ELSE(label,ctr1,ctr2)
			  	}
			  	# register - value > 0
				else
		  		{
					IF_Z_OR_C_ELSE(label,ctr1,ctr2)
			  	}
	  		}
	  	}
	  	else
  		{
			# = OR ==
			if (condition ~ /^=$|^==$|EQ/)
			{
		        MOV_A(register)
	   			JEQ_A2(value,label,ctr1,ctr2)
			}
			else
			{
				# != OR <>
				if (condition ~ /^!=$|^<>$|NE/)
				{
			        MOV_A(register)
			        JNE_A2(value,label,ctr1,ctr2)
				}
				else
				{
					ERR_MSG("M_COMPARE_NJMP -- UNKNOWN CONDITION")
				}
			}
		}
	}
}

#----------------------------------------------------------
function DO_WHILE_END()
{
	# Put a label here for the BREAK statement.
	LABEL("DO_BRK_",_sp_do_level,_sp_do_lbl[_sp_do_level])

	# Increment the successive label counter for the current level.
	_sp_do_lbl[_sp_do_level]++

	# Step down one level.
	_sp_do_level--

	# Error condition
	# BLOCK BEGIN/END NOT MATCHING
	if (_sp_do_level < 0) ERR_MSG("M_DO_WHILE WITHOUT M_DO")
}

#----------------------------------------------------------
# This function is the same for all DO ... WHILE loop forms. It will
# put a unique label at the beginning of the DO ... WHILE loop.
# The condition at the end of the DO ... WHILE loop will jump to this
# label if the condition is met.
function M_DO()
{
	_sp_current_block = "DO"

	# Increment the nesting level.
	_sp_do_level++
	
	# Generate a unique label according to the nesting level and the 
	# number of successive WHILE loops in the current level.

	LABEL("DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])
}
#----------------------------------------------------------

function M_DO_WHILE_BIT(BIT)
{
	#JUMP TO THE BEGINNING OF THE DO WHILE LOOP IF THE BIT IS TRUE
	JMP_BIT_OR_CY(BIT,"DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])

	# Generate the label after the DO ... WHILE loop.
	DO_WHILE_END()
}

function M_DO_WHILE_NOT_BIT(BIT)
{
	JMP_NOT_BIT_OR_CY(BIT,"DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])

	# Generate the label after the DO ... WHILE loop.
	DO_WHILE_END()
}

function M_DO_WHILE_EQU(REGISTER,MASK,VALUE)
{
	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	JEQ_A2(VALUE,"DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])

	# Generate the label after the DO ... WHILE loop.
	DO_WHILE_END()
}

function M_DO_WHILE_NOT_EQU(REGISTER,MASK,VALUE)
{
	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	JNE_A2(VALUE,"DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])

	# Generate the label after the DO ... WHILE loop.
	DO_WHILE_END()
}

function M_DO_WHILE(REGISTER,CONDITION,VALUE)
{
	# IMPLEMENTATION OF DJNZ 
	if ((REGISTER ~ /^--[Rr][0-7]$/) && 
		(CONDITION ~ /^!=$|^<>$|NZ|NE|^>$/ ) && 
		IS_NULL(VALUE)) 
	{
		INST4("DJNZ",substr(REGISTER,3),"DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])
	} 
	else
	{
		M_COMPARE_NJMP(REGISTER,CONDITION,VALUE,"DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])
	}

	# Generate the label after the DO ... WHILE loop.
	DO_WHILE_END()
}

function M_DO_WHILE_DPTR_NE(VALUE_H,VALUE_L)
{
	# (DPH XOR VALUE_H) OR (DPL XOR VALUE_L)

	# Check DPL first since DPTR is usually incremented
	# (There is no decrement DPTR)
	printf("\tMOV\tA,DPL\n")
	printf("\tXRL\tA,%s\n",VALUE_L)

	INST2("JNZ","DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])

	printf("\tMOV\tA,DPH\n")
	printf("\tXRL\tA,%s\n",VALUE_H)

	INST2("JNZ","DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])
	
	# Generate the label after the DO ... WHILE loop.
	DO_WHILE_END()
}

function M_DO_BREAK()
{
	INST2("JMP","DO_BRK_",_sp_do_level,_sp_do_lbl[_sp_do_level])
}

function M_REDO()
{
	INST2("JMP","DO_",_sp_do_level,_sp_do_lbl[_sp_do_level])
}

#----------------------------------------------------------
# This function is the same for all WHILE loop forms. It will
# put a unique label at the beginning of the WHILE loop.
# The WHILE END will generate a jump to this label.
function WHILE_BEGIN()
{
	# This tells what is the current block if we encounter a M_BREAK statement.
	_sp_current_block = "WHILE"

	# Generate a unique label according to the nesting level and the 
	# number of successive WHILE loops in the current level.
	_sp_wh_level++
	LABEL("WHILE_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
}
#----------------------------------------------------------

# To add other M_WHILE macros, put a WHILE_BEGIN() followed by a test that
# jumps if condition is not met. The label to jump to is 
#
# 	"WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level]
#
function M_WHILE_BIT(BIT)
{
	# Generate the label at the beginning of the WHILE loop.
	WHILE_BEGIN()

	JMP_NOT_BIT_OR_CY(BIT,"WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
}

function M_WHILE_NOT_BIT(BIT)
{
	# Generate the label at the beginning of the WHILE loop.
	WHILE_BEGIN()

	JMP_BIT_OR_CY(BIT,"WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
}

function M_WHILE_EQU(REGISTER,MASK,VALUE)
{
	# Generate the label at the beginning of the WHILE loop.
	WHILE_BEGIN()

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	JNE_A2(VALUE,"WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
}

function M_WHILE_NOT_EQU(REGISTER,MASK,VALUE)
{
	# Generate the label at the beginning of the WHILE loop.
	WHILE_BEGIN()

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	JEQ_A2(VALUE,"WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
}

function M_WHILE(REGISTER,CONDITION,VALUE)
{
	# Generate the label at the beginning of the WHILE loop.
	WHILE_BEGIN()

	M_COMPARE_JMP(REGISTER,CONDITION,VALUE,"WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
}

function M_WH_BREAK()
{
	INST2("JMP","WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
}

function M_WH_END()
{
	# Put a jump to the beginning of the WHILE loop
	INST2("JMP","WHILE_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])

	# Put a label here for the BREAK statement.
	LABEL("WH_END_",_sp_wh_level,_sp_wh_lbl[_sp_wh_level])
	
	# Increment the successive label counter for the current level.
	_sp_wh_lbl[_sp_wh_level]++

	# Step down one level.
	_sp_wh_level--

	# Error condition
	# BLOCK BEGIN/END NOT MATCHING
	if (_sp_wh_level < 0) ERR_MSG("M_WH_END WITHOUT M_WHILE_XXX")
}

# This function is the same for all IF THEN ELSE forms.
# At the beginning of the IF block, we are assuming that there is
# an ELSE section. The Jump generated will jump to the ELSE section,
# but if it's not there, a label for the ELSE section will be put by
# the M_ENDIF macro.

#----------------------------------------------------------
function IF_BEGIN()
{
	_sp_current_block = "IF"

	PUSH(_sp_is_else)
	_sp_is_else = 0

	PUSH(_sp_if_nb)
	_sp_if_nb = 1

	_sp_if_level++
}
#----------------------------------------------------------

function M_IF_NOT_BIT(BIT)
{
	IF_BEGIN()

	# JUMP if CONDITION not met to the ELSE section
	JMP_BIT_OR_CY2(BIT,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_IF_BIT(BIT)
{
	IF_BEGIN()

	# JUMP NOT CONDITION
	JMP_NOT_BIT_OR_CY2(BIT,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_IF_NOT_EQU(REGISTER,MASK,VALUE)
{
	IF_BEGIN()

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	# JUMP if CONDITION
	JEQ_A3(VALUE,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_IF_EQU(REGISTER,MASK,VALUE)
{
	IF_BEGIN()

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	# JUMP if CONDITION
	JNE_A3(VALUE,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_IF_EQU_OR2(REGISTER,MASK,VALUE1,VALUE2)
{
	IF_BEGIN()

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	# GENERATE ASSEMBLER CODE
	JNE_OR2(VALUE1,VALUE2,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_IF_EQU_OR3(REGISTER,MASK,VALUE1,VALUE2,VALUE3)
{
	IF_BEGIN()

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	# GENERATE ASSEMBLER CODE
	JNE_OR3(VALUE1,VALUE2,VALUE3,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_IF_EQU_OR4(REGISTER,MASK,VALUE1,VALUE2,VALUE3,VALUE4)
{
	IF_BEGIN()

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)

	# GENERATE ASSEMBLER CODE
	JNE_OR4(VALUE1,VALUE2,VALUE3,VALUE4,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

# REGISTER AND VALUE CAN BE ALL AVAILABLE ADDRESSING MODE
# REGISTER, DIRECT, IMMEDIATE,INDIRECT
function M_IF_CMP(REGISTER,CONDITION,VALUE)
{
	IF_BEGIN()

	M_COMPARE_JMP(REGISTER,CONDITION,VALUE,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ELSE()
{
	# GENERATE ASSEMBLER CODE
	INST2("JMP","ENDIF_",_sp_if_level,_sp_if_lbl[_sp_if_level])

	LABEL2("ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb++)

	_sp_is_else = 1
}

function M_ELSE_IF_BIT(BIT)
{
	M_ELSE()

	# JUMP NOT CONDITION
	JMP_NOT_BIT_OR_CY2(BIT,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ELSE_IF_NOT_BIT(BIT)
{
	M_ELSE()

	# JUMP NOT CONDITION
	JMP_BIT_OR_CY2(BIT,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ELSE_IF_EQU(VALUE)
{
	M_ELSE()

	# JUMP NOT CONDITION
	JNE_A3(VALUE,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ELSE_IF_NOT_EQU(VALUE)
{
	M_ELSE()

	# JUMP NOT CONDITION
	JEQ_A3(VALUE,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ELSE_IF_EQU_OR2(VALUE1,VALUE2)
{
	M_ELSE()

	# GENERATE ASSEMBLER CODE
	JNE_OR2(VALUE1,VALUE2,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ELSE_IF_EQU_OR3(VALUE1,VALUE2,VALUE3)
{
	M_ELSE()

	# GENERATE ASSEMBLER CODE
	JNE_OR3(VALUE1,VALUE2,VALUE3,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ELSE_IF_EQU_OR4(VALUE1,VALUE2,VALUE3,VALUE4)
{
	M_ELSE()

	# GENERATE ASSEMBLER CODE
	JNE_OR4(VALUE1,VALUE2,VALUE3,VALUE4,"ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)
}

function M_ENDIF()
{
	# THE "IF" PLACED A JUMP TO A LABEL "ELSE_..." THAT WHOULD BE 
	# DEFINED IN THE "ELSE"	# MACRO. IF IT WAS NOT DEFINED, 
	# (NO "ELSE" SECTION) DEFINE IT HERE.
	if (_sp_is_else == 0)
		LABEL2("ELSE_",_sp_if_level,_sp_if_lbl[_sp_if_level],_sp_if_nb)

	LABEL("ENDIF_",_sp_if_level,_sp_if_lbl[_sp_if_level])

	# RESTORE FLAGS AND COUNTERS FOR PREVIOUS LEVEL
	_sp_if_nb = POP()
	_sp_is_else = POP()

	# BLOCK BEGIN/END NOT MATCHING
	_sp_if_lbl[_sp_if_level]++
	_sp_if_level--
	if (_sp_if_level < 0) ERR_MSG("M_ENDIF WITHOUT M_IF")
}

function M_IF_BREAK()
{
	# GENERATE ASSEMBLER CODE
	INST2("JMP","ENDIF_",_sp_if_level,_sp_if_lbl[_sp_if_level])
}

#----------------------------------------------------------
function CASE_LABEL()
{
	# GENERATE LABEL FOR CURRENT CASE
	LABEL2("SW_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level],_sp_case_level)

	#INCREMENT CASE COUNTER
	_sp_case_level++
}

function M_SWITCH(REGISTER,MASK)
{
	_sp_current_block = "SWITCH"
	_sp_sw_level++

	# IN CASE OF NESTED SWITCH
	PUSH(_sp_case_level)
	_sp_case_level = 1

	LABEL("SW_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level])

	# GENERATE ASSEMBLER CODE
	MOV_A(REGISTER)
	ANL_A(MASK)
}

function M_CASE(VALUE)
{
	CASE_LABEL()

	# GENERATE ASSEMBLER CODE
	JNE_A3(VALUE,"SW_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level],_sp_case_level)
}

function M_CASE_OR2(VALUE1,VALUE2)
{
	CASE_LABEL()

	# GENERATE ASSEMBLER CODE
	JNE_OR2(VALUE1,VALUE2,"SW_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level],_sp_case_level)
}

function M_CASE_OR3(VALUE1,VALUE2,VALUE3)
{
	CASE_LABEL()

	# GENERATE ASSEMBLER CODE
	JNE_OR3(VALUE1,VALUE2,VALUE3,"SW_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level],_sp_case_level)
}

function M_CASE_OR4(VALUE1,VALUE2,VALUE3,VALUE4)
{
	CASE_LABEL()

	# GENERATE ASSEMBLER CODE
	JNE_OR4(VALUE1,VALUE2,VALUE3,VALUE4,"SW_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level],_sp_case_level)
}

function M_SW_BREAK()
{
	INST2("JMP","SW_END_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level])
}

function M_SW_DEFAULT()
{
	LABEL2("SW_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level],_sp_case_level)
}

function M_SW_END()
{
	LABEL("SW_END_",_sp_sw_level,_sp_sw_lbl[_sp_sw_level])

	_sp_sw_lbl[_sp_sw_level]++
	_sp_case_level = POP()
	_sp_sw_level--

	# BLOCK BEGIN/END NOT MATCHING
	if (_sp_sw_level < 0) ERR_MSG("M_SW_END WITHOUT M_SWITCH")
}

# print the line from the actual token to the end of line
function prt2eol(n) {
	while (n<= NF)
	{
		printf ("%s ",$n)
		n++
	}
	printf("\n")
}

BEGIN {
# INITIALIZE THE STRUCTURED MACROS SYSTEM
# We generate ourselves the local labels to keep track of them for doing the jumps
	_sp_do_level = 0		# DO WHILE LEVEL COUNTER		(nested do while loops)
	_sp_wh_level = 0		# WHILE LEVEL COUNTER			(nested while loops)
	_sp_if_level = 0		# IF ELSE ENDIF LEVEL COUNTER	(nested if then else)
	_sp_sw_level = 0		# SWITCH LEVEL COUNTER			(nested switches)
	_sp_case_level = 0		# SWITCH CASE COUNTER

	_sp_stack_ptr = 0		# STACK POINTER
	_sp_line_number = 1		# LINE COUNTER

	# LETS AWK DO THE PARSING OF ARGUMENTS FOR US
#	FS = "[(,),;]"
}

# CODE EXECUTED FOR EACH LINE OF THE FILE
{
	FS = " "			# This doesn't take effect until the next line is parsed
	$0 = $0			# Force reparse of current line

	# Is it a comment?
	if ($1 ~ /^;/)
	{
		print $0
	}
	else
	{
		FS = "[(,)]"	# Use AWK to parse our macro parameters
		$0 = $0			# Force reparse of current line

	    if ($1 ~ /MOV_A/) {MOV_A($2)}
	    else if ($1 ~ /ANL_A/) {ANL_A($2)}
	    else if ($1 ~ /JNE_A/) {JNE_A($2,$3)}
	    else if ($1 ~ /JEQ_A/) {JEQ_A($2,$3)}

	#=============================================================
	# This is the parser section.
	# You can change macro names as you wish !
	#=============================================================
	    else if ($1 ~ /%M_DO_WHILE_BIT/)	{M_DO_WHILE_BIT($2)}
	    else if ($1 ~ /%M_DO_WHILE_NOT_BIT/) {M_DO_WHILE_NOT_BIT($2)}
	    else if ($1 ~ /%M_DO_WHILE_EQU/)	{M_DO_WHILE_EQU($2,$3,$4)}
	    else if ($1 ~ /%M_DO_WHILE_NOT_EQU/) {M_DO_WHILE_NOT_EQU($2,$3,$4)}
		else if ($1 ~ /%M_DO_WHILE_DPTR_NE/) {M_DO_WHILE_DPTR_NE($2,$3)}

		# Be sure to leave the following %M_DO_WHILE macro here so that it
		# will be parsed last and it will not hide the other macros beginning
		# with %M_DO_WHILE.
		# Same thing for %M_WHILE, %M_ELSE and %M_CASE.

  	    else if ($1 ~ /%M_DO_WHILE/) {M_DO_WHILE($2,$3,$4)}
	    else if ($1 ~ /%M_DO_BREAK/) {M_DO_BREAK()}
	    else if ($1 ~ /%M_REDO/) {M_REDO()}
	    else if ($1 ~ /%M_DO/) {M_DO()}
  
	    else if ($1 ~ /%M_WHILE_BIT/) {M_WHILE_BIT($2)}
	    else if ($1 ~ /%M_WHILE_NOT_BIT/) {M_WHILE_NOT_BIT($2)}
	    else if ($1 ~ /%M_WHILE_EQU/) {M_WHILE_EQU($2,$3,$4)}
	    else if ($1 ~ /%M_WHILE_NOT_EQU/) {M_WHILE_NOT_EQU($2,$3,$4)}
	    else if ($1 ~ /%M_WHILE/) {M_WHILE($2,$3,$4)}
	    else if ($1 ~ /%M_WH_BREAK/) {M_WH_BREAK()}
	    else if ($1 ~ /%M_WH_END/) {M_WH_END()}
  
	    else if ($1 ~ /%M_IF_BIT/) {M_IF_BIT($2)}
	    else if ($1 ~ /%M_IF_NOT_BIT/) {M_IF_NOT_BIT($2)}
	    else if ($1 ~ /%M_IF_EQU_OR2/) {M_IF_EQU_OR2($2,$3,$4,$5)}
	    else if ($1 ~ /%M_IF_EQU_OR3/) {M_IF_EQU_OR3($2,$3,$4,$5,$6)}
	    else if ($1 ~ /%M_IF_EQU_OR4/) {M_IF_EQU_OR4($2,$3,$4,$5,$6,$7)}
	    else if ($1 ~ /%M_IF_EQU/) {M_IF_EQU($2,$3,$4)}
	    else if ($1 ~ /%M_IF_NOT_EQU/) {M_IF_NOT_EQU($2,$3,$4)}
	    else if ($1 ~ /%M_IF_CMP/) {M_IF_CMP($2,$3,$4)}
	    else if ($1 ~ /%M_IF_NOT/) {M_IF_EQU($2,$3,$4)}
	    else if ($1 ~ /%M_ELSE_IF_BIT/) {M_ELSE_IF_BIT($2)}
	    else if ($1 ~ /%M_ELSE_IF_NOT_BIT/) {M_ELSE_IF_NOT_BIT($2)}
	    else if ($1 ~ /%M_ELSE_IF_EQU/) {M_ELSE_IF_EQU($2)}
	    else if ($1 ~ /%M_ELSE_IF_NOT_EQU/) {M_ELSE_IF_NOT_EQU($2)}
	    else if ($1 ~ /%M_ELSE_IF_EQU_OR2/) {M_ELSE_IF_EQU_OR2($2,$3)}
	    else if ($1 ~ /%M_ELSE_IF_EQU_OR3/) {M_ELSE_IF_EQU_OR3($2,$3,$4)}
	    else if ($1 ~ /%M_ELSE_IF_EQU_OR4/) {M_ELSE_IF_EQU_OR4($2,$3,$4,$5)}
	    else if ($1 ~ /%M_ELSE/) {M_ELSE()}
	    else if ($1 ~ /%M_ENDIF/) {M_ENDIF()}

	    else if ($1 ~ /%M_SWITCH/) {M_SWITCH($2,$3)}
	    else if ($1 ~ /%M_CASE_OR2/) {M_CASE_OR2($2,$3)}
	    else if ($1 ~ /%M_CASE_OR3/) {M_CASE_OR3($2,$3,$4)}
	    else if ($1 ~ /%M_CASE_OR4/) {M_CASE_OR4($2,$3,$4,$5)}
	    else if ($1 ~ /%M_CASE/) {M_CASE($2)}
	    else if ($1 ~ /%M_SW_BREAK/) {M_SW_BREAK()}

		# ALLOW SOME FREEDOM
	    else if ($1 ~ /%M_DEFAULT|%M_SW_DEFAULT/) {M_SW_DEFAULT()}

	    else if ($1 ~ /%M_SW_END/) {M_SW_END()}

		else if ($1 ~ /%M_BREAK/)
		{
			if (_sp_current_block ~ "SWITCH") {M_SW_BREAK()}
			else if (_sp_current_block ~ "WHILE") {M_WH_BREAK()}
			else if (_sp_current_block ~ "DO") {M_DO_BREAK()}
			else if (_sp_current_block ~ "IF") {M_IF_BREAK()}
  		}
	#=============================================================
	# End of the parser section.
	#=============================================================
		else
			print $0

	}
	_sp_line_number++
}

END {
}





