#=========================================================================
#    elemental.awk
#
#    Copyright (C) 1998  Jacques Pelletier
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#=========================================================================

# To include this file in a gawk program, use igawk instead of gawk
#
# MACROS NEEDED FOR INTELLIGENT ASSEMBLY
# Constants defined can have these formats:
# 	00h-0ffh
# 	h'00 - h'ff
#	x'00 - x'ff
#	0x'00 - 0x'ff
# 	0x00 - 0xff
#	$00 - $ff
 
# Returns TRUE if value is 0. This is used when doing a compare with a value: if it's 0, replace
# the instruction CJNE by JNZ or JZ depending of the macro.
function IS_NULL(value) { return (match("#NULL #00H #0 #0X00 #H'00 #H'0 #X'00 #X'0 #0X'00 #0X'0 #$00",toupper(value) ) != 0) }

# Return TRUE if there is an argument
function IS_ARG(arg) {return (length(arg))}

# Return TRUE if register specified was not the accumulator
function REG_NOT_ACC(reg) { return (match("A ACC",toupper(reg) ) == 0) }

# Generates a local label
function LOCAL_LABEL()
{
	printf("L_%04d:\n",_sp_label++)
}

# Generates a label with 2 counters
function LABEL(label,ctr1,ctr2)
{
	printf("%s%02d%03d:\n",label,ctr1,ctr2)
}

# Generates a label with 3 counters
function LABEL2(label,ctr1,ctr2,ctr3)
{
	printf("%s%02d%03d%02d:\n",label,ctr1,ctr2,ctr3)
}

# Generates an instruction (jump or similar) to the local label 
function INST_LOCAL(inst)
{
	printf("\t%s\tL_%04d\n",inst,_sp_label)
}

# Generates an instruction (jump or similar) to a label
function INST1(inst,label)
{
	printf("\t%s\t%s\n",inst,label)
}

# Generates an instruction (jump or similar) to a label with 2 counters
function INST2(inst,label,ctr1,ctr2)
{
	printf("\t%s\t%s%02d%03d\n",inst,label,ctr1,ctr2)
}

# Generates an instruction (jump or similar) to a label with 3 counters
function INST3(inst,label,ctr1,ctr2,ctr3)
{
	printf("\t%s\t%s%02d%03d%02d\n",inst,label,ctr1,ctr2,ctr3)
}

# Generates an instruction (jump with operand) to a label with 2 counters
function INST4(inst,op,label,ctr1,ctr2)
{
	printf("\t%s\t%s,%s%02d%03d\n",inst,op,label,ctr1,ctr2)
}

# Generates an instruction (jump with operand) to a label with 3 counters
function INST5(inst,op,label,ctr1,ctr2,ctr3)
{
	printf("\t%s\t%s,%s%02d%03d%02d\n",inst,op,label,ctr1,ctr2,ctr3)
}

function CJNE_A(value,address)
{
	printf("\tCJNE\tA,%s,%s\n",value,address)
}

function CJNE_LOCAL(value)
{
	printf("\tCJNE\tA,%s,L_%04d\n",value,_sp_label)
}

function CJNE_A2(value,address,ctr1,ctr2)
{
	printf("\tCJNE\tA,%s,%s%02d%03d\n",value,address,ctr1,ctr2)
}

function CJNE_A3(value,address,ctr1,ctr2,ctr3)
{
	printf("\tCJNE\tA,%s,%s%02d%03d%02d\n",value,address,ctr1,ctr2,ctr3)
}

# There is no CJEQ instruction in the 8051 (unless it's undocumented) 
# we must do a small spaghetti jump 
function CJEQ_A(value,address)
{
	CJNE_LOCAL(value)
	INST1("JMP",address)

	LOCAL_LABEL()
}

function CJEQ_A2(value,address,ctr1,ctr2)
{
	CJNE_LOCAL(value)
	INST2("JMP",address,ctr1,ctr2)

	LOCAL_LABEL()
}

function CJEQ_A3(value,address,ctr1,ctr2,ctr3)
{
	CJNE_LOCAL(value)
	INST3("JMP",address,ctr1,ctr2,ctr3)

	LOCAL_LABEL()
}

function SUB_A(value)
{
	printf ("\tCLR\tC\n")
	printf ("\tSUBB\tA,%s\n",value)
}

# THESE INSTRUCTIONS ARE GENERATED OR NOT DEPENDING ON THE PARAMETER OR VALUE SPECIFIED'
# THEY SHOULD BE USED INSIDE OF HIGHER LEVEL MACROS'
# DEFINING MOV_A(REGISTER)
function MOV_A(register)
{
# IF REGISTER IS A OR ACC, MOVE IS NOT NECESSARY

	if (REG_NOT_ACC(register))
		printf("\tMOV\tA,%s\n",register)
}

function SUB(register,value)
{
	MOV_A(register)
	SUB_A(value)
}

function MOV_DES(destination)
{
# Destination is in A, move is not necessary

	if (REG_NOT_ACC(destination))
		printf("\tMOV\t%s,A\n",destination)
}

function ANL_A(mask)
{
# IF THE MASK ARGUMENT IS SPECIFIED, DO A LOGICAL AND
	if (IS_ARG(mask))
		printf("\tANL\tA,%s\n",mask)
}

function JNE_A_LOCAL(value)
{
# IF THE VALUE IS "NULL" OR "00H" REPLACE CJNE WITH JNZ
	if (IS_NULL(value))
	{
		INST_LOCAL("JNZ")
	} else
	{
		CJNE_A_LOCAL(value)
	}
}

function JNE_A(value,address)
{
# IF THE VALUE IS "NULL" OR "00H" REPLACE CJNE WITH JNZ
	if (IS_NULL(value))
	{
		INST1("JNZ",address)
	} else
	{
		CJNE_A(value,address)
	}
}

function JEQ_A(value,address)
{
# IF THE VALUE IS "NULL" OR "00H" REPLACE CJNE WITH JZ

	if (IS_NULL(value))
	{
		INST1("JZ",address)
	} else
	{
		CJEQ_A(value,address)
	}
}

function JNE_A2(value,address,ctr1,ctr2)
{
# IF THE VALUE IS "NULL" OR "00H" REPLACE CJNE WITH JNZ
	if (IS_NULL(value))
	{
		INST2("JNZ",address,ctr1,ctr2)
	} else
	{
		CJNE_A2(value,address,ctr1,ctr2)
	}
}

function JEQ_A2(value,address,ctr1,ctr2)
{
# IF THE VALUE IS "NULL" OR "00H" REPLACE CJNE WITH JNZ

	if (IS_NULL(value))
	{
		INST2("JZ",address,ctr1,ctr2)
	} else
	{
		CJEQ_A2(value,address,ctr1,ctr2)
	}
}

function JNE_A3(value,address,ctr1,ctr2,ctr3)
{
# IF THE VALUE IS "NULL" OR "00H" REPLACE CJNE WITH JNZ
	if (IS_NULL(value))
	{
		INST3("JNZ",address,ctr1,ctr2,ctr3)
	} else
	{
		CJNE_A3(value,address,ctr1,ctr2,ctr3)
	}
}

function JEQ_A3(value,address,ctr1,ctr2,ctr3)
{
# IF THE VALUE IS "NULL" OR "00H" REPLACE CJNE WITH JNZ

	if (IS_NULL(value))
	{
		INST3("JZ",address,ctr1,ctr2,ctr3)
	} else
	{
		CJEQ_A3(value,address,ctr1,ctr2,ctr3)
	}
}

function IF_Z_OR_C_ELSE(address,ctr1,ctr2)
{
	INST_LOCAL("JZ")
	INST2("JNC",address,ctr1,ctr2)

	LOCAL_LABEL()
}

function IF_NC_AND_NZ_ELSE(address,ctr1,ctr2)
{
	INST2("JC",address,ctr1,ctr2)
	INST2("JZ",address,ctr1,ctr2)
}

function IF_Z_OR_C_ELSE2(address,ctr1,ctr2,ctr3)
{
	INST_LOCAL("JZ")
	INST3("JNC",address,ctr1,ctr2,ctr3)

	LOCAL_LABEL()
}

function IF_NC_AND_NZ_ELSE2(address,ctr1,ctr2,ctr3)
{
	INST3("JC",address,ctr1,ctr2,ctr3)
	INST3("JZ",address,ctr1,ctr2,ctr3)
}

function JMP_BIT_OR_CY(bit,address,ctr1,ctr2)
{
	if (bit ~ /^C$|^CY$|^c$|^cy$/)
		INST2("JC",address,ctr1,ctr2)
	else
		INST4("JB",bit,address,ctr1,ctr2)
}

function JMP_NOT_BIT_OR_CY(bit,address,ctr1,ctr2)
{
	if (bit ~ /^C$|^CY$|^c$|^cy$/)
		INST2("JNC",address,ctr1,ctr2)
	else
		INST4("JNB",bit,address,ctr1,ctr2)
}

function JMP_BIT_OR_CY2(bit,address,ctr1,ctr2,ctr3)
{
	if (bit ~ /^C$|^CY$|^c$|^cy$/)
		INST3("JC",address,ctr1,ctr2,ctr3)
	else
		INST5("JB",bit,address,ctr1,ctr2,ctr3)
}

function JMP_NOT_BIT_OR_CY2(bit,address,ctr1,ctr2,ctr3)
{
	if (bit ~ /^C$|^CY$|^c$|^cy$/)
		INST3("JNC",address,ctr1,ctr2,ctr3)
	else
		INST5("JNB",bit,address,ctr1,ctr2,ctr3)
}

# Jump to the label if A is not equal to one of 2 values
function JNE_OR2(value1,value2,label,ctr1,ctr2,ctr3,   temp)
{
	temp = "OR2_" _sp_label

	# GENERATE ASSEMBLER CODE
	JEQ_A(value1,temp)
	JNE_A3(value2,label,ctr1,ctr2,ctr3)

	# GENERATE LOCAL LABEL
	printf("%s:\n",temp)

	_sp_label++
}

# Jump to the label if A is not equal to one of 3 values
function JNE_OR3(value1,value2,value3,label,ctr1,ctr2,ctr3,   temp)
{
	temp = "OR3_" _sp_label

	# GENERATE ASSEMBLER CODE
	JEQ_A(value1,temp)
	JEQ_A(value2,temp)
	JNE_A3(value3,label,ctr1,ctr2,ctr3)

	# GENERATE LOCAL LABEL
	printf("%s:\n",temp)

	_sp_label++
}

# Jump to the label if A is not equal to one of 4 values
function JNE_OR4(value1,value2,value3,value4,label,ctr1,ctr2,ctr3,   temp)
{
	temp = "OR4_" _sp_label

	# GENERATE ASSEMBLER CODE
	JEQ_A(value1,temp)
	JEQ_A(value2,temp)
	JEQ_A(value3,temp)
	JNE_A3(value4,label,ctr1,ctr2,ctr3)

	# GENERATE LOCAL LABEL
	printf("%s:\n",temp)

	_sp_label++
}

function ERR_MSG(string)
{
	printf(";MACRO ERROR: %s\n",string)
	printf(";MACRO ERROR AT LINE %05d IN FILE %s: %s\n",_sp_line_number,FILENAME,string) > "/dev/stderr"
}



