################################################################################
#
# Copyright (C) 1997, 2000  James A. Cureington
#                           tonyc@acm.org
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#
# $Id: menu_bar.tcl,v 1.1 2001/03/04 00:08:31 tonycu Exp $
#
# $Log: menu_bar.tcl,v $
# Revision 1.1  2001/03/04 00:08:31  tonycu
# initial check-in at sourceforge...
#
# Revision 1.12  2000/05/24 03:19:08  tony
# Updated all copyright dates.
#
# Revision 1.11  1999/09/13 01:21:06  tony
#
# Updated copyright notice date
#
# Revision 1.10  1997/03/26  07:04:07  tony
#      Fixed defects uncovered from test plan
#
# Revision 1.9  1997/03/22  17:29:27  tony
#      1) changed FSM_BIN_DIR to FSKC_BIN_DIR
#      2) upcased state and event name in find window, so user can
#         enter the names in upper or lower case
#      3) move find from file to edit menu
#
# Revision 1.8  1997/03/08  03:47:49  tony
#      1) remove new-lines (using trimright) in event, state, and action-block
#         edit windows
#      2) the nextState value was not being incremented in the action-blocks
#         when a state was inserted before it if the nextState value was 0
#      3) the copy buffer was not being update when states were deleted
#      4) if the function called in the action-block is vAbortFsk, don't
#         enclose it in a return(), i.e. return(vAbortFsk)
#      5) remove leading and trailing spaces from event and state names
#      6) change no_op function to take a parameter and return it; also,
#         rename it uiNoOp
#      7) change vPrintFskHistory to take a parameter and return it; also,
#         rename it uiPrintFskHistory
#      8) add check for duplicate event and state names
#      9) rename "recursive" variables to "iterative"
#
# Revision 1.7  1997/03/02  19:49:14  tony
#      - Added code to allow user defined error handler
#      - Added call to infinite state check
#
# Revision 1.6  1997/03/01  23:23:54  tony
#      Made the following changes:
#           - Add version to FSK file for validation when reading in
#           - Check for " in parameter list
#           - Add CVS Id to source and header file
#           - Add copyright notice to header files
#           - Add EDIT menu item and associated entries
#           - Make mouse button 3 cause the commnad pop-up to be displayed
#             instead of mouse button 2
#           - Add box to insert header files into
#           - Changed pchz to pch
#           - Allow macros to be used in parameter list
#           - Add function and associated calls to check for infinite state
#             condition; this is a warning, not an error condition
#           - Add over-ride abort function capability to GUI
#           - Changed "modified save pop-up" window to be displayed before
#             any window is displayed to read and RSK, or create a new one
#
# Revision 1.5  1997/02/10  04:00:40  tony
#      1) added code to generate C source code
#      2) added trace history
#
# Revision 1.4  1997/02/08  22:50:12  tony
#      1) added info window
#      2) changed text to fixed width
#      3) validate fsk before running
#      4) added fsk filename to top of main window
#      5) all path names were not being updated on a "save as"
#      6) added code to print various reports...
#
# Revision 1.3  1997/01/26  04:52:24  tony
# Made changes to:
#      - clear, copy, paste, and undo action blocks
#      - comment out Edit menu option
#      - add ".fsk" to files as needed
#      - insert user defined default function next state when a state or
#        event is inserted
#      - check for invalid number of states and events in newFsk
#      - add "save if modified" to quit menu item
#      - add "processing" and "not selectable" cursors
#      - add procs to position state and event rows, positionState &
#        positionEvent
#      - add find state, event, and function commands
#      - add generate "C" code command
#      - add "print" Fsk command
#      - add run Fsk command
#
# Revision 1.2  1997/01/11  22:29:43  tony
#   Added code to edit state, event, and action blocks.
#
# Revision 1.1  1997/01/06  06:32:25  tony
# Initial check-in of PRODUCTION code.
#
# Revision 1.2  1996/12/13  05:23:37  tony
# Made minor comment and format changes
#
# Revision 1.1  1996/04/28  20:50:26  tony
# this file contains code to create the menu bar containing all the available
# sub-menus;  it can be run as a unit test.
#
#
################################################################################

set getFskFileToOpenWindowActive 0
set getFskFileToSaveAsWinActive 0


################################################################################
# NAME    : getFskFileToOpen
#
# PURPOSE : This procedure get the name of the FSK to open, including the path
################################################################################
proc getFskFileToOpen {dir file} {

   # completeFskFileNameToOpen  is the file to open, including the path
   global completeFskFileNameToOpen
   global getFskFileToOpenWindowActive
   global fsm_window
   global fsk
   global notSelectableCursor
   global processingCursor
   global win


   set win .fsk_file_open_window

   # if the window is not active (visable), make it so
   if {$getFskFileToOpenWindowActive == 0} {
      toplevel $win
      wm title $win "Open FSK"
      positionWindow $win 100 150
      raise $win
      focus $win
      grab set $win

      $fsm_window configure -cursor $processingCursor

      set listWidth 30

      if {$dir == "."} {
         set dir [exec pwd]
      } 
      set fsk(fileToOpenPath) $dir

      if {$file == "."} {
         set file ""
      }

      #######
      # create directory entry text field
      #######
      frame $win.top
      pack $win.top -side top

      frame $win.top.dir_entry
      pack $win.top.dir_entry -side top

      label $win.top.dir_entry.label -text "Path :" 
      pack $win.top.dir_entry.label -side left -pady 4
   
      set temp [expr $listWidth*2]
      entry $win.top.dir_entry.entry -width $temp -relief flat  \
            -state disabled -justify left                       \
            -textvariable fsk(fileToOpenPath)
      pack $win.top.dir_entry.entry -fill x -expand y -pady 4

      #######
      # create list of directories
      #######
      frame $win.mid
      pack $win.mid -side top

      frame $win.mid.dir_names
      pack $win.mid.dir_names -side left -pady 4

      label $win.mid.dir_names.label -text "Directories" 
      pack $win.mid.dir_names.label -side top -pady 2

      scrollbar $win.mid.dir_names.scroll -command \
                "$win.mid.dir_names.list yview"
      pack $win.mid.dir_names.scroll -side right -fill y
      listbox $win.mid.dir_names.list -yscroll \
              "$win.mid.dir_names.scroll set" -relief sunken \
              -width $listWidth -height 10 -setgrid yes
      pack $win.mid.dir_names.list -side left -fill both -expand yes


      #######
      # create list of files
      #######
      frame $win.mid.file_names
      pack $win.mid.file_names -side right -pady 4

      label $win.mid.file_names.label -text "FSK Files" 
      pack $win.mid.file_names.label -side top -pady 2

      scrollbar $win.mid.file_names.scroll -command \
                "$win.mid.file_names.list yview"
      pack $win.mid.file_names.scroll -side right -fill y
      listbox $win.mid.file_names.list -yscroll \
              "$win.mid.file_names.scroll set" -relief sunken \
              -width $listWidth -height 10 -setgrid yes
      pack $win.mid.file_names.list -side left -fill both -expand yes

      #######
      # create button frames
      #######
      frame $win.bot
      pack $win.bot -side bottom

      frame $win.bot.buttons
      pack $win.bot.buttons -side left

      button $win.bot.buttons.cancel -text "CANCEL"        \
             -command {set getFskFileToOpenWindowActive 0; \
                       destroy .fsk_file_open_window}

      pack $win.bot.buttons.cancel -side left

      #######
      # setup bindings 
      #######
      bind $win.mid.dir_names.list <Double-Button-1>            \
           {foreach i [selection get]                           \
            {getFskFileToOpen $fsk(fileToOpenPath)/$i ""}}

      bind $win.mid.file_names.list <Double-Button-1>           \
           {foreach i [selection get]                           \
            {getFskFileToOpen $fsk(fileToOpenPath) $i}}

      set getFskFileToOpenWindowActive 1
   }

   $fsm_window configure -cursor $notSelectableCursor

   if [file isdirectory $dir/$file] {

      if {[string match "*/.." $dir]} {
         # backing up one dir
         set index [string last "/" $fsk(fileToOpenPath)]
         set index [expr $index-1]
         set fsk(fileToOpenPath) [string range $fsk(fileToOpenPath) 0 $index]
      } else {
         # going into a dir
         set fsk(fileToOpenPath) $dir
      }

      # delete the current list
      $win.mid.file_names.list delete 0 end
      $win.mid.dir_names.list delete 0 end

      # fill the directory listbox
      $win.mid.dir_names.list insert end ".."
      foreach name [lsort [glob -nocomplain -- $fsk(fileToOpenPath)/*/]] {
         set name [string trimright $name "/"]
         set index [string last "/" $name]
         set index [expr $index+1]
         set name [string range $name $index [string length $name]]
         $win.mid.dir_names.list insert end $name
      }

      # fill the file listbox
      foreach name [lsort [glob -nocomplain -- $fsk(fileToOpenPath)/*.fsk]] {
         set index [string last "/" $name]
         set index [expr $index+1]
         set name [string range $name $index [string length $name]]
         $win.mid.file_names.list insert end $name
      }
   } else {
      # we have an fsk filename
      if [file isfile $fsk(fileToOpenPath)/$file] {
         # file has been selected
         set completeFskFileNameToOpen $fsk(fileToOpenPath)/$file
         set getFskFileToOpenWindowActive 0
         destroy .fsk_file_open_window
         # TODO, should probably move this outside of this proc later
         ReadFsk $completeFskFileNameToOpen IGNORE_MODIFIED_FLAG
         processEventArray
         return
      } else {
         puts stdout "\"$file\" isn't a directory or regular file"
      }
   }


   #######
   # if current FSK has been modified, prompt user for save
   #######
   if {$fsk(modified) != "0"} then {
      
      global tempWindow

      set tempWindow .openWithoutSavePopupWindow

      toplevel $tempWindow
      wm title $tempWindow "FSK MODIFIED"
      positionWindow $tempWindow 100 150
      raise $tempWindow 
      focus $tempWindow 
      grab set $tempWindow 
      $win configure -cursor $notSelectableCursor


      ######
      # create prompt
      ######
      frame $tempWindow.msg
      pack $tempWindow.msg -side top 

      label $tempWindow.msg.label  \
            -text "\n  \"$fsk(completeFileName)\" modified but not saved.  " 
      pack $tempWindow.msg.label -side top

      label $tempWindow.msg.label2  \
            -text "  Save modified FSK?  " 
      pack $tempWindow.msg.label2 -side bottom

      ######
      # create buttons
      ######
      frame $tempWindow.buttons
      pack $tempWindow.buttons -side bottom

      button $tempWindow.buttons.yes -text "YES"                           \
             -command {SaveFsk $fsk(completeFileName) OVERWRITE_OK;        \
                       destroy $tempWindow;                                \
                       processEventArray;                                  \
                       $fsm_window configure -cursor $notSelectableCursor; \
                       $win configure -cursor top_left_arrow;              \
                       grab set $win                                       \
                      }
      pack $tempWindow.buttons.yes -side left

      button $tempWindow.buttons.no -text "NO"                             \
             -command {destroy $tempWindow;                                \
                       $fsm_window configure -cursor $notSelectableCursor; \
                       $win configure -cursor top_left_arrow;              \
                       grab set $win                                       \
                      }

      pack $tempWindow.buttons.no -side left

      button $tempWindow.buttons.cancel -text "CANCEL"                  \
             -command {destroy $tempWindow;                             \
                       destroy $win;                                    \
                       set getFskFileToOpenWindowActive 0;              \
                      }
      pack $tempWindow.buttons.cancel -side left 
   }

}



################################################################################
# NAME    : getFskFileToSaveAs
#
# PURPOSE : This procedure gets the filename to save the FSK to
################################################################################
proc getFskFileToSaveAs {dir} {

   # completeFskFileNameToSaveAs, the filename to save the FSK to, including 
   #                              the path

   global completeFskFileNameToSaveAs
   global getFskFileToSaveAsWinActive
   global fsm_window
   global fsk
   global notSelectableCursor
   global processingCursor


   set win .fsk_file_save_window

   # if the window is not active (visable), make it so
   if {$getFskFileToSaveAsWinActive == 0} {
      toplevel $win
      wm title $win "SAVE AS"
      positionWindow $win 100 150
      raise $win
      focus $win
      grab set $win

      $fsm_window configure -cursor $processingCursor

      set listWidth 30

      if {$dir == "."} {
         set dir [exec pwd]
      } 
      set fsk(fileToSaveAsPath) $dir


      #######
      # create directory entry text field
      #######
      frame $win.top
      pack $win.top -side top

      frame $win.top.file_entry
      pack $win.top.file_entry -side top -anchor w

      frame $win.top.dir_entry
      pack $win.top.dir_entry -side top

      label $win.top.file_entry.label -text "File :" 
      pack $win.top.file_entry.label -side left -pady 4


      set temp [expr $listWidth]
      entry $win.top.file_entry.entry -width $temp -relief sunken \
            -justify left -textvariable fskFileToSaveAs 
      pack $win.top.file_entry.entry -side left -pady 4
      $win.top.file_entry.entry delete 0 end

      label $win.top.dir_entry.label -text "Path :" 
      pack $win.top.dir_entry.label -side left -pady 4

      set temp [expr $listWidth*2]
      entry $win.top.dir_entry.entry -width $temp -relief flat \
            -state disabled -justify left                      \
            -textvariable fsk(fileToSaveAsPath) 
      pack $win.top.dir_entry.entry -side left -pady 4


      #######
      # create list of directories
      #######
      frame $win.mid
      pack $win.mid -side top

      frame $win.mid.dir_names
      pack $win.mid.dir_names -side left

      label $win.mid.dir_names.label -text "Directories" 
      pack $win.mid.dir_names.label -side top -pady 2

      scrollbar $win.mid.dir_names.scroll -command \
                "$win.mid.dir_names.list yview"
      pack $win.mid.dir_names.scroll -side right -fill y -pady 4
      listbox $win.mid.dir_names.list -yscroll               \
              "$win.mid.dir_names.scroll set" -relief sunken \
              -width $listWidth -height 10 -setgrid yes 
      pack $win.mid.dir_names.list -side left -fill both -expand yes


      #######
      # create button frame
      #######
      frame $win.bot
      pack $win.bot -side bottom

      frame $win.bot.buttons
      pack $win.bot.buttons

      button $win.bot.buttons.ok -text "OK"                                   \
             -command {                                                       \
                          set iLen [string length $fskFileToSaveAs];          \
                          set index [expr $iLen-4];                           \
                          set tmpStr [string range $fskFileToSaveAs           \
                                      $index end];                            \
                          if {$tmpStr != ".fsk"} then {                       \
                             set fskFileToSaveAs ${fskFileToSaveAs}.fsk;      \
                          };                                                  \
                          set completeFskFileNameToSaveAs                     \
                              $fsk(fileToSaveAsPath)/$fskFileToSaveAs;        \
                          set getFskFileToSaveAsWinActive 0;                  \
                          destroy .fsk_file_save_window;                      \
                          SaveFsk $completeFskFileNameToSaveAs                \
                                  OVERWRITE_NOT_OK;                           \
                          processEventArray;                                  \
                       }

      pack $win.bot.buttons.ok -side left

      button $win.bot.buttons.cancel -text "CANCEL"           \
             -command {set getFskFileToSaveAsWinActive 0;     \
                       destroy .fsk_file_save_window}
      pack $win.bot.buttons.cancel -side right

      #######
      # setup bindings 
      #######
      bind $win.mid.dir_names.list <Double-Button-1>      \
           {foreach i [selection get]                     \
            {getFskFileToSaveAs $fsk(fileToSaveAsPath)/$i}}

      bind $win.top.file_entry.entry  <Return> {
         # file has been provided
         set iLen [string length $fskFileToSaveAs];
         set index [expr $iLen-4];
         set tmpStr [string range $fskFileToSaveAs $index end];                            \
         if {$tmpStr != ".fsk"} then {
            set fskFileToSaveAs ${fskFileToSaveAs}.fsk;
         };
         set completeFskFileNameToSaveAs                     \
             $fsk(fileToSaveAsPath)/$fskFileToSaveAs;
         set getFskFileToSaveAsWinActive 0;
         destroy .fsk_file_save_window;
         SaveFsk $completeFskFileNameToSaveAs OVERWRITE_NOT_OK;
         processEventArray;
      }


      set getFskFileToSaveAsWinActive 1
   }



   if {[string match "*/.." $dir]} {
      # backing up one dir
      set index [string last "/" $fsk(fileToSaveAsPath)]
      set index [expr $index-1]
      set fsk(fileToSaveAsPath) [string range $fsk(fileToSaveAsPath) 0 $index]
   } else {
      # going into a dir
      set fsk(fileToSaveAsPath) $dir
   }

   # delete the current list
   $win.mid.dir_names.list delete 0 end

   # fill the directory listbox
   $win.mid.dir_names.list insert end ".."
   foreach name [lsort [glob -nocomplain -- $fsk(fileToSaveAsPath)/*/]] {
      set name [string trimright $name "/"]
      set index [string last "/" $name]
      set index [expr $index+1]
      set name [string range $name $index [string length $name]]
      $win.mid.dir_names.list insert end $name
   }

   focus $win.top.file_entry.entry 
   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : getNewFskData
#
# PURPOSE : This procedure gets the filename to save the FSK to
################################################################################
proc getNewFskData {} {

   global fsk
   global fsm_window
   global dText
   global numLines
   global win
   global newFskStates 
   global newFskEvents 
   global newFskReturnType 
   global notSelectableCursor
   global processingCursor


   set win .getNewFskDataWindow

   toplevel $win
   wm title $win "NEW FSK"
   positionWindow $win 100 150
   raise $win
   focus $win
   grab set $win

   $fsm_window configure -cursor $processingCursor

   #######
   # create number entry fields
   #######
   frame $win.num
   pack $win.num -side top -anchor w

   frame $win.num.states
   pack $win.num.states -side top -anchor w

   frame $win.num.events
   pack $win.num.events -side top -anchor w

   frame $win.num.return_type
   pack $win.num.return_type -side bottom -anchor w

   label $win.num.states.label -text      "Number of States :" \
         -font $fsk(Font2)
   pack $win.num.states.label -side left -pady 4

   label $win.num.events.label -text      "Number of Events :" \
         -font $fsk(Font2)
   pack $win.num.events.label -side left -pady 4

   label $win.num.return_type.label -text "Return Type      :" \
         -font $fsk(Font2)
   pack $win.num.return_type.label -side left -pady 4


   entry $win.num.states.entry -width 5 -relief sunken \
         -justify left -textvariable newFskStates
   pack $win.num.states.entry -side left -pady 4
   $win.num.states.entry delete 0 end

   entry $win.num.events.entry -width 5 -relief sunken \
         -justify left -textvariable newFskEvents 
   pack $win.num.events.entry -side left -pady 4
   $win.num.events.entry delete 0 end

   entry $win.num.return_type.entry -width 20 -relief sunken \
         -justify left -textvariable newFskReturnType 
   pack $win.num.return_type.entry -side left -pady 4
   $win.num.return_type.entry delete 0 end


   #######
   # create description entry field
   #######
   set numLines 8
   set numChars 48

   frame $win.description
   pack $win.description -side top -pady 4

   label $win.description.label -text "\nDescription" \
         -font $fsk(Font2)
   pack $win.description.label -side top 

   text $win.description.text -relief sunken -width $numChars -height    \
                              $numLines -wrap word
   pack $win.description.text -side left
   set dText $win.description.text


   #######
   # create buttons
   #######
   frame $win.buttons
   pack $win.buttons -side top

   button $win.buttons.ok -text "OK" \
          -command {                                                           \
                       set newFskEvents [string trim  $newFskEvents " "];      \
                       set newFskStates [string trim  $newFskStates " "];      \
                                                                               \
                       set tmp1 [string first - $newFskEvents];                \
                       set tmp2 [string first - $newFskStates];                \
                       set tmp3 [string toupper $newFskEvents];                \
                       set tmp4 [string toupper $newFskStates];                \
                                                                               \
                       if {$tmp1 != -1 || $tmp2 != -1} then {                  \
                          displayMessage "States and Events must be positive"; \
                          grab set $win;                                       \
                          $fsm_window configure -cursor $notSelectableCursor;  \
                       } elseif {$tmp3 != $newFskEvents ||                     \
                                 $tmp4 != $newFskStates ||                     \
                                 $newFskEvents == ""    ||                     \
                                 $newFskStates == "" } then {                  \
                          displayMessage "Non-numeric state or event given";   \
                          grab set $win;                                       \
                          $fsm_window configure -cursor $notSelectableCursor;  \
                       } elseif {$newFskEvents == 0 ||                         \
                                 $newFskStates == 0} then {                    \
                          displayMessage "State and event must be > 0";        \
                          grab set $win;                                       \
                          $fsm_window configure -cursor $notSelectableCursor;  \
                       } else {                                                \
                          set fsk(description) [$dText get 1.0 $numLines.end]; \
                          set fsk(description)                                 \
                              [string trimright $fsk(description) '\n'];       \
                          destroy .getNewFskDataWindow;                        \
                          SetupNewFsk $newFskEvents $newFskStates              \
                                      $newFskReturnType $fsk(description);     \
                          processEventArray;                                   \
                       }                                                       \
                    }
   pack $win.buttons.ok -side left

   button $win.buttons.cancel -text "CANCEL" \
          -command {destroy .getNewFskDataWindow}
   pack $win.buttons.cancel -side left

   focus $win.num.states.entry


   #######
   # if current FSK has been modified, prompt user for save
   #######
   if {$fsk(modified) != "0"} then {
      
      global tempWindow

      set tempWindow .newWithoutSavePopupWindow

      toplevel $tempWindow
      wm title $tempWindow "FSK MODIFIED"
      positionWindow $tempWindow 100 150
      raise $tempWindow 
      focus $tempWindow 
      grab set $tempWindow 
      $win configure -cursor $notSelectableCursor


      ######
      # create prompt
      ######
      frame $tempWindow.msg
      pack $tempWindow.msg -side top 

      label $tempWindow.msg.label  \
            -text "\n  \"$fsk(completeFileName)\" modified but not saved.  " 
      pack $tempWindow.msg.label -side top

      label $tempWindow.msg.label2  \
            -text "  Save modified FSK?  " 
      pack $tempWindow.msg.label2 -side bottom

      ######
      # create buttons
      ######
      frame $tempWindow.buttons
      pack $tempWindow.buttons -side bottom

      button $tempWindow.buttons.yes -text "YES"                           \
             -command {SaveFsk $fsk(completeFileName) OVERWRITE_OK;        \
                       destroy $tempWindow;                                \
                       processEventArray;                                  \
                       $fsm_window configure -cursor $notSelectableCursor; \
                       $win configure -cursor top_left_arrow;              \
                       grab set $win                                       \
                      }
      pack $tempWindow.buttons.yes -side left

      button $tempWindow.buttons.no -text "NO"                             \
             -command {destroy $tempWindow;                                \
                       $fsm_window configure -cursor $notSelectableCursor; \
                       $win configure -cursor top_left_arrow;              \
                       grab set $win                                       \
                      }
      pack $tempWindow.buttons.no -side left

      button $tempWindow.buttons.cancel -text "CANCEL"                  \
             -command {destroy $tempWindow;                             \
                       destroy $win;                                    \
                      }
      pack $tempWindow.buttons.cancel -side left 
   }

   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : getDefaultInfo
#
# PURPOSE : This procedure gets the default information (default function
#           and nextState) from the user.
################################################################################
proc getDefaultInfo {} {

   global fsk
   global fsm_window
   global win
   global defaultFunctionName
   global defaultNextState
   global actionBlock
   global fText
   global numFuncLines
   global numFuncChars
   global notSelectableCursor
   global processingCursor

   set win .getDefaultDataWindow

   toplevel $win
   wm title $win "FSK DEFAULTS"
   positionWindow $win 100 150
   raise $win
   focus $win
   grab set $win

   $fsm_window configure -cursor $processingCursor

   #######
   # default next state radio button, first, last, or none
   #######
   frame $win.button
   pack $win.button -side top -anchor w

   label $win.button.label -text \
                        "Default Transition State" 
   pack $win.button.label -side top 

   radiobutton $win.button.def_next_state_first -text "First State" \
               -variable defaultNextState -value FIRST_STATE 
   radiobutton $win.button.def_next_state_last  -text "Last State" \
               -variable defaultNextState -value LAST_STATE
   radiobutton $win.button.def_next_state_none  -text "Unassinged" \
               -variable defaultNextState -value UNASSIGNED_STATE

   set defaultNextState $fsk(defaultNextState)

    pack $win.button.def_next_state_first $win.button.def_next_state_last \
         $win.button.def_next_state_none -side top  -anchor s 

   #######
   # function name entry
   #######
   frame $win.spacer1
   pack $win.spacer1 -side top -anchor w
   label $win.spacer1.label -text  " " 
   pack $win.spacer1.label -side left -anchor s 

   frame $win.def_function_name
   pack $win.def_function_name -side top

   label $win.def_function_name.label -text \
                        "Default Function Name & Parameters" 
   pack $win.def_function_name.label -side top 

   text $win.def_function_name.text -relief sunken -width $numFuncChars \
        -height $numFuncLines -wrap word

   set defaultFunctionName $fsk(defaultFunction)
   set defaultFunctionName [string trimright $defaultFunctionName '\n']

   $win.def_function_name.text delete 1.0 end
   $win.def_function_name.text insert 1.0 $defaultFunctionName

   pack $win.def_function_name.text -side left
   set fText $win.def_function_name.text

   #######
   # create buttons
   #######
   frame $win.buttons
   pack $win.buttons -side top

   button $win.buttons.ok -text "OK" \
          -command {                                                          \
                       set fsk(defaultNextState) $defaultNextState;           \
                       set fsk(defaultFunction)                               \
                               [$fText get 1.0 $numFuncLines.end];            \
                       set fsk(defaultFunction)                               \
                           [string trimright $fsk(defaultFunction) '\n'];     \
                       SetFskDefaults;                                        \
                       destroy .getDefaultDataWindow;                         \
                       processEventArray;                                     \
                    }
   pack $win.buttons.ok -side left

   button $win.buttons.cancel -text "CANCEL" \
          -command {destroy .getDefaultDataWindow}
   pack $win.buttons.cancel -side left

   focus $win.def_function_name.text

   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : getSourceSettings
#
# PURPOSE : This procedure gets the source file info (return type and
#           header files) from the user.
################################################################################
proc getSourceSettings {} {

   global fsk
   global fsm_window
   global win
   global headerFiles
   global returnType
   global actionBlock
   global userDefinedErrorHandler
   global hText
   global numHeaderLines
   global numHeaderChars
   global notSelectableCursor
   global processingCursor


   set win .getSourceSettingsWindow

   toplevel $win
   wm title $win "FSK SOURCE SETTINGS"
   positionWindow $win 100 150
   raise $win
   focus $win
   grab set $win

   $fsm_window configure -cursor $processingCursor


   #######
   # fsk return type
   #######
   frame $win.spacer1
   pack $win.spacer1 -side top -anchor w
   label $win.spacer1.label -text  " " 
   pack $win.spacer1.label -side left -anchor s 

   frame $win.return_type
   pack $win.return_type -side top -anchor w

   label $win.return_type.label -text "FSK Return Type" 
   pack $win.return_type.label -side left 

   entry $win.return_type.entry -width 15 -relief sunken  \
         -justify left -textvariable returnType

   set returnType $fsk(returnType)

   pack $win.return_type.entry -side left


   #######
   # user defined error handler checkbutton
   #######
   frame $win.spacer2
   pack $win.spacer2 -side top -anchor w
   label $win.spacer2.label -text  " " 
   pack $win.spacer2.label -side left -anchor s 

   frame $win.button
   pack $win.button -side top -anchor w

   set userDefinedErrorHandler $fsk(userDefinedErrorHandler)

   checkbutton $win.button.handler -text "User Defined Error Handler (0,0)" \
               -variable userDefinedErrorHandler
   pack $win.button.handler -side left  -anchor s -pady 4


   #######
   # header file entry
   #######
   frame $win.spacer3
   pack $win.spacer3 -side top -anchor w
   label $win.spacer3.label -text  " " 
   pack $win.spacer3.label -side left -anchor s 

   frame $win.header_files
   pack $win.header_files -side top

   label $win.header_files.label -text \
                        "FSK Header Files" 
   pack $win.header_files.label -side top 

   text $win.header_files.text -relief sunken -width $numHeaderChars \
        -height $numHeaderLines -wrap word

   set headerFiles $fsk(headerFiles)
   set headerFiles [string trimright $headerFiles '\n']

   $win.header_files.text delete 1.0 end
   $win.header_files.text insert 1.0 $headerFiles

   pack $win.header_files.text -side left
   set hText $win.header_files.text

   frame $win.spacer4
   pack $win.spacer4 -side top -anchor w
   label $win.spacer4.label -text  " " 
   pack $win.spacer4.label -side left -anchor s 


   #######
   # create buttons
   #######
   frame $win.buttons
   pack $win.buttons -side top

   button $win.buttons.ok -text "OK" \
          -command {                                                          \
                       set fsk(headerFiles)                                   \
                               [$hText get 1.0 $numHeaderLines.end];          \
                       set fsk(headerFiles)                                   \
                           [string trimright $fsk(headerFiles) '\n'];         \
                       set fsk(returnType) $returnType;                       \
                       set fsk(returnType)                                    \
                           [string trimright $fsk(returnType) '\n'];          \
                       set fsk(userDefinedErrorHandler)                       \
                               $userDefinedErrorHandler;                      \
                       SetFskSourceSettings;                                  \
                       destroy .getSourceSettingsWindow;                      \
                       processEventArray;                                     \
                    }
   pack $win.buttons.ok -side left

   button $win.buttons.cancel -text "CANCEL" \
          -command {destroy .getSourceSettingsWindow}
   pack $win.buttons.cancel -side left

   focus $win.header_files.text

   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : getFskDescription
#
# PURPOSE : This procedure gets the FSK description from the user
################################################################################
proc getFskDescription {} {

   global fsk
   global fsm_window
   global dText
   global numLines
   global win
   global notSelectableCursor
   global processingCursor


   set win .getFskDescriptionWindow

   toplevel $win
   wm title $win "FSK DESCRIPTION"
   positionWindow $win 100 150
   raise $win
   focus $win
   grab set $win

   $fsm_window configure -cursor $processingCursor

   #######
   # create description entry field
   #######
   set numLines 8
   set numChars 48

   frame $win.description
   pack $win.description -side top -pady 4

   label $win.description.label -text "\nDescription" \
         -font $fsk(Font2)
   pack $win.description.label -side top 

   text $win.description.text -relief sunken -width $numChars -height    \
                              $numLines -wrap word
   pack $win.description.text -side left
   set dText $win.description.text

   set description $fsk(description)
   set description [string trimright $description '\n']

   $win.description.text insert 1.0 $description

   #######
   # create buttons
   #######
   frame $win.buttons
   pack $win.buttons -side top

   button $win.buttons.ok -text "OK" \
          -command {                                                           \
                       set fsk(description) [$dText get 1.0 $numLines.end];    \
                       set fsk(description)                                    \
                           [string trimright $fsk(description) '\n'];          \
                       destroy .getFskDescriptionWindow;                       \
                       SetFskDescription $fsk(description);                    \
                       processEventArray;                                      \
                    }
   pack $win.buttons.ok -side left

   button $win.buttons.cancel -text "CANCEL" \
          -command {destroy .getFskDescriptionWindow}
   pack $win.buttons.cancel -side left

   focus $win.description.text

   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : displayFskInfo
#
# PURPOSE : This procedure displays the default Fsk info.
#
################################################################################
proc displayFskInfo {} {

   global fsk
   global fsm_window
   global win
   global notSelectableCursor
   global processingCursor


   set tmpDFIWin .displayFskInfoWin

   toplevel $tmpDFIWin
   wm title $tmpDFIWin "FSK INFO"
   positionWindow $tmpDFIWin 100 150
   raise $tmpDFIWin
   focus $tmpDFIWin
   grab set $tmpDFIWin

   $fsm_window configure -cursor $processingCursor

   #######
   # get default info
   #######
   SetFskInfo
 
   frame $tmpDFIWin.text1
   pack $tmpDFIWin.text1 -side top -anchor w
   label $tmpDFIWin.text1.fskName1 -text " Name                 : " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text1.fskName1 -side left -pady 2 -anchor e
   label $tmpDFIWin.text1.fskName2 -text "$fsk(fileName) " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text1.fskName2 -side right -pady 2  

   frame $tmpDFIWin.text2
   pack $tmpDFIWin.text2 -side top -anchor w
   label $tmpDFIWin.text2.fskName1 -text " Full Path            : " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text2.fskName1 -side left -pady 2 -anchor e
   label $tmpDFIWin.text2.fskName2 -text "$fsk(completeFileName) " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text2.fskName2 -side right -pady 2  

   frame $tmpDFIWin.text3
   pack $tmpDFIWin.text3 -side top -anchor w
   label $tmpDFIWin.text3.dateModified1 -text " Last Saved           : " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text3.dateModified1 -side left -pady 2 -anchor e
   label $tmpDFIWin.text3.dateModified2 -text "$fsk(dateModified) " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text3.dateModified2 -side right -pady 2 -anchor e

   frame $tmpDFIWin.text4
   pack $tmpDFIWin.text4 -side top -anchor w
   label $tmpDFIWin.text4.numEvents1 -text " Number of Events     : " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text4.numEvents1 -side left -pady 2 -anchor e
   label $tmpDFIWin.text4.numEvents2 -text "$fsk(numEvents) " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text4.numEvents2 -side right -pady 2 -anchor e

   frame $tmpDFIWin.text5
   pack $tmpDFIWin.text5 -side top -anchor w
   label $tmpDFIWin.text5.numStates1 -text " Number of States     : " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text5.numStates1 -side left -pady 2 -anchor e
   label $tmpDFIWin.text5.numStates2 -text "$fsk(numStates) " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text5.numStates2 -side right -pady 2 -anchor e

   frame $tmpDFIWin.text6
   pack $tmpDFIWin.text6 -side top -anchor w
   label $tmpDFIWin.text6.modified1 -text " Current Fsk Modified : " \
         -font $fsk(Font2)
   pack $tmpDFIWin.text6.modified1 -side left -pady 2 -anchor e

   if {$fsk(modified) == 1} then {
      label $tmpDFIWin.text6.modified2 -text "YES " \
         -font $fsk(Font2)
      pack $tmpDFIWin.text6.modified2 -side right -pady 2 -anchor e
   } else {
      label $tmpDFIWin.text6.modified2 -text "NO " \
         -font $fsk(Font2)
      pack $tmpDFIWin.text6.modified2 -side right -pady 2 -anchor e
   }

   focus $tmpDFIWin

   #######
   # create buttons
   #######
   frame $tmpDFIWin.buttons
   pack $tmpDFIWin.buttons -side top

   button $tmpDFIWin.buttons.ok -text "OK" \
          -command { destroy .displayFskInfoWin }
   pack $tmpDFIWin.buttons.ok -side left


   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : quitWithoutSavePopup
#
# PURPOSE : This procedure is invoked to display a popup window to determine
#           if the user wishes to exit the modified FSK without saveing it.
################################################################################
proc quitWithoutSavePopup {} {

   global fsk
   global fsm_window
   global tempWindow
   global completeFskFileNameToOpen
   global notSelectableCursor
   global processingCursor

   set tempWindow .quitWithoutSavePopupWindow

   toplevel $tempWindow
   wm title $tempWindow "FSK MODIFIED"
   positionWindow $tempWindow 100 150
   raise $tempWindow 
   focus $tempWindow 
   grab set $tempWindow 

   $fsm_window configure -cursor $processingCursor

   ######
   # create prompt
   ######
   frame $tempWindow.msg
   pack $tempWindow.msg -side top

   label $tempWindow.msg.label  \
         -text "\n  \"$fsk(completeFileName)\" modified but not saved.  "
   pack $tempWindow.msg.label -side top

   label $tempWindow.msg.label2  \
         -text "  Save modified FSK?  " 
   pack $tempWindow.msg.label2 -side bottom

   ######
   # create buttons
   ######
   frame $tempWindow.buttons
   pack $tempWindow.buttons -side bottom

   #
   # TODO, the below "YES" command could cause a problem if there is a 
   #       problem writting the file.  if there is a problem and the file
   #       is not saved, we process the event array and then read the new file.
   #       hence, the file was never saved...
   #
   button $tempWindow.buttons.yes -text "YES"                                  \
          -command {SaveFsk $fsk(completeFileName) OVERWRITE_OK;               \
                    processEventArray;                                         \
                    exit}
   pack $tempWindow.buttons.yes -side left

   button $tempWindow.buttons.no -text "NO"                                    \
          -command {exit}
   pack $tempWindow.buttons.no -side left

   button $tempWindow.buttons.cancel -text "CANCEL" \
          -command {destroy $tempWindow}
   pack $tempWindow.buttons.cancel -side left 

   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : createMenuBar
#
# PURPOSE : This procedure creates the menu bar along the top of the window
#
################################################################################
proc createMenuBar {the_window} {

   frame $the_window.menuBar -relief raised -bd 2
   pack $the_window.menuBar -side top -fill x

   ######
   # create the File menu item
   ######
   menubutton $the_window.menuBar.file -text File -underline 0\
              -menu $the_window.menuBar.file.menu

   ######
   # create menu items under the File menu
   ######
   menu $the_window.menuBar.file.menu
   $the_window.menuBar.file.menu configure -tearoff 0

   $the_window.menuBar.file.menu add command -label "New..." -underline 0 \
          -command {getNewFskData}

   $the_window.menuBar.file.menu add command -label "Open..." -underline 0 \
          -accelerator "Ctrl+o"                                            \
          -command {getFskFileToOpen $fsk(fileToOpenPath) .}

   bind $the_window <Control-o> {getFskFileToOpen $fsk(fileToOpenPath) .}


   $the_window.menuBar.file.menu add command -label "Save" -underline 0 \
          -accelerator "Ctrl+s"                                         \
          -command {SaveFsk $fsk(completeFileName) OVERWRITE_OK;        \
                     processEventArray}

   bind $the_window <Control-s>  {SaveFsk $fsk(completeFileName)        \
                                  OVERWRITE_OK; processEventArray}


   $the_window.menuBar.file.menu add command -label "Save As..." -underline 5 \
          -command {getFskFileToSaveAs $fsk(fileToSaveAsPath)}


   $the_window.menuBar.file.menu add separator

   $the_window.menuBar.file.menu add command -label "Print..." -underline 0 \
          -accelerator "Ctrl+p"                                             \
          -command {printFsk}

   bind $the_window <Control-p> {printFsk}


   $the_window.menuBar.file.menu add command -label "Generate Code..."     \
           -underline 0                                                    \
          -accelerator "Ctrl+g"                                            \
          -command {generateCode}

   bind $the_window <Control-g> {generateCode}


   $the_window.menuBar.file.menu add separator

   $the_window.menuBar.file.menu add command -label "Run FSK..." -underline 0 \
          -accelerator "Ctrl+r"                                               \
          -command {runFsk}

   bind $the_window <Control-r> {runFsk}


   $the_window.menuBar.file.menu add separator

   $the_window.menuBar.file.menu add command  \
          -label "FSK Info" -underline 4      \
          -command {displayFskInfo}


   $the_window.menuBar.file.menu add separator

   $the_window.menuBar.file.menu add command -label "Quit" -underline 0 \
          -command {   if {$fsk(modified) != "0"} then {               \
                           quitWithoutSavePopup;                        \
                        } else {                                        \
                           exit;                                        \
                        }                                               \
                    }

   pack $the_window.menuBar.file -side left 



   ######
   # create the Edit menu item
   ######
   #
   menubutton $the_window.menuBar.edit -text Edit -underline 0\
              -menu $the_window.menuBar.edit.menu

   ######
   # create menu items under the Edit menu
   ######
   menu $the_window.menuBar.edit.menu
   $the_window.menuBar.edit.menu configure -tearoff 0

   $the_window.menuBar.edit.menu add command -label "Find..." -underline 0 \
          -accelerator "Ctrl+f"                                            \
          -command {findFskItem}

   bind $the_window <Control-f> {findFskItem}

   $the_window.menuBar.edit.menu add separator


   $the_window.menuBar.edit.menu add command                   \
          -label "FSK Defaults..." -underline 0                \
          -command {getDefaultInfo}

   $the_window.menuBar.edit.menu add separator

   $the_window.menuBar.edit.menu add command                   \
          -label "Source Settings..." -underline 0             \
          -command {getSourceSettings}

   $the_window.menuBar.edit.menu add separator

   $the_window.menuBar.edit.menu add command                   \
          -label "FSK Description..." -underline 4             \
          -command {getFskDescription}


   pack $the_window.menuBar.edit -side left 



   ######
   # create the Options menu item
   ######
   menubutton $the_window.menuBar.options -text Options -underline 0 \
              -menu $the_window.menuBar.options.menu

   ######
   # create menu items under the Options menu
   ######
   menu $the_window.menuBar.options.menu
   $the_window.menuBar.options.menu configure -tearoff 0

   $the_window.menuBar.options.menu add command  \
           -label "Show Command Palette" -underline 0 \
          -command {createCommandPalette}

   $the_window.menuBar.options.menu add command  \
           -label "Hide Command Palette" -underline 0 \
          -command {catch {destroy .commandPaletteWindow}}

   pack $the_window.menuBar.options -side left 



   ######
   # create the Help menu item
   ######
   menubutton $the_window.menuBar.help -text Help -underline 0 \
              -menu $the_window.menuBar.help.menu

   ######
   # create menu items under the Help menu
   ######
   menu $the_window.menuBar.help.menu
   $the_window.menuBar.help.menu configure -tearoff 0

   $the_window.menuBar.help.menu add command           \
           -label "On Commands" -underline 0           \
          -command {fskcCommandHelp}

   $the_window.menuBar.help.menu add command        \
           -label "About" -underline 0              \
          -command {fskcAboutBox}

   pack $the_window.menuBar.help -side right 



   ######
   # setup keyboard traversal order for menus
   ######
   tk_menuBar $the_window.menuBar         \
              $the_window.menuBar.file    \
              $the_window.menuBar.options \
              $the_window.menuBar.help 

   focus $the_window.menuBar
}


