#
#!fsm_wish
#
################################################################################
# Copyright (C) 1997, 2000  James A. Cureington
#                           tonyc@acm.org
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#
# $Id: fsmcase.tcl,v 1.1 2001/03/04 00:08:31 tonycu Exp $
#
# $Log: fsmcase.tcl,v $
# Revision 1.1  2001/03/04 00:08:31  tonycu
# initial check-in at sourceforge...
#
# Revision 1.14  2000/05/24 03:19:08  tony
# Updated all copyright dates.
#
# Revision 1.13  1999/09/13 01:21:06  tony
#
# Updated copyright notice date
#
# Revision 1.12  1997/03/26  07:04:07  tony
#      Fixed defects uncovered from test plan
#
# Revision 1.11  1997/03/22  17:29:26  tony
#      1) changed FSM_BIN_DIR to FSKC_BIN_DIR
#      2) upcased state and event name in find window, so user can
#         enter the names in upper or lower case
#      3) move find from file to edit menu
#
# Revision 1.10  1997/03/08  03:47:48  tony
#      1) remove new-lines (using trimright) in event, state, and action-block
#         edit windows
#      2) the nextState value was not being incremented in the action-blocks
#         when a state was inserted before it if the nextState value was 0
#      3) the copy buffer was not being update when states were deleted
#      4) if the function called in the action-block is vAbortFsk, don't
#         enclose it in a return(), i.e. return(vAbortFsk)
#      5) remove leading and trailing spaces from event and state names
#      6) change no_op function to take a parameter and return it; also,
#         rename it uiNoOp
#      7) change vPrintFskHistory to take a parameter and return it; also,
#         rename it uiPrintFskHistory
#      8) add check for duplicate event and state names
#      9) rename "recursive" variables to "iterative"
#
# Revision 1.9  1997/03/02  19:49:14  tony
#      - Added code to allow user defined error handler
#      - Added call to infinite state check
#
# Revision 1.8  1997/03/01  23:23:54  tony
#      Made the following changes:
#           - Add version to FSK file for validation when reading in
#           - Check for " in parameter list
#           - Add CVS Id to source and header file
#           - Add copyright notice to header files
#           - Add EDIT menu item and associated entries
#           - Make mouse button 3 cause the commnad pop-up to be displayed
#             instead of mouse button 2
#           - Add box to insert header files into
#           - Changed pchz to pch
#           - Allow macros to be used in parameter list
#           - Add function and associated calls to check for infinite state
#             condition; this is a warning, not an error condition
#           - Add over-ride abort function capability to GUI
#           - Changed "modified save pop-up" window to be displayed before
#             any window is displayed to read and RSK, or create a new one
#
# Revision 1.7  1997/02/08  22:50:12  tony
#      1) added info window
#      2) changed text to fixed width
#      3) validate fsk before running
#      4) added fsk filename to top of main window
#      5) all path names were not being updated on a "save as"
#      6) added code to print various reports...
#
# Revision 1.6  1997/01/26  04:52:24  tony
# Made changes to:
#      - clear, copy, paste, and undo action blocks
#      - comment out Edit menu option
#      - add ".fsk" to files as needed
#      - insert user defined default function next state when a state or
#        event is inserted
#      - check for invalid number of states and events in newFsk
#      - add "save if modified" to quit menu item
#      - add "processing" and "not selectable" cursors
#      - add procs to position state and event rows, positionState &
#        positionEvent
#      - add find state, event, and function commands
#      - add generate "C" code command
#      - add "print" Fsk command
#      - add run Fsk command
#
# Revision 1.5  1997/01/15  01:08:07  tony
# added code to:
#        - Add undo to TK and lower-level routines for all FSK
#           edit commands
#        - Added "undo" to state/event mouse menu bar.
#
# Revision 1.3  1997/01/13  08:21:25  tony
# added code to:
#        - delete states
#        - insert states
#
# Revision 1.2  1997/01/11  22:29:43  tony
#   Added code to edit state, event, and action blocks.
#
# Revision 1.1  1997/01/06  06:32:25  tony
# Initial check-in of PRODUCTION code.
#
# Revision 1.12  1996/12/13  05:23:37  tony
# Made minor comment and format changes
#
# Revision 1.11  1996/05/01  03:05:41  tony
# added call to LoadFSKVariables to load the values in the state, event, and
# action blocks......
#
# Revision 1.10  1996/04/28  20:54:09  tony
# moved global variables to "global_vars.tcl";  re-named some global variables
# to make more readable; made changes to invoke the procedure to create the
# menu bar.....
#
# Revision 1.9  1996/04/24  02:21:42  tony
# completed the mouse menu for the state blocks;  need to do the
# same for the event and action blocks.....
#
# Revision 1.8  1996/04/23  22:41:03  tony
# incremental check-in of current code with right mouse button activating
# a mouse menu on the state action blocks.
#
# Revision 1.7  1996/04/16  03:32:04  tony
# added capability to high-light the state and event blocks...
# also, reformated code and changed a few variable names...
#
# Revision 1.6  1996/04/15  04:00:41  tony
# finally got the canvases in the right locations.....
#
# Revision 1.5  1996/04/14  22:54:03  tony
# the vertical scrolling now works; need to align the canvases.....have not
# been able to do this....probably a little oversight that'll take days to
# discover....
#
# Revision 1.4  1996/04/14  18:35:33  tony
# changed horizScroll to horizontalScroll
# fixed bug in horizontalScroll
# added comments above procedures and cohesive blocks of code
#
# Revision 1.3  1996/04/14  07:07:26  tony
# added the capability of scrolling two canvas items with one scroll bar...
# still need to adjust the canvas to fit the state blocks, however.
# also need to add the event blocks and add scrolling for them.
#
# Revision 1.2  1996/04/14  03:59:28  tony
# Made some more changes to test concepts......
#
# Revision 1.1  1996/03/31  08:12:47  tony
# A prototype demonstrating Tcl/Tk scrolling capabillity.
#
# 
#
################################################################################



# NOTE: need a procedure to adjust display_num_* based on the number or rows
#       and columns....if num_{rows, columns} < display_num_*, the screen
#       does not fit the blocks correctly.....


################################################################################
# NAME    : positionWindow
#
# PURPOSE : This procedure positions a window at a given location
################################################################################
proc positionWindow {the_window x y} {
              # the_window the name of the window
              # x          the x coordinate to place the window at
              # y          the y coordinate to place the window at
   wm geometry $the_window +$x+$y
}



################################################################################
# NAME    : createBar
#
# PURPOSE : This procedure creates a canvas containing a black bar.  It is used
#           to create a bar between buttons in the mouse menus
################################################################################
proc createBar {bar_name} {
              # bar_name  the name of the bar to create
   canvas $bar_name -height 1m -width 2m 
   pack $bar_name -side top -expand no -fill x
   $bar_name create rect 0c 0c 4c 2c -outline black -fill black

}



################################################################################
# NAME    : hideMouseMenu
#
# PURPOSE : This procedure hides a window from the user
################################################################################
proc hideMouseMenu {the_window} {

                # the_window   the name of the window to hide

   global highlightedBlockID
   global highlighted_canvas
   global oldFill
   global mouseMenuActive
   global fsm_window
   

   # if this is the first time through, don't try to un-highlight any blocks...
   # there are none highlighted yet...

   if {$mouseMenuActive != 0} then {
      $highlighted_canvas itemconfigure $highlightedBlockID -fill $oldFill
      $highlighted_canvas itemconfigure [expr $highlightedBlockID+1] -fill black
   }

   # set mouseMenuActive variable to 0 so the blocks will be highlighted 
   # when the mouse passes over them

   set mouseMenuActive 0

   grab release $the_window

   wm withdraw $the_window

   bind $the_window <B1-ButtonRelease> {}

}



################################################################################
# NAME    : unhideMouseMenu
#
# PURPOSE : This procedure "un-hides" a window from the user that was 
#           previously hidden by hideMouseMenu
################################################################################
proc unhideMouseMenu {the_canvas the_window x y} {

                  # the_window    the name of the window to un-hide
   global highlightedBlockID
   global highlighted_canvas
   global mouseMenuActive
   global currentMouseMenu
   global fsm_window


   # if a mouse menu is currently active, hide it
   if {$mouseMenuActive != 0} then {
      hideMouseMenu $currentMouseMenu
      return
   }


   # find the item that has been selected and color it...
   set highlightedBlockID [$the_canvas find withtag current]
   if {[lsearch [$the_canvas gettags current] allTextTags] >= 0} {
      set highlightedBlockID [expr $highlightedBlockID-1]
   }

   if {[winfo depth $the_canvas] > 1} {
      $the_canvas itemconfigure $highlightedBlockID -fill yellow
   } else {
      $the_canvas itemconfigure $highlightedBlockID -fill red
      $the_canvas itemconfigure [expr $highlightedBlockID+1] -fill white
   }

   # save the canvas id the item is contained within so we can un-highlight it
   # later
   set highlighted_canvas $the_canvas

   # let main window know an action block is active so other blocks will not
   # be highlighted/filled with the mouse moves over them.
   set mouseMenuActive 1

   # save current mouse menu window name so we can hide it if the user snaps
   # the mouse button over a block activating the blockButton procedure
   set currentMouseMenu $the_window

   positionWindow $the_window $x $y

   wm deiconify $the_window

   raise $the_window

   grab set $the_window
   # don't change the mouse pointer cursor, we *want* the user to think
   # the area is selectable.  this allows us to hide the mouse menu 
   # when the mouse button is pressed over a state, event, or action-block

   bind $the_window <B1-ButtonRelease> "hideMouseMenu $the_window"

}



################################################################################
# NAME    : processMouseMenuCmd
#
# PURPOSE : This procedure process the mouse menu commands
################################################################################
proc processMouseMenuCmd {commandToProcess} {

   global highlightedBlockID
   global guiColumnIdToFskState
   global guiRowIdToFskEvent
   global actionBlock

   # highlightedBlockID is set by unhideMouseMenu

   set highlightedTextID [expr {$highlightedBlockID+1}]

   #################
   # DEBUG BEGIN
   # switch -glob $commandToProcess {
   #    *STATE* {
   #       puts "$commandToProcess $guiColumnIdToFskState($highlightedTextID)"
   #    }
   #    *EVENT* {
   #       puts "$commandToProcess $guiRowIdToFskEvent($highlightedTextID)"
   #    }
   # }
   # DEBUG END
   #################

   switch -exact $commandToProcess {

      DELETE_STATE {
         DeleteState $guiColumnIdToFskState($highlightedTextID)
      }

      INSERT_STATE_BEFORE {
         InsertState BEFORE $guiColumnIdToFskState($highlightedTextID)
      }

      INSERT_STATE_AFTER {
         InsertState AFTER $guiColumnIdToFskState($highlightedTextID)
      }

      CUT_STATE {
         CopyState $guiColumnIdToFskState($highlightedTextID)
         processEventArray
         DeleteState $guiColumnIdToFskState($highlightedTextID)
      }

      COPY_STATE {
         CopyState $guiColumnIdToFskState($highlightedTextID)
      }

      PASTE_STATE_BEFORE {
         PasteState BEFORE $guiColumnIdToFskState($highlightedTextID)
      }

      PASTE_STATE_AFTER {
         PasteState AFTER $guiColumnIdToFskState($highlightedTextID)
      }

      DELETE_EVENT {
         DeleteEvent $guiRowIdToFskEvent($highlightedTextID)
      }

      INSERT_EVENT_BEFORE {
         InsertEvent BEFORE $guiRowIdToFskEvent($highlightedTextID)
      }

      INSERT_EVENT_AFTER {
         InsertEvent AFTER $guiRowIdToFskEvent($highlightedTextID)
      }

      CUT_EVENT {
         CopyEvent $guiRowIdToFskEvent($highlightedTextID)
         processEventArray
         DeleteEvent $guiRowIdToFskEvent($highlightedTextID)
      }

      COPY_EVENT {
         CopyEvent $guiRowIdToFskEvent($highlightedTextID)
      }

      PASTE_EVENT_BEFORE {
         PasteEvent BEFORE $guiRowIdToFskEvent($highlightedTextID)
      }

      PASTE_EVENT_AFTER {
         PasteEvent AFTER $guiRowIdToFskEvent($highlightedTextID)
      }

      COPY_ACTION_BLOCK {
         CopyActionBlockData                                                  \
              $guiRowIdToFskEvent($actionBlock($highlightedTextID,eventId))   \
              $guiColumnIdToFskState($actionBlock($highlightedTextID,stateId))
      }

      PASTE_ACTION_BLOCK {
         PasteActionBlockData                                                 \
              $guiRowIdToFskEvent($actionBlock($highlightedTextID,eventId))   \
              $guiColumnIdToFskState($actionBlock($highlightedTextID,stateId))
      }

      CLEAR_ACTION_BLOCK {
         ClearActionBlockData                                                 \
              $guiRowIdToFskEvent($actionBlock($highlightedTextID,eventId))   \
              $guiColumnIdToFskState($actionBlock($highlightedTextID,stateId))
      }

      UNDO_LAST_EDIT {
         UndoLastCommand 
      }

      default {
         puts "INTERNAL ERROR --"
         puts "undefined command in processMouseMenuCmd, $commandToProcess"
         exit 1
      }
   }

   processEventArray
}



################################################################################
# NAME    : createEventStateMouseMenu
#
# PURPOSE : This procedure creates a mouse menu containing actions that can be
#           performed on states and events...                            
################################################################################
proc createEventStateMouseMenu {mouse_menu} {

   toplevel $mouse_menu 

   # the type of block being displayed, STATE or EVENT
   # this variable must be set outside this procedure
   global blockTypeToEdit

   # remove re-size corners and iconify button
   wm overrideredirect $mouse_menu true


   positionWindow $mouse_menu 100 150

   button $mouse_menu.undo -text Undo \
          -command { processMouseMenuCmd UNDO_LAST_EDIT } 

   button $mouse_menu.cut -text Cut \
          -command { processMouseMenuCmd CUT_${blockTypeToEdit} } 
   button $mouse_menu.copy -text Copy \
          -command { processMouseMenuCmd COPY_${blockTypeToEdit} } 
   button $mouse_menu.pasteA -text "Paste After" \
          -command { processMouseMenuCmd PASTE_${blockTypeToEdit}_AFTER } 
   button $mouse_menu.pasteB -text "Paste Before" \
          -command { processMouseMenuCmd PASTE_${blockTypeToEdit}_BEFORE } 

   button $mouse_menu.insertA -text "Insert After" \
          -command { processMouseMenuCmd INSERT_${blockTypeToEdit}_AFTER } 
   button $mouse_menu.insertB -text "Insert Before" \
          -command { processMouseMenuCmd INSERT_${blockTypeToEdit}_BEFORE } 

   button $mouse_menu.delete -text Delete \
          -command { processMouseMenuCmd DELETE_${blockTypeToEdit} } 


   pack $mouse_menu.undo -expand yes -fill both

   createBar $mouse_menu.bar1
   pack $mouse_menu.cut        \
        $mouse_menu.copy       \
        $mouse_menu.pasteA     \
        $mouse_menu.pasteB     \
        -expand yes -fill both 

   createBar $mouse_menu.bar2
   pack $mouse_menu.insertA     \
        $mouse_menu.insertB     \
        -expand yes -fill both

   createBar $mouse_menu.bar3
   pack $mouse_menu.delete -expand yes -fill both

}



################################################################################
# NAME    : createActionBlockMouseMenu
#
# PURPOSE : This procedure creates a mouse menu containing actions that can be
#           performed on action-blocks
################################################################################
proc createActionBlockMouseMenu {mouse_menu} {

   toplevel $mouse_menu 

   # remove re-size corners and iconify button
   wm overrideredirect $mouse_menu true

   positionWindow $mouse_menu 100 150

   button $mouse_menu.undo -text Undo \
          -command { processMouseMenuCmd UNDO_LAST_EDIT } 

   button $mouse_menu.copy -text Copy \
          -command { processMouseMenuCmd COPY_ACTION_BLOCK } 
   button $mouse_menu.paste -text "Paste" \
          -command { processMouseMenuCmd PASTE_ACTION_BLOCK } 

   button $mouse_menu.clear -text Clear \
          -command { processMouseMenuCmd CLEAR_ACTION_BLOCK } 


   pack $mouse_menu.undo -expand yes -fill both

   createBar $mouse_menu.bar1
   pack $mouse_menu.copy       \
        $mouse_menu.paste      \
        -expand yes -fill both 

   createBar $mouse_menu.bar2
   pack $mouse_menu.clear     \
        -expand yes -fill both

}



################################################################################
# NAME    : fskHorizontalScroll
#
# PURPOSE : Scroll the state and action blocks horizonatally
#
# ARGS    : arg1              "scroll" or "moveto"
#
#           arg2              if arg1 is "scroll", arg2 is number of units as 
#                                  positive or negative int value
#                               if arg1 is "moveto", arg2 is number of units 
#                                  as floating point value
#
#           arg3              units, pages, or "" (NULL)
#
# EXAMPLE :                     arg1       arg2         arg3
#                            ----       ----         ----
#                            scroll        1         pages
#                            scroll       -1         pages
#                            scroll        1         units
#                            scroll       -1         units
#                            moveto        0.164..  ""
#                            moveto       -0.073..  ""
#
#
################################################################################
proc fskHorizontalScroll {arg1 arg2 {arg3 "" }} {

   global action_block_canvas
   global state_block_canvas
   global BLOCK_SIZE

   if {$arg3 != ""} {
      $action_block_canvas xview $arg1 $arg2 $arg3
      $state_block_canvas xview $arg1 $arg2 $arg3
   } else {
      $action_block_canvas xview $arg1 $arg2 
      $state_block_canvas xview $arg1 $arg2 
   }
}



################################################################################
# NAME    : fskVerticalScroll
#
# PURPOSE : Scroll the event and action blocks vertically
#
# ARGS    : arg1              "scroll" or "moveto"
#
#           arg2              if arg1 is "scroll", arg2 is number of units as 
#                                  positive or negative int value
#                               if arg1 is "moveto", arg2 is number of units 
#                                  as floating point value
#
#           arg3              units, pages, or "" (NULL)
#
# EXAMPLE :                     arg1       arg2         arg3
#                            ----       ----         ----
#                            scroll        1         pages
#                            scroll       -1         pages
#                            scroll        1         units
#                            scroll       -1         units
#                            moveto        0.164..  ""
#                            moveto       -0.073..  ""
#
#
################################################################################
proc fskVerticalScroll {arg1 arg2 {arg3 "" }} {

   global action_block_canvas
   global event_block_canvas
   global BLOCK_SIZE

   if {$arg3 != ""} {
      $action_block_canvas yview $arg1 $arg2 $arg3
      $event_block_canvas yview $arg1 $arg2 $arg3
   } else {
      $action_block_canvas yview $arg1 $arg2 
      $event_block_canvas yview $arg1 $arg2 
   }
}



################################################################################
# NAME    : adjustScrollRegions
#
# PURPOSE : This procedure adjusts the scroll region when states or events are
#           deleted or inserted.
################################################################################
proc adjustScrollRegions {} {

   global fsk
   global stateScrollRegion
   global eventScrollRegion
   global actionBlockScrollRegion
   global state_block_canvas
   global event_block_canvas
   global action_block_canvas
   global BLOCK_SIZE

   #######
   # re-compute scroll regions
   #######
   set stateScrollRegion   [list 0c                             \
                            0c                                  \
                            [expr $fsk(numStates)*$BLOCK_SIZE]c \
                            ${BLOCK_SIZE}c ] 

   set eventScrollRegion   [list 0c                            \
                            0c                                \
                            ${BLOCK_SIZE}c                    \
                            [expr $fsk(numEvents)*$BLOCK_SIZE]c ]  

   set actionBlockScrollRegion   [list 0c                             \
                                  0c                                  \
                                  [expr $fsk(numStates)*$BLOCK_SIZE]c \
                                  [expr $fsk(numEvents)*$BLOCK_SIZE]c] 

   #######
   # set canvas to new scroll regions
   #######
   $state_block_canvas  configure -scrollregion $stateScrollRegion
   $event_block_canvas configure -scrollregion $eventScrollRegion
   $action_block_canvas configure -scrollregion $actionBlockScrollRegion

}



################################################################################
# NAME    : newFsk
#
# PURPOSE : This procedure loads the new FSK
################################################################################
proc newFsk {} {

   #######
   # canvas
   #######
   global event_block_canvas
   global state_block_canvas
   global action_block_canvas

   #######
   # vars
   #######
   global fsk
   global eventBlock
   global stateBlock
   global actionBlock
   global BLOCK_SIZE
   global stateScrollRegion
   global eventScrollRegion
   global actionBlockScrollRegion

   global guiColumnIdToFskState
   global guiRowIdToFskEvent
   global fskStateToGuiColumnId
   global fskEventToGuiRowId

   #######
   # tags
   #######
   global allTextTags
   global allBlockTags

   # array of tags to a single objects text
   global actionBlockTextTag
   global stateBlockTextTag
   global eventBlockTextTag

   # array of tags to rectangles
   global fskColumnBlockTag
   global fskRowBlockTag


   #######
   # delete all text and blocks
   #######
   $event_block_canvas delete allTextTags
   $event_block_canvas delete allBlockTags
   $state_block_canvas delete allTextTags
   $state_block_canvas delete allBlockTags
   $action_block_canvas delete allTextTags
   $action_block_canvas delete allBlockTags

   
   #######
   # create state blocks
   #######
   set bg [lindex [$state_block_canvas config -bg] 4]

   for {set i 0} {$i < $fsk(numStates)} {incr i} {
      set x [expr {$BLOCK_SIZE*$i}]
      $state_block_canvas create rect ${x}c 0c       \
            [expr $x+$BLOCK_SIZE]c ${BLOCK_SIZE}c    \
            -outline black -fill $bg -width 2        \
            -tags fskColumnBlockTag($i)


      $state_block_canvas addtag allBlockTags withtag \
                                 fskColumnBlockTag($i)

      set tempId                                              \
         [$state_block_canvas create text [expr $x+0.2]c 0.5c \
             -text "$i\n$stateBlock($i,name)"                 \
             -font $fsk(Font1) -anchor w -tags stateBlockTextTag($i)]

      $state_block_canvas addtag allTextTags withtag \
                                 stateBlockTextTag($i)

      set guiColumnIdToFskState($tempId) $i
      set fskStateToGuiColumnId($i) $tempId
   }


   #######
   # create event blocks
   #######
   set bg [lindex [$event_block_canvas config -bg] 4]

   for {set i 0; set y 0} {$i < $fsk(numEvents)} \
      {incr i; incr y $BLOCK_SIZE} {

      $event_block_canvas create rect 0c ${y}c     \
             ${BLOCK_SIZE}c [expr $y+$BLOCK_SIZE]c \
             -outline black -fill $bg -width 2     \
             -tags fskRowBlockTag($i)

      $event_block_canvas addtag allBlockTags withtag \
                                 fskRowBlockTag($i)

      set tempId                                                        \
         [$event_block_canvas create text 0.2c [expr $y+0.5]c           \
             -text "$i\n$eventBlock($i,name)"                           \
             -font $fsk(Font1) -anchor w -tags eventBlockTextTag($i)]

      $event_block_canvas addtag allTextTags withtag \
                                  eventBlockTextTag($i)

      set guiRowIdToFskEvent($tempId) $i
      set fskEventToGuiRowId($i) $tempId
   }


   #######
   # create action blocks
   #######
   set bg [lindex [$action_block_canvas config -bg] 4]

   for {set i 0} {$i < $fsk(numEvents)} {incr i} {
      set y [expr {$BLOCK_SIZE*$i}]
      for {set j 0; set x 0} \
         {$j < $fsk(numStates)} \
         {incr j; incr x $BLOCK_SIZE} {

         $action_block_canvas create rect ${x}c ${y}c                    \
               [expr $x+$BLOCK_SIZE]c [expr $y+$BLOCK_SIZE]c             \
               -outline black -fill $bg -width 2 -tags tempTag($i,$j)

         $action_block_canvas addtag fskColumnBlockTag($j) \
                                          withtag tempTag($i,$j)
         $action_block_canvas addtag fskRowBlockTag($i) \
                                          withtag tempTag($i,$j)
         $action_block_canvas addtag allBlockTags \
                                          withtag tempTag($i,$j)

         if {$actionBlock($i,$j,iterative)} then {
            set iFlag ",I"
         } else {
            set iFlag ""
         }

         set tempId                                                            \
         [$action_block_canvas create text [expr $x+0.2]c [expr $y+1]c         \
               -text                                                           \
"$actionBlock($i,$j,functionName)\n\[$actionBlock($i,$j,nextState)\]$iFlag"    \
               -font $fsk(Font1) -anchor w -tags actionBlockTextTag($i,$j)]

         $action_block_canvas addtag fskColumnBlockTag($j) \
                                          withtag actionBlockTextTag($i,$j)

         $action_block_canvas addtag fskRowBlockTag($i) \
                                          withtag actionBlockTextTag($i,$j)

         $action_block_canvas addtag allTextTags \
                                    withtag actionBlockTextTag($i,$j) 

         set actionBlock($tempId,eventId) $fskEventToGuiRowId($i)
         set actionBlock($tempId,stateId) $fskStateToGuiColumnId($j)

      }
   }


   #######
   # re-compute scroll regions
   #######
   adjustScrollRegions

}



################################################################################
# NAME    : blockEnter
#
# PURPOSE : This procedure changes the color of the item the mouse cursor 
#           is over.
################################################################################
proc blockEnter {canvas} {

   global mouseMenuActive
   global oldFill

   # if the mouse menu is active, don't allow selection of any blocks...
   if {$mouseMenuActive} then {
      return
   }

   set id [$canvas find withtag current]

   if {[lsearch [$canvas gettags current] allTextTags] >= 0} {
      set id [expr $id-1]
   }

   set oldFill [lindex [$canvas itemconfig $id -fill] 4]

   if {[winfo depth $canvas] > 1} {
      $canvas itemconfigure $id -fill SeaGreen1
   } else {
      $canvas itemconfigure $id -fill black
      $canvas itemconfigure [expr $id+1] -fill white
   }
}



################################################################################
# NAME    : blockLeave
#
# PURPOSE : This procedure changes the color of the block back to the color 
#           before it was changes by blockEnter.                           
################################################################################
proc blockLeave {canvas} {

   global mouseMenuActive
   global oldFill

   # if the mouse-menu is active, don't allow selection of any blocks...
   if {$mouseMenuActive} then {
      return
   }

   set id [$canvas find withtag current]
   if {[lsearch [$canvas gettags current] allTextTags] >= 0} {
      set id [expr $id-1]
   }
   $canvas itemconfigure $id -fill $oldFill
   $canvas itemconfigure [expr $id+1] -fill black
}



################################################################################
# NAME    : stateBlockButton
#
# PURPOSE : This procedure is invoked when the left mouse button is pressed
#           over a state block.
################################################################################
proc stateBlockButton {} {

   global mouseMenuActive
   global currentMouseMenu
   global state_block_canvas
   global guiColumnIdToFskState

   # if the mouse menu is active, don't allow selection of any blocks...
   # since the user snapped the mouse, cancel the mouse menu..
   if {$mouseMenuActive} then {
      hideMouseMenu $currentMouseMenu
      return
   }

   set id [$state_block_canvas find withtag current]
   if {[lsearch [$state_block_canvas gettags current] allTextTags] < 0} {
      set id [expr $id+1]
   }

   # puts "fsk state number =  $guiColumnIdToFskState($id)"
   getStateData $guiColumnIdToFskState($id)
   
   # wait for getStateData window to be destroyed
   tkwait window .getStateDataWindow 

}



################################################################################
# NAME    : eventBlockButton
#
# PURPOSE : This procedure is invoked when the left mouse button is pressed
#           over a event block.
################################################################################
proc eventBlockButton {} {

   global mouseMenuActive
   global currentMouseMenu
   global event_block_canvas
   global guiRowIdToFskEvent

   # if the mouse menu is active, don't allow selection of any blocks...
   # since the user snapped the mouse, cancel the mouse menu..
   if {$mouseMenuActive} then {
      hideMouseMenu $currentMouseMenu
      return
   }

   set id [$event_block_canvas find withtag current]
   if {[lsearch [$event_block_canvas gettags current] allTextTags] < 0} {
      set id [expr $id+1]
   }

   # puts "fsk event number =  $guiRowIdToFskEvent($id)"
   getEventData $guiRowIdToFskEvent($id)
   
   # wait for getEventData window to be destroyed
   tkwait window .getEventDataWindow 

}



################################################################################
# NAME    : actionBlockButton
#
# PURPOSE : This procedure is invoked when the left mouse button is pressed
#           over a action-block.
################################################################################
proc actionBlockButton {} {

   global mouseMenuActive
   global currentMouseMenu
   global action_block_canvas
   global guiRowIdToFskEvent
   global guiColumnIdToFskState
   global actionBlock

   # if the mouse menu is active, don't allow selection of any blocks...
   # since the user snapped the mouse, cancel the mouse menu..
   if {$mouseMenuActive} then {
      hideMouseMenu $currentMouseMenu
      return
   }

   set id [$action_block_canvas find withtag current]
   if {[lsearch [$action_block_canvas gettags current] allTextTags] < 0} {
      set id [expr $id+1]
   }

   getActionBlockData $guiRowIdToFskEvent($actionBlock($id,eventId))  \
                      $guiColumnIdToFskState($actionBlock($id,stateId))
   
   # wait for getActionBlockDataWindow to be destroyed
   tkwait window .getActionBlockDataWindow

}



########################################
########################################
#BEGIN need to move below to different file and source it
########################################
########################################

################################################################################
# NAME    : updateStateBlock
#
# PURPOSE : This procedure updates the state name and description in the 
#           GUI from the the values loaded by the lower level routines.
################################################################################
proc updateStateBlock {stateNumber} {

   global fsk
   global fsm_window
   global win
   global state_block_canvas
   global stateBlock

   $state_block_canvas itemconfigure stateBlockTextTag($stateNumber) \
                       -text "$stateNumber\n$stateBlock($stateNumber,name)"


}



################################################################################
# NAME    : getStateData
#
# PURPOSE : This procedure gets the new state name and description from the 
#           user and calls the C function to update the FSK structures.
################################################################################
proc getStateData {stateNumber} {

   global fsk
   global fsm_window
   global win
   global currentStateNumber
   global newStateName
   global newStateDescription
   global stateBlock
   global dText
   global numLines
   global notSelectableCursor
   global processingCursor


   set win .getStateDataWindow

   toplevel $win
   wm title $win "EDIT STATE $stateNumber"
   positionWindow $win 100 150
   raise $win
   focus $win
   grab set $win

   $fsm_window configure -cursor $processingCursor

   set currentStateNumber $stateNumber

   #######
   # create state name entry field
   #######
   frame $win.spacer
   pack $win.spacer -side top -anchor w
   label $win.spacer.label -text  " " 
   pack $win.spacer.label -side left -anchor s 

   frame $win.state_name
   pack $win.state_name -side top -anchor w

   label $win.state_name.label -text  "Name :" 
   pack $win.state_name.label -side left -anchor s 

   entry $win.state_name.entry -width 35 -relief sunken \
         -justify left -textvariable newStateName 
   pack $win.state_name.entry -side left  -anchor s

   $win.state_name.entry delete 0 end
   $win.state_name.entry insert 0 $stateBlock($stateNumber,name)


   #######
   # create description entry field
   #######
   set numLines 8

   frame $win.description
   pack $win.description -side top

   label $win.description.label -text "Description" 
   pack $win.description.label -side top 

   text $win.description.text -relief sunken -width 48 -height $numLines \
        -wrap word

   set description $stateBlock($stateNumber,description)
   set description [string trimright $description '\n']
   $win.description.text delete 1.0 end
   $win.description.text insert 1.0 $description

   pack $win.description.text -side left
   set dText $win.description.text


   #######
   # create buttons
   #######
   frame $win.buttons
   pack $win.buttons -side top

   button $win.buttons.ok -text "OK" \
          -command {                                                          \
                       set newStateName [string toupper $newStateName];       \
                       set newStateDescription [$dText get 1.0 $numLines.end];\
                       set newStateDescription                                \
                           [string trimright $newStateDescription '\n'];      \
                       UpdateStateData $newStateName $newStateDescription     \
                                       $currentStateNumber;                   \
                       destroy .getStateDataWindow;                           \
                       processEventArray;                                     \
                    }
   pack $win.buttons.ok -side left

   button $win.buttons.cancel -text "CANCEL" \
          -command {destroy .getStateDataWindow}
   pack $win.buttons.cancel -side left

   focus $win.state_name.entry

   $fsm_window configure -cursor $notSelectableCursor

}



################################################################################
# NAME    : updateEventBlock
#
# PURPOSE : This procedure updates the event name and description in the 
#           GUI from the the values loaded by the lower level routines.
################################################################################
proc updateEventBlock {eventNumber} {

   global fsk
   global fsm_window
   global win
   global event_block_canvas
   global eventBlock

   $event_block_canvas itemconfigure eventBlockTextTag($eventNumber) \
                       -text "$eventNumber\n$eventBlock($eventNumber,name)"                 


}



################################################################################
# NAME    : getEventData
#
# PURPOSE : This procedure gets the new event name and description from the 
#           user and calls the C function to update the FSK structures.
################################################################################
proc getEventData {eventNumber} {

   global fsk
   global fsm_window
   global win
   global currentEventNumber
   global newEventName
   global newEventDescription
   global eventBlock
   global dText
   global numLines
   global notSelectableCursor
   global processingCursor


   set win .getEventDataWindow

   toplevel $win
   wm title $win "EDIT EVENT $eventNumber"
   positionWindow $win 100 150
   raise $win
   focus $win
   grab set $win

   $fsm_window configure -cursor $processingCursor

   set currentEventNumber $eventNumber

   #######
   # create event name entry field
   #######
   frame $win.spacer
   pack $win.spacer -side top -anchor w
   label $win.spacer.label -text  " " 
   pack $win.spacer.label -side left -anchor s 

   frame $win.event_name
   pack $win.event_name -side top -anchor w

   label $win.event_name.label -text  "Name :" 
   pack $win.event_name.label -side left -anchor s 

   entry $win.event_name.entry -width 35 -relief sunken \
         -justify left -textvariable newEventName 
   pack $win.event_name.entry -side left  -anchor s
   $win.event_name.entry delete 0 end
   $win.event_name.entry insert 0 $eventBlock($eventNumber,name)


   #######
   # create description entry field
   #######
   set numLines 8

   frame $win.description
   pack $win.description -side top

   label $win.description.label -text "Description" 
   pack $win.description.label -side top 

   text $win.description.text -relief sunken -width 48 -height $numLines \
        -wrap word

   set description $eventBlock($eventNumber,description)
   set description [string trimright $description '\n']
   $win.description.text delete 1.0 end
   $win.description.text insert 1.0 $description

   pack $win.description.text -side left
   set dText $win.description.text


   #######
   # create buttons
   #######
   frame $win.buttons
   pack $win.buttons -side top

   button $win.buttons.ok -text "OK" \
          -command {                                                          \
                       set newEventName [string toupper $newEventName];       \
                       set newEventDescription [$dText get 1.0 $numLines.end];\
                       set newEventDescription                                \
                           [string trimright $newEventDescription '\n'];      \
                       UpdateEventData $newEventName $newEventDescription     \
                                       $currentEventNumber;                   \
                       destroy .getEventDataWindow;                           \
                       processEventArray;                                     \
                    }
   pack $win.buttons.ok -side left

   button $win.buttons.cancel -text "CANCEL" \
          -command {destroy .getEventDataWindow}
   pack $win.buttons.cancel -side left

   focus $win.event_name.entry
   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : updateActionBlock
#
# PURPOSE : This procedure updates the action-block variables in the 
#           GUI from the the values loaded by the lower level routines.
################################################################################
proc updateActionBlock {eventNumber stateNumber} {

   global fsk
   global fsm_window
   global win
   global action_block_canvas
   global actionBlock


   if {$actionBlock($eventNumber,$stateNumber,iterative)} then {
      set iFlag ",I"
   } else {
      set iFlag " "
   }

   $action_block_canvas itemconfigure                                   \
               actionBlockTextTag($eventNumber,$stateNumber) -text      \
    "$actionBlock($eventNumber,$stateNumber,functionName)\
\n\[$actionBlock($eventNumber,$stateNumber,nextState)\]$iFlag"

}



################################################################################
# NAME    : getActionBlockData
#
# PURPOSE : This procedure gets the new action-block data from the
#           user and calls the C function to update the FSK structures.
################################################################################
proc getActionBlockData {eventNumber stateNumber} {

   global fsk
   global fsm_window
   global win
   global currentEventNumber
   global currentStateNumber
   global newFunctionName
   global newNextState
   global newIterativeFlag
   global newDescription
   global actionBlock
   global dText
   global fText
   global numDescLines
   global numFuncLines
   global numDescChars
   global numFuncChars
   global notSelectableCursor
   global processingCursor


   set win .getActionBlockDataWindow

   toplevel $win
   wm title $win "EDIT ACTION-BLOCK $eventNumber,$stateNumber"
   positionWindow $win 100 150
   raise $win
   focus $win
   grab set $win

   $fsm_window configure -cursor $processingCursor

   set currentEventNumber $eventNumber
   set currentStateNumber $stateNumber


   #######
   # function name entry
   #######
   frame $win.spacer
   pack $win.spacer -side top -anchor w
   label $win.spacer.label -text  " " 
   pack $win.spacer.label -side left -anchor s 

   frame $win.function_name
   pack $win.function_name -side top

   label $win.function_name.label -text "Function Name & Parameters" 
   pack $win.function_name.label -side top 

   text $win.function_name.text -relief sunken -width $numFuncChars \
        -height $numFuncLines -wrap word

#   set newFunctionName                                       \
#           $actionBlock($eventNumber,$stateNumber,functionName)$actionBlock($eventNumber,$stateNumber,functionParameters)
#
   set newFunctionName $actionBlock($eventNumber,$stateNumber,functionName)
   set newFunctionName \
     $newFunctionName$actionBlock($eventNumber,$stateNumber,functionParameters)

   set newFunctionName [string trimright $newFunctionName '\n']
   $win.function_name.text delete 1.0 end
   $win.function_name.text insert 1.0 $newFunctionName

   pack $win.function_name.text -side left
   set fText $win.function_name.text


   #######
   # next state field
   #######
   frame $win.next_state
   pack $win.next_state -side top -anchor w

   label $win.next_state.label -text     "Transition to State : " 
   pack $win.next_state.label -side left -anchor s -pady 4

   entry $win.next_state.entry -width 10 -relief sunken \
         -justify left -textvariable newNextState 
   pack $win.next_state.entry -side left  -anchor s -pady 4

   $win.next_state.entry delete 0 end
   $win.next_state.entry insert 0 \
           $actionBlock($eventNumber,$stateNumber,nextState)
         

   #######
   # iterative checkbutton
   #######
   frame $win.button
   pack $win.button -side top -anchor w

   set newIterativeFlag $actionBlock($eventNumber,$stateNumber,iterative)

   checkbutton $win.button.iterative -text "Iterative" \
               -variable newIterativeFlag
   pack $win.button.iterative -side left  -anchor s -pady 4


   #######
   # create description entry field
   #######
   frame $win.description
   pack $win.description -side top

   label $win.description.label -text "Description" 
   pack $win.description.label -side top 

   text $win.description.text -relief sunken -width $numDescChars \
        -height $numDescLines -wrap word

   set description $actionBlock($eventNumber,$stateNumber,description)
   set description [string trimright $description '\n']
   $win.description.text delete 1.0 end
   $win.description.text insert 1.0 $description

   pack $win.description.text -side left
   set dText $win.description.text


   #######
   # create buttons
   #######
   frame $win.buttons
   pack $win.buttons -side top

   button $win.buttons.ok -text "OK" \
          -command {                                                          \
                       set newDescription [$dText get 1.0 $numDescLines.end]; \
                       set newDescription                                     \
                           [string trimright $newDescription '\n'];           \
                       set newFunctionName [$fText get 1.0 $numFuncLines.end];\
                       set newFunctionName                                    \
                           [string trimright $newFunctionName '\n'];          \
                       UpdateActionBlockData $newFunctionName $newNextState   \
                                       $newIterativeFlag $newDescription      \
                                       $currentEventNumber                    \
                                       $currentStateNumber;                   \
                       destroy .getActionBlockDataWindow;                     \
                       processEventArray;                                     \
                    }
   pack $win.buttons.ok -side left

   button $win.buttons.cancel -text "CANCEL" \
          -command {destroy .getActionBlockDataWindow}
   pack $win.buttons.cancel -side left

   focus $win.function_name.text
   $fsm_window configure -cursor $notSelectableCursor

}



################################################################################
# NAME    : displayMessage
#
# PURPOSE : This procedure displays the given message in a window with 
#           an OK button.
################################################################################
proc displayMessage {theMessage} {
   global fsm_window
   global tempDMsgWindow
   global notSelectableCursor
   global processingCursor

   set tempDMsgWindow .displayMessageWindow

   toplevel $tempDMsgWindow
   wm title $tempDMsgWindow "MESSAGE"
   positionWindow $tempDMsgWindow 100 150
   raise $tempDMsgWindow 
   focus $tempDMsgWindow 
   grab set $tempDMsgWindow 

   $fsm_window configure -cursor $processingCursor

   ######
   # create message
   ######
   frame $tempDMsgWindow.msg
   pack $tempDMsgWindow.msg -side top

   message $tempDMsgWindow.msg.text  -width 16c -justify left -relief raised   \
         -bd 2 -text "\n$theMessage\n"

   pack $tempDMsgWindow.msg.text -side left

   frame $tempDMsgWindow.buttons
   pack $tempDMsgWindow.buttons -side bottom

   button $tempDMsgWindow.buttons.ok -text "OK" \
          -command {destroy $tempDMsgWindow}
   pack $tempDMsgWindow.buttons.ok 

   $fsm_window configure -cursor $notSelectableCursor

   focus $tempDMsgWindow.buttons.ok 

   tkwait window .displayMessageWindow 
}



################################################################################
# NAME    : displayScrollMessage
#
# PURPOSE : This procedure displays the given message in a scroll window with 
#           an OK button.
################################################################################
proc displayScrollMessage {theMessage} {
   global fsm_window
   global tempDSMsgWindow
   global notSelectableCursor
   global processingCursor

   set tempDSMsgWin .displayScrollMessageWindow
   catch {destroy $tempDSMsgWin}

   toplevel $tempDSMsgWin
   wm title $tempDSMsgWin "MESSAGE"
   positionWindow $tempDSMsgWin 100 150
   raise $tempDSMsgWin 
   focus $tempDSMsgWin 
   grab set $tempDSMsgWin 

   $fsm_window configure -cursor $processingCursor

   ######
   # create scroll message
   ######
   frame $tempDSMsgWin.list_box
   pack $tempDSMsgWin.list_box -side top -fill both -expand yes

   scrollbar $tempDSMsgWin.list_box.scroll            \
             -command "$tempDSMsgWin.list_box.text yview"

   listbox $tempDSMsgWin.list_box.text -relief sunken -width 50 \
           -borderwidth 2 -yscroll "$tempDSMsgWin.list_box.scroll set"

   pack $tempDSMsgWin.list_box.scroll -side right -fill y

   pack $tempDSMsgWin.list_box.text -side left -fill both -expand yes


   foreach i [split $theMessage '\n'] {
       $tempDSMsgWin.list_box.text insert end $i
   }

   frame $tempDSMsgWin.buttons
   pack $tempDSMsgWin.buttons -side bottom

   button $tempDSMsgWin.buttons.ok -text "OK" \
          -command {destroy .displayScrollMessageWindow}
   pack $tempDSMsgWin.buttons.ok 
   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : waitOnDisplayScrollMessage
#
# PURPOSE : This procedure waits for the displayScrollMessage window to be    
#           destroyed.  The fsm window's cursor is set to notSelectableCursor
#           after this call.
################################################################################
proc waitOnDisplayScrollMessage {callingWindow} {

   global fsm_window
   global notSelectableCursor
   global .displayScrollMessageWindow

   # catch errors in case window has not be created yet
   catch {$callingWindow configure -cursor $notSelectableCursor}
   catch {grab set .displayScrollMessageWindow}

   # wait for dispalyScrollMessage window to be destroyed
   catch {tkwait window .displayScrollMessageWindow}

   catch {$callingWindow configure -cursor top_left_arrow}
   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : deleteStateAndUpdate
#
# PURPOSE : This procedure deletes the last state in the GUI FSK.  It then 
#           updates all state and action-blocks to reflect the deleted state.
################################################################################
proc deleteStateAndUpdate {} {

   global fsk
   global fsm_window
   global win
   global state_block_canvas
   global action_block_canvas

   # at this time we have one more state than the actual FSK maintained by
   # the lower level routines.  so we delete the Nth state, not the Nth+1
   # state (our last state, since we are zero based and fsk(numStates) is 
   # one based).  this also preserves all tags 

   set stateToDelete $fsk(numStates)

   set numEvents $fsk(numEvents)
   set numStates $fsk(numStates)

   # delete the text in the state and action-blocks
   $state_block_canvas delete stateBlockTextTag($stateToDelete)
   $state_block_canvas delete fskColumnBlockTag($stateToDelete)
   $action_block_canvas delete fskColumnBlockTag($stateToDelete)

   # update all states and action-blocks
   for {set iLoop 0} {$iLoop < $numStates} {incr iLoop} {

      updateStateBlock $iLoop

      for {set iLoop2 0} {$iLoop2 < $numEvents} {incr iLoop2} {
         updateActionBlock $iLoop2 $iLoop
      }
   }

   adjustScrollRegions
}



################################################################################
# NAME    : insertStateAndUpdate
#
# PURPOSE : This procedure inserts a state at the end of the GUI FSK.  It then 
#           updates all state and action-blocks to reflect the inserted state.
################################################################################
proc insertStateAndUpdate {} {

   global fsk
   global win
   global fsm_window
   global state_block_canvas
   global action_block_canvas

   #######
   # canvas
   #######
   global state_block_canvas
   global action_block_canvas

   #######
   # vars
   #######
   global fsk
   global actionBlock
   global BLOCK_SIZE

   global guiColumnIdToFskState
   global fskStateToGuiColumnId
   global fskEventToGuiRowId

   #######
   # tags
   #######
   global allTextTags
   global allBlockTags

   # array of tags to a single objects text
   global actionBlockTextTag
   global stateBlockTextTag

   # array of tags to rectangles
   global fskColumnBlockTag
   global fskRowBlockTag


   # at this time the lower level routines have one more state than we have
   # in the current GUI FSK, so we insert a state at the Nth-1 state
   set stateToInsertAt [expr $fsk(numStates)-1]

   set numEvents $fsk(numEvents)
   set numStates $fsk(numStates)

   #######
   # create a state block at the end of the GUI FSK
   # note: no need to insert actual text since the complete FSK will be updated
   #######
   set bg [lindex [$state_block_canvas config -bg] 4]

   set x [expr {$BLOCK_SIZE*$stateToInsertAt}]
   $state_block_canvas create rect ${x}c 0c       \
         [expr $x+$BLOCK_SIZE]c ${BLOCK_SIZE}c    \
         -outline black -fill $bg -width 2        \
         -tags fskColumnBlockTag($stateToInsertAt)


   $state_block_canvas addtag allBlockTags withtag \
                              fskColumnBlockTag($stateToInsertAt)

   set tempId                                                           \
      [$state_block_canvas create text [expr $x+0.2]c 0.5c              \
          -text " "                                                     \
          -font $fsk(Font1) -anchor w -tags stateBlockTextTag($stateToInsertAt)]

   $state_block_canvas addtag allTextTags withtag \
                              stateBlockTextTag($stateToInsertAt)

   set guiColumnIdToFskState($tempId) $stateToInsertAt
   set fskStateToGuiColumnId($stateToInsertAt) $tempId


   #######
   # create action blocks for the new state
   # note: no need to insert actual text since the complete FSK will be updated
   #######
   set bg [lindex [$action_block_canvas config -bg] 4]

   set x [expr $BLOCK_SIZE*$stateToInsertAt]

   for {set i 0} {$i < $fsk(numEvents)} {incr i} {

      set y [expr $BLOCK_SIZE*$i]

      $action_block_canvas create rect ${x}c ${y}c                    \
            [expr $x+$BLOCK_SIZE]c [expr $y+$BLOCK_SIZE]c             \
            -outline black -fill $bg -width 2                         \
            -tags tempTag($i,$stateToInsertAt)

      $action_block_canvas addtag fskColumnBlockTag($stateToInsertAt) \
                                       withtag tempTag($i,$stateToInsertAt)
      $action_block_canvas addtag fskRowBlockTag($i) \
                                       withtag tempTag($i,$stateToInsertAt)
      $action_block_canvas addtag allBlockTags       \
                                       withtag tempTag($i,$stateToInsertAt)

      set tempId                                                       \
      [$action_block_canvas create text [expr $x+0.2]c [expr $y+1]c    \
            -text " "                                                  \
            -font $fsk(Font1) -anchor w                                \
            -tags actionBlockTextTag($i,$stateToInsertAt)]

      $action_block_canvas addtag fskColumnBlockTag($stateToInsertAt) \
                        withtag actionBlockTextTag($i,$stateToInsertAt)

      $action_block_canvas addtag fskRowBlockTag($i) \
                        withtag actionBlockTextTag($i,$stateToInsertAt)

      $action_block_canvas addtag allTextTags \
                        withtag actionBlockTextTag($i,$stateToInsertAt) 

      set actionBlock($tempId,eventId) $fskEventToGuiRowId($i)
      set actionBlock($tempId,stateId) \
                      $fskStateToGuiColumnId($stateToInsertAt)
   }

   # update all states and action-blocks
   for {set iLoop 0} {$iLoop < $numStates} {incr iLoop} {

      updateStateBlock $iLoop

      for {set iLoop2 0} {$iLoop2 < $numEvents} {incr iLoop2} {
         updateActionBlock $iLoop2 $iLoop
      }
   }

   adjustScrollRegions

}



################################################################################
# NAME    : insertEventAndUpdate
#
# PURPOSE : This procedure inserts an event at the end of the GUI FSK.  It then 
#           updates all event and action-blocks to reflect the inserted event.
################################################################################
proc insertEventAndUpdate {} {

   global fsk
   global win
   global fsm_window
   global event_block_canvas
   global action_block_canvas

   #######
   # canvas
   #######
   global event_block_canvas
   global action_block_canvas

   #######
   # vars
   #######
   global fsk
   global actionBlock
   global BLOCK_SIZE

   global guiRowIdToFskEvent
   global fskStateToGuiColumnId
   global fskEventToGuiRowId

   #######
   # tags
   #######
   global allTextTags
   global allBlockTags

   # array of tags to a single objects text
   global actionBlockTextTag
   global eventBlockTextTag

   # array of tags to rectangles
   global fskColumnBlockTag
   global fskRowBlockTag


   # at this time the lower level routines have one more event than we have
   # in the current GUI FSK, so we insert an event at the Nth-1 state

   set eventToInsertAt [expr $fsk(numEvents)-1]

   set numEvents $fsk(numEvents)
   set numStates $fsk(numStates)

   #######
   # create an event block at the end of the GUI FSK
   # note: no need to insert actual text since the complete FSK will be updated
   #######
   set bg [lindex [$event_block_canvas config -bg] 4]

   set y [expr {$BLOCK_SIZE*$eventToInsertAt}]
   $event_block_canvas create rect 0c ${y}c       \
         ${BLOCK_SIZE}c [expr $y+$BLOCK_SIZE]c    \
         -outline black -fill $bg -width 2        \
         -tags fskRowBlockTag($eventToInsertAt)


   $event_block_canvas addtag allBlockTags withtag \
                              fskRowBlockTag($eventToInsertAt)

#      [$event_block_canvas create text 0.5c [expr $y+0.2]c              

   set tempId                                                           \
      [$event_block_canvas create text 0.2c [expr $y+0.5]c              \
          -text " "                                                     \
          -font $fsk(Font1) -anchor w -tags eventBlockTextTag($eventToInsertAt)]

   $event_block_canvas addtag allTextTags withtag \
                              eventBlockTextTag($eventToInsertAt)

   set guiRowIdToFskEvent($tempId) $eventToInsertAt
   set fskEventToGuiRowId($eventToInsertAt) $tempId


   #######
   # create action blocks for the new event
   # note: no need to insert actual text since the complete FSK will be updated
   #######
   set bg [lindex [$action_block_canvas config -bg] 4]

   set y [expr $BLOCK_SIZE*$eventToInsertAt]

   for {set j 0} {$j < $fsk(numStates)} {incr j} {

      set x [expr $BLOCK_SIZE*$j]

      $action_block_canvas create rect ${x}c ${y}c                    \
            [expr $x+$BLOCK_SIZE]c [expr $y+$BLOCK_SIZE]c             \
            -outline black -fill $bg -width 2                         \
            -tags tempTag($eventToInsertAt,$j)

      $action_block_canvas addtag fskRowBlockTag($eventToInsertAt)    \
                                       withtag tempTag($eventToInsertAt,$j)
      $action_block_canvas addtag fskColumnBlockTag($j) \
                                       withtag tempTag($eventToInsertAt,$j)
      $action_block_canvas addtag allBlockTags       \
                                       withtag tempTag($eventToInsertAt,$j)

#      [$action_block_canvas create text [expr $x+0.2]c [expr $y+1]c    

      set tempId                                                       \
      [$action_block_canvas create text [expr $x+0.2]c [expr $y+1]c    \
            -text " "                                                  \
            -font $fsk(Font1) -anchor w                                \
            -tags actionBlockTextTag($eventToInsertAt,$j)]

      $action_block_canvas addtag fskRowBlockTag($eventToInsertAt) \
                        withtag actionBlockTextTag($eventToInsertAt,$j)

      $action_block_canvas addtag fskColumnBlockTag($j) \
                        withtag actionBlockTextTag($eventToInsertAt,$j)

      $action_block_canvas addtag allTextTags \
                        withtag actionBlockTextTag($eventToInsertAt,$j)

      set actionBlock($tempId,stateId) $fskStateToGuiColumnId($j)
      set actionBlock($tempId,eventId) \
                      $fskEventToGuiRowId($eventToInsertAt)
   }

   # update all events and action-blocks
   for {set iLoop 0} {$iLoop < $numEvents} {incr iLoop} {

      updateEventBlock $iLoop

      for {set iLoop2 0} {$iLoop2 < $numStates} {incr iLoop2} {
         updateActionBlock $iLoop $iLoop2
      }
   }

   adjustScrollRegions

}



################################################################################
# NAME    : deleteEventAndUpdate
#
# PURPOSE : This procedure deletes the last event in the GUI FSK.  It then 
#           updates all event and action-blocks to reflect the deleted event.
################################################################################
proc deleteEventAndUpdate {} {

   global fsk
   global fsm_window
   global win
   global event_block_canvas
   global action_block_canvas

   # at this time we have one more event than the actual FSK maintained by
   # the lower level routines.  so we delete the Nth event, not the Nth+1
   # event (our last event, since we are zero based and fsk(numEvents) is 
   # one based).  this also preserves all tags 

   set eventToDelete $fsk(numEvents)

   set numEvents $fsk(numEvents)
   set numStates $fsk(numStates)

   # delete the text in the event and action-blocks
   $event_block_canvas delete eventBlockTextTag($eventToDelete)
   $event_block_canvas delete fskRowBlockTag($eventToDelete)
   $action_block_canvas delete fskRowBlockTag($eventToDelete)

   # update all events and action-blocks
   for {set iLoop 0} {$iLoop < $numEvents} {incr iLoop} {

      updateEventBlock $iLoop

      for {set iLoop2 0} {$iLoop2 < $numStates} {incr iLoop2} {
         updateActionBlock $iLoop $iLoop2
      }
   }

   adjustScrollRegions
}



################################################################################
# NAME    : updateActionBlock
#
# PURPOSE : This procedure updates the action-block variables in the 
#           GUI from the the values loaded by the lower level routines.
################################################################################
proc updateActionBlock {eventNumber stateNumber} {

   global fsk
   global fsm_window
   global win
   global action_block_canvas
   global actionBlock


   if {$actionBlock($eventNumber,$stateNumber,iterative)} then {
      set iFlag ",I"
   } else {
      set iFlag " "
   }

   $action_block_canvas itemconfigure                                          \
               actionBlockTextTag($eventNumber,$stateNumber) -text             \
    "$actionBlock($eventNumber,$stateNumber,functionName)\
\n\[$actionBlock($eventNumber,$stateNumber,nextState)\]$iFlag"

}



################################################################################
# NAME    : overwriteFskPopup
#
# PURPOSE : This procedure is invoked to display a popup window to determine
#           if the user wishes to overwrite the existing FSK.
################################################################################
proc overwriteFskPopup {} {

   global fsk
   global fsm_window
   global completeFskFileNameToSaveAs
   global tmpOWFPWin
   set tmpOWFPWin .overwriteFskPopupWindow

   toplevel $tmpOWFPWin
   wm title $tmpOWFPWin "FSK EXISTS"
   positionWindow $tmpOWFPWin 100 150
   raise $tmpOWFPWin 
   focus $tmpOWFPWin 
   grab set $tmpOWFPWin 
   global notSelectableCursor
   global processingCursor

   ######
   # create prompt
   ######
   frame $tmpOWFPWin.msg
   pack $tmpOWFPWin.msg -side top

   $fsm_window configure -cursor $processingCursor

   label $tmpOWFPWin.msg.label \
         -text "  \"$completeFskFileNameToSaveAs\" exists  " 
   pack $tmpOWFPWin.msg.label -side left

   ######
   # create buttons
   ######
   frame $tmpOWFPWin.buttons
   pack $tmpOWFPWin.buttons -side bottom

   button $tmpOWFPWin.buttons.state -text "OVERWRITE"     \
          -command {                                                      \
                      SaveFsk $completeFskFileNameToSaveAs OVERWRITE_OK;  \
                      destroy $tmpOWFPWin;                                \
                      processEventArray                                   \
                   }

   pack $tmpOWFPWin.buttons.state -side left

   button $tmpOWFPWin.buttons.cancel -text "CANCEL" \
          -command {destroy $tmpOWFPWin}

   pack $tmpOWFPWin.buttons.cancel -side left 
   $fsm_window configure -cursor $notSelectableCursor
}



################################################################################
# NAME    : readWithoutSavePopup
#
# PURPOSE : This procedure is invoked to display a popup window to determine
#           if the user wishes to save the current modified FSK before reading
#           in the new one.
################################################################################
proc readWithoutSavePopup {} {

   global fsk
   global fsm_window
   global tmpRWSPWin
   global completeFskFileNameToOpen
   global notSelectableCursor
   global processingCursor

   set tmpRWSPWin .readWithoutSavePopupWindow

   toplevel $tmpRWSPWin
   wm title $tmpRWSPWin "FSK MODIFIED"
   positionWindow $tmpRWSPWin 100 150
   raise $tmpRWSPWin 
   focus $tmpRWSPWin 
   grab set $tmpRWSPWin 

   $fsm_window configure -cursor $processingCursor

   ######
   # create prompt
   ######
   frame $tmpRWSPWin.msg
   pack $tmpRWSPWin.msg -side top

   label $tmpRWSPWin.msg.label  \
         -text "\n  \"$fsk(completeFileName)\" modified but not saved.  "
   pack $tmpRWSPWin.msg.label -side top

   label $tmpRWSPWin.msg.label2  \
         -text "  Save modified FSK?  " 
   pack $tmpRWSPWin.msg.label2 -side bottom

   ######
   # create buttons
   ######
   frame $tmpRWSPWin.buttons
   pack $tmpRWSPWin.buttons -side bottom

   #
   # TODO, the below "YES" command could cause a problem if there is a 
   #       problem writting the file.  if there is a problem and the file
   #       is not saved, we process the event array and then read the new file.
   #       hence, the file was never saved...
   #
   button $tmpRWSPWin.buttons.yes -text "YES"                                  \
          -command {SaveFsk $fsk(completeFileName) OVERWRITE_OK;               \
                    processEventArray;                                         \
                    ReadFsk $completeFskFileNameToOpen  IGNORE_MODIFIED_FLAG;  \
                    destroy $tmpRWSPWin; processEventArray}
   pack $tmpRWSPWin.buttons.yes -side left

   button $tmpRWSPWin.buttons.no -text "NO"                                    \
          -command {ReadFsk $completeFskFileNameToOpen  IGNORE_MODIFIED_FLAG;  \
                    destroy $tmpRWSPWin; processEventArray}
   pack $tmpRWSPWin.buttons.no -side left

   button $tmpRWSPWin.buttons.cancel -text "CANCEL" \
          -command {destroy $tmpRWSPWin}
   pack $tmpRWSPWin.buttons.cancel -side left 

   $fsm_window configure -cursor $notSelectableCursor
}


########################################
#END need to move below to different file and source it
########################################


################################################################################
# NAME    : processEventArray
#
# PURPOSE : This procedure processes the events in the event array
################################################################################
proc processEventArray {} {

   global fsk
   global fsm_window
   global actionBlock

   # TODO, at a later date it would be good to load the event array onto 
   # another array so later additions will not corrupt the current events on 
   # the array

   for {set iLoop 0} {$iLoop < $fsk(eventArrayLength)} {incr iLoop} {
      switch -exact $fsk(eventArray,$iLoop) {

         UPDATE_STATE_BLOCK {
            updateStateBlock $fsk(stateToUpdate)
         }

         UPDATE_EVENT_BLOCK {
            updateEventBlock $fsk(eventToUpdate)
         }

         UPDATE_ACTION_BLOCK {
            updateActionBlock $actionBlock(rowToUpdate)    \
                              $actionBlock(columnToUpdate)
         }

         DISPLAY_MESSAGE {
            displayMessage $fsk(message)
         }

         DISPLAY_SCROLL_MESSAGE {
            displayScrollMessage $fsk(scrollMessage)
         }

         # all the below commands do the same thing, insert a state and the
         # end of the fsk and update all states and action blocks...
         INSERT_STATE_BEFORE -
         INSERT_STATE_AFTER  -
         PASTE_STATE_BEFORE  -
         PASTE_STATE_AFTER {
            insertStateAndUpdate
         }

         DELETE_STATE {
            deleteStateAndUpdate
         }

         # all the below commands do the same thing, insert an event and the
         # end of the fsk and update all events and action blocks...
         INSERT_EVENT_BEFORE -
         INSERT_EVENT_AFTER  -
         PASTE_EVENT_BEFORE  -
         PASTE_EVENT_AFTER {
            insertEventAndUpdate
         }


         DELETE_EVENT {
            deleteEventAndUpdate
         }

         PASTE_ACTION_BLOCK {
            puts "not implemented yet, $fsk(eventArray,$iLoop)"
         }

         FSK_MODIFIED {
            set fsk(modified) 1
         }

         FSK_NOT_MODIFIED {
            set fsk(modified) 0
         }

         POSITION_STATE_BLOCK {
            positionState $fsk(stateToPosition)
         }

         POSITION_EVENT_BLOCK {
            positionEvent $fsk(eventToPosition)
         }

         UPDATE_FSK_NAME {
            wm title $fsm_window "$fsk(FSK_CASE_NAME) - ($fsk(fileName))"
         }

         READ_WITHOUT_SAVE_POPUP {
            readWithoutSavePopup
         }

         OVERWRITE_FSK_POPUP {
            overwriteFskPopup
         }

         OVERWRITE_STATE_TABLE_POPUP {
            puts "not implemented yet, $fsk(eventArray,$iLoop)"
         }

         OVERWRITE_STATES_POPUP {
            puts "not implemented yet, $fsk(eventArray,$iLoop)"
         }

         OVERWRITE_EVENTS_POPUP {
            puts "not implemented yet, $fsk(eventArray,$iLoop)"
         }

         OVERWRITE_SOURCE_POPUP {
            puts "not implemented yet, $fsk(eventArray,$iLoop)"
         }

         DISPLAY_TUTORIAL_WINDOW {
            puts "not implemented yet, $fsk(eventArray,$iLoop)"
         }

         UPDATE_TUTORIAL_WINDOW {
            puts "not implemented yet, $fsk(eventArray,$iLoop)"
         }

         NEW_FSK {
            newFsk
         }


         default {
            puts "INTERNAL ERROR --"
            puts "undefined event in processEventArray, $fsk(eventArray,$iLoop)"
         }

      }
   }

   set fsk(eventArrayLength) 0
}

################################################################################
# NAME    : destroy
#
# PURPOSE : Handles all destroy calls; the real destroy is renamed to 
#           realDestroy and is called within this proc.  The main purpose of
#           this proc is to reset the cursor to "xterm".  We do this because
#           the "wm protocol <window> WM_DELETE_WINDOW" did not work...
################################################################################
rename destroy realDestroy
proc destroy {theWindow} {
   global fsm_window
   $fsm_window configure -cursor top_left_arrow
   realDestroy $theWindow
}


################################################################################
# 
# setup window title, icon name, menu bar, scroll bars, etc.
#
# the MAIN program starts here..
################################################################################

### BEGIN MAIN

######
# set the fsk(binDirectory) var to the env var FSKC_BIN_DIR
######
SetFskBinDir

######
# must source global_vars before any other files...and call 
# initGlobalVars after sourcing files.....
######
source $fsk(binDirectory)/global_vars.tcl
source $fsk(binDirectory)/cmd_palette.tcl
source $fsk(binDirectory)/menu_bar.tcl
source $fsk(binDirectory)/fskc_help.tcl
initGlobalVars

InitFsk
processEventArray

set fsm_window .fsm
catch {destroy $fsm_window}
toplevel $fsm_window
wm title $fsm_window $fsk(FSK_CASE_NAME)
wm iconname $fsm_window "FSM CASE"

# hide the parent window that pops up...
wm withdraw .

set top_canvas $fsm_window.top_canvas

set status_canvas $fsm_window.s_canvas
set state_block_canvas $fsm_window.sb_canvas
set event_block_canvas $fsm_window.eb_canvas
set action_block_canvas $fsm_window.ab_canvas


positionWindow $fsm_window 150 50
createMenuBar $fsm_window


scrollbar $fsm_window.hscroll -orient horiz \
             -command "fskHorizontalScroll" 
pack $fsm_window.hscroll -side bottom -fill x


scrollbar $fsm_window.vscroll \
            -command "fskVerticalScroll" 
pack $fsm_window.vscroll -side right -fill y



#######
# create a top canvas to insert the status and state canvas into
#######
canvas $top_canvas -relief sunken -borderwidth 2       \
          -height ${BLOCK_SIZE}c                       \
          -width [expr $BLOCK_SIZE*$fsk(numStates)]c

pack $top_canvas -side top -expand no -fill x



#######
# create status canvas
#######
canvas $status_canvas -relief sunken -borderwidth 2    \
          -height ${BLOCK_SIZE}c                       \
          -width ${BLOCK_SIZE}c                      

 
pack $status_canvas -in $top_canvas -anchor n -side left \
                       -expand no -fill none

# create a rectangle in the status canvas
$status_canvas create rect 0c 0c                  \
             [expr $BLOCK_SIZE]c [expr $BLOCK_SIZE]c \
             -outline black -fill gray 

#######
# create state canvas
#######
set stateScrollRegion   [list 0c                             \
                         0c                                  \
                         [expr $fsk(numStates)*$BLOCK_SIZE]c \
                         ${BLOCK_SIZE}c ] 

######
#   BUG:
#
#   the below does not work for the -width command
#         set DISPLAY_NUM_STATES       [expr 4*$BLOCK_SIZE]
#         -width ${DISPLAY_NUM_STATES}c 
#
#   the below statement works, however
#         -width [expr 6*$BLOCK_SIZE]c 
######

canvas $state_block_canvas -relief sunken -borderwidth 2     \
          -scrollregion $stateScrollRegion                   \
          -xscrollincrement ${BLOCK_SIZE}c                   \
          -width [expr $DISPLAY_NUM_STATES*$BLOCK_SIZE]c     \
          -height ${BLOCK_SIZE}c                             \
          -xscrollcommand "$fsm_window.hscroll set"       

pack $state_block_canvas -in $top_canvas -side left -expand yes -fill x

#######
# create event canvas
#######
set eventScrollRegion   [list 0c                            \
                         0c                                \
                         ${BLOCK_SIZE}c                    \
                         [expr $fsk(numEvents)*$BLOCK_SIZE]c ]  

canvas $event_block_canvas -relief sunken -borderwidth 2      \
          -scrollregion $eventScrollRegion                    \
          -yscrollincrement ${BLOCK_SIZE}c                    \
          -width ${BLOCK_SIZE}c                               \
          -height [expr $DISPLAY_NUM_EVENTS*$BLOCK_SIZE]c     \
          -yscrollcommand "$fsm_window.vscroll set"       

pack $event_block_canvas -side left -expand no -fill y

#######
# create action-block canvas
#######

set actionBlockScrollRegion   [list 0c                             \
                               0c                                  \
                               [expr $fsk(numStates)*$BLOCK_SIZE]c \
                               [expr $fsk(numEvents)*$BLOCK_SIZE]c] 


######
# BUG:  the addition of xscrollincrement and yscrollincrement below cause the 
#       scrollable region to be larger than the number of action blocks.
#       if these options are removed, the scrollable region matches the 
#       number of action blocks....the space is at the left and bottom of
#       the action blocks...it is probably ok....                        
######

canvas $action_block_canvas -relief sunken -borderwidth 2       \
          -scrollregion $actionBlockScrollRegion                \
          -width  [expr $DISPLAY_NUM_STATES*$BLOCK_SIZE]c       \
          -height [expr $DISPLAY_NUM_EVENTS*$BLOCK_SIZE]c       \
          -xscrollincrement ${BLOCK_SIZE}c                      \
          -yscrollincrement ${BLOCK_SIZE}c                      \
          -xscrollcommand "$fsm_window.hscroll set"             \
          -yscrollcommand "$fsm_window.vscroll set"

pack $action_block_canvas -expand yes -fill both

#######
# bind mouse buttons to procedures for state blocks
#######
$state_block_canvas bind all <Any-Enter> "blockEnter $state_block_canvas"
$state_block_canvas bind all <Any-Leave> "blockLeave $state_block_canvas"
$state_block_canvas bind all <1> "stateBlockButton"
$state_block_canvas bind all <3>                                   \
                         { set blockTypeToEdit STATE;              \
                           unhideMouseMenu $state_block_canvas     \
                                           $STATE_MENU %X %Y       \
                         }

#######
# bind mouse buttons to procedures for event blocks
#######
$event_block_canvas bind all <Any-Enter> "blockEnter $event_block_canvas"
$event_block_canvas bind all <Any-Leave> "blockLeave $event_block_canvas"
$event_block_canvas bind all <1> "eventBlockButton"
$event_block_canvas bind all <3>                                   \
                         { set blockTypeToEdit EVENT;              \
                           unhideMouseMenu $event_block_canvas     \
                                           $EVENT_MENU %X %Y       \
                         }

#######
# bind mouse buttons to procedures for action blocks
#######
$action_block_canvas bind all <Any-Enter> "blockEnter $action_block_canvas"
$action_block_canvas bind all <Any-Leave> "blockLeave $action_block_canvas"
$action_block_canvas bind all <1> "actionBlockButton"
$action_block_canvas bind all <3>                                     \
                         { unhideMouseMenu $action_block_canvas       \
                                           $ACTION_BLOCK_MENU %X %Y   \
                         }

#######
# create state and event mouse menus 
#######
createEventStateMouseMenu $STATE_MENU
hideMouseMenu $STATE_MENU

createEventStateMouseMenu $EVENT_MENU
hideMouseMenu $EVENT_MENU

createActionBlockMouseMenu $ACTION_BLOCK_MENU
hideMouseMenu $ACTION_BLOCK_MENU

newFsk

### END MAIN
